/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseMenuItemModifierSpec;
import com.floreantpos.model.dao.ModifierGroupDAO;
import com.floreantpos.ui.model.ModifierPricingRule;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@XmlRootElement(name = "menuItemModifierSpec")
@XmlAccessorType(XmlAccessType.FIELD)
@JsonIgnoreProperties(ignoreUnknown = true, value = { "defaultModifiers", "modifiers", "modifierGroup" })
public class MenuItemModifierSpec extends BaseMenuItemModifierSpec implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;
	public static final String JSON_PROP_PRICE_RULES = "priceRules";
	public static final String JSON_PROP_ENABLE_GROUP_PRICE = "enableGroupPricing";
	private transient JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public MenuItemModifierSpec() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuItemModifierSpec(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	private ModifierGroup modifierGroup;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private Map<Integer, Double> priceMap;
	private ArrayList<ModifierPricingRule> pricingRules;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String toString() {
		return getName();
	}

	public String getUniqueId() {
		return ("menuitem_modifiergroup_" + toString() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	@XmlTransient
	public List<MenuItemModifierPageItem> getMenuItemModifierPageItemsAsList() {
		return this.getMenuItemModifierPageItemsAsSet();
	}

	@XmlTransient
	public List<MenuItemModifierPageItem> getMenuItemModifierPageItemsAsSet() {
		Set<MenuItemModifierPage> modifierPages = this.getModifierPages();
		List<MenuItemModifierPageItem> menuItemModifierPageItems = new ArrayList<>();
		for (MenuItemModifierPage page : modifierPages) {
			List<MenuItemModifierPageItem> items = page.getPageItems();
			if (items != null) {
				menuItemModifierPageItems.addAll(items);
			}
//			if (items != null) {
//				Collections.sort(items, (item1, item2) -> item1.getCol().compareTo(item2.getCol()));
//				Collections.sort(items, (item1, item2) -> item1.getRow().compareTo(item2.getRow()));
//				for (MenuItemModifierPageItem pageItem : items) {
//					menuItemModifierPageItems.add(pageItem);
//				}
//			}
		}
		return menuItemModifierPageItems;
	}

	@XmlTransient
	public Set<MenuModifier> getModifiers() {
		Set<MenuItemModifierPage> modifierPages = getModifierPages();
		Set<MenuModifier> menuModifiers = new HashSet<>();
		if (modifierPages != null) {
			for (MenuItemModifierPage page : modifierPages) {
				if (page.getPageItems() != null) {
					for (MenuItemModifierPageItem pageItem : page.getPageItems()) {
						MenuModifier modifier = pageItem.getMenuModifier();
						if (modifier != null) {
							modifier.setPageItemId(pageItem.getId());
							menuModifiers.add(modifier);
						}
					}
				}
			}
		}
		return menuModifiers;
	}

	public boolean hasMenuItemModifierPageItem() {
		Set<MenuItemModifierPage> modifierPages = getModifierPages();
		if (modifierPages == null || modifierPages.isEmpty()) {
			return false;
		}
		for (MenuItemModifierPage menuItemModifierPage : modifierPages) {
			List<MenuItemModifierPageItem> pageItems = menuItemModifierPage.getPageItems();
			if (pageItems != null && !pageItems.isEmpty()) {
				return true;
			}
		}
		return false;
	}

	public String getDisplayName() {
		return super.getName();
	}

	@XmlTransient
	@JsonIgnoreProperties
	public Set<MenuModifier> getDefaultModifiers() {
		Set<MenuModifier> defaultModifiers = new HashSet<>();
		List<DefaultMenuModifier> defaultModifierList = super.getDefaultModifierList();
		if (defaultModifierList != null) {
			for (DefaultMenuModifier defaultMenuModifier : defaultModifierList) {
				defaultModifiers.add(defaultMenuModifier.getModifier());
			}
		}
		return defaultModifiers;
	}

	public ModifierGroup getModifierGroup() {
		if (modifierGroup != null)
			return modifierGroup;
		String modifierGroupId = super.getModifierGroupId();
		if (StringUtils.isEmpty(modifierGroupId))
			return null;
		return modifierGroup = ModifierGroupDAO.getInstance().get(modifierGroupId);
	}

	public void setModifierGroup(ModifierGroup modifierGroup) {
		this.modifierGroup = modifierGroup;
		String modifierGroupId = null;
		if (modifierGroup != null) {
			modifierGroupId = modifierGroup.getId();
		}
		setModifierGroupId(modifierGroupId);
	}

	public MenuItemModifierSpec deepClone() throws Exception {
		MenuItemModifierSpec newSpec = new MenuItemModifierSpec();
		BeanUtils.copyProperties(newSpec, this);
		newSpec.setId(null);
		newSpec.setInstruction(this.getInstruction());
		newSpec.setModifierPages(null);
		newSpec.setDefaultModifierList(null);
		//		newSpec.setAutoShow(this.isAutoShow());
		//		newSpec.setEnable(this.isEnable());
		//		newSpec.setExclusive(this.isExclusive());
		//		newSpec.setJumpGroup(this.isJumpGroup());

		Set<MenuItemModifierPage> modifierPages2 = getModifierPages();
		if (modifierPages2 != null) {
			for (MenuItemModifierPage menuItemModifierPage : modifierPages2) {
				newSpec.addTomodifierPages(menuItemModifierPage.deepClone());
			}
		}
		List<DefaultMenuModifier> defaultModifierList2 = getDefaultModifierList();
		if (defaultModifierList2 != null) {
			List<DefaultMenuModifier> list = new ArrayList<>(defaultModifierList2.size());
			for (DefaultMenuModifier defaultMenuModifier : defaultModifierList2) {
				DefaultMenuModifier d2 = new DefaultMenuModifier();
				BeanUtils.copyProperties(d2, defaultMenuModifier);
				list.add(d2);
			}
			newSpec.setDefaultModifierList(list);
		}
		return newSpec;
	}

	public Boolean getAutoShow() {
		return super.isAutoShow();
	}

	public Boolean getEnable() {
		return super.isEnable();
	}

	public Boolean getExclusive() {
		return super.isExclusive();
	}

	public Boolean getJumpGroup() {
		return super.isJumpGroup();
	}

	public Boolean getRequired() {
		return super.isRequired();
	}

	public void setEnableGroupPricing(boolean isEnableGroupWisePricing) {
		addProperty(MenuItemModifierSpec.JSON_PROP_ENABLE_GROUP_PRICE, String.valueOf(isEnableGroupWisePricing));
	}

	public boolean isEnableGroupPricing() {
		return hasProperty(JSON_PROP_ENABLE_GROUP_PRICE) && getBooleanProperty(JSON_PROP_ENABLE_GROUP_PRICE, false);
	}

	public boolean hasPriceRules() {
		return isEnableGroupPricing() && hasProperty(JSON_PROP_PRICE_RULES);
	}

	public Double getPrice(Integer quantity) {
		if (!hasPriceRules()) {
			return null;
		}
		if (priceMap != null) {
			return priceMap.get(quantity);
		}
		buildPriceRuleMap();
		return priceMap.get(quantity);
	}

	private void buildPriceRuleMap() {
		priceMap = new HashMap<Integer, Double>();
		List<ModifierPricingRule> priceList = getPriceRules();
		if (isUseModifierGroupSettings()) {
			priceList = getModifierGroup().getPriceRules();
		}
		if (priceList != null && priceList.size() > 0) {
			for (ModifierPricingRule pricing : priceList) {
				for (int i = pricing.getFromQty(); i <= pricing.getToQty(); i++) {
					priceMap.put(i, pricing.getPrice());
				}
			}
		}
	}

	public List getPriceRules() {
		try {
			if (pricingRules == null) {
				pricingRules = new ArrayList<ModifierPricingRule>();
			}

			String properties = getProperties();
			if (StringUtils.isEmpty(properties)) {
				return pricingRules;
			}
			ObjectMapper mapper = new ObjectMapper();
			JsonNode readTree = mapper.readTree(properties);
			if (readTree != null) {
				JsonNode arrayNode = readTree.findValue(JSON_PROP_PRICE_RULES);
				if (arrayNode != null) {
					pricingRules = mapper.readValue(arrayNode.toString(), new TypeReference<List<ModifierPricingRule>>() {
					});
					return pricingRules;
				}
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return pricingRules;
	}
	
	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}
		
		String jsonProprties = super.getProperties();
		if (StringUtils.isEmpty(jsonProprties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProprties, com.google.gson.JsonObject.class);
		}
		return propertiesContainer;
	}
	
	@JsonIgnore
	public boolean isUseModifierGroupSettings() {
		return getBooleanProperty("enableModifierGroupSettings", true); //$NON-NLS-1$
	}

	public void setUseModifierGroupSettings(boolean useModifierGroupSettings) {
		addProperty("enableModifierGroupSettings", String.valueOf(useModifierGroupSettings)); //$NON-NLS-1$
	}

	public void copyModifierGroupProperties() {
		ModifierGroup modifierGroup = getModifierGroup();

		setMinQuantity(modifierGroup.getMinQuantity());
		setMaxQuantity(modifierGroup.getMaxQuantity());
		setAutoShow(modifierGroup.isAutoShow());
		setEnable(modifierGroup.isEnable());
		setJumpGroup(modifierGroup.isJumpGroup());
	}
}