package com.floreantpos.model.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.apache.commons.lang3.tuple.Pair;

import com.floreantpos.PosException;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.DayOfWeek;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.DoctorTimeSchedule;
import com.floreantpos.model.dao.AppointmentDAO;
import com.floreantpos.model.dao.StoreDAO;

public class AppointmentUtil {

	public static final SimpleDateFormat simpleDateFormat = new SimpleDateFormat("hh:mm a"); //$NON-NLS-1$

	public static Map<String, Appointment> getAppointmentTimelineMap(Date fromDate, Date toDate, Doctor doctor) {
		List<Appointment> appointments = AppointmentDAO.getInstance().getAppointments(fromDate, toDate, doctor, false, false);
		Map<String, Appointment> timeMap = new HashMap<String, Appointment>();
		if (appointments != null && appointments.size() > 0) {
			for (Appointment appointment : appointments) {
				String timeValue = simpleDateFormat.format(appointment.getFromDate());
				timeMap.put(appointment.getDoctorId() + "_" + timeValue, appointment); //$NON-NLS-1$
			}
		}
		return timeMap;
	}

	public static Date getNextTimeSlot(Date selectedDate, DoctorTimeSchedule doctorTimeSchedule, Doctor doctor) {
		if (selectedDate == null || doctorTimeSchedule == null || doctor == null) {
			return null;
		}

		Calendar c = Calendar.getInstance();
		c.setTime(selectedDate);
		if (!doctorTimeSchedule.getDays().contains(c.get(Calendar.DAY_OF_WEEK))) {
			return null;
		}

		Map<String, Appointment> timeMap = getAppointmentTimelineMap(selectedDate, selectedDate, doctor);

		Calendar start = Calendar.getInstance();

		try {
			start.setTime(simpleDateFormat.parse(doctorTimeSchedule.getStartTime()));
		} catch (ParseException e) {
		}
		start.set(Calendar.SECOND, 0);
		start.set(Calendar.MILLISECOND, 0);
		start.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		start.set(Calendar.MONTH, c.get(Calendar.MONTH));
		start.set(Calendar.YEAR, c.get(Calendar.YEAR));

		Calendar end = Calendar.getInstance();
		try {
			end.setTime(simpleDateFormat.parse(doctorTimeSchedule.getEndTime()));
		} catch (ParseException e) {
		}
		end.set(Calendar.SECOND, 0);
		end.set(Calendar.MILLISECOND, 0);
		end.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		end.set(Calendar.MONTH, c.get(Calendar.MONTH));
		end.set(Calendar.YEAR, c.get(Calendar.YEAR));

		//long totalMinutes = TimeUnit.MILLISECONDS.toMinutes(end.getTimeInMillis() - start.getTimeInMillis());
		//Integer consultationTime = 0;
		//if (doctorTimeSchedule.getPatientCount() > 0) {
		//	consultationTime = (int) (totalMinutes / doctorTimeSchedule.getPatientCount());
		//}
		Integer consultationTime = doctorTimeSchedule.getConsultationTime();
		if (consultationTime <= 0) {
			consultationTime = 15;
		}
		if (consultationTime <= 0) {
			consultationTime = 15;
		}

		do {
			String key = doctor.getId() + "_" + simpleDateFormat.format(start.getTime()); //$NON-NLS-1$
			Appointment appointment = timeMap.get(key);
			if (appointment != null || start.getTime().before(DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp()))) {
				start.add(Calendar.MINUTE, consultationTime);
				continue;
			}

			return start.getTime();

		} while (start.getTime().before(end.getTime()));

		Appointment appointment = timeMap.values().stream().sorted(Comparator.comparing(Appointment::getFromDate).reversed()).findFirst().orElse(null);

		if (appointment == null || !appointment.getShiftId().equals(doctorTimeSchedule.getId())) {
			return null;
		}

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(appointment.getFromDate());
		calendar.add(Calendar.MINUTE, consultationTime);

		return calendar.getTime();
	}

	public static Date convertStringTimeToDate(String value) {
		return convertStringTimeToDate(null, value);
	}

	public static Date convertStringTimeToDate(Date startDate, String value) {
		try {
			LocalDate fromDate = LocalDate.now();
			Calendar calendar1 = Calendar.getInstance();
			if (value != null) {
				Date date = simpleDateFormat.parse(value);
				calendar1.setTime(date);
			}
			calendar1.set(Calendar.DAY_OF_MONTH, fromDate.getDayOfMonth());
			calendar1.set(Calendar.MONTH, fromDate.getMonthValue() - 1);
			calendar1.set(Calendar.YEAR, fromDate.getYear());
			if (startDate != null) {
				Calendar startC = Calendar.getInstance();
				startC.setTime(startDate);
				int amPM = startC.get(Calendar.AM_PM);
				if (amPM == Calendar.PM) {
					int cAmPM = calendar1.get(Calendar.AM_PM);
					if (cAmPM == Calendar.AM) {
						calendar1.add(Calendar.DAY_OF_MONTH, 1);
					}
				}
			}
			return calendar1.getTime();
		} catch (ParseException e) {
		}
		return null;
	}

	public static Pair<List<DoctorTimeSchedule>, Set<Integer>> updateSchedules(Appointment appointment, Doctor doctor, Date selectedDate) {
		List<DoctorTimeSchedule> avaiableSchedules = new ArrayList<>();
		Set<Integer> availableDays = new HashSet<Integer>();
		if (doctor != null) {
			List<DoctorTimeSchedule> appointmentTimes = doctor.getSchedules(appointment.getOutletId());
			if (appointmentTimes != null && !appointmentTimes.isEmpty()) {
				if (selectedDate != null) {
					for (DoctorTimeSchedule doctorTimeSchedule : appointmentTimes) {
						if (hasSchedule(appointment, doctorTimeSchedule, selectedDate, doctor)) {
							avaiableSchedules.add(doctorTimeSchedule);
						}
						availableDays.addAll(doctorTimeSchedule.getDays());
					}
				}
			}
		}

		if (avaiableSchedules.size() > 1) {
			avaiableSchedules.sort((o1, o2) -> {
				try {
					return simpleDateFormat.parse(o1.getStartTime()).compareTo(simpleDateFormat.parse(o2.getStartTime()));
				} catch (ParseException e) {
				}
				return 0;
			});
		}
		return Pair.of(avaiableSchedules, availableDays);
	}

	public static void doCheckshedulesValidation(Doctor doctor, Date selectedDate, List<DoctorTimeSchedule> avaiableSchedules, Set<Integer> availableDays) {
		if (doctor != null && avaiableSchedules.isEmpty() && selectedDate != null) {
			String nameOfDays = availableDays.stream().sorted().map(day -> DayOfWeek.getDayOfWeek(day).name().substring(0, 3))
					.collect(Collectors.joining(", ")); //$NON-NLS-1$
			throw new PosException("Available days: " + nameOfDays);
		}
	}

	public static boolean hasSchedule(Appointment appointment, DoctorTimeSchedule schedule, Date selectedDate, Doctor doctor) {
		if (schedule != null && !schedule.getActive()) {
			if (StringUtils.isBlank(appointment.getId()) || appointment.getShiftId() == null || !appointment.getShiftId().equals(schedule.getId())) {
				return false;
			}
		}
		if (doctor == null) {
			return false;
		}
		List<DoctorTimeSchedule> appointmentTimes = doctor.getSchedules();
		if (appointmentTimes == null || appointmentTimes.isEmpty()) {
			return false;
		}
		if (selectedDate == null) {
			return false;
		}
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(selectedDate);
		int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
		List<Integer> appointmentDays = schedule.getDays();
		return appointmentDays.contains(dayOfWeek);
	}

	public static String scheduleSelected(Appointment appointment, DoctorTimeSchedule schedule, Date selectedFromDate, Doctor doctor) {
		if (schedule == null) {
			return null;
		}

		Calendar c = Calendar.getInstance();
		c.setTime(selectedFromDate);

		Calendar start = Calendar.getInstance();
		try {
			start.setTime(simpleDateFormat.parse(schedule.getStartTime()));
		} catch (ParseException e) {
		}
		start.set(Calendar.SECOND, 0);
		start.set(Calendar.MILLISECOND, 0);
		start.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		start.set(Calendar.MONTH, c.get(Calendar.MONTH));
		start.set(Calendar.YEAR, c.get(Calendar.YEAR));

		Calendar end = Calendar.getInstance();
		try {
			end.setTime(simpleDateFormat.parse(schedule.getEndTime()));
		} catch (ParseException e) {
		}
		end.set(Calendar.SECOND, 59);
		end.set(Calendar.MILLISECOND, 999);
		end.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		end.set(Calendar.MONTH, c.get(Calendar.MONTH));
		end.set(Calendar.YEAR, c.get(Calendar.YEAR));

		Integer consultationTime = schedule.getConsultationTime();
		if (consultationTime <= 0) {
			consultationTime = 15;
		}

		boolean regenarateAppointmentId = !(StringUtils.isNotBlank(appointment.getAppoinmentId())
				&& (appointment.getFromDate() != null && selectedFromDate != null && DateUtils.isSameDay(appointment.getFromDate(), selectedFromDate))
				&& (schedule != null && schedule.getId().equals(appointment.getShiftId())));
		if (regenarateAppointmentId) {
			Date nextTimeSlot = getNextTimeSlot(selectedFromDate, schedule, doctor);
			if (nextTimeSlot != null) {
				return simpleDateFormat.format(nextTimeSlot);
			}
		}
		else if (appointment.getFromDate() != null) {
			return simpleDateFormat.format(appointment.getFromDate());
		}
		return null;
	}

	public static int getAppointmentIdBySchedule(Appointment appointment, Map<String, Appointment> timeMap, String selectedTimeSlot, Date selectedDate,
			DoctorTimeSchedule doctorTimeSchedule) {
		Calendar c = Calendar.getInstance();
		c.setTime(selectedDate);
		if (!doctorTimeSchedule.getDays().contains(c.get(Calendar.DAY_OF_WEEK))) {
			throw new PosException("Doctor has no appointment on selected day.");
		}
		Calendar start = Calendar.getInstance();
		try {
			start.setTime(simpleDateFormat.parse(doctorTimeSchedule.getStartTime()));
		} catch (ParseException e) {
		}
		start.set(Calendar.SECOND, 0);
		start.set(Calendar.MILLISECOND, 0);
		start.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		start.set(Calendar.MONTH, c.get(Calendar.MONTH));
		start.set(Calendar.YEAR, c.get(Calendar.YEAR));

		Calendar end = Calendar.getInstance();
		try {
			end.setTime(simpleDateFormat.parse(doctorTimeSchedule.getEndTime()));
		} catch (ParseException e) {
		}
		end.set(Calendar.SECOND, 0);
		end.set(Calendar.MILLISECOND, 0);
		end.set(Calendar.DAY_OF_MONTH, c.get(Calendar.DAY_OF_MONTH));
		end.set(Calendar.MONTH, c.get(Calendar.MONTH));
		end.set(Calendar.YEAR, c.get(Calendar.YEAR));

		boolean dayChanged = false;
		int amPM = start.get(Calendar.AM_PM);
		if (amPM == Calendar.PM) {
			int cAmPM = end.get(Calendar.AM_PM);
			if (cAmPM == Calendar.AM) {
				dayChanged = true;
			}
		}
		if (dayChanged) {
			end.add(Calendar.DAY_OF_MONTH, 1);
		}
		int appointmentId = getdAppointmentId(timeMap, selectedTimeSlot, doctorTimeSchedule, start, end, null, appointment);
		return appointmentId;
	}

	private static int getdAppointmentId(Map<String, Appointment> timeMap, String selectedTimeSlot, DoctorTimeSchedule doctorTimeSchedule, Calendar start,
			Calendar end, Integer previousCount, Appointment originalAppointment) {
		Integer consultationTime = doctorTimeSchedule.getConsultationTime();
		if (consultationTime < 15) {
			consultationTime = 15;
		}
		int count = previousCount != null ? previousCount : 1;
		do {
			String key = simpleDateFormat.format(start.getTime()); //$NON-NLS-1$
			Appointment appointment = timeMap.get(key);
			if (key.equalsIgnoreCase(selectedTimeSlot)) {
				if (appointment != null) {
					if (appointment.equals(originalAppointment)) {
						return count;
					}
					if (originalAppointment.getFromDate() != null) {
						Calendar appFromDate = Calendar.getInstance();
						appFromDate.setTime(originalAppointment.getFromDate());

						if (originalAppointment.getShiftId().equals(appointment.getShiftId())
								&& appFromDate.get(Calendar.DAY_OF_MONTH) == start.get(Calendar.DAY_OF_MONTH)) {
							return count;
						}
					}
				}
				return count;
			}
			count++;
			start.add(Calendar.MINUTE, consultationTime);
		} while (start.getTime().before(end.getTime()));
		return count;
	}

	public static Map<String, Appointment> getAppointmentTimelineMap(Date fromDate, Date toDate, Doctor doctor, DoctorTimeSchedule doctorTimeSchedule) {
		List<Appointment> appointments = AppointmentDAO.getInstance().getAppointments(fromDate, toDate, doctor, false, false, doctorTimeSchedule.getId());
		Map<String, Appointment> timeMap = new HashMap<String, Appointment>();
		if (appointments != null && appointments.size() > 0) {
			for (Appointment appointment : appointments) {
				String timeValue = simpleDateFormat.format(appointment.getFromDate());
				timeMap.put(timeValue, appointment); //$NON-NLS-1$
			}
		}
		return timeMap;
	}

	public static Date getSelectedFromDate(Date selectedDate, String selectedTime) {
		try {
			LocalDate fromDate = DateUtil.convertDateToLocalDate(selectedDate);
			if (fromDate == null) {
				fromDate = LocalDate.now();
			}

			Calendar calendar1 = Calendar.getInstance();
			if (selectedTime != null) {
				Date date = simpleDateFormat.parse(selectedTime);
				calendar1.setTime(date);
			}
			calendar1.set(Calendar.DAY_OF_MONTH, fromDate.getDayOfMonth());
			calendar1.set(Calendar.MONTH, fromDate.getMonthValue() - 1);
			calendar1.set(Calendar.YEAR, fromDate.getYear());
			return calendar1.getTime();
		} catch (ParseException e) {
		}
		return null;
	}
}
