/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.AdvanceAdjustmentTransaction;
import com.floreantpos.model.AdvanceTransaction;
import com.floreantpos.model.Challan;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.dao.ChallanDAO;
import com.floreantpos.model.dao.PosTransactionDAO;

public class UpdateDBTo469 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo469(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$
		updateTotalAmountField();
		PosLog.info(getClass(), "Challan update complete."); //$NON-NLS-1$ //$NON-NLS-2$
		updateAdvanceTransactionsTicketId();
	}

	public void updateTotalAmountField() {

		int totalChallan = rowCount();
		PosLog.info(UpdateDBTo469.class, "Found total challans: " + totalChallan); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < totalChallan) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<Challan> pagination = new Pagination<Challan>(count, increment);
				pagination.setNumRows(totalChallan);
				findChallans(pagination, session);

				List<Challan> challans = pagination.getDataList();
				for (Challan challan : challans) {
					double totalAmount = challan.getDoubleProperty("total.amount");
					challan.setTotalAmount(totalAmount);
					ChallanDAO.getInstance().saveOrUpdate(challan, session);
				}
				transaction.commit();

				count += increment;
			}
		}

	}

	private void findChallans(Pagination<Challan> pagination, Session session) {
		Criteria criteria = createChallanCriteria(session);
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int rowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createChallanCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createChallanCriteria(Session session) {
		Criteria criteria = session.createCriteria(Challan.class);
		return criteria;
	}

	public void updateAdvanceTransactionsTicketId() {

		int total = rowTransactionCount();
		PosLog.info(UpdateDBTo469.class, "Found total Transaction: " + total); //$NON-NLS-1$

		int increment = 100;
		int count = 0;
		while (count < total) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<PosTransaction> pagination = new Pagination<PosTransaction>(count, increment);
				pagination.setNumRows(total);
				findTransactions(pagination, session);

				List<PosTransaction> transactions = pagination.getDataList();
				for (PosTransaction advanceTransaction : transactions) {
					String ticketId = advanceTransaction.getProperty("ticketId", StringUtils.EMPTY);
					if (StringUtils.isBlank(ticketId)) {
						continue;
					}

					advanceTransaction.setEntityId(ticketId);
					advanceTransaction.setEntityType(Ticket.REF);
					PosTransactionDAO.getInstance().saveOrUpdate(advanceTransaction, session);
				}

				transaction.commit();

				count += increment;
			}
		}

	}

	private void findTransactions(Pagination<PosTransaction> pagination, Session session) {
		Criteria criteria = createTransactionCriteria(session);
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(), String.format("Searching transaction from: %s to %s", pagination.getCurrentRowIndex(), //$NON-NLS-1$
				(pagination.getCurrentRowIndex() + pagination.getPageSize())));

		pagination.setRows(criteria.list());

	}

	public int rowTransactionCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = createTransactionCriteria(session);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}

	private Criteria createTransactionCriteria(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		Disjunction disjunction = Restrictions.disjunction();
		disjunction.add(Property.forName("class").eq(AdvanceTransaction.class));
		disjunction.add(Property.forName("class").eq(AdvanceAdjustmentTransaction.class));
		criteria.add(disjunction);
		return criteria;
	}
}