package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.ChemoCycle;
import com.floreantpos.model.Chemotherapy;
import com.orocube.medlogics.chemo.ChemoCycleStatus;

public class ChemoCycleDAO extends BaseChemoCycleDAO {

	public ChemoCycleDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		ChemoCycle bean = (ChemoCycle) obj;
		if (bean == null) {
			throw new PosException("ChemoCycle not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<ChemoCycle> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(ChemoCycle.PROP_CYCLE_NUMBER));
			return criteria.list();
		}
	}

	/**
	 * Find all cycles for a specific chemotherapy
	 * @param chemotherapy the chemotherapy
	 * @return list of chemo cycles
	 */
	public List<ChemoCycle> findByChemotherapy(Chemotherapy chemotherapy) {
		if (chemotherapy == null || chemotherapy.getId() == null) {
			throw new IllegalArgumentException("Chemotherapy must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ChemoCycle.PROP_CHEMOTHERAPY, chemotherapy));
			criteria.addOrder(Order.asc(ChemoCycle.PROP_CYCLE_NUMBER));
			return criteria.list();
		}
	}

	/**
	 * Find all cycles for a specific chemotherapy by ID
	 * @param chemotherapyId the chemotherapy ID
	 * @return list of chemo cycles
	 */
	public List<ChemoCycle> findByChemotherapyId(String chemotherapyId) {
		if (chemotherapyId == null) {
			throw new IllegalArgumentException("Chemotherapy ID must not be null");
		}
		Chemotherapy chemotherapy = ChemotherapyDAO.getInstance().get(chemotherapyId);
		if (chemotherapy == null) {
			throw new PosException("Chemotherapy not found with ID: " + chemotherapyId);
		}
		return findByChemotherapy(chemotherapy);
	}

	/**
	 * Find cycles by status
	 * @param status the cycle status
	 * @return list of chemo cycles
	 */
	public List<ChemoCycle> findByStatus(ChemoCycleStatus status) {
		if (status == null) {
			throw new IllegalArgumentException("Status must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ChemoCycle.PROP_STATUS, status.name()));
			criteria.addOrder(Order.asc(ChemoCycle.PROP_CYCLE_NUMBER));
			return criteria.list();
		}
	}

	/**
	 * Find a specific cycle by chemotherapy and cycle number
	 * @param chemotherapy the chemotherapy
	 * @param cycleNumber the cycle number
	 * @return the chemo cycle or null
	 */
	public ChemoCycle findByChemotherapyAndCycleNumber(Chemotherapy chemotherapy, Integer cycleNumber) {
		if (chemotherapy == null || chemotherapy.getId() == null) {
			throw new IllegalArgumentException("Chemotherapy must not be null and must have an ID");
		}
		if (cycleNumber == null) {
			throw new IllegalArgumentException("Cycle number must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ChemoCycle.PROP_CHEMOTHERAPY, chemotherapy));
			criteria.add(Restrictions.eq(ChemoCycle.PROP_CYCLE_NUMBER, cycleNumber));
			return (ChemoCycle) criteria.uniqueResult();
		}
	}
}
