/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.LogicalExpression;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.AttendenceHistory;
import com.floreantpos.model.DeclaredTips;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Shift;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.AttendanceReportData;
import com.floreantpos.report.PayrollReportData;
import com.floreantpos.report.WeeklyPayrollReportData;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class AttendenceHistoryDAO extends BaseAttendenceHistoryDAO {

	private final static SimpleDateFormat keyDateFormat = new SimpleDateFormat("dd_MM_yyyy"); //$NON-NLS-1$
	private Map<String, User> userMap = new HashMap<String, User>();

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public AttendenceHistoryDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public Set<User> findNumberOfClockedInUserAtHour(Date fromDay, Date toDay, int hour, Terminal terminal, Outlet outlet) {
		Set<User> users = new HashSet<User>();
		Session session = null;

		List<Date> betweenDates = POSUtil.getDaysBetweenDates(fromDay, toDay);
		try {

			if (betweenDates != null) {

				for (Date date : betweenDates) {

					Calendar c = Calendar.getInstance();
					c.setTime(date);
					c.set(Calendar.HOUR_OF_DAY, hour);
					c.set(Calendar.MINUTE, 0);
					c.set(Calendar.SECOND, 0);
					Date time1 = c.getTime();

					c.add(Calendar.HOUR_OF_DAY, 1);
					Date time2 = c.getTime();

					session = getSession();
					Criteria criteria = session.createCriteria(getReferenceClass());
					buildAttendenceCriteria(time1, time2, criteria);

					if (terminal != null) {
						criteria.add(Restrictions.eq(AttendenceHistory.PROP_TERMINAL_ID, terminal.getId()));
					}

					if (outlet != null) {
						criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, outlet.getId()));

					}
					List list = criteria.list();
					for (Object object : list) {
						AttendenceHistory history = (AttendenceHistory) object;
						users.add(history.getUser());
					}
				}
			}
			return users;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.2"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	private void buildAttendenceCriteria(Date startTime, Date endTime, Criteria criteria) {
		Criterion outTimeNotNull = Restrictions.isNotNull(AttendenceHistory.PROP_CLOCK_OUT_TIME);
		LogicalExpression geClockOut1 = Restrictions.and(outTimeNotNull, Restrictions.ge(AttendenceHistory.PROP_CLOCK_OUT_TIME, startTime));
		LogicalExpression geClockOut2 = Restrictions.and(outTimeNotNull, Restrictions.ge(AttendenceHistory.PROP_CLOCK_OUT_TIME, endTime));
		LogicalExpression leClockOut3 = Restrictions.and(outTimeNotNull, Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, endTime));

		LogicalExpression expression1 = Restrictions.and(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, startTime), geClockOut1);
		LogicalExpression expression2 = Restrictions.and(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, endTime), geClockOut2);
		LogicalExpression expression3 = Restrictions.and(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, startTime), leClockOut3);
		criteria.add(Restrictions.or(expression1, expression2, expression3));
	}

	public List<User> findNumberOfClockedInUserAtHour(Date fromDay, Date toDay, int hour, Outlet outlet) {
		Session session = null;

		ArrayList<User> users = new ArrayList<User>();

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, fromDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, toDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_HOUR, new Short((short) hour)));

			if (outlet != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, outlet.getId()));
			}

			List list = criteria.list();
			for (Object object : list) {
				AttendenceHistory history = (AttendenceHistory) object;

				if (!history.isClockedOut()) {
					users.add(history.getUser());
				}
				else if (history.getClockOutHour() >= hour) {
					users.add(history.getUser());
				}
			}
			return users;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.2"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public List<User> findNumberOfClockedInUserAtShift(Date fromDay, Date toDay, Shift shift, Outlet outlet) {
		Session session = null;

		ArrayList<User> users = new ArrayList<User>();

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, fromDay));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, toDay));

			if (shift != null) {
				criteria.add(Restrictions.le(AttendenceHistory.PROP_SHIFT_ID, shift.getId()));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, outlet.getId()));
			}

			List list = criteria.list();
			for (Object object : list) {
				AttendenceHistory history = (AttendenceHistory) object;
				users.add(history.getUser());
			}
			return users;
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.5"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public AttendenceHistory findByLastClockInTime(User user) {
		try (Session session = createNewSession()) {
			return findByLastClockInTime(user, session);
		}
	}

	@Deprecated
	public AttendenceHistory findHistoryByClockedInTime(User user) {
		Session session = null;

		try {
			session = createNewSession();
			return findHistoryByClockedInTime(user, session);
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	public AttendenceHistory findByLastClockInTime(User user, Session session) {
		Criteria criteria = session.createCriteria(AttendenceHistory.class);
		if (user != null) {
			criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, user.getOutletId()));
		}
		//criteria.add(Restrictions.isNull(AttendenceHistory.PROP_CLOCK_OUT_TIME));
		criteria.addOrder(Order.desc(AttendenceHistory.PROP_CLOCK_IN_TIME));
		criteria.setFirstResult(0);
		criteria.setMaxResults(1);

		List<AttendenceHistory> list = criteria.list();
		if (list.size() > 0) {
			return list.get(0);
		}

		return null;
	}

	@Deprecated
	public AttendenceHistory findHistoryByClockedInTime(User user, Session session) {
		Criteria criteria = session.createCriteria(AttendenceHistory.class);
		if (user != null) {
			criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
		}
		criteria.add(Restrictions.isNull(AttendenceHistory.PROP_CLOCK_OUT_TIME));
		criteria.addOrder(Order.desc(AttendenceHistory.PROP_CLOCK_IN_TIME));
		criteria.setFirstResult(0);
		criteria.setMaxResults(1);

		List<AttendenceHistory> list = criteria.list();
		if (list.size() > 0) {
			return list.get(0);
		}

		return null;
	}

	public List<PayrollReportData> findPayroll(Date from, Date to, User user, Outlet outlet) {
		userMap.clear();

		ArrayList<PayrollReportData> reportDataList = new ArrayList<PayrollReportData>();
		Map<String, List<AttendenceHistory>> attendenceHistoryMap = new HashMap<String, List<AttendenceHistory>>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
			if (user != null) {
				criteria.add(Restrictions.in(AttendenceHistory.PROP_USER_ID, user.getRoleIds()));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, outlet.getId()));
			}
			populateReportDataListMap(attendenceHistoryMap, criteria.list());

			for (List<AttendenceHistory> attendenceHistories : attendenceHistoryMap.values()) {
				if (attendenceHistories != null && !attendenceHistories.isEmpty()) {
					populateReportDataList(reportDataList, attendenceHistories);
				}
			}
			return reportDataList;
		}
	}

	private void populateReportDataListMap(Map<String, List<AttendenceHistory>> attendenceHistoryMap, List searchResultList) {
		for (Iterator iterator = searchResultList.iterator(); iterator.hasNext();) {
			AttendenceHistory history = (AttendenceHistory) iterator.next();

			String formatDateKey = keyDateFormat.format(history.getClockInTime()) + "_" + history.getUser().getId(); //$NON-NLS-1$
			List<AttendenceHistory> attendenceHistories = attendenceHistoryMap.get(formatDateKey);
			if (attendenceHistories == null || attendenceHistories.isEmpty()) {
				attendenceHistories = new ArrayList<AttendenceHistory>();
			}
			attendenceHistories.add(history);
			attendenceHistoryMap.put(formatDateKey, attendenceHistories);
		}
		calculateWorkHourMap(attendenceHistoryMap);
	}

	private void calculateWorkHourMap(Map<String, List<AttendenceHistory>> attendenceHistoryMap) {

		//Sort data
		for (List<AttendenceHistory> attendenceHistories : attendenceHistoryMap.values()) {
			Comparator<AttendenceHistory> comparator = Comparator.comparing(AttendenceHistory::getClockInTime, Comparator.nullsLast(Comparator.naturalOrder()));
			attendenceHistories.sort(comparator);
		}

		Set<String> keySet = attendenceHistoryMap.keySet();
		for (String key : keySet) {
			long totalWorkHourMs = 0;
			long previousTotalWorkHourMs = 0;

			List<AttendenceHistory> attendenceHistories = attendenceHistoryMap.get(key);
			for (ListIterator<AttendenceHistory> iterator = attendenceHistories.listIterator(); iterator.hasNext();) {
				boolean hasPrevious = iterator.hasPrevious();
				int previousIndex = iterator.previousIndex();
				AttendenceHistory attendenceHistory = (AttendenceHistory) iterator.next();
				long historyTotalWorkHourMs = attendenceHistory.getClockOutTime().getTime() - attendenceHistory.getClockInTime().getTime();

				if (hasPrevious) {
					AttendenceHistory previousAttenHistory = attendenceHistories.get(previousIndex);
					long peviousHistoryTime = previousAttenHistory.getClockOutTime().getTime() - previousAttenHistory.getClockInTime().getTime();
					previousTotalWorkHourMs += peviousHistoryTime;
					attendenceHistory.setPreviousTotalWorkHourMs(previousTotalWorkHourMs);
					totalWorkHourMs = previousTotalWorkHourMs + historyTotalWorkHourMs;
				}
				else {
					totalWorkHourMs += historyTotalWorkHourMs;
				}
				attendenceHistory.setDailyTotalWorkHourMs(totalWorkHourMs);
			}
		}
	}

	private void populateReportDataList(ArrayList<PayrollReportData> reportDataList, List searchResultList) {
		for (Iterator iterator = searchResultList.iterator(); iterator.hasNext();) {
			AttendenceHistory history = (AttendenceHistory) iterator.next();
			PayrollReportData data = new PayrollReportData();
			data.setClockIn(history.getClockInTime());
			data.setClockOut(history.getClockOutTime());
			data.setDate(history.getClockInTime());
			User user = getUserFromHistory(history);
			data.setUser(user);
			data.setUserIdSortKey(StringUtils.isBlank(user.getParentUserId()) ? user.getId() : user.getParentUserId());
			data.setDailyTotalWorkHourMs(history.getDailyTotalWorkHourMs());
			data.setPreviousTotalWorkHourMs(history.getPreviousTotalWorkHourMs());
			data.calculate();

			reportDataList.add(data);
		}
	}

	public List<WeeklyPayrollReportData> findWeeklyPayroll(Date from, Date to, User user, Outlet outlet, int firstDayOfWeek) {
		userMap.clear();
		Session session = null;
		List<WeeklyPayrollReportData> reportDataList = new ArrayList<WeeklyPayrollReportData>();
		List<Date[]> searchDateList = new ArrayList<>();
		List<User> activeUserList = UserDAO.getInstance().findAll();
		if (user != null) {
			activeUserList.clear();
			activeUserList.add(user);
			List<User> linkedUsers = user.getLinkedUser();
			for (User linkedUser : linkedUsers) {
				if (linkedUser.getId().equals(user.getId())) {
					continue;
				}
				activeUserList.add(linkedUser);
			}
		}
		List<String> userIdList = POSUtil.getStringIds(activeUserList, User.class);
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);

			Calendar calendar = Calendar.getInstance();
			calendar.setTime(from);
			Date[] dateRange = new Date[2];
			dateRange[0] = from;
			while (!from.after(to)) {
				int dayOfWeek = calendar.get(Calendar.DAY_OF_WEEK);
				int difference = dayOfWeek - (firstDayOfWeek - 1);
				calendar.add(Calendar.DATE, 1);

				if (difference == 0 || difference == 7) {
					calendar.add(Calendar.DATE, -1);
					Date toDate = calendar.getTime();
					dateRange[1] = toDate;
					searchDateList.add(dateRange);
					dateRange = new Date[2];
					calendar.add(Calendar.DATE, 1);
					from = calendar.getTime();
					dateRange[0] = from;
				}
				else if (calendar.getTime().after(to)) {
					dateRange[1] = to;
				}
			}

			for (Date[] dateRangeArray : searchDateList) {
				Map<String, List<AttendenceHistory>> attendenceHistoryMap = new HashMap<String, List<AttendenceHistory>>();
				criteria = session.createCriteria(AttendenceHistory.class);
				Date startOfDay = dateRangeArray[0];
				Date endOfDay = dateRangeArray[1];

				criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, startOfDay));
				criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_IN_TIME, endOfDay));
				criteria.add(Restrictions.isNotNull(AttendenceHistory.PROP_CLOCK_OUT_TIME));
				if (!userIdList.isEmpty()) {
					criteria.add(Restrictions.in(AttendenceHistory.PROP_USER_ID, userIdList));
				}
				if (outlet != null) {
					criteria.add(Restrictions.eq(AttendenceHistory.PROP_OUTLET_ID, outlet.getId()));
				}
				List weeklyPayrollList = criteria.list();
				populateReportDataListMap(attendenceHistoryMap, weeklyPayrollList);

				List<AttendenceHistory> aLLAttendenceHistories = new ArrayList<>();
				for (List<AttendenceHistory> attendenceHistories : attendenceHistoryMap.values()) {
					if (attendenceHistories != null && !attendenceHistories.isEmpty()) {
						aLLAttendenceHistories.addAll(attendenceHistories);
					}
				}
				List<WeeklyPayrollReportData> populateWeeklyPayrollReportDataList = populateWeeklyPayrollReportDataList(aLLAttendenceHistories, startOfDay,
						endOfDay, firstDayOfWeek, activeUserList, userIdList, outlet);
				reportDataList.addAll(populateWeeklyPayrollReportDataList);
			}

			return reportDataList;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private List<WeeklyPayrollReportData> populateWeeklyPayrollReportDataList(List attendenceHistoryResultList, Date from, Date to, int firstDayOfWeek,
			List<User> activeUserList, List<String> activeUserIdList, Outlet outlet) {

		from = DateUtil.convertServerTimeToBrowserTime(from);
		to = DateUtil.convertServerTimeToBrowserTime(to);

		Map<String, WeeklyPayrollReportData> reportDataMap = new HashMap<String, WeeklyPayrollReportData>();
		for (Iterator iterator = attendenceHistoryResultList.iterator(); iterator.hasNext();) {
			AttendenceHistory history = (AttendenceHistory) iterator.next();

			User user = getUserFromHistory(history);
			if (user == null)
				continue;

			WeeklyPayrollReportData weeklyUserData = reportDataMap.get(user.getId());
			if (weeklyUserData == null) {
				weeklyUserData = new WeeklyPayrollReportData();
				weeklyUserData.setFirstDayOfWeek(firstDayOfWeek);
				reportDataMap.put(user.getId(), weeklyUserData);
			}

			double costPerHourRate = user.getCostPerHourRate();
			double overtimePerHourRate = user.getOvertimePerHourRate();
			long historyWorkHourMs = history.getClockOutTime().getTime() - history.getClockInTime().getTime();
			long weeklyWorkHourMs = (long) (user.getRegularWorkHourWeekly() * 3600000d);

			long totalWeeklyWorkHourMs = weeklyUserData.getTotalWorkHourMs() + historyWorkHourMs;

			long regularweeklyWorkHourMs = totalWeeklyWorkHourMs > weeklyWorkHourMs ? weeklyWorkHourMs : totalWeeklyWorkHourMs;
			long overtimeWorkHourMs = totalWeeklyWorkHourMs > weeklyWorkHourMs ? (totalWeeklyWorkHourMs - weeklyWorkHourMs) : 0;

			weeklyUserData.setTotalWorkHourMs(totalWeeklyWorkHourMs);

			weeklyUserData.setRegularWorkHourMs(regularweeklyWorkHourMs);
			weeklyUserData.setOvertimeMs(overtimeWorkHourMs);
			double regularPayment = NumberUtil.round((weeklyUserData.getRegularWorkHourMs() / 3600000d) * costPerHourRate);
			weeklyUserData.setRegularPayment(regularPayment);
			double overTimePayment = NumberUtil.round((weeklyUserData.getOvertimeMs() / 3600000d) * overtimePerHourRate);
			weeklyUserData.setOvertimePayment(overTimePayment);

			weeklyUserData.setFromDateOfWeek(from);
			weeklyUserData.setToDateOfWeek(to);
			weeklyUserData.setUser(user);
		}

		//Show all user without attendance history.
		for (User user : activeUserList) {
			WeeklyPayrollReportData weeklyPayrollReportData = reportDataMap.get(user.getId());
			if (weeklyPayrollReportData == null) {
				WeeklyPayrollReportData data = new WeeklyPayrollReportData();
				data.setFirstDayOfWeek(firstDayOfWeek);
				data.setTotalWorkHourMs(0);
				data.setFromDateOfWeek(from);
				data.setToDateOfWeek(to);
				data.setUser(user);
				reportDataMap.put(user.getId(), data);
			}
		}

		Map<String, Double> declaredTipsMap = doFindDeclaredTips(from, to, reportDataMap);

		Collection<WeeklyPayrollReportData> values = reportDataMap.values();
		for (WeeklyPayrollReportData data : values) {

			data.calculate();

			User user = data.getUser();
			data.setUserIdSortKey(StringUtils.isBlank(user.getParentUserId()) ? user.getId() : user.getParentUserId());
			Double totalDeclaredTips = declaredTipsMap.get(user.getId());
			data.setTotalDeclaredTips(totalDeclaredTips == null ? 0D : totalDeclaredTips);
			//data.setTotalNoCashTips(PosTransactionDAO.getInstance().calculateTipsByUser(from, to, user, outlet, null));

		}
		return new ArrayList<WeeklyPayrollReportData>(values);
	}

	private User getUserFromHistory(AttendenceHistory history) {
		String userId = history.getUserId();
		String outletId = history.getOutletId();
		if (StringUtils.isBlank(userId) || StringUtils.isBlank(outletId)) {
			return null;
		}
		String key = userId + "_" + outletId;
		User user = userMap.get(key);
		if (user != null) {
			return user;
		}
		user = UserDAO.getInstance().get(userId, outletId);
		userMap.put(key, user);
		return user;
	}

	private Map<String, Double> doFindDeclaredTips(Date from, Date to, Map<String, WeeklyPayrollReportData> reportDataMap) {
		//----search declared Tips from db.-------
		Map<String, Double> declaredTipsMap = new HashMap<String, Double>();
		try (Session session = DeclaredTipsDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(DeclaredTips.class);
			criteria.add(Restrictions.ge(DeclaredTips.PROP_DECLARED_TIME, from));
			criteria.add(Restrictions.le(DeclaredTips.PROP_DECLARED_TIME, to));
			if (!reportDataMap.keySet().isEmpty()) {
				criteria.add(Restrictions.in(DeclaredTips.PROP_OWNER_ID, reportDataMap.keySet()));
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(DeclaredTips.PROP_AMOUNT), DeclaredTips.PROP_AMOUNT);
			projectionList.add(Projections.groupProperty(DeclaredTips.PROP_OWNER_ID), DeclaredTips.PROP_OWNER_ID);
			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(DeclaredTips.class));
			List list = criteria.list();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				DeclaredTips declaredTips = (DeclaredTips) iterator.next();
				declaredTipsMap.put(declaredTips.getOwnerId(), declaredTips.getAmount());
			}
		}
		return declaredTipsMap;
	}

	public List<AttendanceReportData> findAttendance(Date from, Date to, User user) {
		Session session = null;

		ArrayList<AttendanceReportData> list = new ArrayList<AttendanceReportData>();

		try {

			session = createNewSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.between(AttendenceHistory.PROP_CLOCK_IN_TIME, from, to));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_USER_ID));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_CLOCK_IN_TIME));
			addUserWithAllRoleCriteria(criteria, user, AttendenceHistory.PROP_USER_ID);

			List list2 = criteria.list();

			for (Iterator iterator = list2.iterator(); iterator.hasNext();) {
				AttendenceHistory history = (AttendenceHistory) iterator.next();
				AttendanceReportData data = new AttendanceReportData();
				data.setClockIn(history.getClockInTime());
				data.setClockOut(history.getClockOutTime());
				data.setUser(history.getUser());
				data.setName(history.getUser().getFirstName());
				data.calculate();

				list.add(data);
			}

			return list;
		} catch (Exception e) {
			throw new PosException("Unable to find Attendance", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<AttendenceHistory> findHistory(Date from, Date toDate, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.between(AttendenceHistory.PROP_CLOCK_IN_TIME, from, toDate));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_ID));
			addUserWithAllRoleCriteria(criteria, user, AttendenceHistory.PROP_USER_ID);

			List list2 = criteria.list();

			return list2;
		} catch (Exception e) {
			throw new PosException("Unable to find History", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<AttendenceHistory> findAttendanceHistory(Date from, Date to, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(AttendenceHistory.class);
			criteria.add(Restrictions.ge(AttendenceHistory.PROP_CLOCK_IN_TIME, from));
			criteria.add(Restrictions.le(AttendenceHistory.PROP_CLOCK_OUT_TIME, to));
			criteria.addOrder(Order.asc(AttendenceHistory.PROP_ID));
			if (user != null) {
				criteria.add(Restrictions.eq(AttendenceHistory.PROP_USER_ID, user.getId()));
			}

			List list2 = criteria.list();

			return list2;
		} catch (Exception e) {
			throw new PosException("Unable to find History", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateAttendenceHistoryList(List<AttendenceHistory> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<AttendenceHistory> iterator = dataList.iterator(); iterator.hasNext();) {
				AttendenceHistory item = (AttendenceHistory) iterator.next();
				AttendenceHistory existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public static AttendenceHistory createNewHistory(User user, Terminal terminal, Shift shift) {
		AttendenceHistory attendenceHistory = new AttendenceHistory();
		Date lastClockInTime = user.getLastClockInTime();
		Calendar c = Calendar.getInstance();
		if (lastClockInTime != null) {
			c.setTime(lastClockInTime);
		}
		attendenceHistory.setClockInTime(lastClockInTime);
		attendenceHistory.setClockInHour(Short.valueOf((short) c.get(Calendar.HOUR)));
		attendenceHistory.setUser(user);
		attendenceHistory.setTerminal(terminal);
		attendenceHistory.setOutletId(user.getOutletId());
		attendenceHistory.setShift(shift);
		return attendenceHistory;
	}

	public List<AttendenceHistory> findAttendenceReport(Date fromDay, Date toDay) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			buildAttendenceCriteria(fromDay, toDay, criteria);

			return criteria.list();
		} catch (Exception e) {
			throw new PosException(Messages.getString("AttendenceHistoryDAO.2"), e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}
}