/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Set;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Interval;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.DurationType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Store;
import com.floreantpos.model.dao.StoreDAO;

public class DateUtil {
	private final static SimpleDateFormat dateFormat = new SimpleDateFormat("dd-MM-yy");
	private final static DateFormat localeFormat = DateFormat.getDateInstance(DateFormat.SHORT);
	private static SimpleDateFormat monthDateFormat = new SimpleDateFormat("MMM dd");//$NON-NLS-1$
	public static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("hh:mm a"); //$NON-NLS-1$

	public static String formatDateAsLocale(Date date) {
		String dateString = localeFormat.format(date);
		return dateString;
	}

	public static String formatReportDateAsString(Date date) {
		return formatDateWithTime(date);
	}

	public static String formatDateAsString(Date date) {
		return formatDateWithTime(date);
	}

	public static String formatDateAsUTCString(Date date) {
		//		dateFormat9.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		//		String dateString = dateFormat9.format(date);

		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static Date parseUTCString(String utcDate) throws ParseException {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		return simpleDateFormat.parse(utcDate);
	}

	public static Date parseByFullDate(String date) throws Exception {
		return formatDateWithTimeAndSec(date);
	}

	public static String formatFullDateAsString(Date date) {
		return formatDate(date);
	}

	public static Date parseBySortDate(String date) throws Exception {
		return formatDate(date);
	}

	public static Date startOfDay(Date date) {
		//		Calendar cal = Calendar.getInstance();
		//		cal.setTime(date == null ? new Date() : date);
		//		cal.set(Calendar.HOUR_OF_DAY, 0);
		//		cal.set(Calendar.MINUTE, 0);
		//		cal.set(Calendar.SECOND, 0);
		//
		//		return new Date(cal.getTimeInMillis());

		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.systemDefault());
		return Date.from(startOfDay.toInstant());
	}

	public static Date getUTCStartOfDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), getClientZoneId());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(getClientZoneId());
		return Date.from(startOfDay.toInstant());
	}

	public static Date getUTCEndOfDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), getClientZoneId());
		dateTime = dateTime.plusDays(1);
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(getClientZoneId());
		return Date.from(startOfDay.toInstant());
	}

	private static ZoneId getClientZoneId() {
		int timeZoneOffset = DataProvider.get().getTimeZoneOffset();
		if (timeZoneOffset == 0) {
			return ZoneId.systemDefault();
		}
		return ZoneId.ofOffset("UTC", ZoneOffset.ofTotalSeconds(timeZoneOffset / 1000)); //$NON-NLS-1$
	}

	public static Date startOfNextDay(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		dateTime = dateTime.plusDays(1);

		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.systemDefault());
		return Date.from(startOfDay.toInstant());
	}

	public static Date endOfDay(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date == null ? new Date() : date);
		cal.set(Calendar.HOUR_OF_DAY, 23);
		cal.set(Calendar.MINUTE, 59);
		cal.set(Calendar.SECOND, 59);
		cal.set(Calendar.MILLISECOND, 999);

		return new Date(cal.getTimeInMillis());
	}

	public static Date startOfDayInUTC(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.of("UTC")); //$NON-NLS-1$

		return Date.from(startOfDay.toInstant());
	}

	public static Date endOfDayInUTC(Date date) {
		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		ZonedDateTime atStartOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.of("UTC")); //$NON-NLS-1$
		ZonedDateTime enDateTime = atStartOfDay.plusDays(1);

		return Date.from(enDateTime.toInstant());
	}

	public static Date toUTC(Date date) {
		int rawOffset = TimeZone.getDefault().getRawOffset();
		return new Date(date.getTime() - rawOffset);
	}

	public static Date startOfMonth(Date date) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(date);
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.DAY_OF_MONTH, 1);

		return new Date(cal.getTimeInMillis());
	}

	public static Date endOfMonth(Date date) {
		//		Calendar cal = Calendar.getInstance();
		//		cal.setTime(date);
		//		cal.set(Calendar.HOUR_OF_DAY, 23);
		//		cal.set(Calendar.MINUTE, 59);
		//		cal.set(Calendar.SECOND, 59);
		//		cal.set(Calendar.DAY_OF_MONTH, cal.getActualMaximum(Calendar.DAY_OF_MONTH));
		//
		//		return new Date(cal.getTimeInMillis());

		LocalDateTime dateTime = LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
		dateTime = dateTime.plusMonths(1);

		ZonedDateTime startOfDay = dateTime.toLocalDate().atStartOfDay(ZoneId.systemDefault());
		return Date.from(startOfDay.toInstant());
	}

	public static Date copyTime(Date copyTo, Date copyFrom) {

		Calendar calendar = Calendar.getInstance();
		calendar.setTime(copyFrom);
		int hour = calendar.get(Calendar.HOUR_OF_DAY);
		int min = calendar.get(Calendar.MINUTE);

		calendar.setTime(copyTo);
		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, min);
		return calendar.getTime();

	}

	public static boolean between(Date startDate, Date endDate, Date guniping) {
		if (startDate == null || endDate == null) {
			return false;
		}

		return (guniping.equals(startDate) || guniping.after(startDate)) && (guniping.equals(endDate) || guniping.before(endDate));
	}

	public static String formatSmall(Date date) {
		return formatDate(date);
	}

	public static String getReportDate() {
		return formatDateWithTimeAndSec(new Date());
	}

	public static boolean isToday(Date date) {
		return isSameDay(date, Calendar.getInstance().getTime());
	}

	public static boolean isToday(Calendar cal) {
		return isSameDay(cal, Calendar.getInstance());
	}

	public static String formatFullDateAndTimeAsString(Date date) {
		String dateString = date == null ? "" : formatDateWithTimeAndSec(date); //$NON-NLS-1$
		return dateString;
	}

	public static String formatFullDateAndTimeWithoutYearAsString(Date date) {
		return formatDateWithTimeAndSec(date);
	}

	public static boolean isSameDay(Date date1, Date date2) {
		Calendar cal1 = Calendar.getInstance();
		cal1.setTime(date1);
		Calendar cal2 = Calendar.getInstance();
		cal2.setTime(date2);
		return isSameDay(cal1, cal2);
	}

	public static boolean isSameDay(Calendar cal1, Calendar cal2) {
		return (cal1.get(Calendar.ERA) == cal2.get(Calendar.ERA) && cal1.get(Calendar.YEAR) == cal2.get(Calendar.YEAR)
				&& cal1.get(Calendar.DAY_OF_YEAR) == cal2.get(Calendar.DAY_OF_YEAR));
	}

	public static String getOnlyFormattedDate(Date date) {
		return formatDateWithTime(date);
	}

	public static String getElapsedTime(Date oldTime, Date newTime) {
		DateTime startDate = new DateTime(oldTime);
		DateTime endDate = new DateTime(newTime);
		Interval interval = new Interval(startDate, endDate);
		long days = interval.toDuration().getStandardDays();
		long hours = interval.toDuration().getStandardHours();
		long minutes = interval.toDuration().getStandardMinutes();
		long seconds = interval.toDuration().getStandardSeconds();

		hours = hours % 24;
		minutes = minutes % 60;
		seconds = seconds % 60;

		String strDays = days + "d, ";//$NON-NLS-1$
		String strHours = hours + "hr, ";//$NON-NLS-1$
		String strMins = minutes + "mn";//$NON-NLS-1$
		String strSec = seconds + "secs";//$NON-NLS-1$
		String strAgo = "";//$NON-NLS-1$

		String fullTime = strDays + strHours + strMins + strAgo;
		String timeWithoutDay = strHours + strMins + strAgo;
		String timeWithoutHour = strMins + strAgo;
		String timeWithoutMin = strSec + strAgo;

		if (days != 0) {
			return fullTime;
		}
		else if (hours != 0) {
			return timeWithoutDay;
		}
		else if (minutes != 0) {
			return timeWithoutHour;
		}
		else if (seconds != 0) {
			return timeWithoutMin;
		}
		else {
			return "not printed yet";//$NON-NLS-1$
		}
	}

	public static Calendar getLocalTimeCalendar() {
		return Calendar.getInstance();
	}

	public static Calendar getServerTimeCalendar() {
		Calendar calendar = Calendar.getInstance();
		calendar.setTime(StoreDAO.getServerTimestamp());
		return calendar;
	}

	public static Date getGmtTime() {
		return Calendar.getInstance(TimeZone.getTimeZone("GMT")).getTime();//$NON-NLS-1$
	}

	public static Calendar getGmtCalendar() {
		return Calendar.getInstance(TimeZone.getTimeZone("GMT"));//$NON-NLS-1$
	}

	//Dynamic Date Format
	public static String getStoreDateFormat() {
		Store store = DataProvider.get().getStore();
		return store.getDateFormat();
	}

	public static SimpleDateFormat getSimpleDateFormatWithTime() {
		Store store = DataProvider.get().getStore();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(store.getDateFormat() + " hh:mm a");
		return simpleDateFormat;

	}

	private static SimpleDateFormat getSimpleDateFormat() {
		String dateFormat = getStoreDateFormat();
		if (StringUtils.isNotBlank(dateFormat)) {
			return new SimpleDateFormat(dateFormat);
		}
		return new SimpleDateFormat();

	}

	public static String formatDate(Date date) {
		SimpleDateFormat simpleDateFormat = getSimpleDateFormat();
		return simpleDateFormat.format(date);

	}

	public static Date formatDate(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = getSimpleDateFormat();
		return simpleDateFormat.parse(date);

	}

	/**
	 * ************************************************************************
	 * * inputFormat Can't be null
	 * ************************************************************************
	 */
	public static String parseStringDateToString(String inputFormat, String date) throws Exception {
		String outputFormat = getStoreDateFormat();
		SimpleDateFormat simpleDateOutputFormat = new SimpleDateFormat(outputFormat);

		SimpleDateFormat simpleDateInputFormat = new SimpleDateFormat(inputFormat);

		Date parseDate = simpleDateInputFormat.parse(date);

		return simpleDateOutputFormat.format(parseDate);
	}

	public static String formatAsShortDate(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		return simpleDateFormat.format(date);

	}

	public static String formatDateWithTime(Date date) {
		if (date == null) {
			return "";
		}
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static String simplifyDateAndTime(Date date, String format) {
		if (date == null) {
			return "";
		}
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static String tweentyFourHoursDateFormat(Date date) {
		return new SimpleDateFormat(getStoreDateFormat() + " HH:mm", Locale.UK).format(date); //$NON-NLS-1$
	}

	public static Date formatDateWithTime(String date) throws ParseException {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.parse(date);

	}

	public static String formatDateWithTimeAndSec(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static Date formatDateWithTimeAndSec(String date) throws ParseException {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.parse(date);

	}

	public static Date formatDateWithDefaultTimeAndSec(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy hh:mm:ss a"); //$NON-NLS-1$
		try {
			Date parse = simpleDateFormat.parse(date);
			return parse;
		} catch (Exception e) {
		}
		return null;
	}

	public static String formatDateWithDefaultTimeAndSec(Date date) {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy hh:mm:ss a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);

	}

	public static String formatSyncTime(Date date) {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy HH:mm:ss"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static Date parseSyncTime(String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("MM-dd-yyyy HH:mm:ss"); //$NON-NLS-1$
		simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC")); //$NON-NLS-1$
		try {
			Date parse = simpleDateFormat.parse(date);
			return parse;
		} catch (Exception e) {

		}
		return DateUtil.formatDateWithDefaultTimeAndSec(date);
	}

	public static String formatAsDefaultMonthDate(Date date) {
		return monthDateFormat.format(date);
	}

	/**
	 * * This method is used to format date to string date without second by using store format.
	 * 
	 * @param date Date value
	 * 
	 */
	public static String formatReportDateWithStoreTimeZone(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}

		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return formatReportDateWithStoreTimeZone(date, simpleDateFormat);
	}

	/**
	 * * This method is used to format date to string date without second by using store format.
	 * 
	 * @param date Date value
	 * 
	 */
	public static String formatDateWithoutTime(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}

		return formatReportDateWithStoreTimeZone(date, dateFormat);
	}

	public static String formatReportDateWithStore24TimeZone(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}

		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " HH:mm"); //$NON-NLS-1$
		return formatReportDateWithStoreTimeZone(date, simpleDateFormat);
	}

	/**
	 * * This method is used to format date to string date.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static String formatReportDateWithStoreTimeZone(Date date, SimpleDateFormat simpleDateFormat) {
		Outlet outlet = DataProvider.get().getOutlet();
		if (outlet != null) {
			TimeZone timezone = outlet.getTimeZone();
			if (timezone != null) {
				simpleDateFormat.setTimeZone(timezone);
				return simpleDateFormat.format(date);
			}
		}

		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	/**
	 * * This method is used to format date to string short date.
	 * 
	 * @param date Date value
	 * 
	 */
	public static String formatReportShortDateWithBrowserTimeOffset(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format); //$NON-NLS-1$
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	/**
	 * * This method is used to format date to string date without second.
	 * 
	 * @param date Date value
	 * 
	 */
	public static String formatReportDateWithBrowserTimeOffset(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	/**
	 * * This method is used to format date to string date.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static String formatDateWithBrowserTimeOffset(Date date) {
		if (date == null) {
			return ""; //$NON-NLS-1$
		}
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	public static String formatDateWithBrowserTimeOffset(Date date, SimpleDateFormat simpleDateFormat) {
		if (date == null) {
			return null;
		}
		int timezoneOffset = DataProvider.get().getTimeZoneOffset();
		if (timezoneOffset == 0) {
			return simpleDateFormat.format(date);
		}
		String[] availableIDs = TimeZone.getAvailableIDs(timezoneOffset);
		String timezone = null;
		if (availableIDs.length > 0) {
			timezone = availableIDs[0];
		}
		if (StringUtils.isNotEmpty(timezone)) {
			simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timezone));
		}
		return simpleDateFormat.format(date);
	}

	/**
	 * * This method is used to format string date to date.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static Date formatDateWithBrowserTimeOffset(String date) {
		try {
			String format = DataProvider.get().getStore().getDateFormat();
			SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a"); //$NON-NLS-1$

			int timezoneOffset = DataProvider.get().getTimeZoneOffset();
			if (timezoneOffset == 0) {
				return simpleDateFormat.parse(date);
			}
			String[] availableIDs = TimeZone.getAvailableIDs(timezoneOffset);
			String timezone = null;
			if (availableIDs.length > 0) {
				timezone = availableIDs[0];
			}

			if (StringUtils.isNotEmpty(timezone)) {
				simpleDateFormat.setTimeZone(TimeZone.getTimeZone(timezone));
			}
			return simpleDateFormat.parse(date);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return null;
		}
	}
	//
	//	public static String convertDateToBrowserTimeZone(Date date) {
	//		if (date == null) {
	//			return ""; //$NON-NLS-1$
	//		}
	//		String format = DataProvider.get().getStore().getDateFormat();
	//		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm:ss a z"); //$NON-NLS-1$
	//		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	//	}

	/**
	 * * This method is used for search by Browser date from DB.
	 * 
	 * @param date Date value
	 * 
	 */
	public static Date convertDateToServerTime(Date date) {
		try {
			int timezoneOffset = DataProvider.get().getTimeZoneOffset();
			return new Date(date.getTime() - timezoneOffset);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return date;
		}
	}

	/**
	 * * This method is used to format date to string date without second value.
	 * 
	 * @param date Date value
	 * @param timezoneOffset Millisecond
	 * 
	 */
	public static String formatDateWithBrowserTimeOffsetWithoutSec(Date date) {
		String format = DataProvider.get().getStore().getDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		return formatDateWithBrowserTimeOffset(date, simpleDateFormat);
	}

	//	public static Date startOfDayInUTC(Date date) {
	//		date = startOfDay(date);
	////		int rawOffset = TimeZone.getDefault().getRawOffset();
	////		date = new Date(date.getTime() + rawOffset);
	//
	//		return date;
	//	}
	//
	//	public static Date endOfDayInUTC(Date date) {
	//		date = endOfDay(date);
	////		int rawOffset = TimeZone.getDefault().getRawOffset();
	////		date = new Date(date.getTime() + rawOffset);
	//
	//		return date;
	//	}

	public static String formatReportDate(Date date) {
		String format = getStoreDateFormat();
		//SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a z"); //$NON-NLS-1$
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static Date parseDate(String format, String date) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format); //$NON-NLS-1$
		return simpleDateFormat.parse(date);

	}

	public static String format(String pattern, Date date) {
		return new SimpleDateFormat(pattern).format(date);
	}

	public static void validateDate(Date fromDate, Date toDate) {
		if (fromDate == null) {
			throw new PosException(POSConstants.FROM_DATE_IS_NULL);
		}
		if (toDate == null) {
			throw new PosException(POSConstants.TO_DATE_IS_NULL);
		}
		Date startOfDay = DateUtil.startOfDay(fromDate);
		Date endOfDay = DateUtil.endOfDay(toDate);
		if (startOfDay.after(endOfDay)) {
			throw new PosException(POSConstants.FROM_DATE_CANNOT_BE_GREATER_THAN_TO_DATE_);
		}
	}

	public static Date convertLocalDateToStartOfDayDate(LocalDate localDate) {
		if (localDate != null) {
			Instant instant = Instant.from(localDate.atStartOfDay(ZoneId.of("GMT"))); //$NON-NLS-1$
			return Date.from(instant);
		}
		return null;
	}
	
	public static Date convertLocalDateToEndOfDayDate(LocalDate localDate) {
	    if (localDate != null) {
	        Instant instant = Instant.from(localDate.atTime(LocalTime.MAX).atZone(ZoneId.of("GMT")));
	        return Date.from(instant);
	    }
	    return null;
	}

	public static Date convertLocalDateAndLocalTimeToDate(LocalDate localDate, LocalTime localTime) {
		return convertLocalDateTimeToDate(LocalDateTime.of(localDate, localTime));
	}

	public static Date convertLocalDateTimeToDate(LocalDateTime localDateTime) {
		if (localDateTime != null) {
			Instant instant = localDateTime.atZone(ZoneId.of("GMT")).toInstant(); //$NON-NLS-1$
			return Date.from(instant);
		}
		return null;
	}

	public static Date convertLocalDateTimeToLocalDate(LocalDateTime localDateTime) {
		if (localDateTime != null) {
			Instant instant = localDateTime.atZone(getBrowserTimeZoneId()).toInstant();
			return Date.from(instant);
		}
		return null;
	}

	public static LocalDateTime convertLocalDateToLocalDateTime(Date date) {
		if (date == null) {
			return null;
		}
		LocalDateTime localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
		return localDate;
	}

	public static LocalDate convertDateToLocalDate(Date date) {
		if (date == null) {
			return null;
		}
		LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
		return localDate;
	}

	public static LocalTime convertDateToLocalTime(Date date) {
		if (date == null) {
			return null;
		}
		LocalTime localTime = date.toInstant().atZone(ZoneId.systemDefault()).toLocalTime();
		return localTime;
	}

	public static Date convertLocalTimeToDate(LocalTime localTime) {
		if (localTime != null) {
			Instant instant = localTime.atDate(LocalDate.now()).atZone(ZoneId.systemDefault()).toInstant();
			return Date.from(instant);
		}
		return null;
	}

	/**
	 * * This method is used to format DB date to date for user display.
	 * 
	 * @param date Date value
	 * 
	 */
	public static Date convertServerTimeToBrowserTime(Date date) {
		try {
			int timezoneOffset = DataProvider.get().getTimeZoneOffset();
			return new Date(date.getTime() + timezoneOffset);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return date;
		}
	}

	public static Date convertDateByTimeZone(Date date, int timezoneOffset) {
		try {
			return new Date(date.getTime() + timezoneOffset);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return date;
		}
	}

	public static Date convertDateToGmtTimeZone(Date date, int timezoneOffset) {
		try {
			return new Date(date.getTime() - timezoneOffset);
		} catch (Exception e) {
			PosLog.error(DateUtil.class, e);
			return date;
		}
	}

	public static String formatReportDateWithTime(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " hh:mm a z"); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static String getCurrentYear() {
		Date now = new Date();
		SimpleDateFormat yearFormatter = new SimpleDateFormat("YYYY"); //$NON-NLS-1$
		String currentYear = yearFormatter.format(now);
		return currentYear;
	}

	public static String format24HoursDate(Date date) {
		String format = getStoreDateFormat();
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format + " HH:mm"); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	public static String formatTime(Date date) {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("hh:mm a"); //$NON-NLS-1$
		return simpleDateFormat.format(date);
	}

	/**
	 * * This method is used to string (HH:mm) format to date.
	 * 
	 * @param String (HH:mm) format
	 * 
	 */
	public static Date prepareDate(String dateTime) {
		if (StringUtils.isBlank(dateTime) && !dateTime.contains(":")) { //$NON-NLS-1$
			return null;
		}

		String[] split = dateTime.split(":"); //$NON-NLS-1$
		int hour = Integer.valueOf(split[0]);
		int minute = 0;
		if (split.length > 1) {
			minute = Integer.valueOf(split[1]);
		}
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.HOUR_OF_DAY, hour);
		calendar.add(Calendar.MINUTE, minute);
		calendar.set(Calendar.SECOND, 0);

		int currentMinute = calendar.get(Calendar.MINUTE);

		if (currentMinute < 15) {
			calendar.set(Calendar.MINUTE, 0);
		}
		else if (currentMinute <= 30) {
			calendar.set(Calendar.MINUTE, 30);
		}
		else {
			calendar.set(Calendar.MINUTE, 0);
			calendar.add(Calendar.HOUR_OF_DAY, 1);
		}
		return calendar.getTime();
	}

	public static Date prepareDateToBrowserTime(String dateTime) {
		Date prepareDate = prepareDate(dateTime);
		if (prepareDate == null) {
			return null;
		}
		return convertServerTimeToBrowserTime(prepareDate);
	}

	public static String calculateAge(Date bornDate) {
		Period period = Period.between(bornDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate(), LocalDate.now());
		int years = period.getYears();
		int months = period.getMonths();
		int days = period.getDays();

		if (years > 0) {
			String year = years + " Y"; //$NON-NLS-1$
			if (months > 0) {
				return year + " " + months + " M"; //$NON-NLS-1$ //$NON-NLS-2$
			}
			return year;
		}

		if (months > 0) {
			String month = months + " M"; //$NON-NLS-1$
			if (days > 0) {
				return month + " " + days + " D"; //$NON-NLS-1$ //$NON-NLS-2$
			}
			return month;
		}

		return period.getDays() + " D";
	}

	//	public static String formatSmallDate(Date date) {
	//		String pattern = "yyyy-MM-dd HnH:mm:ss"; //$NON-NLS-1$
	//		return format(pattern, date);
	//	}

	public static String format24HourLocalTime(LocalTime localTime) {
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
		return localTime.format(formatter);
	}

	public static LocalTime parse24HourLocalTime(String localTimeString) {
		DateTimeFormatter formatter = DateTimeFormatter.ofPattern("HH:mm");
		return LocalTime.parse(localTimeString, formatter);
	}

	public static int calculatePatientAge(LocalDate dob) {
		LocalDate curDate = LocalDate.now();
		if ((dob != null) && (curDate != null)) {
			return Period.between(dob, curDate).getYears();
		}
		else {
			return 0;
		}

	}

	public static LocalDate convertDateToBrowserLocalDate(Date date) {
		if (date == null) {
			return null;
		}
		ZoneId zoneId = getBrowserTimeZoneId();
		if (zoneId == null) {
			zoneId = ZoneId.systemDefault();
		}
		LocalDate localDate = date.toInstant().atZone(zoneId).toLocalDate();
		return localDate;
	}

	public static ZoneId getBrowserTimeZoneId() {
		long offsetMillis = DataProvider.get().getTimeZoneOffset();

		// Convert milliseconds to seconds
		int offsetSeconds = (int) (offsetMillis / 1000);

		// Create a ZoneOffset from the offset in seconds
		ZoneOffset zoneOffset = ZoneOffset.ofTotalSeconds(offsetSeconds);

		// (Optional) Find all ZoneIds that match this offset at the current time
		Instant now = Instant.now();
		Set<String> allZoneIds = ZoneId.getAvailableZoneIds();
		for (String zoneId : allZoneIds) {
			ZoneId zone = ZoneId.of(zoneId);
			ZoneOffset currentOffset = ZonedDateTime.ofInstant(now, zone).getOffset();

			if (currentOffset.equals(zoneOffset)) {
				return zone;
				//System.out.println("ZoneId: " + zone + " has offset " + currentOffset + " at this time.");
			}
		}

		return null;
	}

	public static int getDaysBetween(Date fromDate, Date toDate) {
		long daysBetween = 0;
		if (fromDate != null) {
			long differenceInMillis = toDate.getTime() - fromDate.getTime();
			daysBetween = differenceInMillis / (1000 * 60 * 60 * 24);
		}

		return (int) daysBetween;
	}

	public static int getRevisitIntervalInDays(Integer revisitInterval, DurationType durationType) {
		if (revisitInterval == null || durationType == null)
			return 0;

		switch (durationType) {
			case DAY:
				return revisitInterval;
			case MONTH:
				return revisitInterval * 30; // Approximate month
			case YEAR:
				return revisitInterval * 365; // Approximate year
			default:
				return 7;
		}
	}

	public static Date mergeDateAndTime(Date date, String timeStr) {
		if (date == null || StringUtils.isBlank(timeStr)) {
			return date;
		}

		try {
			LocalTime time = LocalTime.parse(timeStr, TIME_FORMATTER);
			LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
			return convertLocalDateAndLocalTimeToDate(localDate, time);

		} catch (DateTimeParseException e) {
			return date;
		}
	}

	public static boolean isWithinLastHours(Date date, int hours) {
		Instant createInstant = date.toInstant();
		Instant now = Instant.now();
		Duration duration = Duration.between(createInstant, now);
		return duration.toHours() < hours;
	}
}
