package com.orocube.accounting.reportservice;

import java.util.Date;
import java.util.HashMap;
import java.util.List;

import com.floreantpos.POSConstants;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.Outlet;
import com.floreantpos.report.ReportConstants;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.report.model.BalanceSheetLedgerEntryModel;
import com.floreantpos.report.model.BalanceSheetReportModel;
import com.floreantpos.report.model.IncomeStatementReportModel;
import com.floreantpos.report.model.LedgerEntryReportModel;
import com.floreantpos.report.model.TrialBalanceReportModel;
import com.floreantpos.services.report.ReportService;
import com.floreantpos.util.NumberUtil;
import com.orocube.siiopa.accounting.client.service.BalanceSheetService;
import com.orocube.siiopa.accounting.client.service.IncomeStatementService;

import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class AccountingReport {

	public static JasperPrint getIncomeStatementReportPrint(Date fromDate, Date toDate, Outlet outlet, IncomeStatementService incomeStatementService)
			throws Exception {

		JasperReport report = ReportUtil.getReport("income_statement_report"); //$NON-NLS-1$

		HashMap properties = new HashMap();
		ReportUtil.populateReportTime(properties, fromDate, toDate);
		ReportUtil.populateReportFooter(properties);
		ReportUtil.populateRestaurantProperties(properties, true, outlet);

		populateReportParamsForIncomeStatementReport(properties, incomeStatementService);
		properties.put("reportDate", //$NON-NLS-1$
				makeBoldString("Report on:") + ReportService.formatFullDate(new Date())); //$NON-NLS-1$

		IncomeStatementReportModel reportModel = new IncomeStatementReportModel();
		reportModel.setRows(incomeStatementService.getExpenseList());

		JasperPrint print = JasperFillManager.fillReport(report, properties, new JRTableModelDataSource(reportModel));
		return print;
	}

	private static void populateReportParamsForIncomeStatementReport(HashMap properties, IncomeStatementService incomeStatementService) {
		properties.put("reportTitle", "Income statement report");
		properties.put("colAccountName", "Account Name");
		properties.put("colAmount", "Amount");

		properties.put("sales", "Sales");
		properties.put("cost", "Cost");
		properties.put("salesAmount", NumberUtil.formatToAccountingAmount(incomeStatementService.getTotalSales() * (-1)));
		properties.put("costAmount", NumberUtil.formatToAccountingAmount(incomeStatementService.getTotalDirectCost() * (-1)));

		properties.put("grossProfit", "Gross profit");
		properties.put("grossProfitAmount", NumberUtil.formatToAccountingAmount(incomeStatementService.getGrossProfit() * (-1)));

		properties.put("colOperatingExpenses", "Oparating expenses");
		properties.put("totalOperatingExpenses", "Total oparating expenses");
		properties.put("totalOperatingExpensesAmount", NumberUtil.formatToAccountingAmount(incomeStatementService.getTotalExpenseAamount() * (-1)));

		properties.put("netProfit", "Net profit");
		properties.put("netProfitAmount", NumberUtil.formatToAccountingAmount(incomeStatementService.getTotalOparatingProfit() * (-1)));
	}

	public static JasperPrint getBalanceSheetReportPrint(Date fromDate, Date toDate, Outlet outlet, BalanceSheetService balanceSheetService)
			throws JRException {
		JasperReport assetSubreport = ReportUtil.getReport(ReportConstants.BALANCE_SHEET_SUB_REPORT);
		JasperReport liabliSubreport = ReportUtil.getReport(ReportConstants.BALANCE_SHEET_SUB_REPORT);
		JasperReport equitySubreport = ReportUtil.getReport(ReportConstants.BALANCE_SHEET_SUB_REPORT);

		HashMap map = new HashMap();
		ReportUtil.populateRestaurantProperties(map, true, outlet);
		ReportUtil.populateReportTime(map, fromDate, toDate);
		ReportUtil.populateReportFooter(map);
		if (outlet != null) {
			map.put("outletName", ReportUtil.reportLabelWithBoldTag("Outlet: ") + " " + outlet.getName()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		map.put("reportTitle", "Balance sheet report");
		map.put("colLeftHeading", "Asset"); //$NON-NLS-1$
		map.put("colRightHeading1", "Liabilities"); //$NON-NLS-1$
		map.put("colRightHeading2", "Equity"); //$NON-NLS-1$
		map.put("colTotal", "Total"); //$NON-NLS-1$
		map.put("colGroupTotal", "Total"); //$NON-NLS-1$

		BalanceSheetReportModel leftDataSourceModel = new BalanceSheetReportModel();
		List<BalanceSheetLedgerEntryModel> totalAssetList = balanceSheetService.getCurrentAssetList();
		totalAssetList.addAll(balanceSheetService.getAssetList());
		leftDataSourceModel.setRows(totalAssetList);

		map.put("leftDataSource", new JRTableModelDataSource(leftDataSourceModel)); //$NON-NLS-1$
		map.put("leftSubReport", assetSubreport); //$NON-NLS-1$
		map.put("colLeftTotal", "Total Asset"); //$NON-NLS-1$

		BalanceSheetReportModel rightDataSourceModel1 = new BalanceSheetReportModel();
		List<BalanceSheetLedgerEntryModel> totalLiabilityList = balanceSheetService.getCurrentLiabilityList();
		totalLiabilityList.addAll(balanceSheetService.getLiabilitiesList());
		rightDataSourceModel1.setRows(totalLiabilityList);

		map.put("rightDataSource1", new JRTableModelDataSource(rightDataSourceModel1)); //$NON-NLS-1$
		map.put("rightSubReport1", liabliSubreport); //$NON-NLS-1$
		map.put("colRightTotal1", "Total Liabilities"); //$NON-NLS-1$

		BalanceSheetReportModel rightDataSourceModel2 = new BalanceSheetReportModel();
		BalanceSheetLedgerEntryModel retainedEarning = balanceSheetService.getRetainedEarning();
		List<BalanceSheetLedgerEntryModel> equityList = balanceSheetService.getEquityList();
		if (equityList.isEmpty()) {
			BalanceSheetLedgerEntryModel model = new BalanceSheetLedgerEntryModel();
			model.setName("Equity");
			model.setAmount(0);
			equityList.add(model);
		}
		equityList.add(retainedEarning);
		rightDataSourceModel2.setRows(equityList);

		map.put("rightDataSource2", new JRTableModelDataSource(rightDataSourceModel2)); //$NON-NLS-1$
		map.put("rightSubReport2", equitySubreport); //$NON-NLS-1$
		map.put("colRightTotal2", "Total Equity"); //$NON-NLS-1$
		map.put("totalLiabilitiesAndEquity", "Total liabilities & equity"); //$NON-NLS-1$
		map.put("totalLiabilitiesAndEquityAmount", NumberUtil.formatToAccountingAmount(balanceSheetService.getTotalExpenseAamount()));

		JasperReport masterReport = ReportUtil.getReport(ReportConstants.BALANCE_SHEET_REPORT);
		JasperPrint print = JasperFillManager.fillReport(masterReport, map, new JREmptyDataSource());
		return print;
	}

	public static JasperPrint getTrialBalanceReportPrint(Date fromDate, Date toDate, Outlet outlet, List trialBalanceEntries) throws Exception {

		JasperReport report = ReportUtil.getReport("trial_balance_report"); //$NON-NLS-1$

		HashMap properties = new HashMap();
		ReportUtil.populateReportTime(properties, fromDate, toDate);
		ReportUtil.populateReportFooter(properties);
		ReportUtil.populateRestaurantProperties(properties, true, outlet);

		populateReportParamsForTrialBalanceReport(properties);

		TrialBalanceReportModel reportModel = new TrialBalanceReportModel();
		reportModel.setRows(trialBalanceEntries);

		JasperPrint print = JasperFillManager.fillReport(report, properties, new JRTableModelDataSource(reportModel));
		return print;
	}

	private static void populateReportParamsForTrialBalanceReport(HashMap properties) {
		properties.put("reportTitle", "Trial balance report"); //$NON-NLS-1$
		properties.put("colAccountName", "Account"); //$NON-NLS-1$
		properties.put("colDebit", "Debit"); //$NON-NLS-1$
		properties.put("colCredit", "Credit"); //$NON-NLS-1$
		properties.put("colTotal", POSConstants.TOTAL + ":"); //$NON-NLS-1$ //$NON-NLS-2$
	}

	private static String makeBoldString(String string) {
		return "<b>" + string + "</b> "; //$NON-NLS-1$ //$NON-NLS-2$
	}

	public static JasperPrint getLedgerEntryReportPrint(Date fromDate, Date toDate, Outlet outlet, List<LedgerEntry> dataList) throws Exception {

		if (dataList != null) {
			dataList.sort((o1, o2) -> o1.getAccountId().compareTo(o2.getAccountId()));
		}
		JasperReport report = ReportUtil.getReport("ledger_entry_report"); //$NON-NLS-1$

		HashMap properties = new HashMap();
		ReportUtil.populateReportTime(properties, fromDate, toDate);
		ReportUtil.populateReportFooter(properties);
		ReportUtil.populateRestaurantProperties(properties, true, outlet);

		populateReportParamsForLedgerEntryReport(properties);
		properties.put("reportDate", //$NON-NLS-1$
				makeBoldString("Report on:") + ReportService.formatFullDate(new Date())); //$NON-NLS-1$

		LedgerEntryReportModel reportModel = new LedgerEntryReportModel();
		reportModel.setRows(dataList);

		JasperPrint print = JasperFillManager.fillReport(report, properties, new JRTableModelDataSource(reportModel));
		return print;
	}

	private static void populateReportParamsForLedgerEntryReport(HashMap properties) {
		properties.put("reportTitle", "General ledger report");
		properties.put("colDate", "Date");
		properties.put("colRefNo", "Ref No");
		properties.put("colDescription", "Description");
		properties.put("colDebit", "Debit");
		properties.put("colCredit", "Credit");
		properties.put("footerNetMovement", "Net movement");
	}

}
