package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.model.Chemotherapy;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginatedListModel;
import com.orocube.medlogics.chemo.ChemotherapyStatus;

public class ChemotherapyDAO extends BaseChemotherapyDAO {

	public ChemotherapyDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Chemotherapy bean = (Chemotherapy) obj;
		if (bean == null) {
			throw new PosException("Chemotherapy not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<Chemotherapy> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific patient
	 * @param patient the patient
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific patient by ID
	 * @param patientId the patient ID
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByPatientId(String patientId) {
		if (patientId == null) {
			throw new IllegalArgumentException("Patient ID must not be null");
		}
		Customer patient = CustomerDAO.getInstance().get(patientId);
		if (patient == null) {
			throw new PosException("Patient not found with ID: " + patientId);
		}
		return findByPatient(patient);
	}

	/**
	 * Find chemotherapy records by status
	 * @param status the chemotherapy status
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByStatus(ChemotherapyStatus status) {
		if (status == null) {
			throw new IllegalArgumentException("Status must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_STATUS, status.name()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find ongoing chemotherapy records for a patient
	 * @param patient the patient
	 * @return list of ongoing chemotherapy records
	 */
	public List<Chemotherapy> findOngoingByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.add(Restrictions.eq(Chemotherapy.PROP_STATUS, ChemotherapyStatus.ONGOING.name()));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all chemotherapy records for a specific outlet
	 * @param outletId the outlet ID
	 * @return list of chemotherapy records
	 */
	public List<Chemotherapy> findByOutlet(String outletId) {
		if (outletId == null) {
			throw new IllegalArgumentException("Outlet ID must not be null");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_OUTLET_ID, outletId));
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));
			return criteria.list();
		}
	}

	/**
	 * Count total chemotherapy records for a patient
	 * @param patient the patient
	 * @return count of chemotherapy records
	 */
	public Long countByPatient(Customer patient) {
		if (patient == null || patient.getId() == null) {
			throw new IllegalArgumentException("Patient must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patient.getId()));
			criteria.setProjection(org.hibernate.criterion.Projections.rowCount());
			return (Long) criteria.uniqueResult();
		}
	}

	/**
	 * Load chemotherapy records with filtering and pagination support
	 * @param tableModel the paginated list model to populate
	 * @param chemotherapyId the chemotherapy ID filter (optional)
	 * @param doctor the doctor filter (optional)
	 * @param patientId the patient ID filter (optional)
	 * @param selectedOutletId the outlet ID filter (optional)
	 * @param fromDate the start date filter (optional)
	 * @param toDate the end date filter (optional)
	 */
	public void loadChemotherapies(PaginatedListModel<Chemotherapy> tableModel, String chemotherapyId, Doctor doctor, String patientId,
			String selectedOutletId, Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(chemotherapyId)) {
				criteria.add(Restrictions.eq(Chemotherapy.PROP_ID, chemotherapyId));
			}
			if (StringUtils.isNotBlank(selectedOutletId)) {
				criteria.add(Restrictions.eq(Chemotherapy.PROP_OUTLET_ID, selectedOutletId));
			}
			if (fromDate != null) {
				criteria.add(Restrictions.ge(Chemotherapy.PROP_CREATED_DATE, DateUtil.startOfDay(fromDate)));
			}
			if (toDate != null) {
				criteria.add(Restrictions.le(Chemotherapy.PROP_CREATED_DATE, DateUtil.endOfDay(toDate)));
			}
			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patientId));
			}
			if (doctor != null) {
				criteria.add(Restrictions.eq(Chemotherapy.PROP_DOCTOR_ID, doctor.getId()));
			}

			// Set total row count
			Criteria countCriteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(countCriteria);
			if (StringUtils.isNotBlank(chemotherapyId)) {
				countCriteria.add(Restrictions.eq(Chemotherapy.PROP_ID, chemotherapyId));
			}
			if (StringUtils.isNotBlank(selectedOutletId)) {
				countCriteria.add(Restrictions.eq(Chemotherapy.PROP_OUTLET_ID, selectedOutletId));
			}
			if (fromDate != null) {
				countCriteria.add(Restrictions.ge(Chemotherapy.PROP_CREATED_DATE, DateUtil.startOfDay(fromDate)));
			}
			if (toDate != null) {
				countCriteria.add(Restrictions.le(Chemotherapy.PROP_CREATED_DATE, DateUtil.endOfDay(toDate)));
			}
			if (StringUtils.isNotBlank(patientId)) {
				countCriteria.add(Restrictions.eq(Chemotherapy.PROP_PATIENT_ID, patientId));
			}
			if (doctor != null) {
				countCriteria.add(Restrictions.eq(Chemotherapy.PROP_DOCTOR_ID, doctor.getId()));
			}
			countCriteria.setProjection(Projections.rowCount());
			Long count = (Long) countCriteria.uniqueResult();
			tableModel.setNumRows(count != null ? count.intValue() : 0);

			// Project only necessary fields for performance
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.property(Chemotherapy.PROP_ID), Chemotherapy.PROP_ID);
			projectionList.add(Projections.property(Chemotherapy.PROP_PATIENT_ID), Chemotherapy.PROP_PATIENT_ID);
			projectionList.add(Projections.property(Chemotherapy.PROP_DOCTOR_ID), Chemotherapy.PROP_DOCTOR_ID);
			projectionList.add(Projections.property(Chemotherapy.PROP_CREATED_DATE), Chemotherapy.PROP_CREATED_DATE);
			projectionList.add(Projections.property(Chemotherapy.PROP_OUTLET_ID), Chemotherapy.PROP_OUTLET_ID);
			projectionList.add(Projections.property(Chemotherapy.PROP_STATUS), Chemotherapy.PROP_STATUS);
			projectionList.add(Projections.property(Chemotherapy.PROP_CHEMOTHERAPY_NAME), Chemotherapy.PROP_CHEMOTHERAPY_NAME);
			projectionList.add(Projections.property(Chemotherapy.PROP_PROPERTIES), Chemotherapy.PROP_PROPERTIES);
			projectionList.add(Projections.property(Chemotherapy.PROP_NUMBER_OF_CYCLES), Chemotherapy.PROP_NUMBER_OF_CYCLES);
			projectionList.add(Projections.property(Chemotherapy.PROP_NUMBER_OF_INITIAL_CYCLES), Chemotherapy.PROP_NUMBER_OF_INITIAL_CYCLES);
			criteria.setProjection(projectionList);

			// Order by creation date descending
			criteria.addOrder(Order.desc(Chemotherapy.PROP_CREATED_DATE));

			// Apply pagination
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.setResultTransformer(Transformers.aliasToBean(Chemotherapy.class));

			tableModel.setData(criteria.list());
		}
	}
}
