package com.floreantpos.model.dao;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.DocumentMeta;

public class DocumentMetaDAO extends BaseDocumentMetaDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public DocumentMetaDAO() {
	}

	@Override
	public List<DocumentMeta> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(DocumentMeta.class);
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		DocumentMeta documentMeta = (DocumentMeta) obj;
		if (documentMeta == null) {
			throw new PosException("Document meta data not found!");
		}

		documentMeta.setDeleted(Boolean.TRUE);
		saveOrUpdate(documentMeta, session);
	}

	/**
	 * Find all document metadata by entity
	 * @param entityModel the entity model type
	 * @param entityId the entity ID
	 * @return list of document metadata
	 */
	public List<DocumentMeta> findByEntity(String entityModel, String entityId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());

			if (StringUtils.isNotBlank(entityModel)) {
				criteria.add(Restrictions.eq(DocumentMeta.PROP_ENTITY_MODEL, entityModel));
			}
			if (StringUtils.isNotBlank(entityId)) {
				criteria.add(Restrictions.eq(DocumentMeta.PROP_ENTITY_ID, entityId));
			}

			criteria.add(Restrictions.or(Restrictions.isNull(DocumentMeta.PROP_DELETED), Restrictions.eq(DocumentMeta.PROP_DELETED, false)));

			criteria.addOrder(Order.desc(DocumentMeta.PROP_PERFORM_DATE));

			return criteria.list();
		}
	}

	/**
	 * Find all document metadata by document ID
	 * @param documentId the document ID
	 * @return list of document metadata
	 */
	public List<DocumentMeta> findByDocumentId(String documentId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());

			if (StringUtils.isNotBlank(documentId)) {
				criteria.add(Restrictions.eq(DocumentMeta.PROP_DOCUMENT_ID, documentId));
			}

			criteria.add(Restrictions.or(Restrictions.isNull(DocumentMeta.PROP_DELETED), Restrictions.eq(DocumentMeta.PROP_DELETED, false)));

			criteria.addOrder(Order.desc(DocumentMeta.PROP_PERFORM_DATE));

			return criteria.list();
		}
	}

}
