package com.floreantpos.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;
import org.hibernate.Transaction;

import com.floreantpos.PosLog;
import com.floreantpos.model.Agent;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.Allergy;
import com.floreantpos.model.AllergyCategory;
import com.floreantpos.model.Department;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Gender;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.InventoryUnitGroup;
import com.floreantpos.model.InventoryVendor;
import com.floreantpos.model.InventoryVendorItems;
import com.floreantpos.model.MenuCategory;
import com.floreantpos.model.MenuGroup;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.RecepieItem;
import com.floreantpos.model.Specimen;
import com.floreantpos.model.TestItem;
import com.floreantpos.model.TestItemGroup;
import com.floreantpos.model.base.BaseMenuItem;
import com.floreantpos.model.base.BaseRecepieItem;
import com.floreantpos.model.dao.AgentDAO;
import com.floreantpos.model.dao.AllergyCategoryDAO;
import com.floreantpos.model.dao.AllergyDAO;
import com.floreantpos.model.dao.DepartmentDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.InventoryUnitGroupDAO;
import com.floreantpos.model.dao.InventoryVendorDAO;
import com.floreantpos.model.dao.InventoryVendorItemsDAO;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.MenuGroupDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.PatientDAO;
import com.floreantpos.model.dao.RecepieDAO;
import com.floreantpos.model.dao.RecepieItemDAO;
import com.floreantpos.model.dao.SpecimenDAO;
import com.floreantpos.model.dao.TestItemDAO;
import com.floreantpos.model.dao.TestItemGroupDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.ReferralCommissionType;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.FieldNamingStrategy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import com.orocube.rest.service.JsonBeanFieldExcluder;

public class TemplateData {
	private static File[] jsonFiles;

	public void dataCopyFromTemplateJson(final String jsonPath) {
		try {
			if (jsonFiles == null) {
				jsonFiles = new File(jsonPath).listFiles((dir, name) -> name.endsWith(".json"));
				if (jsonFiles == null || jsonFiles.length < 1) {
					throw new Exception("Json file not found.");
				}
			}

			doStartCopyFromJson();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	enum JsonFileAndModelClass {
		// @formatter:off
		ALLERGY_CATEGORY("allergy_category.json", AllergyCategory.class),
		ALLERGY_ITEM("allergy.json", Allergy.class),
		TEST_ITEM_GROUP("test_item_group.json", TestItemGroup.class),
		TEST_ITEM("test_item.json", TestItem.class),
		SPECIMEN("specimen.json", Specimen.class),
		MENU_CATEGORY("menu_category.json", MenuCategory.class),
		MENU_GROUP("menu_group.json", MenuGroup.class),
		INVENTORY_UNIT_GROUP("inventory_unit_group.json", InventoryUnitGroup.class),
		INVENTORY_UNIT("inventory_unit.json", InventoryUnit.class),
		MENU_ITEM("menu_item.json", MenuItem.class),
		INVENTORY_VENDOR("inventory_vendor.json", InventoryVendor.class),
		INVENTORY_VENDOR_ITEMS("inventory_vendor_items.json", InventoryVendorItems.class),
		RECIPE("recipe.json", Recepie.class),
		RECIPE_ITEM("recipe_item.json", RecepieItem.class),
		DEPARTMENT("department.json", Department.class);
		// @formatter:on

		final String jsonFileName;
		final Class<?> clazz;

		JsonFileAndModelClass(String jsonFileName, Class<?> clazz) {
			this.jsonFileName = jsonFileName;
			this.clazz = clazz;
		}
	}

	private void doStartCopyFromJson() throws Exception {
		Transaction transaction = null;

		try (Session session = GenericDAO.getInstance().createNewSession()) {

			for (JsonFileAndModelClass entry : JsonFileAndModelClass.values()) {
				String jsonFileName = entry.jsonFileName;
				Class<?> modelClass = entry.clazz;

				Optional<File> targetFile = getJsonFile(jsonFileName);
				if (!targetFile.isPresent()) {
					continue;
				}

				PosLog.debug(modelClass, "import from json");

				GsonBuilder gsonBuilder = new GsonBuilder().setFieldNamingStrategy(new FieldNameMapping())
						.setExclusionStrategies(new JsonBeanFieldExcluder(MenuShift.class));

				if (modelClass == InventoryVendorItems.class) {
					gsonBuilder.registerTypeAdapter(MenuItem.class, new MenuItemAdapter());
					gsonBuilder.registerTypeAdapter(InventoryVendor.class, new VendorAdapter());
				}
				else if (modelClass == RecepieItem.class) {
					gsonBuilder.registerTypeAdapter(MenuItem.class, new MenuItemAdapter());
					gsonBuilder.registerTypeAdapter(Recepie.class, new RecepieAdapter());
				}
				else if (modelClass == Recepie.class) {
					gsonBuilder.registerTypeAdapter(MenuItem.class, new MenuItemAdapter());
				}

				Type type = createType(entry);

				Gson gson = gsonBuilder.create();

				try (InputStream inputStream = new FileInputStream(targetFile.get())) {

					String json = IOUtils.toString(inputStream, "UTF-8");

					Map<String, ?> objectMap = gson.fromJson(json, type);
					for (Object object : objectMap.values()) {
						transaction = session.beginTransaction();
						if (object == null) {
							continue;
						}
						save(entry, object, session);
						transaction.commit();
						PosLog.debug(entry.clazz, "Saved successfully");
					}
				}
			}

			transaction = session.beginTransaction();
			saveDemoPatient_DemoAgent_DemoDoctor(session);
			transaction.commit();

		} catch (Exception e) {
			if (transaction != null) {
				transaction.rollback();
			}
			throw e;
		}
	}

	class FieldNameMapping implements FieldNamingStrategy {

		@Override
		public String translateName(Field f) {
			String name = f.getName();

			switch (name) {
				case "version":
					return "version_no";

				case "lastUpdateTime":
					return "last_update_time";

				case "lastSyncTime":
					return "last_sync_time";

				case "baseUnit":
					return "base_unit";

				case "conversionRate":
					return "conversion_rate";

				case "unitGroupId":
					return "unit_group_id";

				case "outletId":
					return "outlet_id";

				case "sortOrder":
					return "sort_order";

				case "normalValue":
					return "normal_value";

				case "approvedNote":
					return "approved_note";

				case "testItemGroupId":
					return "test_item_group_id";

				case "unitCode":
					return "unit_code";

				case "inventoryDeductable":
					return "inventory_deductable";

				case "inventoryItem":
					if (f.getDeclaringClass() == BaseRecepieItem.class) {
						//recipe_item
						return "inventory_item";
					}
					else if (f.getDeclaringClass() == BaseMenuItem.class) {
						//menu_item
						return "inventory";
					}

				case "recepie":
					return "recepie_id";

				case "yieldUnit":
					return "yield_unit";

				case "portionUnit":
					return "portion_unit";

				case "adjustmentAmount":
					return "adjustment_amount";

				case "laborCost":
					return "labor_cost";

				case "batchProcess":
					return "batch_process";

				case "cookingTime":
					return "cooking_time";

				case "menuItem":
					return "menu_item_id";

				case "translatedName":
					return "translated_name";

				case "buttonColorCode":
					return "btn_color";

				case "textColorCode":
					return "text_color";

				case "onlineOrderingEnable":
					return "online_enabled";

				case "cloudSynced":
					return "cloud_synced";

				case "hasSyncError":
					return "has_sync_error";

				case "menuCategoryId":
					return "category_id";

				case "menuCategoryName":
					return "category_name";

				case "unitName":
					return "unit_name";

				case "brandId":
					return "brand_id";

				case "discountRate":
					return "discount_rate";

				case "rawMaterial":
					return "raw_material";

				case "averageUnitPurchasePrice":
					return "avg_unit_purchase_price";

				case "reorderLevel":
					return "reorder_level";

				case "replenishLevel":
					return "replenish_level";

				case "disableWhenStockAmountIsZero":
					return "disable_when_stock_amount_is_zero";

				case "shouldPrintToKitchen":
					return "print_to_kitchen";

				case "printKitchenSticker":
					return "print_kitchen_sticker";

				case "hasVariant":
					return "has_variant";

				case "variant":
					return "is_variant";

				case "service":
					return "is_service";

				case "serviceType":
					return "service_type";

				case "imageId":
					return "image_id";

				case "showImageOnly":
					return "show_image_only";

				case "fractionalUnit":
					return "fractional_unit";

				case "pizzaType":
					return "pizza_type";

				case "defaultSellPortion":
					return "default_sell_portion";

				case "lastPurchasedCost":
					return "last_purchase_cost";

				case "avgCost":
					return "avg_cost";

				case "serviceCharge":
					return "service_charge";

				case "serviceChargeApplicable":
					return "service_charge_applicable";

				case "taxOnServiceCharge":
					return "tax_on_service_charge";

				case "discountApplicable":
					return "discount_applicable";

				case "ticketDiscountApplicable":
					return "ticket_discount_applicable";

				case "serviceChargeRefundable":
					return "service_charge_refundable";

				case "menuGroupId":
					return "group_id";

				case "menuGroupName":
					return "group_name";

				case "courseId":
					return "course_id";

				case "unitId":
					return "unit_id";

				case "taxGroupId":
					return "tax_group_id";

				case "printerGroupId":
					return "pg_id";

				case "reportGroupId":
					return "report_group_id";

				case "parentMenuItemId":
					return "parent_menu_item_id";

				case "hasModifiers":
					return "has_modifiers";

				case "hasMandatoryModifiers":
					return "has_mandatory_modifiers";

				case "propertiesJson":
					return "properties_json";

				case "defaultRecipeId":
					return "default_recipe_id";

				case "editablePrice":
					return "editable_price";

				case "enableOnlineOrdering":
					return "enable_online_ordering";

				case "productType":
					return "product_type";

				case "specimenId":
					return "specimen_id";

				case "canBeSold":
					return "can_be_sold";

				case "canBePurchased":
					return "can_be_purchased";

				case "canBeRented":
					return "can_be_rented";

				case "digitalProduct":
					return "digital_product";

				case "canBeDownloaded":
					return "can_be_downloaded";

				case "canBeShipped":
					return "can_be_shipped";

				case "machineCode":
					return "machine_code";

				case "categoryType":
					return "category_type";

				case "allergyCategoryId":
					return "category_id";

				default:
					return name;
			}
		}

	}

	class MenuItemAdapter implements JsonDeserializer<MenuItem> {

		@Override
		public MenuItem deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
			MenuItem item = new MenuItem();
			item.setId(json.getAsString());
			return item;
		}

	}

	class VendorAdapter implements JsonDeserializer<InventoryVendor> {

		@Override
		public InventoryVendor deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
			InventoryVendor item = new InventoryVendor();
			item.setId(json.getAsString());
			return item;
		}

	}

	class RecepieAdapter implements JsonDeserializer<Recepie> {

		@Override
		public Recepie deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
			Recepie item = new Recepie();
			item.setId(json.getAsString());
			return item;
		}

	}

	private void saveDemoPatient_DemoAgent_DemoDoctor(Session session) {
		Patient patient = new Patient();
		patient.setOutletId(DataProvider.get().getOutletId());
		patient.setFirstName("Demo patient");
		patient.setGender(Gender.Male.name()); 
		patient.setMobileNo("0123456789");
		patient.setEmail("example@mail.com");
		patient.setDateOfBirth(new Date(1217855559000L));

		Doctor doctor = new Doctor();
		doctor.setOutletId(DataProvider.get().getOutletId());
		doctor.setFirstName("Demo doctor");
		doctor.setMobileNo("0123456789");

		Agent agent = new Agent();
		agent.setOutletId(DataProvider.get().getOutletId());
		agent.setFirstName("Demo agent");
		agent.setAgentType(AgentTypeEnum.INDOOR_AGENT.name());
		agent.putRfOnReportType(ReferralCommissionType.PERCENTAGE.name());
		agent.putRfRateOnReport(NumberUtil.formatAmount(10));

		if (PatientDAO.getInstance().findByPhoneOrName(patient.getFirstName(), session) == null) {
			PatientDAO.getInstance().save(patient, session);
			AgentDAO.getInstance().save(agent, session);
			doctor.setDoctorAgentId(agent.getId());
			DoctorDAO.getInstance().save(doctor, session);
		}
	}

	@SuppressWarnings("unchecked")
	private void save(JsonFileAndModelClass entry, Object object, final Session session) throws Exception {
		switch (entry) {
			case ALLERGY_CATEGORY:
				PosLog.debug(entry.clazz, "Saving...");
				List<AllergyCategory> allergyCategories = (List<AllergyCategory>) object;
				for (AllergyCategory category : allergyCategories) {
					if (StringUtils.isNotBlank(category.getId())) {
						AllergyCategory allergyCategory = AllergyCategoryDAO.getInstance().get(category.getId(), session);

						if (allergyCategory == null) {
							session.save(entry.clazz.getSimpleName(), category);
						}
						else {
							session.evict(allergyCategory);
							session.update(entry.clazz.getSimpleName(), category);
						}
					}
				}
				break;
			case ALLERGY_ITEM:
				PosLog.debug(entry.clazz, "Saving...");
				List<Allergy> allergies = (List<Allergy>) object;
				for (Allergy allergy : allergies) {
					if (StringUtils.isNotBlank(allergy.getId())) {
						Allergy allergyItem = AllergyDAO.getInstance().get(allergy.getId(), session);

						if (allergyItem == null) {
							session.save(entry.clazz.getSimpleName(), allergy);
						}
						else {
							session.evict(allergyItem);
							session.update(entry.clazz.getSimpleName(), allergy);
						}
					}
				}
				break;

			case TEST_ITEM_GROUP:
				PosLog.debug(entry.clazz, "Saving...");
				List<TestItemGroup> itemGroups = (List<TestItemGroup>) object;
				for (TestItemGroup testItemGroup : itemGroups) {
					if (StringUtils.isNotBlank(testItemGroup.getId())) {
						TestItemGroup group = TestItemGroupDAO.getInstance().get(testItemGroup.getId(), session);

						if (group == null) {
							session.save(entry.clazz.getSimpleName(), testItemGroup);
						}
						else {
							session.evict(group);
							session.update(entry.clazz.getSimpleName(), testItemGroup);
						}
					}
				}
				break;

			case TEST_ITEM:
				PosLog.debug(entry.clazz, "Saving...");
				List<TestItem> testItems = (List<TestItem>) object;
				for (TestItem testItem : testItems) {
					if (StringUtils.isNotBlank(testItem.getId())) {
						TestItem t_item = TestItemDAO.getInstance().get(testItem.getId(), session);

						if (t_item == null) {
							session.save(entry.clazz.getSimpleName(), testItem);
						}
						else {
							session.evict(t_item);
							session.update(entry.clazz.getSimpleName(), testItem);
						}
					}
				}
				break;

			case SPECIMEN:
				PosLog.debug(entry.clazz, "Saving...");
				List<Specimen> specimens = (List<Specimen>) object;
				for (Specimen specimen : specimens) {
					if (StringUtils.isNotBlank(specimen.getName())) {
						Specimen specimen_ = SpecimenDAO.getInstance().findByName(specimen.getName(), session);

						if (specimen_ == null) {
							session.save(entry.clazz.getSimpleName(), specimen);
						}
						else {
							session.evict(specimen_);
							session.update(entry.clazz.getSimpleName(), specimen);
						}
					}
				}
				break;

			case MENU_CATEGORY:
				PosLog.debug(entry.clazz, "Saving...");
				List<MenuCategory> menuCategories = (List<MenuCategory>) object;
				for (MenuCategory menuCategory : menuCategories) {
					if (StringUtils.isNotBlank(menuCategory.getId())) {
						MenuCategory category = MenuCategoryDAO.getInstance().get(menuCategory.getId(), session);

						if (category == null) {
							session.save(entry.clazz.getSimpleName(), menuCategory);
						}
						else {
							session.evict(category);
							menuCategory.setVersion(category.getVersion());
							session.update(entry.clazz.getSimpleName(), menuCategory);
						}
					}
				}
				break;

			case MENU_GROUP:
				PosLog.debug(entry.clazz, "Saving...");
				List<MenuGroup> menuGroups = (List<MenuGroup>) object;
				for (MenuGroup menuGroup : menuGroups) {
					if (StringUtils.isNotBlank(menuGroup.getId())) {
						MenuGroup group = MenuGroupDAO.getInstance().get(menuGroup.getId(), session);

						if (group == null) {
							session.save(entry.clazz.getSimpleName(), menuGroup);
						}
						else {
							session.evict(group);
							menuGroup.setVersion(group.getVersion());
							session.update(entry.clazz.getSimpleName(), menuGroup);
						}
					}
				}
				break;

			case INVENTORY_UNIT_GROUP:
				PosLog.debug(entry.clazz, "Saving...");
				List<InventoryUnitGroup> inventoryUnitGroups = (List<InventoryUnitGroup>) object;
				for (InventoryUnitGroup inventoryUnitGroup : inventoryUnitGroups) {
					if (StringUtils.isNotBlank(inventoryUnitGroup.getId())) {
						InventoryUnitGroup group = InventoryUnitGroupDAO.getInstance().get(inventoryUnitGroup.getId(), session);

						if (group == null) {
							session.save(entry.clazz.getSimpleName(), inventoryUnitGroup);
						}
						else {
							session.evict(group);
							inventoryUnitGroup.setVersion(group.getVersion());
							session.update(entry.clazz.getSimpleName(), inventoryUnitGroup);
						}
					}
				}
				break;

			case INVENTORY_UNIT:
				PosLog.debug(entry.clazz, "Saving...");
				List<InventoryUnit> inventoryUnits = (List<InventoryUnit>) object;
				for (InventoryUnit inventoryUnit : inventoryUnits) {
					if (StringUtils.isNotBlank(inventoryUnit.getId())) {
						InventoryUnit unit = InventoryUnitDAO.getInstance().get(inventoryUnit.getId(), session);

						if (unit == null) {
							session.save(entry.clazz.getSimpleName(), inventoryUnit);
						}
						else {
							session.evict(unit);
							inventoryUnit.setVersion(unit.getVersion());
							session.update(entry.clazz.getSimpleName(), inventoryUnit);
						}
					}
				}
				break;

			case MENU_ITEM:
				PosLog.debug(entry.clazz, "Saving...");
				List<MenuItem> menuItems = (List<MenuItem>) object;
				for (MenuItem menuItem : menuItems) {
					if (StringUtils.isNotBlank(menuItem.getId())) {
						MenuItem item = MenuItemDAO.getInstance().get(menuItem.getId(), session);

						if (item == null) {
							menuItem.setTestItems(getTestItemByMenuItemId(menuItem.getId(), session));
							session.save(entry.clazz.getSimpleName(), menuItem);
						}
						else {
							session.evict(item);
							menuItem.setTestItems(getTestItemByMenuItemId(menuItem.getId(), session));
							menuItem.setVersion(item.getVersion());
							session.update(entry.clazz.getSimpleName(), menuItem);
						}
					}
				}
				break;

			case INVENTORY_VENDOR:
				PosLog.debug(entry.clazz, "Saving...");
				List<InventoryVendor> inventoryVendors = (List<InventoryVendor>) object;
				for (InventoryVendor inventoryVendor : inventoryVendors) {
					if (StringUtils.isNotBlank(inventoryVendor.getId())) {
						InventoryVendor vendor = InventoryVendorDAO.getInstance().get(inventoryVendor.getId(), session);

						if (vendor == null) {
							session.save(entry.clazz.getSimpleName(), inventoryVendor);
						}
						else {
							session.evict(vendor);
							session.update(entry.clazz.getSimpleName(), inventoryVendor);
						}
					}
				}
				break;

			case INVENTORY_VENDOR_ITEMS:
				PosLog.debug(entry.clazz, "Saving...");
				List<InventoryVendorItems> inventoryVendorItems = (List<InventoryVendorItems>) object;
				for (InventoryVendorItems inventoryVendorItem : inventoryVendorItems) {
					if (StringUtils.isNotBlank(inventoryVendorItem.getId())) {
						InventoryVendorItems v_item = InventoryVendorItemsDAO.getInstance().get(inventoryVendorItem.getId(), session);

						if (v_item == null) {
							session.save(entry.clazz.getSimpleName(), inventoryVendorItem);
						}
						else {
							session.evict(v_item);
							inventoryVendorItem.setVersion(v_item.getVersion());
							session.update(entry.clazz.getSimpleName(), inventoryVendorItem);
						}
					}
				}
				break;

			case RECIPE_ITEM:
				PosLog.debug(entry.clazz, "Saving...");
				List<RecepieItem> recepieItems = (List<RecepieItem>) object;
				for (RecepieItem recepieItem : recepieItems) {
					if (StringUtils.isNotBlank(recepieItem.getId())) {
						RecepieItem r_item = RecepieItemDAO.getInstance().get(recepieItem.getId(), session);

						if (r_item == null) {
							session.save(entry.clazz.getSimpleName(), recepieItem);
						}
						else {
							session.evict(r_item);
							recepieItem.setVersion(r_item.getVersion());
							session.update(entry.clazz.getSimpleName(), recepieItem);
						}
					}
				}
				break;

			case RECIPE:
				PosLog.debug(entry.clazz, "Saving...");
				List<Recepie> recepies = (List<Recepie>) object;
				for (Recepie recepie : recepies) {
					if (StringUtils.isNotBlank(recepie.getId())) {
						Recepie rcp = RecepieDAO.getInstance().get(recepie.getId(), session);

						if (rcp == null) {
							session.save(entry.clazz.getSimpleName(), recepie);
						}
						else {
							session.evict(rcp);
							recepie.setVersion(rcp.getVersion());
							session.update(entry.clazz.getSimpleName(), recepie);
						}
					}
				}
				break;

			case DEPARTMENT:
				PosLog.debug(entry.clazz, "Saving...");
				List<Department> departments = (List<Department>) object;
				for (Department department : departments) {
					if (StringUtils.isNotBlank(department.getId())) {
						Department dept = DepartmentDAO.getInstance().get(department.getId(), session);

						if (dept == null) {
							session.save(entry.clazz.getSimpleName(), department);
						}
						else {
							session.evict(dept);
							department.setVersion(dept.getVersion());
							session.update(entry.clazz.getSimpleName(), department);
						}
					}
				}
				break;

			default:
				break;
		}
	}

	private Type createType(JsonFileAndModelClass entry) {
		switch (entry) {
			case ALLERGY_CATEGORY:
				return new TypeToken<Map<String, List<AllergyCategory>>>() {
				}.getType();

			case ALLERGY_ITEM:
				return new TypeToken<Map<String, List<Allergy>>>() {
				}.getType();

			case TEST_ITEM_GROUP:
				return new TypeToken<Map<String, List<TestItemGroup>>>() {
				}.getType();

			case TEST_ITEM:
				return new TypeToken<Map<String, List<TestItem>>>() {
				}.getType();

			case SPECIMEN:
				return new TypeToken<Map<String, List<Specimen>>>() {
				}.getType();

			case MENU_CATEGORY:
				return new TypeToken<Map<String, List<MenuCategory>>>() {
				}.getType();

			case MENU_GROUP:
				return new TypeToken<Map<String, List<MenuGroup>>>() {
				}.getType();

			case INVENTORY_UNIT_GROUP:
				return new TypeToken<Map<String, List<InventoryUnitGroup>>>() {
				}.getType();

			case INVENTORY_UNIT:
				return new TypeToken<Map<String, List<InventoryUnit>>>() {
				}.getType();

			case MENU_ITEM:
				return new TypeToken<Map<String, List<MenuItem>>>() {
				}.getType();

			case INVENTORY_VENDOR:
				return new TypeToken<Map<String, List<InventoryVendor>>>() {
				}.getType();

			case INVENTORY_VENDOR_ITEMS:
				return new TypeToken<Map<String, List<InventoryVendorItems>>>() {
				}.getType();

			case RECIPE_ITEM:
				return new TypeToken<Map<String, List<RecepieItem>>>() {
				}.getType();

			case RECIPE:
				return new TypeToken<Map<String, List<Recepie>>>() {
				}.getType();

			case DEPARTMENT:
				return new TypeToken<Map<String, List<Department>>>() {
				}.getType();

			default:
				return null;
		}
	}

	private Optional<File> getJsonFile(final String jsonFileName) throws Exception {
		return Stream.of(jsonFiles).filter(f -> f.getName().equalsIgnoreCase(jsonFileName)).findFirst();
	}

	private List<TestItem> getTestItemByMenuItemId(String menuItemId, Session session) throws Exception {
		Optional<File> menuItemAndTestItemFile = getJsonFile("menu_item_test_item.json");
		if (!menuItemAndTestItemFile.isPresent()) {
			return null;
		}
		try (InputStream inputStream = new FileInputStream(menuItemAndTestItemFile.get())) {
			List<TestItem> testItems = null;
			String json = IOUtils.toString(inputStream);

			Map<String, List<MenuItemAndTestItemDto>> objectMap = new GsonBuilder().setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES).create()
					.fromJson(json, new TypeToken<Map<String, List<MenuItemAndTestItemDto>>>() {
					}.getType());

			Collection<List<MenuItemAndTestItemDto>> menuItemAndTestItemList = objectMap.values();

			if (menuItemAndTestItemList != null && !menuItemAndTestItemList.isEmpty()) {
				for (List<MenuItemAndTestItemDto> list : menuItemAndTestItemList) {
					List<MenuItemAndTestItemDto> collect = list.stream().filter(mtDto -> menuItemId.equals(mtDto.getMenuItemId())).collect(Collectors.toList());
					if (collect != null) {
						testItems = new ArrayList<>();
						for (MenuItemAndTestItemDto menuItemAndTestItemDto : collect) {
							TestItem testItem = session.get(TestItem.class, menuItemAndTestItemDto.getTestItemId());
							if (testItem != null) {
								testItems.add(testItem);
							}
						}
					}
				}
			}

			return testItems;
		}
	}

	class MenuItemAndTestItemDto {

		private String menuItemId;
		private String testItemId;
		private Integer sortOrder;

		MenuItemAndTestItemDto() {
		}

		public String getMenuItemId() {
			return menuItemId;
		}

		public void setMenuItemId(String menuItemId) {
			this.menuItemId = menuItemId;
		}

		public String getTestItemId() {
			return testItemId;
		}

		public void setTestItemId(String testItemId) {
			this.testItemId = testItemId;
		}

		public Integer getSortOrder() {
			return sortOrder;
		}

		public void setSortOrder(Integer sortOrder) {
			this.sortOrder = sortOrder;
		}

	}
}
