package com.orocube.cloudflare.r2;

import java.io.InputStream;
import java.util.List;
import java.util.UUID;

import com.floreantpos.PosLog;
import com.floreantpos.model.DocumentMeta;
import com.floreantpos.model.DocumentType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.StoreDAO;

/**
 * Utility class for Cloudflare R2 document management operations
 * Provides static methods for upload, download, delete, and list operations
 * No UI dependencies - pure business logic
 */
public class CloudflareR2Util {

	private static final String DOCUMENT_CATEGORY_PREFIX = "DOCUMENTS";

	/**
	 * Upload a document to R2 storage
	 *
	 * @param storeId the store/tenant id
	 * @param outletId the outlet ID
	 * @param entityId the customer ID
	 * @param customerName the customer name
	 * @param fileName the file name
	 * @param fileContent the file content as byte array
	 * @param mimeType the MIME type (optional, will be auto-detected if null)
	 * @param documentType the document type
	 * @param description optional description
	 * @param performer the User
	 * @return DocumentMeta object with upload details
	 * @throws RuntimeException if upload fails
	 */
	public static DocumentMeta uploadDocument(String storeId, String outletId, String entityModel, String entityId, String entityName, String customerName,
			String fileName, byte[] fileContent, String mimeType, DocumentType documentType, String description, User performer) {

		try {
			R2StorageService r2StorageService = R2StorageService.getInstance();
			String documentId = UUID.randomUUID().toString();

			// Build R2 category path: PATIENT_DOCUMENTS/{storeName}/{customerId}/{documentId}
			String category = buildCategoryPath(storeId, entityModel, entityId, documentId);

			PosLog.debug(CloudflareR2Util.class, "Upload request - fileName: " + fileName + ", mimeType: " + mimeType + ", documentType: " + documentType);

			// Auto-detect content type if not provided
			if (mimeType == null || mimeType.isEmpty()) {
				mimeType = R2HelperService.detectContentType(fileName);
				PosLog.debug(CloudflareR2Util.class, "Auto-detected mimeType: " + mimeType + " for fileName: " + fileName);
			}

			// Upload to R2
			R2UploadResponse uploadResponse = r2StorageService.uploadFile(fileContent, fileName, mimeType, category);

			if (!uploadResponse.isSuccess()) {
				throw new RuntimeException("R2 upload failed: " + uploadResponse.getErrorMessage());
			}

			// Build metadata
			DocumentMeta metadata = new DocumentMeta();
			metadata.setDocumentId(documentId);
			metadata.setR2Key(uploadResponse.getR2Key());
			metadata.setFileName(fileName);
			metadata.setMimeType(mimeType);
			metadata.setFileSize(uploadResponse.getFileSize());
			metadata.setDocumentTypeName(documentType.name());
			metadata.setEntityModel(entityModel);
			metadata.setEntityId(entityId);
			metadata.setEntityName(entityName);
			metadata.setOutletId(outletId);
			if (performer != null) {
				metadata.setPerformerId(performer.getId());
				metadata.setPerformerName(performer.getFullName());
			}
			metadata.setPerformDate(uploadResponse.getUploadedDate() != null ? uploadResponse.getUploadedDate() : StoreDAO.getServerTimestamp());
			metadata.setDescription(description);
			metadata.setFilePath(category);

			PosLog.debug(CloudflareR2Util.class, "Document uploaded successfully to R2: " + fileName + " for customer " + entityId);

			return metadata;

		} catch (Exception e) {
			PosLog.error(CloudflareR2Util.class, e);
			throw new RuntimeException("Failed to upload document: " + e.getMessage(), e);
		}
	}

	/**
	 * Download a document from R2 storage by R2 key
	 *
	 * @param r2Key the R2 key
	 * @return InputStream of the document content
	 * @throws RuntimeException if download fails
	 */
	public static InputStream downloadDocument(String r2Key) {
		try {
			R2StorageService r2StorageService = R2StorageService.getInstance();
			return r2StorageService.downloadFileAsStream(r2Key);
		} catch (Exception e) {
			PosLog.error(CloudflareR2Util.class, "Failed to download document from R2 by key: " + r2Key, e);
			throw new RuntimeException("Failed to download document: " + e.getMessage(), e);
		}
	}

	/**
	 * Download document by metadata
	 *
	 * @param metadata the document metadata
	 * @return InputStream of the document content
	 * @throws RuntimeException if download fails
	 */
	public static InputStream downloadDocument(DocumentMeta metadata) {
		if (metadata == null || metadata.getR2Key() == null) {
			throw new IllegalArgumentException("Invalid metadata or R2 key is null");
		}
		return downloadDocument(metadata.getR2Key());
	}

	/**
	 * Delete a document from R2 storage by R2 key
	 *
	 * @param r2Key the R2 key
	 * @return true if deleted successfully, false otherwise
	 * @throws RuntimeException if delete fails
	 */
	public static boolean deleteDocument(String r2Key) {
		try {
			R2StorageService r2StorageService = R2StorageService.getInstance();
			boolean deleted = r2StorageService.deleteFile(r2Key);

			if (deleted) {
				PosLog.debug(CloudflareR2Util.class, "Document deleted successfully from R2: " + r2Key);
			}
			else {
				PosLog.debug(CloudflareR2Util.class, "Failed to delete file from R2: " + r2Key);
			}

			return deleted;
		} catch (Exception e) {
			PosLog.error(CloudflareR2Util.class, "Failed to delete document from R2: " + r2Key, e);
			throw new RuntimeException("Failed to delete document: " + e.getMessage(), e);
		}
	}

	/**
	 * Delete document by metadata
	 *
	 * @param metadata the document metadata
	 * @return true if deleted successfully, false otherwise
	 * @throws RuntimeException if delete fails
	 */
	public static boolean deleteDocument(DocumentMeta metadata) {
		if (metadata == null || metadata.getR2Key() == null) {
			throw new IllegalArgumentException("Invalid metadata or R2 key is null");
		}
		return deleteDocument(metadata.getR2Key());
	}

	/**
	 * List all documents for a customer in R2 storage
	 *
	 * @param storeId the store/tenant id
	 * @param entityId the customer ID
	 * @return List of R2StorageMetadata for the customer's documents
	 */
	public static List<R2StorageMetadata> listCustomerDocuments(String storeId, String entityModel, String entityId) {
		try {
			R2StorageService r2StorageService = R2StorageService.getInstance();
			String prefix = buildCustomerPrefix(storeId, entityModel, entityId);

			List<R2StorageMetadata> files = r2StorageService.listFiles(prefix);
			PosLog.debug(CloudflareR2Util.class, "Found " + files.size() + " documents for customer " + entityId + " in store " + storeId);

			return files;
		} catch (Exception e) {
			PosLog.error(CloudflareR2Util.class, "Failed to list customer documents", e);
			throw new RuntimeException("Failed to list documents: " + e.getMessage(), e);
		}
	}

	/**
	 * Generate presigned URL for temporary access to a document
	 *
	 * @param r2Key the R2 key
	 * @param expirationMinutes expiration time in minutes
	 * @return presigned URL
	 * @throws RuntimeException if URL generation fails
	 */
	public static String generatePresignedUrl(String r2Key, int expirationMinutes) {
		try {
			R2StorageService r2StorageService = R2StorageService.getInstance();
			return r2StorageService.generatePresignedUrl(r2Key, expirationMinutes);
		} catch (Exception e) {
			PosLog.error(CloudflareR2Util.class, "Failed to generate presigned URL for key: " + r2Key, e);
			throw new RuntimeException("Failed to generate presigned URL: " + e.getMessage(), e);
		}
	}

	/**
	 * Generate presigned URL from metadata
	 *
	 * @param metadata the document metadata
	 * @param expirationMinutes expiration time in minutes
	 * @return presigned URL
	 * @throws RuntimeException if URL generation fails
	 */
	public static String generatePresignedUrl(DocumentMeta metadata, int expirationMinutes) {
		if (metadata == null || metadata.getR2Key() == null) {
			throw new IllegalArgumentException("Invalid metadata or R2 key is null");
		}
		return generatePresignedUrl(metadata.getR2Key(), expirationMinutes);
	}

	/**
	 * Build R2 category path for a document
	 * Format: PATIENT_DOCUMENTS/{storeName}/{customerId}/{documentId}
	 *
	 * @param tenantId the store/tenant name
	 * @param entityId the customer ID
	 * @param documentId the document ID
	 * @param entityModel 
	 * @return category path
	 */
	public static String buildCategoryPath(String tenantId, String entityModel, String entityId, String documentId) {
		return DOCUMENT_CATEGORY_PREFIX + "/" + tenantId + "/" + entityModel + "/" + entityId + "/" + documentId;
	}

	/**
	 * Build R2 prefix for customer documents
	 * Format: PATIENT_DOCUMENTS/{storeName}/{customerId}/
	 *
	 * @param storeName the store/tenant name
	 * @param customerId the customer ID
	 * @return prefix path
	 */
	public static String buildCustomerPrefix(String storeName, String entityModel, String entityId) {
		return DOCUMENT_CATEGORY_PREFIX + "/" + storeName + "/" + entityModel + "/" + entityId + "/";
	}
}
