package com.orocube.cloudflare.r2;

import java.util.UUID;

/**
 * Utility class for R2 storage operations.
 * Provides helper methods for file handling and key generation.
 */
public class R2HelperService {

	private R2HelperService() {
		// Utility class - private constructor
	}

	/**
	 * Extract file extension from filename.
	 *
	 * @param fileName file name
	 * @return file extension (without dot) or empty string if no extension
	 */
	public static String extractFileExtension(String fileName) {
		if (fileName == null || fileName.isEmpty()) {
			return "";
		}
		int lastDotIndex = fileName.lastIndexOf('.');
		if (lastDotIndex > 0 && lastDotIndex < fileName.length() - 1) {
			return fileName.substring(lastDotIndex + 1).toLowerCase();
		}
		return "";
	}

	/**
	 * Detect content type based on file extension.
	 *
	 * @param fileName file name
	 * @return content type (MIME type)
	 */
	public static String detectContentType(String fileName) {
		String extension = extractFileExtension(fileName);
		switch (extension) {
			case "jpg":
			case "jpeg":
				return "image/jpeg";
			case "png":
				return "image/png";
			case "gif":
				return "image/gif";
			case "bmp":
				return "image/bmp";
			case "webp":
				return "image/webp";
			case "pdf":
				return "application/pdf";
			case "txt":
				return "text/plain";
			case "html":
				return "text/html";
			case "json":
				return "application/json";
			case "xml":
				return "application/xml";
			case "zip":
				return "application/zip";
			case "csv":
				return "text/csv";
			default:
				return "application/octet-stream";
		}
	}

	/**
	 * Check if file is a valid image file based on extension.
	 *
	 * @param fileName file name
	 * @return true if image file, false otherwise
	 */
	public static boolean isValidImageFile(String fileName) {
		String extension = extractFileExtension(fileName);
		return extension.equals("jpg") || extension.equals("jpeg") ||
		       extension.equals("png") || extension.equals("gif") ||
		       extension.equals("bmp") || extension.equals("webp");
	}

	/**
	 * Format R2 URL from bucket name and R2 key.
	 *
	 * @param r2Key R2 object key
	 * @return formatted URL
	 */
	public static String formatR2Url(String r2Key) {
		if (r2Key == null || r2Key.isEmpty()) {
			return "";
		}
		String bucketName = R2Config.getBucketName();
		String endpointUrl = R2Config.getEndpointUrl();
		if (endpointUrl != null && endpointUrl.endsWith("/")) {
			endpointUrl = endpointUrl.substring(0, endpointUrl.length() - 1);
		}
		return endpointUrl + "/" + bucketName + "/" + r2Key;
	}

	/**
	 * Generate unique R2 key for file storage.
	 * Format: {category}/{uuid}.{extension}
	 *
	 * @param category file category (e.g., "PRODUCTS", "SIGNATURE")
	 * @param fileName original file name
	 * @return unique R2 key
	 */
	public static String generateUniqueKey(String category, String fileName) {
		String extension = extractFileExtension(fileName);
		String uuid = UUID.randomUUID().toString();

		if (category != null && !category.isEmpty()) {
			if (extension != null && !extension.isEmpty()) {
				return category + "/" + uuid + "." + extension;
			}
			return category + "/" + uuid;
		}

		if (extension != null && !extension.isEmpty()) {
			return uuid + "." + extension;
		}
		return uuid;
	}

	/**
	 * Sanitize file name to be safe for storage.
	 * Removes special characters and spaces.
	 *
	 * @param fileName original file name
	 * @return sanitized file name
	 */
	public static String sanitizeFileName(String fileName) {
		if (fileName == null || fileName.isEmpty()) {
			return "unnamed";
		}
		// Replace spaces and special characters with underscores
		return fileName.replaceAll("[^a-zA-Z0-9._-]", "_");
	}

	/**
	 * Convert file size in bytes to human-readable format.
	 *
	 * @param bytes file size in bytes
	 * @return formatted file size (e.g., "1.5 MB")
	 */
	public static String formatFileSize(long bytes) {
		if (bytes < 1024) {
			return bytes + " B";
		} else if (bytes < 1024 * 1024) {
			return String.format("%.2f KB", bytes / 1024.0);
		} else if (bytes < 1024 * 1024 * 1024) {
			return String.format("%.2f MB", bytes / (1024.0 * 1024.0));
		} else {
			return String.format("%.2f GB", bytes / (1024.0 * 1024.0 * 1024.0));
		}
	}
}
