package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.Consent;
import com.floreantpos.model.InvoiceTemplate;
import com.floreantpos.model.InvoiceTemplateType;
import com.floreantpos.swing.PaginatedListModel;

public class InvoiceTemplateDAO extends BaseInvoiceTemplateDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InvoiceTemplateDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	public void delete(Object obj, Session s) throws HibernateException {
		InvoiceTemplate template = (InvoiceTemplate) obj;
		template.setDeleted(true);
		update(template, s);
	}

	public InvoiceTemplate findInvoiceTemplateByName(String templateName, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InvoiceTemplate.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(InvoiceTemplate.PROP_TEMPLATE_NAME, templateName));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_OUTLET_ID, outletId));
			}
			List list = criteria.list();
			if (!list.isEmpty()) {

				return (InvoiceTemplate) list.get(0);
			}
		}
		return null;
	}

	public void findAll(InvoiceTemplateType type, String searchTemplate, PaginatedListModel<InvoiceTemplate> dataModel, String outletId) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(InvoiceTemplate.class);
			this.addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_OUTLET_ID, outletId));
			}
			if (StringUtils.isNotBlank(searchTemplate)) {
				criteria.add(Restrictions.ilike(InvoiceTemplate.PROP_TEMPLATE_NAME, searchTemplate, MatchMode.ANYWHERE));
			}
			criteria.add(Restrictions.eq(InvoiceTemplate.PROP_TEMPLATE_TYPE, type.name()).ignoreCase());

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());

			criteria.addOrder(Order.desc(Consent.PROP_CREATE_DATE));
			List<InvoiceTemplate> list = criteria.list();

			dataModel.setData(list);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public InvoiceTemplate markAsDefaultTemplate(String outletId, String selectedTemplateId, InvoiceTemplateType type) {
		if (StringUtils.isBlank(selectedTemplateId)) {
			return null;
		}

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			session.createSQLQuery("UPDATE INVOICE_TEMPLATE SET DEFAULT_TEMPLATE = FALSE WHERE OUTLET_ID = :outletId AND TEMPLATE_TYPE = :type")
					.setParameter("outletId", outletId).setParameter("type", type.name()).executeUpdate();

			session.createSQLQuery("UPDATE INVOICE_TEMPLATE SET DEFAULT_TEMPLATE = TRUE WHERE ID = :templateId").setParameter("templateId", selectedTemplateId)
					.executeUpdate();

			tx.commit();

			return get(selectedTemplateId, session);
		} catch (Exception e) {
			if (tx != null)
				tx.rollback();
			PosLog.error(InvoiceTemplateDAO.class, e);
			throw e;
		}
	}

	public InvoiceTemplate getDefaultTemplate(String outletId, InvoiceTemplateType type) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InvoiceTemplate.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_OUTLET_ID, outletId));
			}

			criteria.add(Restrictions.eq(InvoiceTemplate.PROP_TEMPLATE_TYPE, type.name()).ignoreCase());
			criteria.add(Restrictions.sqlRestriction("DEFAULT_TEMPLATE = TRUE"));

			List<InvoiceTemplate> list = criteria.list();
			if (!list.isEmpty()) {
				return list.get(0);
			}
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public InvoiceTemplate findByTemplateId(String outletId, String templateId) {
		if (StringUtils.isBlank(templateId)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InvoiceTemplate.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_OUTLET_ID, outletId));
			}

			criteria.add(Restrictions.eq(InvoiceTemplate.PROP_ID, templateId));
			criteria.setMaxResults(1);

			return (InvoiceTemplate) criteria.uniqueResult();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

	public List<InvoiceTemplate> findAllInvoiceTemplates(String templateType, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(InvoiceTemplate.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_OUTLET_ID, outletId));
			}

			if (StringUtils.isNotBlank(templateType)) {
				criteria.add(Restrictions.eq(InvoiceTemplate.PROP_TEMPLATE_TYPE, templateType).ignoreCase());
			}

			criteria.addOrder(Order.desc(InvoiceTemplate.PROP_CREATE_DATE));

			return criteria.list();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return null;
	}

}
