package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;

import com.floreantpos.DuplicateDataException;
import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.Shift;

public class MenuShiftDAO extends BaseMenuShiftDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public MenuShiftDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		validateData((MenuShift) obj, s);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		validateData((MenuShift) obj, s);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		validateData((MenuShift) obj, s);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private void validateData(MenuShift menuShift, Session session) {
		GenericDAO.getInstance().checkIdOrNameExists(menuShift.getId(), menuShift.getName(), MenuShift.class, session);
		if (menuShiftExists(menuShift.getId(), menuShift.getStartHour(), menuShift.getStartMin(), menuShift.getEndHour(), menuShift.getEndMin(),
				menuShift.getDaysOfWeek(), session)) {
			throw new DuplicateDataException(Messages.getString("MenuShiftDAO.0")); //$NON-NLS-1$
		}
	}

	@Override
	public List<MenuShift> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public List<MenuShift> findByOutletId(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Shift.PROP_OUTLET_ID, outletId));
			return criteria.list();
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		ShiftDAO.getInstance().delete(obj, session);
	}

	public boolean exists(String shiftName) throws PosException {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Shift.class);
			criteria.add(Restrictions.eq(Shift.PROP_NAME, shiftName));
			List list = criteria.list();
			if (list != null && list.size() > 0) {
				return true;
			}
			return false;
		} catch (Exception e) {
			throw new PosException("An error occured while trying to check price shift duplicacy", e); //$NON-NLS-1$
		} finally {
			if (session != null) {
				try {
					session.close();
				} catch (Exception e) {
				}
			}
		}
	}

	private boolean menuShiftExists(String existingId, Integer startHour, Integer startMin, Integer endHour, Integer endMin, String daysOfWeek, Session session)
			throws PosException {

		Criteria criteria = session.createCriteria(getReferenceClass());
		if (StringUtils.isNotEmpty(existingId)) {
			SimpleExpression existingIdExp = Restrictions.ne("id", existingId); //$NON-NLS-1$
			criteria.add(existingIdExp);
		}
		addDeletedFilter(criteria);
		SimpleExpression startHourExp = Restrictions.eq(MenuShift.PROP_START_HOUR, startHour);
		SimpleExpression startMinExp = Restrictions.eq(MenuShift.PROP_START_MIN, startMin);
		SimpleExpression endHourExp = Restrictions.eq(MenuShift.PROP_END_HOUR, endHour);
		SimpleExpression endMinExp = Restrictions.eq(MenuShift.PROP_END_MIN, endMin);
		SimpleExpression daysOfWeekExp = Restrictions.eq(MenuShift.PROP_DAYS_OF_WEEK, daysOfWeek);
		Conjunction shiftTimeExp = Restrictions.and(startHourExp, startMinExp, endHourExp, endMinExp, daysOfWeekExp);
		criteria.add(shiftTimeExp);

		List list = criteria.list();
		if (list != null && list.size() > 0) {
			return true;
		}
		return false;
	}

	public List<MenuShift> findAllActive() {
		Session session = null;
		try {
			session = getSession();
			return findAllActive(session);
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<MenuShift> findAllActive(Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.add(Restrictions.eq(MenuShift.PROP_ENABLE, Boolean.TRUE));
		criteria.addOrder(Order.asc(MenuShift.PROP_PRIORITY));
		return criteria.list();
	}
}