package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.Project;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.ProjectIdGenerator;

public class ProjectDAO extends BaseProjectDAO {

	public ProjectDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		performPreSaveOperations(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		performPreSaveOperations(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		performPreSaveOperations(obj);
		super.saveOrUpdate(obj, s);
	}

	private void performPreSaveOperations(Object obj) {
		updateTime(obj);
		
		Project project = (Project) obj;
		boolean newProject = StringUtils.isEmpty(project.getId());
		if (newProject) {
			project.setId(ProjectIdGenerator.generateProjectId());
		}
	}

	public boolean projectExists(String name, String id) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne(Project.PROP_ID, id));
			}
			criteria.add(Restrictions.eq(Project.PROP_NAME, name).ignoreCase());
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			closeSession(session);
		}
	}

	@Override
	protected void delete(Object obj, Session session) {
		Project bean = (Project) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("PayoutReasonDAO.0")); //$NON-NLS-1$
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<Project> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Project.PROP_NAME).ignoreCase());
			return criteria.list();
		}
	}

	public void findByName(String projectName, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Project.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(projectName)) {
				criteria.add(Restrictions.ilike(Project.PROP_NAME, projectName, MatchMode.ANYWHERE));
			}
			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Project.PROP_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}
}