package com.orocube.medlogics.chemo;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.ChemoCycle;
import com.floreantpos.model.Chemotherapy;

/**
 * Utility class to automatically calculate and update Chemotherapy status based on ChemoCycle statuses
 */
public class ChemotherapyStatusCalculator {

	/**
	 * Calculate and return the appropriate Chemotherapy status based on cycle statuses
	 * Rules:
	 * - New chemotherapy (no ID) -> PLANNED
	 * - Cycles are sorted by start date
	 * - If no cycle has started (all SCHEDULED or no status) -> PLANNED
	 * - If first cycle is COMPLETED/SKIPPED/DELAYED and more cycles exist -> check next cycle
	 * - If all cycles are COMPLETED/SKIPPED/DELAYED -> COMPLETED
	 * - If current active cycle is IN_PROGRESS -> ONGOING
	 * - If current active cycle is SCHEDULED -> PLANNED
	 *
	 * @param chemotherapy The chemotherapy object
	 * @return The calculated ChemotherapyStatus
	 */
	public static ChemotherapyStatus calculateStatus(Chemotherapy chemotherapy) {
		if (chemotherapy == null) {
			return ChemotherapyStatus.PLANNED;
		}

		// New chemotherapy (not yet saved) -> PLANNED
		if (StringUtils.isBlank(chemotherapy.getId())) {
			return ChemotherapyStatus.PLANNED;
		}

		List<ChemoCycle> cycles = chemotherapy.getChemoCycles();

		// No cycles -> PLANNED
		if (cycles == null || cycles.isEmpty()) {
			return ChemotherapyStatus.PLANNED;
		}

		// Filter out deleted cycles
		List<ChemoCycle> activeCycles = new ArrayList<>();
		for (ChemoCycle cycle : cycles) {
			if (cycle.getDeleted() == null || !cycle.getDeleted()) {
				activeCycles.add(cycle);
			}
		}

		// No active cycles -> PLANNED
		if (activeCycles.isEmpty()) {
			return ChemotherapyStatus.PLANNED;
		}

		// Sort cycles by start date (ascending - earliest first)
		activeCycles.sort((c1, c2) -> {
			Date date1 = c1.getStartDate();
			Date date2 = c2.getStartDate();
			if (date1 == null && date2 == null) return 0;
			if (date1 == null) return 1; // null dates go to end
			if (date2 == null) return -1;
			return date1.compareTo(date2);
		});

		// Count finished cycles and find first non-finished cycle
		int totalCycles = activeCycles.size();
		int finishedCycles = 0;
		ChemoCycle firstActiveCycle = null;

		for (ChemoCycle cycle : activeCycles) {
			String statusStr = cycle.getStatus();

			// No status or blank -> consider as not finished (PLANNED)
			if (StringUtils.isBlank(statusStr)) {
				if (firstActiveCycle == null) {
					firstActiveCycle = cycle;
				}
				continue;
			}

			ChemoCycleStatus cycleStatus = ChemoCycleStatus.fromString(statusStr);

			// Invalid status -> consider as not finished
			if (cycleStatus == null) {
				if (firstActiveCycle == null) {
					firstActiveCycle = cycle;
				}
				continue;
			}

			// Check if cycle is finished (COMPLETED/SKIPPED/DELAYED)
			if (cycleStatus == ChemoCycleStatus.COMPLETED ||
				cycleStatus == ChemoCycleStatus.SKIPPED ||
				cycleStatus == ChemoCycleStatus.DELAYED) {
				finishedCycles++;
			} else {
				// First non-finished cycle found
				if (firstActiveCycle == null) {
					firstActiveCycle = cycle;
				}
			}
		}

		// All cycles finished -> COMPLETED
		if (finishedCycles == totalCycles && totalCycles > 0) {
			return ChemotherapyStatus.COMPLETED;
		}

		// No active cycle found (all scheduled or no status) -> PLANNED
		if (firstActiveCycle == null) {
			return ChemotherapyStatus.PLANNED;
		}

		// Check status of first active (non-finished) cycle
		String statusStr = firstActiveCycle.getStatus();
		if (StringUtils.isBlank(statusStr)) {
			return ChemotherapyStatus.PLANNED;
		}

		ChemoCycleStatus cycleStatus = ChemoCycleStatus.fromString(statusStr);
		if (cycleStatus == null) {
			return ChemotherapyStatus.PLANNED;
		}

		// Return status based on first active cycle
		switch (cycleStatus) {
			case IN_PROGRESS:
				return ChemotherapyStatus.ONGOING;
			case SCHEDULED:
				return ChemotherapyStatus.PLANNED;
			default:
				return ChemotherapyStatus.PLANNED;
		}
	}

	/**
	 * Update the chemotherapy status based on cycle statuses and save to database
	 * @param chemotherapy The chemotherapy object to update
	 */
	public static void updateChemotherapyStatus(Chemotherapy chemotherapy) {
		if (chemotherapy == null) {
			return;
		}

		ChemotherapyStatus calculatedStatus = calculateStatus(chemotherapy);
		chemotherapy.setStatus(calculatedStatus.name());
	}

	/**
	 * Get the display string for chemotherapy status
	 * @param chemotherapy The chemotherapy object
	 * @return The display string for the status
	 */
	public static String getStatusDisplayString(Chemotherapy chemotherapy) {
		if (chemotherapy == null || StringUtils.isBlank(chemotherapy.getStatus())) {
			return "";
		}

		ChemotherapyStatus status = ChemotherapyStatus.fromString(chemotherapy.getStatus());
		return status != null ? status.getDisplayString() : chemotherapy.getStatus();
	}

	/**
	 * Get the display string for chemotherapy status by analyzing cycles at runtime
	 * (Does not use the stored status field)
	 * @param chemotherapy The chemotherapy object with cycles loaded
	 * @return The display string for the calculated status
	 */
	public static String getStatusDisplayStringFromCycles(Chemotherapy chemotherapy) {
		if (chemotherapy == null) {
			return "";
		}

		ChemotherapyStatus status = calculateStatus(chemotherapy);
		return status != null ? status.getDisplayString() : "";
	}
}
