package com.floreantpos.model;

import java.util.ArrayList;
import java.util.List;

import javax.xml.bind.annotation.XmlRootElement;

import org.apache.commons.lang.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseChemotherapy;
import com.floreantpos.model.dao.StoreDAO;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import com.orocube.medlogics.chemo.ExtraVital;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "propertiesContainer", "patient" })
@XmlRootElement
public class Chemotherapy extends BaseChemotherapy implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private transient JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Chemotherapy() {
		setCreatedDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public Chemotherapy(java.lang.String id) {
		super(id);
		setCreatedDate(StoreDAO.getServerTimestamp());
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	@Override
	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	@Override
	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	@Override
	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	// Helper methods for Patient Vitals (List<ExtraVital>)
	public List<ExtraVital> getPatientVitalsList() {
		String vitals = super.getPatientVitals();
		if (StringUtils.isEmpty(vitals)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(vitals, new TypeToken<List<ExtraVital>>() {
		}.getType());
	}

	public void setPatientVitalsList(List<ExtraVital> vitals) {
		if (vitals == null || vitals.isEmpty()) {
			super.setPatientVitals(null);
		}
		else {
			super.setPatientVitals(new Gson().toJson(vitals));
		}
	}

	public void addPatientVital(ExtraVital vital) {
		List<ExtraVital> vitals = getPatientVitalsList();
		vitals.add(vital);
		setPatientVitalsList(vitals);
	}

	public List<PrescriptionItem> getPreChemoMedicationList() {
		String medications = super.getPreChemoMedicationJson();
		if (StringUtils.isEmpty(medications)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(medications, new TypeToken<List<PrescriptionItem>>() {
		}.getType());
	}

	public void setPreChemoMedicationList(List<PrescriptionItem> medications) {
		if (medications == null || medications.isEmpty()) {
			super.setPreChemoMedicationJson(null);
		}
		else {
			super.setPreChemoMedicationJson(new Gson().toJson(medications));
		}
	}

	public void addPreChemoMedication(PrescriptionItem medication) {
		List<PrescriptionItem> medications = getPreChemoMedicationList();
		medications.add(medication);
		setPreChemoMedicationList(medications);
	}

	// Helper methods for Chemo Medication (List<PrescriptionItem>)
	public List<PrescriptionItem> getChemoMedicationList() {
		String medications = super.getChemoMedicationJson();
		if (StringUtils.isEmpty(medications)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(medications, new TypeToken<List<PrescriptionItem>>() {
		}.getType());
	}

	public void setChemoMedicationList(List<PrescriptionItem> medications) {
		if (medications == null || medications.isEmpty()) {
			super.setChemoMedicationJson(null);
		}
		else {
			super.setChemoMedicationJson(new Gson().toJson(medications));
		}
	}

	public void addChemoMedication(PrescriptionItem medication) {
		List<PrescriptionItem> medications = getChemoMedicationList();
		medications.add(medication);
		setChemoMedicationList(medications);
	}

	// Helper methods for Post Chemo Medication (List<PrescriptionItem>)
	public List<PrescriptionItem> getPostChemoMedicationList() {
		String medications = super.getPostChemoMedicationJson();
		if (StringUtils.isEmpty(medications)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(medications, new TypeToken<List<PrescriptionItem>>() {
		}.getType());
	}

	public void setPostChemoMedicationList(List<PrescriptionItem> medications) {
		if (medications == null || medications.isEmpty()) {
			super.setPostChemoMedicationJson(null);
		}
		else {
			super.setPostChemoMedicationJson(new Gson().toJson(medications));
		}
	}

	public void addPostChemoMedication(PrescriptionItem medication) {
		List<PrescriptionItem> medications = getPostChemoMedicationList();
		medications.add(medication);
		setPostChemoMedicationList(medications);
	}

	// Helper methods for Other Associated Medication (List<PrescriptionItem>)
	public List<PrescriptionItem> getOtherAssociatedMedicationList() {
		String medications = super.getOtherAssociatedMedicationJson();
		if (StringUtils.isEmpty(medications)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(medications, new TypeToken<List<PrescriptionItem>>() {
		}.getType());
	}

	public void setOtherAssociatedMedicationList(List<PrescriptionItem> medications) {
		if (medications == null || medications.isEmpty()) {
			super.setOtherAssociatedMedicationJson(null);
		}
		else {
			super.setOtherAssociatedMedicationJson(new Gson().toJson(medications));
		}
	}

	public void addOtherAssociatedMedication(PrescriptionItem medication) {
		List<PrescriptionItem> medications = getOtherAssociatedMedicationList();
		medications.add(medication);
		setOtherAssociatedMedicationList(medications);
	}

	// Helper methods for Protocol/Regimen information
	public String getProtocolName() {
		return getProperty("protocolName", "");
	}

	public void setProtocolName(String protocolName) {
		if (StringUtils.isEmpty(protocolName)) {
			removeProperty("protocolName");
		}
		else {
			addProperty("protocolName", protocolName);
		}
	}

	public String getRegimenName() {
		return getProperty("regimenName", "");
	}

	public void setRegimenName(String regimenName) {
		if (StringUtils.isEmpty(regimenName)) {
			removeProperty("regimenName");
		}
		else {
			addProperty("regimenName", regimenName);
		}
	}

	public String getCycleName() {
		return getProperty("cycleName", "");
	}

	public void setCycleName(String cycleName) {
		if (StringUtils.isEmpty(cycleName)) {
			removeProperty("cycleName");
		}
		else {
			addProperty("cycleName", cycleName);
		}
	}

	public String getDiagnosisJson() {
		return getProperty("diagnosis_json", "");
	}

	public void putDiagnosisJson(String cycleName) {
		if (StringUtils.isEmpty(cycleName)) {
			removeProperty("diagnosis_json");
		}
		else {
			addProperty("diagnosis_json", cycleName);
		}
	}

	public List<PrescriptionItem> getDiagnosisList() {
		String medications = getDiagnosisJson();
		if (StringUtils.isEmpty(medications)) {
			return new ArrayList<>();
		}
		return new Gson().fromJson(medications, new TypeToken<List<PrescriptionItem>>() {
		}.getType());
	}

	public void setDiagnosisList(List<PrescriptionItem> medications) {
		if (medications == null || medications.isEmpty()) {
			putDiagnosisJson(null);
		}
		else {
			putDiagnosisJson(new Gson().toJson(medications));
		}
	}

	@Override
	public String toString() {
		String patientInfo = getPatientId() != null ? getPatientId() : "Unknown";
		return "Chemotherapy[" + getId() + "] - Patient ID: " + patientInfo + " - " + getStatus();
	}
}
