package com.floreantpos.report;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TransactionReason;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.dao.BalanceUpdateTransactionDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.TicketItemDAO;
import com.floreantpos.model.util.ServiceType;
import com.floreantpos.util.POSUtil;

public class SummaryReportService {

	public DailyDetailsReportData.PettyCashReceive findPettyCashBalance(Date fromDate, String outletId, boolean isOpeningBalance) {
		PosLog.info(getClass(), "findPettyCashBalance: opeinging balance startTime: " + fromDate); //$NON-NLS-1$ //$NON-NLS-2$
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			BalanceUpdateTransactionDAO.getInstance().addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outletId));
			}

			if (isOpeningBalance) {
				criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate));
			}

			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

			Double totalBalance = (Double) criteria.uniqueResult();

			DailyDetailsReportData.PettyCashReceive pettyCash = new DailyDetailsReportData.PettyCashReceive();
			pettyCash.setAccountName("Closing Balance");
			pettyCash.setAmount(totalBalance != null ? totalBalance : 0d);
			pettyCash.setRemarks("");
			pettyCash.setDate(fromDate);

			return pettyCash;
		}
	}

	public double getPettyCashReceiveAmount(Date from, Date to, Outlet outlet) {
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.ADD_BALANCE.name()));

			if (from != null && to != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, from, to));
			}
			else if (from != null) {
				criteria.add(Restrictions.ge(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, from));
			}
			else if (to != null) {
				criteria.add(Restrictions.le(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, to));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return ((Number) uniqueResult).doubleValue();
			}
		}
		return 0.0;
	}

	public List<DailyDetailsReportData.Expenditure> findExpenditureTransactions(Outlet outlet, Date fromDate, Date toDate) {
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			BalanceUpdateTransactionDAO.getInstance().addDeletedFilter(criteria);

			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			}
			else if (fromDate != null) {
				criteria.add(Restrictions.ge(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate));
			}
			else if (toDate != null) {
				criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, toDate));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.EXPENSE.name()));

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(BalanceUpdateTransaction.PROP_TRANSACTION_TIME), "date"); //$NON-NLS-1$
			pList.add(Projections.property(BalanceUpdateTransaction.PROP_RECEPIENT_ID), "item"); //$NON-NLS-1$
			pList.add(Projections.property(BalanceUpdateTransaction.PROP_PAYMENT_TYPE_STRING), "paymentType"); //$NON-NLS-1$
			pList.add(Projections.property(BalanceUpdateTransaction.PROP_AMOUNT), "totalPrice"); //$NON-NLS-1$
			pList.add(Projections.property(BalanceUpdateTransaction.PROP_REASON_ID), "reasonId"); //$NON-NLS-1$

			criteria.setProjection(pList);
			criteria.setResultTransformer(Transformers.aliasToBean(DailyDetailsReportData.Expenditure.class));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public double findDoctorPaymentTransactions(Outlet outlet, Date fromDate, Date toDate, boolean isUnpaid) {
		try (Session session = PosTransactionDAO.getInstance().createNewSession()) {
			if (isUnpaid) {
				Criteria criteria = session.createCriteria(Ticket.class);
				criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_LAB_DOCTOR_FEE_PAID, Boolean.FALSE));

				if (fromDate != null && toDate != null) {
					criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, fromDate, toDate));
				}
				else if (fromDate != null) {
					criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
				}
				else if (toDate != null) {
					criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));
				}

				if (outlet != null) {
					criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
				}

				Projection projection = Projections.sqlProjection("SUM(lab_doctor_fee - lab_doctor_fee_paid_amount) as unpaidAmount",
						new String[] { "unpaidAmount" }, new Type[] { DoubleType.INSTANCE });

				criteria.setProjection(projection);
				return POSUtil.getDoubleAmount(criteria.uniqueResult());
			}
			else {
				Criteria criteria = session.createCriteria(PosTransaction.class);
				PosTransactionDAO.getInstance().addDeletedFilter(criteria);

				if (fromDate != null && toDate != null) {
					criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
				}
				else if (fromDate != null) {
					criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
				}
				else if (toDate != null) {
					criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
				}

				if (outlet != null) {
					criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
				}

				criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
				criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_REASON, TransactionReason.LDF_PAY.name()));
				criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

				return POSUtil.getDoubleAmount(criteria.uniqueResult());
			}
		}
	}

	public double findDoctorOpeningBalance(Outlet outlet, Date fromDate) {
		try (Session session = PosTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_LAB_DOCTOR_FEE_PAID, Boolean.FALSE));

			if (fromDate != null) {
				criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, fromDate));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			Projection projection = Projections.sqlProjection("SUM(lab_doctor_fee - lab_doctor_fee_paid_amount) as unpaidAmount",
					new String[] { "unpaidAmount" }, new Type[] { DoubleType.INSTANCE });

			criteria.setProjection(projection);
			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	public double findCashAdvanceBalance(Outlet outlet, Date fromDate, Date toDate) {
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			BalanceUpdateTransactionDAO.getInstance().addDeletedFilter(criteria);

			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			}
			else if (fromDate != null) {
				criteria.add(Restrictions.ge(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate));
			}
			else if (toDate != null) {
				criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, toDate));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.CUSTOMER.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, TransactionSubType.BALANCE_ADDED.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	public DailyDetailsReportData.Income getSalesIncomeData(Date fromDate, Date toDate, List<ProductType> productTypes, String outletId,
			List<ServiceType> serviceTypes, String itemName) {
		try (Session session = TicketItemDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(TicketItem.PROP_PARENT_TICKET_OUTLET_ID, outletId));
			}

			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, fromDate, toDate));
			}
			else if (fromDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, fromDate));
			}
			else if (toDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, toDate));
			}
			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));

			if (productTypes != null && !productTypes.isEmpty()) {
				Disjunction productTypeDisjunction = Restrictions.disjunction();
				for (ProductType productType : productTypes) {
					productTypeDisjunction.add(Restrictions.eq(TicketItem.PROP_PRODUCT_TYPE, productType.name()));
				}
				criteria.add(productTypeDisjunction);
			}

			if (serviceTypes != null && !serviceTypes.isEmpty()) {
				Disjunction serviceTypeDisjunction = Restrictions.disjunction();
				for (ServiceType serviceType : serviceTypes) {
					if (serviceType != null) {
						serviceTypeDisjunction.add(Restrictions.eq(TicketItem.PROP_SERVICE_TYPE, serviceType.name()));
					}
					else {
						serviceTypeDisjunction.add(Restrictions.isNull(TicketItem.PROP_SERVICE_TYPE));
					}
				}

				criteria.add(serviceTypeDisjunction);
			}

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(TicketItem.PROP_LAB_DOCTOR_FEE));
			projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_TOTAL));
			projectionList.add(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_REPORT));
			projectionList.add(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_NET_SALES));

			criteria.setProjection(projectionList);

			Object[] result = (Object[]) criteria.uniqueResult();

			DailyDetailsReportData.Income income = new DailyDetailsReportData.Income();
			income.setItem(StringUtils.isNotBlank(itemName) ? itemName : "Sales"); //$NON-NLS-1$

			if (result != null) {
				double doctorAmount = result[0] != null ? ((Number) result[0]).doubleValue() : 0.0;
				double adjustedTotalAmount = result[1] != null ? ((Number) result[1]).doubleValue() : 0.0;
				double rfOnReport = result[2] != null ? ((Number) result[2]).doubleValue() : 0.0;
				double rfOnNetSales = result[2] != null ? ((Number) result[3]).doubleValue() : 0.0;
				double totalAmount = adjustedTotalAmount + doctorAmount + rfOnReport + rfOnNetSales;
				double hospitalAmount = adjustedTotalAmount - doctorAmount - (rfOnReport + rfOnNetSales);

				income.setTotalAmount(totalAmount);
				income.setDoctorAmount(doctorAmount);
				income.setHospitalAmount(hospitalAmount);
				income.setRfAmount(rfOnReport + rfOnNetSales);
			}
			else {
				income.setTotalAmount(0.0);
				income.setDoctorAmount(0.0);
				income.setHospitalAmount(0.0);
				income.setRfAmount(0.0);
			}

			return income;

		} catch (Exception e) {
			PosLog.error(getClass(), e);
			DailyDetailsReportData.Income income = new DailyDetailsReportData.Income();
			income.setItem(StringUtils.isNotBlank(itemName) ? itemName : "Sales"); //$NON-NLS-1$
			income.setTotalAmount(0.0);
			income.setDoctorAmount(0.0);
			income.setHospitalAmount(0.0);
			income.setRfAmount(0.0);
			return income;
		}
	}

	public DailyDetailsReportData.Income getAdmissionIncomeData(Date startDate, Date endDate, String outletId) {
		try (Session session = TicketItemDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(TicketItem.PROP_PARENT_TICKET_OUTLET_ID, outletId));
			}

			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.between(TicketItem.PROP_CREATE_DATE, startDate, endDate));
			}
			else if (startDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, startDate));
			}
			else if (endDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			}

			criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(TicketItem.PROP_NAME, Ticket.ADMISSION_CHARGE));

			criteria.setProjection(Projections.sum(TicketItem.PROP_ADJUSTED_TOTAL));

			Object result = criteria.uniqueResult();

			DailyDetailsReportData.Income income = new DailyDetailsReportData.Income();
			income.setItem(Ticket.ADMISSION_CHARGE);

			double totalAmount = result != null ? ((Number) result).doubleValue() : 0.0;

			income.setTotalAmount(totalAmount);
			income.setDoctorAmount(0.0);
			income.setHospitalAmount(totalAmount);

			return income;

		} catch (Exception e) {
			PosLog.error(getClass(), e);
			DailyDetailsReportData.Income income = new DailyDetailsReportData.Income();
			income.setItem(Ticket.ADMISSION_CHARGE);
			income.setTotalAmount(0.0);
			income.setHospitalAmount(0.0);
			income.setDoctorAmount(0.0);
			return income;
		}
	}

	public double getBankAccountBalanceAccordingType(Date fromDate, Date toDate, String outletId, BalanceType balanceType, TransactionType transactionType,
			List<BalanceSubType> balanceSubTypes, boolean isBankOpeningBalance) {
		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);

			if (!isBankOpeningBalance) {
				if (fromDate != null && toDate != null) {
					criteria.add(Restrictions.between(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
				}
				else if (fromDate != null) {
					criteria.add(Restrictions.ge(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate));
				}
				else if (toDate != null) {
					criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, toDate));
				}
			}

			if (balanceType != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, balanceType.name()));
			}

			if (balanceSubTypes != null && !balanceSubTypes.isEmpty()) {
				Disjunction balanceSubTypeDisjunction = Restrictions.disjunction();
				for (BalanceSubType balanceSubType : balanceSubTypes) {
					balanceSubTypeDisjunction.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, balanceSubType.name()));
				}
				criteria.add(balanceSubTypeDisjunction);
			}

			if (transactionType != null) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_TYPE, transactionType.name()));

			}
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_OUTLET_ID, outletId));
			}

			if (isBankOpeningBalance) {
				criteria.add(Restrictions.lt(BalanceUpdateTransaction.PROP_TRANSACTION_TIME, fromDate));
			}

			criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));

			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult != null && (uniqueResult instanceof Number)) {
				return POSUtil.getDoubleAmount(((Number) uniqueResult).doubleValue());
			}
		}
		return 0.0;
	}
}
