package com.orocube.cloudflare.r2;

import java.util.Date;

/**
 * Response object for R2 file upload operations.
 * Contains upload result information including success status, R2 key, and error details.
 */
public class R2UploadResponse {

	private boolean success;
	private String r2Key;
	private String url;
	private Long fileSize;
	private Date uploadedDate;
	private String errorMessage;

	public R2UploadResponse() {
	}

	/**
	 * Create a successful upload response.
	 *
	 * @param r2Key R2 object key
	 * @param fileSize file size in bytes
	 * @return successful upload response
	 */
	public static R2UploadResponse success(String r2Key, Long fileSize) {
		R2UploadResponse response = new R2UploadResponse();
		response.success = true;
		response.r2Key = r2Key;
		response.fileSize = fileSize;
		response.uploadedDate = new Date();
		return response;
	}

	/**
	 * Create a failed upload response.
	 *
	 * @param errorMessage error message
	 * @return failed upload response
	 */
	public static R2UploadResponse failure(String errorMessage) {
		R2UploadResponse response = new R2UploadResponse();
		response.success = false;
		response.errorMessage = errorMessage;
		return response;
	}

	/**
	 * Check if upload was successful.
	 *
	 * @return true if successful, false otherwise
	 */
	public boolean isSuccess() {
		return success;
	}

	/**
	 * Set success status.
	 *
	 * @param success success status
	 */
	public void setSuccess(boolean success) {
		this.success = success;
	}

	/**
	 * Get R2 object key.
	 *
	 * @return R2 key
	 */
	public String getR2Key() {
		return r2Key;
	}

	/**
	 * Set R2 object key.
	 *
	 * @param r2Key R2 key
	 */
	public void setR2Key(String r2Key) {
		this.r2Key = r2Key;
	}

	/**
	 * Get public URL (if applicable).
	 *
	 * @return URL
	 */
	public String getUrl() {
		return url;
	}

	/**
	 * Set public URL.
	 *
	 * @param url URL
	 */
	public void setUrl(String url) {
		this.url = url;
	}

	/**
	 * Get file size in bytes.
	 *
	 * @return file size
	 */
	public Long getFileSize() {
		return fileSize;
	}

	/**
	 * Set file size.
	 *
	 * @param fileSize file size in bytes
	 */
	public void setFileSize(Long fileSize) {
		this.fileSize = fileSize;
	}

	/**
	 * Get upload date.
	 *
	 * @return uploaded date
	 */
	public Date getUploadedDate() {
		return uploadedDate;
	}

	/**
	 * Set upload date.
	 *
	 * @param uploadedDate upload date
	 */
	public void setUploadedDate(Date uploadedDate) {
		this.uploadedDate = uploadedDate;
	}

	/**
	 * Get error message (if upload failed).
	 *
	 * @return error message
	 */
	public String getErrorMessage() {
		return errorMessage;
	}

	/**
	 * Set error message.
	 *
	 * @param errorMessage error message
	 */
	public void setErrorMessage(String errorMessage) {
		this.errorMessage = errorMessage;
	}

	@Override
	public String toString() {
		if (success) {
			return "R2UploadResponse{success=true, r2Key='" + r2Key + "', fileSize=" + fileSize + "}";
		} else {
			return "R2UploadResponse{success=false, error='" + errorMessage + "'}";
		}
	}
}
