package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.ResultTransformer;

import com.floreantpos.Messages;
import com.floreantpos.PosLog;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.DrawerPullVoidEntry;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.ext.KitchenStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.CopyUtil;
import com.floreantpos.util.GlobalIdGenerator;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class VoidItemDAO extends BaseVoidItemDAO {

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public VoidItemDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void saveOrUpdateVoidItem(List<VoidItem> voidItemList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (voidItemList == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<VoidItem> iterator = voidItemList.iterator(); iterator.hasNext();) {
				VoidItem item = (VoidItem) iterator.next();
				VoidItem existingItem = get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getMenuItemName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem);
				}
				else {
					item.setVersion(0);
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		}
	}

	public VoidItem get(String id, String outletId) {
		return get(new VoidItem(id, outletId));
	}

	public void saveVoidItems(List<VoidItem> voidItems, Ticket ticket) {
		Session session = null;
		Transaction tx = null;

		try {
			CashDrawer activeDrawer = DataProvider.get().getCurrentUser().getCurrentCashDrawer();
			session = createNewSession();
			tx = session.beginTransaction();
			for (VoidItem voidItem : voidItems) {
				Double totalPrice = Math.abs(voidItem.getTotalPrice());
				ticket.setVoidAmount(ticket.getVoidAmount() + totalPrice);
				//				ticket.setRefundableAmount(ticket.getRefundableAmount() + totalPrice);
				voidItem.setTicketId(ticket.getId());
				voidItem.setCashDrawerId(activeDrawer.getId());
				if (voidItem.getItemId() != null) {
					List<KitchenTicketItem> kitchenTicketItems = KitchenTicketItemDAO.getInstance().find(voidItem.getItemId(), voidItem.isModifier(), session);
					if (kitchenTicketItems != null) {
						for (KitchenTicketItem kitchenItem : kitchenTicketItems) {
							KitchenTicket kitchenTicket = kitchenItem.getKitchenTicket();
							if (voidItem.getQuantity() < kitchenItem.getQuantity()) {
								KitchenTicketItem newKitchenTicketItem = (KitchenTicketItem) CopyUtil.deepCopy(kitchenItem);
								newKitchenTicketItem.setId(null);
								newKitchenTicketItem.setQuantity(kitchenItem.getQuantity() - voidItem.getQuantity());
								kitchenTicket.addToticketItems(newKitchenTicketItem);
								session.save(newKitchenTicketItem);
							}
							if (kitchenItem.getStatus().equalsIgnoreCase(KitchenStatus.BUMP.name())) {
								voidItem.setCooked(true);
							}
							kitchenItem.setVoided(true);
							session.update(kitchenItem);
						}
					}
				}
				session.saveOrUpdate(voidItem);
			}
			TicketDAO.getInstance().updateStock(ticket, session);
			session.saveOrUpdate(ticket);
			tx.commit();
		} catch (Exception e) {
			if (tx != null) {
				tx.rollback();
			}

			throwException(e);

		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveVoidItems(List<VoidItem> voidItems, Session session) {
		for (VoidItem voidItem : voidItems) {
			session.saveOrUpdate(voidItem);
		}
	}

	public List<VoidItem> findByDate(Date startDate, Date endDate, Outlet outlet1) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(VoidItem.PROP_VOID_DATE, startDate));
			criteria.add(Restrictions.le(VoidItem.PROP_VOID_DATE, endDate));

			if (outlet1 != null) {
				criteria.add(Restrictions.eq(VoidItem.PROP_OUTLET_ID, outlet1.getId()));
			}
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<DrawerPullVoidEntry> getVoidEntries(CashDrawer cashDrawer) {
		List<String> cashDrawerIds = new ArrayList<>();
		cashDrawerIds.add(cashDrawer.getId());
		return getVoidEntries(cashDrawerIds);
	}

	public List<DrawerPullVoidEntry> getVoidEntries(List<String> cashDrawerIds) {
		if (cashDrawerIds.isEmpty())
			return new ArrayList<>();
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.countDistinct(VoidItem.PROP_TICKET_ID));
			projectionList.add(Projections.sum(VoidItem.PROP_QUANTITY));
			projectionList.add(Projections.sum(VoidItem.PROP_TOTAL_PRICE));
			projectionList.add(Projections.groupProperty(VoidItem.PROP_ITEM_WASTED));
			if (cashDrawerIds.size() == 1) {
				criteria.add(Restrictions.eq(VoidItem.PROP_CASH_DRAWER_ID, cashDrawerIds.get(0)));
			}
			else {
				Disjunction disjunction = Restrictions.disjunction();
				for (String cashDrawerId : cashDrawerIds) {
					disjunction.add(Restrictions.eq(VoidItem.PROP_CASH_DRAWER_ID, cashDrawerId));
				}
				criteria.add(Restrictions.and(Restrictions.isNotNull(VoidItem.PROP_CASH_DRAWER_ID), disjunction));
			}
			ResultTransformer transformer = new ResultTransformer() {

				public Object transformTuple(Object[] row, String[] arg1) {
					DrawerPullVoidEntry voidEntry = new DrawerPullVoidEntry();
					voidEntry.setCheckCount(Double.valueOf("" + row[0]));//$NON-NLS-1$
					voidEntry.setQuantity(Double.valueOf("" + row[1]));//$NON-NLS-1$
					voidEntry.setAmount(Double.valueOf("" + row[2]));//$NON-NLS-1$
					voidEntry.setReason((Boolean.valueOf("" + row[3]) == true ? Messages.getString("TOTAL_WASTE") : Messages.getString("TOTAL_VOID"))//$NON-NLS-1$//$NON-NLS-2$
							+ (voidEntry.getCheckCount() > 0 ? " (" + NumberUtil.trimDecilamIfNotNeeded(voidEntry.getCheckCount()) + ")" : ""));//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
					return voidEntry;
				}

				public List transformList(List arg0) {
					return arg0;
				}
			};
			criteria.setProjection(projectionList).setResultTransformer(transformer);
			return criteria.list();
		} finally {
			closeSession(session);
		}

	}

	public List<VoidItem> getVoidItems(String ticketId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(VoidItem.PROP_TICKET_ID, ticketId));

			List list = criteria.list();
			if (list.isEmpty()) {
				return null;
			}
			return list;

		} finally {
			closeSession(session);
		}
	}

	public void saveAndSentToKitchenIfNeeded(VoidItem voidItem, boolean printedToKitchen, Ticket ticket, Session session) {
		Double totalPrice = Math.abs(voidItem.getTotalPrice());
		ticket.setVoidAmount(ticket.getVoidAmount() + totalPrice);
		//ticket.setRefundableAmount(ticket.getRefundableAmount() + totalPrice);
		voidItem.setTicketId(ticket.getId());
		if (voidItem.getItemId() != null/* && printedToKitchen*/) {
			saveAndSentToKitchenIfNeeded(voidItem, session);
		}
		if (voidItem.getId() == null) {
			voidItem.setId(GlobalIdGenerator.generateGlobalId());
		}
		if (voidItem.getOutletId() == null) {
			voidItem.setOutletId(ticket.getOutletId());
		}
		saveOrUpdate(voidItem, session);
	}

	public void saveAndSentToKitchenIfNeeded(VoidItem voidItem, Session session) {
		List<KitchenTicketItem> kitchenTicketItems = KitchenTicketItemDAO.getInstance().find(voidItem.getItemId(), voidItem.isModifier(), session);
		if (kitchenTicketItems != null) {
			Set<KitchenTicket> kitchenTickets = new HashSet<>();
			for (KitchenTicketItem kitchenItem : kitchenTicketItems) {
				KitchenTicket kitchenTicket = kitchenItem.getKitchenTicket();
				kitchenTickets.add(kitchenTicket);

				if (voidItem.getQuantity() < kitchenItem.getQuantity()) {
					try {
						KitchenTicketItem newKitchenTicketItem = (KitchenTicketItem) CopyUtil.deepCopy(kitchenItem);
						newKitchenTicketItem.setId(null);
						newKitchenTicketItem.setQuantity(kitchenItem.getQuantity() - voidItem.getQuantity());
						kitchenTicket.addToticketItems(newKitchenTicketItem);
						//save(newKitchenTicketItem, session);
					} catch (Exception e) {
						PosLog.error(getClass(), e);
					}
				}
				if (kitchenItem.getStatus().equalsIgnoreCase(KitchenStatus.BUMP.name())) {
					voidItem.setCooked(true);
				}
				kitchenItem.setVoided(true);
				//update(kitchenItem, session);
			}
			for (KitchenTicket kitchenTicket : kitchenTickets) {
				KitchenTicketDAO.getInstance().update(kitchenTicket, session);
			}
		}
	}

	public void deleteAndSentToKitchenIfNeeded(VoidItem voidItem, boolean printedToKitchen, Ticket ticket, Session session) {
		voidItem.setTicketId(ticket.getId());
		if (voidItem.getItemId() != null) {
			saveAndSentToKitchenIfNeeded(voidItem, session);
		}
		delete(voidItem, session);
	}
}