package com.orocube.cloudflare.r2;

/**
 * Configuration holder for Cloudflare R2 storage settings.
 * Configuration must be set by calling application via setConfiguration() before using R2StorageService.
 * Thread-safe implementation.
 */
public class R2Config {

	private static volatile String endpointUrl;
	private static volatile String accessKey;
	private static volatile String secretKey;
	private static volatile String bucketName;
	private static volatile String region;
	private static volatile boolean configured = false;

	/**
	 * Set R2 configuration. Must be called before using R2StorageService.
	 * Thread-safe.
	 *
	 * @param endpointUrl R2 endpoint URL (e.g., https://account-id.r2.cloudflarestorage.com)
	 * @param accessKey R2 access key ID
	 * @param secretKey R2 secret access key
	 * @param bucketName R2 bucket name
	 * @param region R2 region (typically "auto" for Cloudflare R2)
	 */
	public static synchronized void setConfiguration(String endpointUrl, String accessKey,
	                                                 String secretKey, String bucketName, String region) {
		R2Config.endpointUrl = endpointUrl;
		R2Config.accessKey = accessKey;
		R2Config.secretKey = secretKey;
		R2Config.bucketName = bucketName;
		R2Config.region = region;
		R2Config.configured = true;
	}

	/**
	 * Clear R2 configuration. Useful for testing or reconfiguration.
	 */
	public static synchronized void clearConfiguration() {
		R2Config.endpointUrl = null;
		R2Config.accessKey = null;
		R2Config.secretKey = null;
		R2Config.bucketName = null;
		R2Config.region = null;
		R2Config.configured = false;
	}

	/**
	 * Check if R2 configuration has been set.
	 *
	 * @return true if configuration is set, false otherwise
	 */
	public static boolean isConfigured() {
		return configured;
	}

	/**
	 * Get R2 endpoint URL.
	 *
	 * @return endpoint URL
	 */
	public static String getEndpointUrl() {
		return endpointUrl;
	}

	/**
	 * Get R2 access key ID.
	 *
	 * @return access key
	 */
	public static String getAccessKey() {
		return accessKey;
	}

	/**
	 * Get R2 secret access key.
	 *
	 * @return secret key
	 */
	public static String getSecretKey() {
		return secretKey;
	}

	/**
	 * Get R2 bucket name.
	 *
	 * @return bucket name
	 */
	public static String getBucketName() {
		return bucketName;
	}

	/**
	 * Get R2 region.
	 *
	 * @return region
	 */
	public static String getRegion() {
		return region;
	}
}
