package com.orocube.cloudflare.r2;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.floreantpos.PosLog;

import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.ResponseBytes;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.DeleteObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.HeadObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.S3Exception;
import software.amazon.awssdk.services.s3.model.S3Object;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.s3.presigner.model.GetObjectPresignRequest;
import software.amazon.awssdk.services.s3.presigner.model.PresignedGetObjectRequest;

/**
 * Core service for Cloudflare R2 storage operations.
 * Provides upload, download, list, and delete functionality for R2.
 * Singleton pattern - use getInstance() to access.
 */
public class R2StorageService {

	private static R2StorageService instance;
	private S3Client s3Client;

	private R2StorageService() {
		// Private constructor for singleton
	}

	/**
	 * Get singleton instance of R2StorageService.
	 *
	 * @return R2StorageService instance
	 */
	public static synchronized R2StorageService getInstance() {
		if (instance == null) {
			instance = new R2StorageService();
		}
		return instance;
	}

	/**
	 * Initialize S3 client for R2 operations.
	 * Checks configuration and creates client.
	 *
	 * @return S3Client instance
	 * @throws R2StorageException if configuration is not set
	 */
	private S3Client getS3Client() {
		if (!R2Config.isConfigured()) {
			throw new R2StorageException("R2 configuration not set. Call R2Config.setConfiguration() first.");
		}

		if (s3Client == null) {
			synchronized (this) {
				if (s3Client == null) {
					try {
						s3Client = S3Client.builder()
								.endpointOverride(URI.create(R2Config.getEndpointUrl()))
								.region(Region.of(R2Config.getRegion()))
								.credentialsProvider(StaticCredentialsProvider.create(
										AwsBasicCredentials.create(
												R2Config.getAccessKey(),
												R2Config.getSecretKey()
										)
								))
								.build();
						PosLog.info(getClass(), "R2 S3Client initialized successfully");
					} catch (Exception e) {
						PosLog.error(getClass(), "Failed to initialize R2 S3Client", e);
						throw new R2StorageException("Failed to initialize R2 client", e);
					}
				}
			}
		}
		return s3Client;
	}

	/**
	 * Upload file to R2 from byte array.
	 *
	 * @param fileData file data as byte array
	 * @param fileName original file name
	 * @param contentType MIME type (e.g., "image/jpeg")
	 * @param category file category (e.g., "PRODUCTS", "SIGNATURE")
	 * @return R2UploadResponse with upload result
	 */
	public R2UploadResponse uploadFile(byte[] fileData, String fileName, String contentType, String category) {
		if (fileData == null || fileData.length == 0) {
			return R2UploadResponse.failure("File data is empty");
		}
		return uploadFile(new ByteArrayInputStream(fileData), fileName, contentType, category, (long) fileData.length);
	}

	/**
	 * Upload file to R2 from InputStream.
	 *
	 * @param inputStream file input stream
	 * @param fileName original file name
	 * @param contentType MIME type (e.g., "image/jpeg")
	 * @param category file category (e.g., "PRODUCTS", "SIGNATURE")
	 * @return R2UploadResponse with upload result
	 */
	public R2UploadResponse uploadFile(InputStream inputStream, String fileName, String contentType, String category) {
		try {
			// Read all bytes from input stream (Java 8 compatible)
			byte[] buffer = new byte[8192];
			int bytesRead;
			java.io.ByteArrayOutputStream output = new java.io.ByteArrayOutputStream();
			while ((bytesRead = inputStream.read(buffer)) != -1) {
				output.write(buffer, 0, bytesRead);
			}
			byte[] fileData = output.toByteArray();
			return uploadFile(fileData, fileName, contentType, category);
		} catch (IOException e) {
			PosLog.error(getClass(), "Failed to read input stream", e);
			return R2UploadResponse.failure("Failed to read input stream: " + e.getMessage());
		}
	}

	/**
	 * Internal upload method with file size.
	 */
	private R2UploadResponse uploadFile(InputStream inputStream, String fileName, String contentType,
	                                    String category, Long fileSize) {
		try {
			// Generate unique R2 key
			String r2Key = R2HelperService.generateUniqueKey(category, fileName);

			// Auto-detect content type if not provided
			if (contentType == null || contentType.isEmpty()) {
				contentType = R2HelperService.detectContentType(fileName);
			}

			// Prepare metadata
			Map<String, String> metadata = new HashMap<>();
			metadata.put("original-filename", R2HelperService.sanitizeFileName(fileName));
			metadata.put("category", category != null ? category : "UNCATEGORIZED");
			metadata.put("upload-date", new Date().toString());

			// Create put object request
			PutObjectRequest putRequest = PutObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.contentType(contentType)
					.contentLength(fileSize)
					.metadata(metadata)
					.build();

			// Upload to R2
			getS3Client().putObject(putRequest, RequestBody.fromInputStream(inputStream, fileSize));

			PosLog.info(getClass(), "Successfully uploaded file to R2: " + r2Key + " (" +
					R2HelperService.formatFileSize(fileSize) + ")");

			return R2UploadResponse.success(r2Key, fileSize);

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 upload failed (S3Exception): " + e.awsErrorDetails().errorMessage(), e);
			return R2UploadResponse.failure("R2 upload failed: " + e.awsErrorDetails().errorMessage());
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 upload failed (Unexpected error): " + e.getMessage(), e);
			return R2UploadResponse.failure("Upload failed: " + e.getMessage());
		}
	}

	/**
	 * Download file from R2 as byte array.
	 *
	 * @param r2Key R2 object key
	 * @return file data as byte array
	 * @throws R2StorageException if download fails
	 */
	public byte[] downloadFile(String r2Key) {
		try {
			GetObjectRequest getRequest = GetObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			ResponseBytes<GetObjectResponse> objectBytes = getS3Client().getObjectAsBytes(getRequest);
			byte[] data = objectBytes.asByteArray();

			PosLog.info(getClass(), "Successfully downloaded file from R2: " + r2Key +
					" (" + R2HelperService.formatFileSize(data.length) + ")");

			return data;

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 download failed: " + e.awsErrorDetails().errorMessage(), e);
			throw new R2StorageException("Failed to download file from R2: " + e.awsErrorDetails().errorMessage(), e);
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 download failed: " + e.getMessage(), e);
			throw new R2StorageException("Failed to download file from R2", e);
		}
	}

	/**
	 * Download file from R2 as InputStream.
	 *
	 * @param r2Key R2 object key
	 * @return file input stream
	 * @throws R2StorageException if download fails
	 */
	public InputStream downloadFileAsStream(String r2Key) {
		try {
			GetObjectRequest getRequest = GetObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			InputStream stream = getS3Client().getObject(getRequest);
			PosLog.info(getClass(), "Successfully opened stream for R2 file: " + r2Key);
			return stream;

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 download stream failed: " + e.awsErrorDetails().errorMessage(), e);
			throw new R2StorageException("Failed to download file stream from R2: " + e.awsErrorDetails().errorMessage(), e);
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 download stream failed: " + e.getMessage(), e);
			throw new R2StorageException("Failed to download file stream from R2", e);
		}
	}

	/**
	 * List files in R2 by category.
	 *
	 * @param category file category (e.g., "PRODUCTS")
	 * @return list of R2StorageMetadata
	 */
	public List<R2StorageMetadata> listFiles(String category) {
		return listFiles(category + "/", 1000);
	}

	/**
	 * List files in R2 with prefix and max results.
	 *
	 * @param prefix object key prefix
	 * @param maxResults maximum number of results
	 * @return list of R2StorageMetadata
	 */
	public List<R2StorageMetadata> listFiles(String prefix, int maxResults) {
		try {
			ListObjectsV2Request listRequest = ListObjectsV2Request.builder()
					.bucket(R2Config.getBucketName())
					.prefix(prefix)
					.maxKeys(maxResults)
					.build();

			ListObjectsV2Response listResponse = getS3Client().listObjectsV2(listRequest);
			List<R2StorageMetadata> metadataList = new ArrayList<>();

			for (S3Object s3Object : listResponse.contents()) {
				R2StorageMetadata metadata = new R2StorageMetadata();
				metadata.setR2Key(s3Object.key());
				metadata.setFileSize(s3Object.size());
				metadata.setUploadedDate(Date.from(s3Object.lastModified()));

				// Extract category from key
				String key = s3Object.key();
				if (key.contains("/")) {
					String cat = key.substring(0, key.indexOf("/"));
					metadata.setCategory(cat);
				}

				metadataList.add(metadata);
			}

			PosLog.info(getClass(), "Listed " + metadataList.size() + " files from R2 with prefix: " + prefix);
			return metadataList;

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 list failed: " + e.awsErrorDetails().errorMessage(), e);
			throw new R2StorageException("Failed to list files from R2: " + e.awsErrorDetails().errorMessage(), e);
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 list failed: " + e.getMessage(), e);
			throw new R2StorageException("Failed to list files from R2", e);
		}
	}

	/**
	 * Delete file from R2.
	 *
	 * @param r2Key R2 object key
	 * @return true if deleted successfully, false otherwise
	 */
	public boolean deleteFile(String r2Key) {
		try {
			DeleteObjectRequest deleteRequest = DeleteObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			getS3Client().deleteObject(deleteRequest);
			PosLog.info(getClass(), "Successfully deleted file from R2: " + r2Key);
			return true;

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 delete failed: " + e.awsErrorDetails().errorMessage(), e);
			return false;
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 delete failed: " + e.getMessage(), e);
			return false;
		}
	}

	/**
	 * Check if file exists in R2.
	 *
	 * @param r2Key R2 object key
	 * @return true if file exists, false otherwise
	 */
	public boolean fileExists(String r2Key) {
		try {
			HeadObjectRequest headRequest = HeadObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			getS3Client().headObject(headRequest);
			return true;

		} catch (S3Exception e) {
			if (e.statusCode() == 404) {
				return false;
			}
			PosLog.error(getClass(), "R2 file exists check failed: " + e.awsErrorDetails().errorMessage(), e);
			return false;
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 file exists check failed: " + e.getMessage(), e);
			return false;
		}
	}

	/**
	 * Get file metadata from R2.
	 *
	 * @param r2Key R2 object key
	 * @return R2StorageMetadata or null if not found
	 */
	public R2StorageMetadata getFileMetadata(String r2Key) {
		try {
			HeadObjectRequest headRequest = HeadObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			HeadObjectResponse headResponse = getS3Client().headObject(headRequest);

			R2StorageMetadata metadata = new R2StorageMetadata();
			metadata.setR2Key(r2Key);
			metadata.setFileSize(headResponse.contentLength());
			metadata.setContentType(headResponse.contentType());
			metadata.setUploadedDate(Date.from(headResponse.lastModified()));

			// Extract category from key
			if (r2Key.contains("/")) {
				String category = r2Key.substring(0, r2Key.indexOf("/"));
				metadata.setCategory(category);
			}

			// Extract original filename from metadata
			if (headResponse.metadata() != null && headResponse.metadata().containsKey("original-filename")) {
				metadata.setFileName(headResponse.metadata().get("original-filename"));
			}

			return metadata;

		} catch (S3Exception e) {
			if (e.statusCode() == 404) {
				PosLog.info(getClass(), "File not found in R2: " + r2Key);
				return null;
			}
			PosLog.error(getClass(), "R2 get metadata failed: " + e.awsErrorDetails().errorMessage(), e);
			throw new R2StorageException("Failed to get file metadata from R2: " + e.awsErrorDetails().errorMessage(), e);
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 get metadata failed: " + e.getMessage(), e);
			throw new R2StorageException("Failed to get file metadata from R2", e);
		}
	}

	/**
	 * Generate presigned URL for temporary file access.
	 *
	 * @param r2Key R2 object key
	 * @param expirationMinutes URL expiration time in minutes
	 * @return presigned URL
	 */
	public String generatePresignedUrl(String r2Key, int expirationMinutes) {
		try {
			S3Presigner presigner = S3Presigner.builder()
					.endpointOverride(URI.create(R2Config.getEndpointUrl()))
					.region(Region.of(R2Config.getRegion()))
					.credentialsProvider(StaticCredentialsProvider.create(
							AwsBasicCredentials.create(
									R2Config.getAccessKey(),
									R2Config.getSecretKey()
							)
					))
					.build();

			GetObjectRequest getRequest = GetObjectRequest.builder()
					.bucket(R2Config.getBucketName())
					.key(r2Key)
					.build();

			GetObjectPresignRequest presignRequest = GetObjectPresignRequest.builder()
					.signatureDuration(Duration.ofMinutes(expirationMinutes))
					.getObjectRequest(getRequest)
					.build();

			PresignedGetObjectRequest presignedRequest = presigner.presignGetObject(presignRequest);
			String url = presignedRequest.url().toString();

			presigner.close();

			PosLog.info(getClass(), "Generated presigned URL for R2 file: " + r2Key +
					" (expires in " + expirationMinutes + " minutes)");

			return url;

		} catch (S3Exception e) {
			PosLog.error(getClass(), "R2 presigned URL generation failed: " + e.awsErrorDetails().errorMessage(), e);
			throw new R2StorageException("Failed to generate presigned URL: " + e.awsErrorDetails().errorMessage(), e);
		} catch (Exception e) {
			PosLog.error(getClass(), "R2 presigned URL generation failed: " + e.getMessage(), e);
			throw new R2StorageException("Failed to generate presigned URL", e);
		}
	}

	/**
	 * Close S3 client and cleanup resources.
	 * Should be called when application shuts down.
	 */
	public void shutdown() {
		if (s3Client != null) {
			try {
				s3Client.close();
				PosLog.info(getClass(), "R2 S3Client closed successfully");
			} catch (Exception e) {
				PosLog.error(getClass(), "Error closing R2 S3Client", e);
			}
		}
	}
}
