/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.model.base.BasePurchaseOrder;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.payment.common.Payable;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

@XmlRootElement
@JsonIgnoreProperties(ignoreUnknown = true, value = { "statusDisplay", "orderStatusDisplay" })
public class PurchaseOrder extends BasePurchaseOrder implements TimedModel, Payable {
	private static final long serialVersionUID = 1L;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public PurchaseOrder() {
	}

	/**
	 * Constructor for primary key
	 */
	public PurchaseOrder(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public static final int ORDER_PENDING = 0;
	public static final int ORDER_VERIFIED = 1;
	public static final int ORDER_FULLY_INVOICED = 2;
	public static final int ORDER_PARTIALLY_INVOICED = 3;
	public static final int ORDER_FULLY_RECEIVED = 4;
	public static final int ORDER_PARTIALLY_RECEIVED = 5;
	public static final int ORDER_PARTIALLY_RECEIVED_AND_INVOICED = 6;
	public static final int ORDER_CLOSED = 7;
	public static final int ORDER_CANCELLED = 8;
	public static final int ORDER_RETURNED = 9;
	public static final int ORDER_TASK_COMPLETED = 10;
	public static final int ORDER_PAID = 11;
	public static final int SERVICE_ORDER_OPEN = 12;
	private transient Map<String, ActionHistory> events = new LinkedHashMap<>();

	public static final String[] ORDER_STATUS = { Messages.getString("PurchaseOrder.0"), Messages.getString("PurchaseOrder.1"), //$NON-NLS-1$//$NON-NLS-2$
			Messages.getString("PurchaseOrder.2"), Messages.getString("PurchaseOrder.3"), Messages.getString("PurchaseOrder.4"), //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			Messages.getString("PurchaseOrder.5"), //$NON-NLS-1$
			Messages.getString("PurchaseOrder.6"), Messages.getString("PurchaseOrder.7"), Messages.getString("PurchaseOrder.8"), "Returned", //$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$
			"Task completed", "Paid", "Open" };

	public static final String PO_VERIFIED_BY = Messages.getString("PurchaseOrder.9"); //$NON-NLS-1$
	public static final String PO_SENT_BY = Messages.getString("PurchaseOrder.10"); //$NON-NLS-1$
	public static final String PO_RECEIVED_BY = Messages.getString("PurchaseOrder.11"); //$NON-NLS-1$
	public static final String PO_INVOICED_BY = Messages.getString("PurchaseOrder.12"); //$NON-NLS-1$
	public static final String PO_CLOSED_BY = Messages.getString("PurchaseOrder.13"); //$NON-NLS-1$

	public String orderStatusDisplay;

	public static final String DEBIT = Messages.getString("PurchaseOrder.14"); //$NON-NLS-1$
	public static final String CREDIT = Messages.getString("PurchaseOrder.15"); //$NON-NLS-1$
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	public String statusDisplay;

	public static final String JSON_PROP_CUSTOMER_NAME = "customer_name"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_EMAIL = "customer_email"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_PHONE = "customer_phone"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_ADDRESS = "customer_address"; //$NON-NLS-1$

	private Set<String> emailList;
	private transient Project project;
	private transient Requisition requisition;
	private boolean isReturnedFromPurchaseCrudView;
	private double totalPurchaseAmount;
	private double totalPurchaseTaxAmount;

	@XmlTransient
	@JsonIgnore
	public String getOrderStatusDisplay() {
		return ORDER_STATUS[getStatus()];
	}

	public void setOrderStatusDisplay(String orderStatusDisplay) {
		this.orderStatusDisplay = ORDER_STATUS[getStatus()];
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public List<PurchaseOrderItem> getOrderItems() {
		List<PurchaseOrderItem> items = super.getOrderItems();

		if (items == null) {
			items = new ArrayList<PurchaseOrderItem>();
			super.setOrderItems(items);
		}
		return items;
	}

	public void calculatePrice() {
		List<PurchaseOrderItem> items = getOrderItems();
		if (items == null || items.isEmpty()) {
			return;
		}

		calculateTax(items);
		calculateDiscount(items);

		double subtotalAmount = 0;
		double discountAmount = 0;
		double taxAmount = 0;
		boolean isReturned = PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.RETURN)
				|| PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.SERVICE_ORDER_RETURN);

		double totalPurchaseQuantity = 0;
		double totalReturnedQuantity = 0;
		double previousReturnedAmount = 0;
		double previousReturnedTaxAmount = 0;

		for (Iterator<PurchaseOrderItem> iterator = items.iterator(); iterator.hasNext();) {
			PurchaseOrderItem item = iterator.next();
			totalPurchaseQuantity += item.getTotalPurchaseQty();
			if (isReturned && item.getQuantityToReceive() == 0) {
				continue;
			}
			item.calculatePrice();
			double previousReturnedQuantity = item.getReturnedQty();
			totalReturnedQuantity += item.getQuantityToReceive() + previousReturnedQuantity;

			if (!iterator.hasNext() && getVatType() == DiscountType.AMOUNT && !isReturned) {
				item.setTaxAmount(getVatValue() - taxAmount);
			}

			previousReturnedAmount += item.getTotalReturnedAmount();
			previousReturnedTaxAmount += item.getTotalReturnedTaxAmount();

			subtotalAmount += item.getSubtotalAmount();
			discountAmount += item.getDiscountAmount();
			taxAmount += item.getTaxAmount();

			subtotalAmount = NumberUtil.round(subtotalAmount);
		}

		setSubtotalAmount(subtotalAmount);
		setDiscountAmount(discountAmount);

		if (isReturned && taxAmount > 0) {
			taxAmount = taxAmount * (-1);
		}

		setTaxAmount(taxAmount);

		double totalAmount = NumberUtil.round(Math.abs(subtotalAmount) - Math.abs(discountAmount) + Math.abs(taxAmount));

		if (isReturned && Math.abs(subtotalAmount) > 0 && totalPurchaseQuantity == totalReturnedQuantity && getVatType() == DiscountType.AMOUNT
				&& getDiscountType() == DiscountType.AMOUNT) {
			totalAmount = NumberUtil.round(totalPurchaseAmount - Math.abs(previousReturnedAmount));
			taxAmount = NumberUtil.round(totalPurchaseTaxAmount - Math.abs(previousReturnedTaxAmount));
		}

		if (isReturned && taxAmount > 0) {
			taxAmount = taxAmount * (-1);
		}

		setTaxAmount(taxAmount);

		if (!isReturned) {
			totalAmount = NumberUtil.round(fixInvalidAmount(totalAmount));
		}
		else if (isReturned && totalAmount > 0) {
			totalAmount = totalAmount * (-1);
		}
		setTotalAmount(totalAmount);

		setDueAmount(NumberUtil.round(totalAmount - getPaidAmount()));
	}

	private void calculateDiscount(List<PurchaseOrderItem> orderItems) {
		double amount = getDiscountValue();

		DiscountType discountType = getDiscountType();
		if (discountType == null || amount < 0.0) {
			amount = 0;
		}

		if (discountType == DiscountType.PERCENTAGE) {
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				double discount = purchaseOrderItem.getSubtotalAmount() * (amount / 100.0);
				purchaseOrderItem.setDiscountAmount(discount);
			}
		}

		else if (isReturnedFromPurchaseCrudView && PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.RETURN)) {
			double totalQuantity = orderItems.stream().mapToDouble(t -> t.getTotalPurchaseQty()).sum();
			double totalReturnedQuantity = 0;
			double totalDiscountAmount = 0;
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				double discount = NumberUtil.round(amount / totalQuantity) * purchaseOrderItem.getQuantityToReceive();
				totalReturnedQuantity = totalReturnedQuantity + purchaseOrderItem.getQuantityToReceive();
				if (totalQuantity == totalReturnedQuantity) {
					discount = amount - totalDiscountAmount;
				}
				purchaseOrderItem.setDiscountAmount(discount * (-1));
				totalDiscountAmount += discount;
			}
		}
		else {
			double subTotalAmount = orderItems.stream().mapToDouble(t -> t.getSubtotalAmount()).sum();
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				double ratio = NumberUtil.round(purchaseOrderItem.getSubtotalAmount() / Math.abs(subTotalAmount));
				double discount = 0;
				if (PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.RETURN)) {
					discount = ratio * amount;
				}
				else {
					discount = fixInvalidAmount(ratio) * amount;
				}
				purchaseOrderItem.setDiscountAmount(discount);
			}
		}
	}

	private void calculateTax(List<PurchaseOrderItem> orderItems) {
		double amount = getVatValue();
		DiscountType vatType = getVatType();
		boolean isReturned = PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.RETURN);
		if ((isReturned && isReturnedFromPurchaseCrudView) || vatType == null || amount < 0.0) {
			return;
		}

		if (vatType == DiscountType.PERCENTAGE) {
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				purchaseOrderItem.setTaxRate(amount);
			}
		}
		else {
			double subTotalAmount = orderItems.stream().mapToDouble(t -> t.getSubtotalAmount()).sum();
			double vatRate = (amount / subTotalAmount) * 100;
			for (PurchaseOrderItem purchaseOrderItem : orderItems) {
				purchaseOrderItem.setTaxRate(vatRate);
			}

		}
	}

	private double fixInvalidAmount(double tax) {
		if (tax < 0 || Double.isNaN(tax)) {
			tax = 0;
		}
		return tax;
	}

	public void addProperty(String name, String value) {
		if (getProperties() == null) {
			setProperties(new HashMap<String, String>());
		}

		getProperties().put(name, value);
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public String getProperty(String key) {
		if (getProperties() == null) {
			return null;
		}

		return getProperties().get(key);
	}

	public String getProperty(String key, String defaultValue) {
		if (getProperties() == null) {
			return null;
		}

		String string = getProperties().get(key);
		if (StringUtils.isEmpty(string)) {
			return defaultValue;
		}

		return string;
	}

	public void removeProperty(String propertyName) {
		Map<String, String> properties = getProperties();
		if (properties == null) {
			return;
		}

		properties.remove(propertyName);
	}

	@XmlTransient
	@JsonIgnore
	public String getStatusDisplay() {
		int status = getStatus();
		if (status == PurchaseOrder.ORDER_PENDING)
			return Messages.getString("PurchaseOrder.16"); //$NON-NLS-1$
		else if (status == PurchaseOrder.ORDER_VERIFIED || status == PurchaseOrder.ORDER_PARTIALLY_RECEIVED)
			return Messages.getString("PurchaseOrder.17"); //$NON-NLS-1$
		else
			return ""; //$NON-NLS-1$
	}

	public void setStatusDisplay(String statusDisplay) {
		this.statusDisplay = statusDisplay;
	}

	@XmlTransient
	@JsonIgnore
	public String getCreatedDateAsString() {
		if (super.getCreatedDate() == null) {
			return ""; //$NON-NLS-1$
		}
		return DateUtil.formatAsShortDate(super.getCreatedDate());
	}

	public void setCreatedDateAsString(String createdDateAsString) {
	}

	@XmlTransient
	@JsonIgnore
	public String getReceivedDateAsString() {
		if (super.getReceivingDate() == null) {
			return ""; //$NON-NLS-1$
		}
		return DateUtil.formatAsShortDate(super.getReceivingDate());
	}

	public void setReceivedDateAsString(String receivedDateAsString) {
	}

	@XmlTransient
	@JsonIgnore
	public String getVerificationDateAsString() {
		if (super.getVarificationDate() == null) {
			return ""; //$NON-NLS-1$
		}
		return DateUtil.formatAsShortDate(super.getVarificationDate());
	}

	public void setVerificationDateAsString(String verificationDateAsString) {
	}

	@XmlTransient
	@JsonIgnore
	public String getDueDateAsString() {
		if (super.getExpectedDate() == null) {
			return ""; //$NON-NLS-1$
		}
		return DateUtil.formatAsShortDate(super.getExpectedDate());
	}

	public void setDueDateAsString(String dueDateAsString) {
	}

	@XmlTransient
	@JsonIgnore
	public String getCustomerName() {
		return getProperty(JSON_PROP_CUSTOMER_NAME, "");
	}

	public void setCustomerName(String customerName) {
		addProperty(JSON_PROP_CUSTOMER_NAME, customerName);
	}

	@XmlTransient
	@JsonIgnore
	public String getCustomerEmail() {
		return getProperty(JSON_PROP_CUSTOMER_EMAIL, "");
	}

	public void setCustomerEmail(String customerEmail) {
		addProperty(JSON_PROP_CUSTOMER_EMAIL, customerEmail);
	}

	@XmlTransient
	@JsonIgnore
	public String getCustomerPhone() {
		return getProperty(JSON_PROP_CUSTOMER_PHONE, "");
	}

	public void setCustomerPhone(String customerPhone) {
		addProperty(JSON_PROP_CUSTOMER_PHONE, customerPhone);
	}

	@XmlTransient
	@JsonIgnore
	public String getCustomerAddress() {
		return getProperty(JSON_PROP_CUSTOMER_ADDRESS, "");
	}

	public void setCustomerAddress(String customerAddress) {
		addProperty(JSON_PROP_CUSTOMER_ADDRESS, customerAddress);
	}

	@XmlTransient
	@JsonIgnore
	public boolean getEmailed() {
		return POSUtil.getBoolean(getProperty("emailed"), false);
	}

	public void setEmailed(boolean isEmailed) {
		addProperty("emailed", String.valueOf(isEmailed));
	}

	public boolean isFullPaid() {
		return Boolean.parseBoolean(getProperty("full.paid", Boolean.FALSE.toString())); //$NON-NLS-1$
	}

	public void putFullPaid(Boolean isFullPaid) {
		addProperty("full.paid", isFullPaid.toString()); //$NON-NLS-1$
	}

	@Override
	public String getEntityId() {
		return getId();
	}

	@Override
	public PosTransaction createTransaction() {
		boolean isServiceOrder = PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.SERVICE_ORDER)
				|| PurchaseOrderType.match(getPurchaseOrderType(), PurchaseOrderType.SERVICE_ORDER_RETURN);
		if (getDueValue() < 0.0) {
			PurchaseRefundTransaction transaction = new PurchaseRefundTransaction();
			transaction.setEntityId(getId());
			transaction.setEntityNo(getOrderId());
			transaction.setEntityType(EntityType.PURCHASE.name());
			transaction.setTransactionTime(StoreDAO.getServerTimestamp());
			transaction.putServiceOrderTransaction(isServiceOrder);
			return transaction;
		}

		PurchaseTransaction transaction = new PurchaseTransaction();
		transaction.setEntityId(getId());
		transaction.setEntityNo(getOrderId());
		transaction.setEntityType(EntityType.PURCHASE.name());
		transaction.setTransactionTime(StoreDAO.getServerTimestamp());
		transaction.putServiceOrderTransaction(isServiceOrder);
		String coaId = getCOAId();
		if (StringUtils.isNotBlank(coaId)) {
			ChartOfAccounts chartOfAccounts = DataProvider.get().getCOAFromMap(coaId);
			transaction.setChartOfAccounts(chartOfAccounts);
		}
		return transaction;
	}

	@Override
	public void setDueValue(double d) {
		setDueAmount(d);
		setPaidAmount(getSubtotalAmount() - d);
	}

	@Override
	public Double getDueValue() {
		return NumberUtil.round(getTotalAmount() - getPaidAmount());
	}

	@Override
	public double getSubTotal() {
		return getSubtotalAmount();
	}

	public List<PosTransaction> getTransactions() {
		return DataProvider.get().getTransactionsByEntityId(getEntityId());
	}

	@Override
	public Double getPaidAmount() {
		List<PosTransaction> srcTrans = getTransactions();
		if (srcTrans == null || srcTrans.isEmpty()) {
			return 0.0;
		}

		List<String> debitType = Arrays.asList(TransactionType.DEBIT.name(), TransactionType.OUT.name());
		return srcTrans.stream().mapToDouble(t -> debitType.contains(t.getTransactionType()) ? t.getAmount() : -t.getAmount()).sum();
	}

	public void addToEmailList(String emailId) {
		emailList = getEmailList();
		if (emailList == null) {
			emailList = new HashSet<String>();
		}
		emailList.add(emailId);

		buildEmailList();
	}

	public Set<String> getEmailList() {
		if (emailList == null) {
			emailList = new HashSet<String>();

			String property = getProperty("EMAIL_LIST"); //$NON-NLS-1$
			if (StringUtils.isNotEmpty(property)) {
				JSONArray jsonArray = new JSONArray(property);
				for (int i = 0; i < jsonArray.length(); i++) {
					emailList.add(jsonArray.getString(i));
				}
			}
		}
		return emailList;
	}

	public void buildEmailList() {
		if (emailList == null || emailList.isEmpty()) {
			removeProperty("EMAIL_LIST"); //$NON-NLS-1$
			return;
		}
		JSONArray jsonArray = new JSONArray();
		for (String emailId : emailList) {
			jsonArray.put(emailId);
		}
		addProperty("EMAIL_LIST", jsonArray.toString()); //$NON-NLS-1$
	}

	public boolean hasMailSent() {
		return !getEmailList().isEmpty();
	}

	public void addRandomEvent(String eventName, String description) {
		ActionHistory actionHistory = ActionHistory.createHistory(null, eventName, description, DataProvider.get().getCurrentUser(),
				DataProvider.get().getCurrentOutletId());
		getEvents().put(actionHistory.getActionName() + Math.random(), actionHistory);
	}

	public Map<String, ActionHistory> getEvents() {
		if (this.events == null) {
			this.events = new LinkedHashMap<>();
		}
		return this.events;
	}

	public String getBackOrderShortId() {
		return getProperty("backorder.short.id"); //$NON-NLS-1$
	}

	public void putBackOrderShortId(String orderId) {
		addProperty("backorder.short.id", orderId); //$NON-NLS-1$
	}

	public String getBackorderLongId() {
		return getProperty("backorder.long.id"); //$NON-NLS-1$
	}

	public void putBackOrderLongId(String id) {
		addProperty("backorder.long.id", id); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getVendorDisplay() {
		InventoryVendor vendor = getVendor();
		if (vendor != null) {
			return vendor.getName();
		}
		return "";
	}

	public void setVendorDisplay(String vendorDisplay) {
	}

	@XmlTransient
	@JsonIgnore
	public String getLocationDisplay() {
		InventoryLocation invLocation = getInventoryLocation();
		if (invLocation != null) {
			return invLocation.getName();
		}
		return "";
	}

	public void setLocationDisplay(String locationDisplay) {
	}

	public String getProjectNameDisplay() {
		if (StringUtils.isNotBlank(getProjectId())) {
			Project project = (Project) DataProvider.get().getObjectOf(Project.class, getProjectId());
			if (project != null) {
				return project.getNameDisplay();
			}
		}
		return null;
	}

	public String getTermsAndConditionsId() {
		return getProperty("termsAndConditionsId"); //$NON-NLS-1$
	}

	public void putTermsAndConditionsId(String id) {
		addProperty("termsAndConditionsId", id); //$NON-NLS-1$
	}

	public String getCreatedByUserId() {
		return getProperty("created_by", StringUtils.EMPTY); //$NON-NLS-1$
	}

	public void putCreatedByUserId(String userId) {
		addProperty("created_by", userId); //$NON-NLS-1$
	}

	public String getApprovedByUserId() {
		return getProperty("approved_by", StringUtils.EMPTY); //$NON-NLS-1$
	}

	public void putApprovedByUserId(String userId) {
		addProperty("approved_by", userId); //$NON-NLS-1$
	}

	public String getReceivedByUserId() {
		return getProperty("received_by", StringUtils.EMPTY); //$NON-NLS-1$
	}

	public void putReceivedByUserId(String userId) {
		addProperty("received_by", userId); //$NON-NLS-1$
	}

	public void setProject(Project project) {
		this.project = project;
		if (project == null) {
			removeProperty("project_name");
			setProjectId(null);
		}
		else {
			addProperty("project_name", project.getName());
			setProjectId(project.getId());
		}
	}

	public Project getProject() {
		if (this.project != null) {
			return this.project;
		}
		String projectId = getProjectId();
		if (StringUtils.isNotBlank(projectId)) {
			this.project = (Project) DataProvider.get().getObjectOf(Project.class, projectId);
		}

		return this.project;
	}

	public String getRequisitionId() {
		return getProperty("requisition.id");
	}

	public void putRequisitionId(String requisitionId) {
		addProperty("requisition.id", requisitionId);
	}

	public void setRequisition(Requisition requisition) {
		this.requisition = requisition;
		if (requisition != null) {
			putRequisitionId(requisition.getId());
		}
		else {
			putRequisitionId(null);
		}
	}

	public Requisition getRequisition() {
		return requisition;
	}

	public DiscountType getDiscountType() {
		String property = getProperty("discountType");
		if (StringUtils.isBlank(property)) {
			return null;
		}
		return DiscountType.valueOf(property);
	}

	public void putDiscountType(DiscountType discountType) {
		if (discountType != null) {
			addProperty("discountType", discountType.name());
		}
	}

	public double getDiscountValue() {
		return NumberUtil.parseOrGetZero(getProperty("discountValue", "0.0")).doubleValue();
	}

	public void putDiscountValue(double discountValue) {
		addProperty("discountValue", String.valueOf(discountValue));
	}

	public double getVatValue() {
		return NumberUtil.parseOrGetZero(getProperty("vatValue", "0.0")).doubleValue();
	}

	public void putVatValue(double vatValue) {
		addProperty("vatValue", String.valueOf(vatValue));
	}

	public DiscountType getVatType() {
		String property = getProperty("vatType");
		if (StringUtils.isBlank(property)) {
			return null;
		}
		return DiscountType.valueOf(property);
	}

	public void putVatType(DiscountType vatType) {
		addProperty("vatType", vatType == null ? null : vatType.name());
	}

	public boolean isAdjustedInventory() {
		return POSUtil.getBoolean(getProperty("isAdjustedInventory"), true);
	}

	public void putAdjustedInventory(boolean isAdjustedInventory) {
		addProperty("isAdjustedInventory", String.valueOf(isAdjustedInventory));
	}

	public String getSourcePOId() {
		return getProperty("sourcePOId");
	}

	public void putSourcePOId(String sourcePOId) {
		addProperty("sourcePOId", String.valueOf(sourcePOId));
	}

	public boolean hasFixedAssetItems() {
		return POSUtil.getBoolean(getProperty("hasFixedAssetItems"), false);
	}

	public void puthasFixedAssetItems(boolean hasFixedAssetItems) {
		addProperty("hasFixedAssetItems", String.valueOf(hasFixedAssetItems));
	}

	public String getCOAId() {
		if (!hasProperty("coaId")) {
			return "1005";
		}
		return getProperty("coaId");
	}

	public void putCOAId(String coaId) {
		addProperty("coaId", String.valueOf(coaId));
	}

	public String getChartOfAccountsName() {
		String coaCode = getCOAId();
		if (StringUtils.isNotBlank(coaCode)) {
			ChartOfAccounts coaAccount = DataProvider.get().getCOAFromMap(coaCode);
			return coaAccount.getNameDisplay();
		}
		return StringUtils.EMPTY;
	}

	public void putIsReturnedFromPurchaseCrudView(boolean isReturnedFromPurchaseCrudView) {
		this.isReturnedFromPurchaseCrudView = isReturnedFromPurchaseCrudView;
	}

	public boolean isReturnedFromPurchaseCrudView() {
		return isReturnedFromPurchaseCrudView;
	}

	public void putTotalPurchaseAmount(double totalPurchaseAmount) {
		this.totalPurchaseAmount = totalPurchaseAmount;
	}

	public double getTotalPurchaseAmount() {
		return totalPurchaseAmount;
	}

	public void putTotalPurchaseTaxAmount(double totalPurchaseTaxAmount) {
		this.totalPurchaseTaxAmount = totalPurchaseTaxAmount;
	}

	public double getTotalPurchaseTaxAmount() {
		return totalPurchaseTaxAmount;
	}
}