package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.ChemoCycle;
import com.floreantpos.model.ChemoProcessNote;

public class ChemoProcessNoteDAO extends BaseChemoProcessNoteDAO {

	public ChemoProcessNoteDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		ChemoProcessNote bean = (ChemoProcessNote) obj;
		if (bean == null) {
			throw new PosException("ChemoProcessNote not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<ChemoProcessNote> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.desc(ChemoProcessNote.PROP_PROCESS_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all process notes for a specific chemo cycle
	 * @param chemoCycle the chemo cycle
	 * @return list of process notes
	 */
	public List<ChemoProcessNote> findByChemoCycle(ChemoCycle chemoCycle) {
		if (chemoCycle == null || chemoCycle.getId() == null) {
			throw new IllegalArgumentException("ChemoCycle must not be null and must have an ID");
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(ChemoProcessNote.PROP_CHEMO_CYCLE, chemoCycle));
			criteria.addOrder(Order.asc(ChemoProcessNote.PROP_PROCESS_DATE));
			return criteria.list();
		}
	}

	/**
	 * Find all process notes for a specific chemo cycle by ID
	 * @param chemoCycleId the chemo cycle ID
	 * @return list of process notes
	 */
	public List<ChemoProcessNote> findByChemoCycleId(String chemoCycleId) {
		if (chemoCycleId == null) {
			throw new IllegalArgumentException("ChemoCycle ID must not be null");
		}
		ChemoCycle chemoCycle = ChemoCycleDAO.getInstance().get(chemoCycleId);
		if (chemoCycle == null) {
			throw new PosException("ChemoCycle not found with ID: " + chemoCycleId);
		}
		return findByChemoCycle(chemoCycle);
	}
}
