package com.floreantpos.model;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.PosLog;
import com.floreantpos.model.base.BasePatientHistory;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.GsonUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PatientHistory extends BasePatientHistory implements TimedModel, PropertyContainer2 {
	private static final long serialVersionUID = 1L;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;
	private transient JsonObject propertiesContainer;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public PatientHistory() {
	}

	/**
	 * Constructor for primary key
	 */
	public PatientHistory(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public JsonObject getPropertyStore() {
		if (propertiesContainer != null) {
			return propertiesContainer;
		}

		String jsonProperties = super.getProperties();
		if (StringUtils.isEmpty(jsonProperties)) {
			propertiesContainer = new JsonObject();
		}
		else {
			propertiesContainer = new Gson().fromJson(jsonProperties, JsonObject.class);
		}
		return propertiesContainer;
	}

	public String getHeaderTitle() {
		String historyType = getHistoryRefType();

		if (historyType == null) {
			return "History";
		}

		if (HistoryRefType.BOOKING_INFO.name().equals(historyType)) {
			return formatTitle("Admission", "");
		}
		if (HistoryRefType.DOCTOR_VISIT.name().equals(historyType)) {
			return formatTitle("Doctor Visit", getDoctorName());
		}
		if (HistoryRefType.NURSE_VISIT.name().equals(historyType)) {
			return formatTitle("Nurse Visit", getNurseName());
		}
		if (HistoryRefType.PRESCRIPTION.name().equals(historyType)) {
			return formatTitle("Prescription", getDoctorName());
		}
		if (HistoryRefType.CHEMOTHERAPY.name().equals(historyType)) {
			return formatTitle("Chemotherapy", getChemoCycleTitle());
		}

		return "History";
	}

	/**
	 * Get ChemoCycle title for display (e.g., "Cycle #1")
	 */
	private String getChemoCycleTitle() {
		try {
			com.floreantpos.model.ChemoCycle chemoCycle = getChemoCycle();
			if (chemoCycle != null && chemoCycle.getCycleNumber() != null) {
				return "Cycle #" + chemoCycle.getCycleNumber();
			}
		} catch (Exception e) {
			// Ignore - ChemoCycle might not be available
		}
		return "";
	}

	/**
	 * Get the associated ChemoCycle object (lazy load)
	 */
	public com.floreantpos.model.ChemoCycle getChemoCycle() {
		if (HistoryRefType.CHEMOTHERAPY.name().equals(getHistoryRefType()) && StringUtils.isNotBlank(getRefId())) {
			try {
				return com.floreantpos.model.dao.ChemoCycleDAO.getInstance().get(getRefId());
			} catch (Exception e) {
				PosLog.error(getClass(), "Error loading ChemoCycle: " + e.getMessage());
			}
		}
		return null;
	}

	private String formatTitle(String prefix, String name) {
		return prefix + (StringUtils.isBlank(name) ? "" : ": " + name);
	}

	public String getPatientVitalHtml() {
		StringBuilder builder = new StringBuilder();
		PatientVital patientVital = getPatientVital();
		if (patientVital != null) {
			String age = patientVital.age;
			String height = patientVital.height;
			String weight = patientVital.weight;
			String temperature = patientVital.temperature;
			String bpDown = patientVital.bpDown;
			String bpUp = patientVital.bpUp;
			String bpm = patientVital.bpm;
			builder.append("<p><b><i><u>Patient vital</u></i></b></p>");
			builder.append("<ul>");
			if (StringUtils.isNotBlank(age)) {
				builder.append("<div><strong>Age:</strong> " + age + "</div>");
			}
			if (StringUtils.isNotBlank(height)) {
				builder.append("<div><strong>Height:</strong> " + height + " cm</div>");
			}
			if (StringUtils.isNotBlank(weight)) {
				builder.append("<div><strong>Weight:</strong> " + weight + " KG</div>");
			}
			if (StringUtils.isNotBlank(temperature)) {
				builder.append("<div><strong>Temperature:</strong> " + temperature + " ℉</div>");
			}
			if (StringUtils.isNotBlank(bpDown) && StringUtils.isNotBlank(bpUp)) {
				builder.append("<div><strong>Blood pressure:</strong> " + bpDown + "/" + bpUp + "</div>");
			}
			if (StringUtils.isNotBlank(bpm)) {
				builder.append("<div><strong>BPM:</strong> " + bpm + "</div>");
			}
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public PatientVital getPatientVital() {
		String vitalJson = getVitalJson();
		if (StringUtils.isBlank(vitalJson)) {
			return null;
		}
		return GsonUtil.createGson().fromJson(vitalJson, PatientVital.class);
	}

	public void putPatientVital(PatientVital patientVital) {
		setVitalJson(GsonUtil.createGson().toJson(patientVital));
	}

	public String getProblemsAsHtml() {
		StringBuilder builder = new StringBuilder();
		List<String> problems = getProblems();
		if (problems != null && !problems.isEmpty()) {
			builder.append("<p><b><i><u>Problems:</u></i></b></p>");
			builder.append("<ul>");
			for (String problem : problems) {
				builder.append("<li>" + problem + "</li>");
			}
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public List<String> getProblems() {
		String problemsJson = getProblemsJson();
		if (StringUtils.isBlank(problemsJson)) {
			return null;
		}
		return GsonUtil.convertJSONToList(problemsJson, String.class);
	}

	public void putProblems(List<String> problems) {
		setProblemsJson(GsonUtil.createGson().toJson(problems));
	}

	public String getAllergiesAsHtml() {
		StringBuilder builder = new StringBuilder();
		List<PatientAllergy> allergies = getAllergies();
		if (allergies != null && !allergies.isEmpty()) {
			builder.append("<p><b><i><u>Allergy:</u></i></b></p>");
			builder.append("<ul>");
			String allergiesString = allergies.stream().map(t -> t.name + " (" + t.severity + ")").collect(Collectors.joining(", "));
			builder.append(allergiesString);
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public List<PatientAllergy> getAllergies() {
		String allergiesJson = getAllergiesJson();
		if (StringUtils.isBlank(allergiesJson)) {
			return null;
		}
		return GsonUtil.convertJSONToList(allergiesJson, PatientAllergy.class);
	}

	public void putAllergies(List<PatientAllergy> allergies) {
		setAllergiesJson(GsonUtil.createGson().toJson(allergies));
	}

	public String getPatientDiagnosesAsHtml() {
		StringBuilder builder = new StringBuilder();
		List<PatientDiagnosis> patientDiagnoses = getPatientDiagnoses();
		if (patientDiagnoses != null && !patientDiagnoses.isEmpty()) {
			builder.append("<p><b><i><u>Diagnoses:</u></i></b></p>");
			builder.append("<ul>");
			for (PatientDiagnosis patientDiagnosis : patientDiagnoses) {
				builder.append("<li>" + patientDiagnosis.name + "</li>");
			}
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public List<PatientDiagnosis> getPatientDiagnoses() {
		String diagnosesJson = getDiagnosesJson();
		if (StringUtils.isBlank(diagnosesJson)) {
			return null;
		}
		return GsonUtil.convertJSONToList(diagnosesJson, PatientDiagnosis.class);
	}

	public void putPatientDiagnoses(List<PatientDiagnosis> patientDiagnoses) {
		setDiagnosesJson(GsonUtil.createGson().toJson(patientDiagnoses));
	}

	public String getPatientMedicationsAsHtml() {
		StringBuilder builder = new StringBuilder();
		List<PatientMedication> patientMedications = getPatientMedications();
		if (patientMedications != null && !patientMedications.isEmpty()) {
			builder.append("<p><b><i><u>Medications</u></i></b></p>");
			builder.append("<ul>");
			for (PatientMedication patientMedication : patientMedications) {
				builder.append("<li>" + patientMedication.name + "</li>");
				builder.append(patientMedication.instruction);
			}
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public List<PatientMedication> getPatientMedications() {
		String medicationsJson = getMedicationsJson();
		if (StringUtils.isBlank(medicationsJson)) {
			return null;
		}
		return GsonUtil.convertJSONToList(medicationsJson, PatientMedication.class);
	}

	public void putPatientMedications(List<PatientMedication> patientMedications) {
		setMedicationsJson(GsonUtil.createGson().toJson(patientMedications));
	}

	public String getInstructionsAsHtml() {
		StringBuilder builder = new StringBuilder();
		List<String> instructions = getInstructions();
		if (instructions != null && !instructions.isEmpty()) {
			builder.append("<p><b><i><u>Instructions:</u></i></b></p>");
			builder.append("<ul>");
			for (String ins : instructions) {
				builder.append("<li>" + ins + "</li>");
			}
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public List<String> getInstructions() {
		String instructionsJson = getInstructionsJson();
		if (StringUtils.isBlank(instructionsJson)) {
			return null;
		}
		return GsonUtil.convertJSONToList(instructionsJson, String.class);
	}

	public void putInstructions(List<String> instructionsJson) {
		setInstructionsJson(GsonUtil.createGson().toJson(instructionsJson));
	}

	public String getNoteAsHtml() {
		StringBuilder builder = new StringBuilder();
		String note = getNote();
		if (StringUtils.isNotBlank(note)) {
			builder.append("<p><b><i><u>Note:</u></i></b></p>");
			builder.append("<ul>");
			builder.append(note);
			builder.append("</ul>");
		}
		return builder.toString();
	}

	public class PatientMedication {
		public final String name;
		public final String instruction;
		public final Date takingTime;
		public final String dosage;
		public final String status;
		public final String note;

		public PatientMedication(String name, String instruction) {
			this(name, instruction, null, null, null, null);
		}

		public PatientMedication(String name, String instruction, Date takingTime, String dosage, String status, String note) {
			this.name = name;
			this.instruction = instruction;
			this.takingTime = takingTime;
			this.dosage = dosage;
			this.status = status;
			this.note = note;
		}

		public String getName() {
			return name;
		}

		public String getInstruction() {
			return instruction;
		}

		public Date getTakingTime() {
			return takingTime;
		}

		public String getDosage() {
			return dosage;
		}

		public String getStatus() {
			return status;
		}

		public String getNote() {
			return note;
		}

	}

	public class PatientDiagnosis {
		final String name;

		public PatientDiagnosis(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}
	}

	public class PatientAllergy {
		final String name;
		final String severity;

		public PatientAllergy(String name, String severity) {
			this.name = name;
			this.severity = severity;
		}

		public String getName() {
			return name;
		}

		public String getSeverity() {
			return severity;
		}
	}

	public class PatientVital {
		public final String age;
		public final String height;
		public final String weight;
		public final String bpUp;
		public final String bpDown;
		public final String temperature;
		public final String bpm;

		public PatientVital(String age, String height, String weight, String bpUp, String bpDown, String temperature, String bpm) {
			this.age = age;
			this.height = height;
			this.weight = weight;
			this.bpUp = bpUp;
			this.bpDown = bpDown;
			this.temperature = temperature;
			this.bpm = bpm;
		}

		public String getAge() {
			return age;
		}

		public String getHeight() {
			return height;
		}

		public String getWeight() {
			return weight;
		}

		public String getBpUp() {
			return bpUp;
		}

		public String getBpDown() {
			return bpDown;
		}

		public String getTemperature() {
			return temperature;
		}

		public String getBpm() {
			return bpm;
		}
	}

	public enum HistoryRefType {
		BOOKING_INFO, DOCTOR_VISIT, NURSE_VISIT, PRESCRIPTION, CHEMOTHERAPY;
	}

	public String getFormattedCreatedDate() {
		if (super.getCreateDate() == null) {
			return null;
		}
		return DateUtil.formatReportDateWithStoreTimeZone(super.getCreateDate());
	}

	public void putDoctorId(String doctorId) {
		if (doctorId == null) {
			removeProperty("doctorId"); //$NON-NLS-1$
			return;
		}
		addProperty("doctorId", doctorId); //$NON-NLS-1$
	}

	public String getDoctorId() {
		return getProperty("doctorId"); //$NON-NLS-1$
	}

	public void putDoctorName(String doctorName) {
		if (doctorName == null) {
			removeProperty("doctorName"); //$NON-NLS-1$
			return;
		}
		addProperty("doctorName", doctorName); //$NON-NLS-1$
	}

	public String getDoctorName() {
		return getProperty("doctorName"); //$NON-NLS-1$
	}

	public void putNurseName(String nurseName) {
		if (nurseName == null) {
			removeProperty("nurseName"); //$NON-NLS-1$
			return;
		}
		addProperty("nurseName", nurseName); //$NON-NLS-1$
	}

	public String getNurseName() {
		return getProperty("nurseName"); //$NON-NLS-1$
	}

	public void putBookingReleaseDate(Date dischargeDate) {
		addProperty("bookingReleaseDate", //$NON-NLS-1$
				dischargeDate == null ? null : new SimpleDateFormat(BaseDataServiceDao.GSON_PARSING_DATE_FORMAT).format(dischargeDate));

	}

	public Date getBookingReleaseDate() {
		String dischargeDateString = getProperty("bookingReleaseDate"); //$NON-NLS-1$
		if (StringUtils.isBlank(dischargeDateString)) {
			return null;
		}
		try {
			return new SimpleDateFormat(BaseDataServiceDao.GSON_PARSING_DATE_FORMAT).parse(dischargeDateString);
		} catch (Exception e) {
			PosLog.error(getClass(), e.getMessage());
		}
		return null;
	}

	public String getFormattedBookingReleaseDate() {
		if (getBookingReleaseDate() == null) {
			return null;
		}
		return DateUtil.formatReportDateWithStoreTimeZone(getBookingReleaseDate());
	}

}