package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.NullPrecedence;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.AppointmentStatus;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Customer;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.swing.PaginationSupport;

public class PatientDAO extends BasePatientDAO {

	public PatientDAO() {
	}

	public Patient loadFullPatient(String patientId) {
		try (Session session = createNewSession()) {
			Patient patient = (Patient) get(Patient.class, patientId, session);
			Hibernate.initialize(patient.getPatientAllergies());
			Hibernate.initialize(patient.getDeliveryAddresses());
			return patient;
		}
	}

	public Patient loadFullPatient(Patient patient) {
		try (Session session = createNewSession()) {
			session.refresh(patient);
			Hibernate.initialize(patient.getPatientAllergies());
			Hibernate.initialize(patient.getDeliveryAddresses());
			return patient;
		}
	}

	@Override
	public List<Patient> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			return criteria.list();
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		CustomerDAO.doCheckValidation(obj);
		CustomerDAO.doSaveCreateDate(obj);
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		Patient patient = (Patient) obj;
		if (patient == null) {
			throw new PosException("Patient not found");
		}

		Criteria criteria = session.createCriteria(Appointment.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.ne(Appointment.PROP_STATUS, AppointmentStatus.CANCELLED.name()));
		criteria.add(Restrictions.ge(Appointment.PROP_FROM_DATE, DateUtil.convertServerTimeToBrowserTime(StoreDAO.getServerTimestamp())));
		criteria.setMaxResults(1);
		Appointment appointment = (Appointment) criteria.uniqueResult();
		if (appointment != null) {
			throw new PosException(
					"The patient cannot be deleted because an appointment is booked on " + DateUtil.formatDateAsString(appointment.getFromDate()));
		}

		patient.setDeleted(Boolean.TRUE);
		saveOrUpdate(patient, session);
		ActionHistoryDAO.saveHistory(ActionHistory.PATIENT_DELETED, "Agent name: " + patient.getFirstName() + ", id: " + patient.getId() + ", phone: "
				+ patient.getWorkPhoneNo() + ", email: " + patient.getEmail());
	}

	public void findBy(String patientNameOrPhoneOrId, PaginationSupport tableModel, String sortValue, String status) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			//			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(patientNameOrPhoneOrId)) {
				Disjunction or = Restrictions.disjunction();
				or.add(Restrictions.ilike(Customer.PROP_NAME, patientNameOrPhoneOrId, MatchMode.ANYWHERE));
				or.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, patientNameOrPhoneOrId, MatchMode.END));
				or.add(Restrictions.ilike(Customer.PROP_ID, patientNameOrPhoneOrId, MatchMode.END));
				criteria.add(or);
			}

			patientStatus(status, criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			if (sortValue != null) {
				if (sortValue.equals("Name")) { //$NON-NLS-1$
					criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
				}
				if (sortValue.equals("Last update")) { //$NON-NLS-1$
					criteria.addOrder(Order.desc(Customer.PROP_LAST_UPDATE_TIME).nulls(NullPrecedence.LAST));
				}
			}
			tableModel.setRows(criteria.list());
		}
	}

	private void patientStatus(String status, Criteria criteria) {
		if (StringUtils.isNotEmpty(status)) {
			if (status.equals(POSConstants.ACTIVE)) {
				criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, true));
			}
			else if (status.equals(POSConstants.DEACTIVE)) {
				criteria.add(Restrictions.eq(Patient.PROP_ACTIVE, false));
			}
		}
	}

	public void findByNameOrInvoiceOrPhoneOrEmail(String nameString, String phoneOrEmailString, String invoiceNo, List<String> bookingIds,
			PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(nameString)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, nameString, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotEmpty(phoneOrEmailString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_MOBILE_NO, phoneOrEmailString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_EMAIL, phoneOrEmailString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			if (StringUtils.isNotEmpty(invoiceNo)) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Ticket.class);
				detachedCriteria.add(Restrictions.ilike(Ticket.PROP_ID, invoiceNo, MatchMode.END));
				detachedCriteria.setProjection(Projections.property(Ticket.PROP_CUSTOMER_ID));
				criteria.add(Property.forName(Patient.PROP_ID).in(detachedCriteria));
			}

			if (bookingIds != null && !bookingIds.isEmpty()) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(BookingInfo.class);
				detachedCriteria.add(Property.forName(BookingInfo.PROP_BOOKING_ID).in(bookingIds));
				detachedCriteria.setProjection(Projections.property(BookingInfo.PROP_CUSTOMER_ID));
				criteria.add(Property.forName(Patient.PROP_ID).in(detachedCriteria));
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findByPhoneOrName(String searchString, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, null, tableModel);
	}

	public void findByPhoneOrName(String searchString, String status, PaginationSupport tableModel) {
		findByPhoneOrName(searchString, status, null, tableModel);
	}

	public void findByPhoneOrName(String searchString, String status, String gender, PaginationSupport tableModel) {
		findByPhoneOrName(null, searchString, status, gender, tableModel);
	}

	public void findByPhoneOrName(Outlet outlet, String searchString, String status, PaginationSupport tableModel) {
		findByPhoneOrName(outlet, searchString, status, null, tableModel);
	}

	public void findByPhoneOrName(Outlet outlet, String searchString, String status, String gender, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);

			if (outlet != null) {
				criteria.add(Restrictions.eq(Patient.PROP_OUTLET_ID, outlet.getId()));
			}

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			patientStatus(status, criteria);

			// Gender filter
			if (StringUtils.isNotEmpty(gender)) {
				criteria.add(Restrictions.eq(Patient.PROP_GENDER, gender).ignoreCase());
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public List<Patient> findByPhoneOrName(String searchString) {
		try (Session session = createNewSession()) {
			return findByPhoneOrName(searchString, session);
		}
	}

	public List<Patient> findByPhoneOrName(String searchString, int offset, int limit) {
		try (Session session = createNewSession()) {
			return findByPhoneOrName(searchString, offset, limit, session);
		}
	}

	public List<Patient> findByPhoneOrName(String searchString, Session session) {
		return findByPhoneOrName(searchString, 0, -1, session);
	}

	public List<Patient> findByPhoneOrName(String searchString, int offset, int limit, Session session) {
		Criteria criteria = session.createCriteria(Patient.class);

		addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

		if (StringUtils.isNotEmpty(searchString)) {
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
					Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
			criteria.add(disjunction);
		}
		criteria.addOrder(Order.asc(Customer.PROP_NAME).ignoreCase());
		if (offset > 0) {
			criteria.setFirstResult(offset);
		}
		if (limit > 0) {
			criteria.setMaxResults(limit);
		}

		List<Patient> result = criteria.list();
		if (result == null || result.isEmpty()) {
			return Collections.emptyList();
		}
		return result;
	}

	public int countByPhoneOrName(String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}
			criteria.setProjection(Projections.rowCount());
			Number count = (Number) criteria.uniqueResult();
			return count == null ? 0 : count.intValue();
		}
	}

	public Patient findByNameAndPhone(String phoneNo, String name) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(phoneNo)) {
				criteria.add(Restrictions.eq(Customer.PROP_MOBILE_NO, phoneNo));
			}

			if (StringUtils.isNotBlank(name)) {
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, name));
			}

			criteria.setMaxResults(1);
			Patient patient = (Patient) criteria.uniqueResult();
			return patient;
		}
	}

	public boolean isPatientExists(String phoneNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(phoneNo)) {
				criteria.add(Restrictions.eq(Customer.PROP_MOBILE_NO, phoneNo));
			}

			criteria.setMaxResults(1);
			Patient patient = (Patient) criteria.uniqueResult();
			return patient != null;
		}
	}

	public Patient findByEmailOrPhone(String searchString) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			addDeletedFilter(criteria);

			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.ilike(Customer.PROP_MOBILE_NO, searchString));
			disjunction.add(Restrictions.ilike(Customer.PROP_EMAIL, searchString));
			criteria.add(disjunction);

			criteria.setMaxResults(1);
			Patient patient = (Patient) criteria.uniqueResult();
			return patient;
		}
	}

	public boolean isMatchedMobileNoAndId(String phoneNumber, String patientId) {
		if (StringUtils.isBlank(phoneNumber) || StringUtils.isBlank(patientId)) {
			return false;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);
			criteria.add(Restrictions.eq(Patient.PROP_ID, patientId));
			criteria.add(Restrictions.eq(Patient.PROP_MOBILE_NO, phoneNumber));
			criteria.setReadOnly(true);
			criteria.setProjection(Projections.id());
			criteria.setMaxResults(1);
			return criteria.uniqueResult() != null;
		}
	}

	public void findByName(String searchString, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Patient.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));

			if (StringUtils.isNotEmpty(searchString)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, searchString, MatchMode.ANYWHERE)));
				criteria.add(disjunction);
			}

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(Customer.PROP_FIRST_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public List<Patient> findByIds(Collection<String> patientIds) {
		if (patientIds == null || patientIds.isEmpty()) {
			return new ArrayList<Patient>();
		}
		try (Session session = createNewSession()) {
			return findByIds(patientIds, session);
		}
	}

	@SuppressWarnings("unchecked")
	public List<Patient> findByIds(Collection<String> patientIds, Session session) {
		Criteria criteria = session.createCriteria(Patient.class);
		addDeletedFilter(criteria);
		criteria.add(Restrictions.in(Patient.PROP_ID, patientIds));
		criteria.add(Restrictions.eq(Customer.PROP_ACTIVE, true));
		return criteria.list();
	}

}
