/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosLog;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.util.NumberUtil;

public class UpdateDBTo475 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo475(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}

	public void update() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		try (Session session = sessionFactory.openSession()) {
			Transaction transaction = session.beginTransaction();

			// Query for unique accountNumbers with sum of amounts
			Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.ACCOUNTS_MANAGER_BALANCE.name()));
			criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.isNotNull(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.groupProperty(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER), "accountNumber"); //$NON-NLS-1$
			projectionList.add(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT), "totalAmount"); //$NON-NLS-1$
			criteria.setProjection(projectionList);

			@SuppressWarnings("unchecked")
			List<Object[]> results = criteria.list();

			PosLog.info(getClass(), "Found " + results.size() + " unique accountNumbers with ACCOUNTS_MANAGER_BALANCE transactions.");

			double tolerance = 0.01;
			int updateCount = 0;
			int activatedCount = 0;

			for (Object[] result : results) {
				String accountNumber = (String) result[0];
				Double sumAmount = result[1] != null ? ((Number) result[1]).doubleValue() : 0.0;
				sumAmount = NumberUtil.round(sumAmount);

				// Find corresponding User where id = accountNumber
				User user = null;
				try {
					// Try to get user using Criteria in the same session
					Criteria userCriteria = session.createCriteria(User.class);
					userCriteria.add(Restrictions.eq(User.PROP_ID, accountNumber));
					user = (User) userCriteria.uniqueResult();
				} catch (Exception e) {
					PosLog.error(getClass(), "Error retrieving user for accountNumber: " + accountNumber, e);
				}

				if (user != null) {
					boolean userNeedsUpdate = false;
					
					// Check if user is active, if not, make it active (only if sum is not 0)
					if (sumAmount != 0.0) {
						Boolean isActive = user.isActive();
						if (isActive == null || !isActive) {
							user.setActive(Boolean.TRUE);
							userNeedsUpdate = true;
							activatedCount++;
							PosLog.info(getClass(), "Activated user: " + user.getFullName() + " (accountNumber: " + accountNumber + ")");
						}
						
						if (!user.isAccountsManager() && !user.isAccountsOwner() && !user.isExpenseOfficer()) {
							user.setExpenseOfficer(true);
							user.putAccountsRoleActive(true);
							
							PosLog.info(getClass(), "Accounts role activated user: " + user.getFullName() + " (accountNumber: " + accountNumber + ")");
						}
					}
					
					Double userAccountManagerAmount = user.getAccountsManagerAmount();
					if (userAccountManagerAmount == null) {
						userAccountManagerAmount = 0.0;
					}

					// Compare values with tolerance
					double difference = Math.abs(sumAmount - userAccountManagerAmount);
					boolean valuesMatch = difference <= tolerance;

					if (valuesMatch) {
						if (userNeedsUpdate) {
							// Save user if we activated them
							UserDAO.getInstance().saveOrUpdate(user, session);
						}
						PosLog.info(getClass(), "AccountNumber: " + accountNumber + 
								" | User: " + user.getFullName() + 
								" | Sum from transactions: " + sumAmount + 
								" | User.accountManagerAmount: " + userAccountManagerAmount + 
								" | ✅ Values match");
					} else {
						PosLog.info(getClass(), "AccountNumber: " + accountNumber + 
								" | User: " + user.getFullName() + 
								" | Sum from transactions: " + sumAmount + 
								" | User.accountManagerAmount: " + userAccountManagerAmount + 
								" | ❌ Values do NOT match (difference: " + difference + ")");
						
						// Update User.accountManagerAmount to match the sum
						user.setAccountsManagerAmount(sumAmount);
						UserDAO.getInstance().saveOrUpdate(user, session);
						updateCount++;
						
						PosLog.info(getClass(), "Updated User.accountManagerAmount from " + userAccountManagerAmount + 
								" to " + sumAmount + " for user: " + user.getFullName());
					}
				} else {
					PosLog.info(getClass(), "AccountNumber: " + accountNumber + 
							" | Sum from transactions: " + sumAmount + 
							" | User not found");
				}
			}

			PosLog.info(getClass(), "Updated " + updateCount + " user(s) with mismatched accountManagerAmount values.");
			PosLog.info(getClass(), "Activated " + activatedCount + " inactive user(s).");

			transaction.commit();
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}

}

