package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosLog;
import com.floreantpos.main.Application;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.teminaltype.TerminalCategory;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class TerminalTypeDAO extends BaseTerminalTypeDAO {

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	private static final String EMPTY_NEWLINE_STRING = "\n"; //$NON-NLS-1$
	private static final String EMPTY_DOT_STRING = "."; //$NON-NLS-1$
	private static final String EMPTY_SPACE_STRING = " "; //$NON-NLS-1$

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public TerminalTypeDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	private String constructExceptionDetailsByOrderType(TerminalType terminalType, List<OrderType> orderTypes) {
		if (orderTypes != null && !orderTypes.isEmpty()) {
			StringBuilder builder = new StringBuilder(terminalType.getName() + EMPTY_SPACE_STRING + Messages.getString("TerminalTypeDAO.5")); //$NON-NLS-1$
			for (int i = 0; i < orderTypes.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + orderTypes.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	private String constructExceptionDetailsByTerminal(TerminalType terminalType, List<Terminal> terminals) {
		if (terminals != null && !terminals.isEmpty()) {
			StringBuilder builder = new StringBuilder(terminalType.getName() + EMPTY_SPACE_STRING + Messages.getString("TerminalTypeDAO.6")); //$NON-NLS-1$
			for (int i = 0; i < terminals.size(); i++) {
				String message = (i + 1) + EMPTY_DOT_STRING + EMPTY_SPACE_STRING + terminals.get(i).getName();
				builder.append(EMPTY_NEWLINE_STRING).append(message);
			}
			return builder.toString();
		}
		return EMPTY_STRING;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<TerminalType> findAll() {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public void initialize(TerminalType terminalType) {
		if (terminalType == null || terminalType.getId() == null)
			return;

		if (Hibernate.isInitialized(terminalType.getOrderTypes()) && Hibernate.isInitialized(terminalType.getCategories())) {
			return;
		}
		Session session = null;

		try {
			session = createNewSession();
			session.refresh(terminalType);

			Hibernate.initialize(terminalType.getOrderTypes());
			Hibernate.initialize(terminalType.getCategories());
		} finally {
			closeSession(session);
		}
	}

	public boolean nameExists(TerminalType terminalType, String name) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(TerminalType.PROP_NAME, name).ignoreCase());
			if (terminalType.getId() != null) {
				criteria.add(Restrictions.ne(TerminalType.PROP_ID, terminalType.getId()));
			}
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		}
	}

	@SuppressWarnings("unchecked")
	public TerminalType getTerminalType(TerminalCategory category) {
		if (category == null) {
			TerminalType thisTerminalType = Application.getInstance().getTerminal().getTerminalType();
			if (thisTerminalType != null) {
				return thisTerminalType;
			}
			category = TerminalCategory.REGULAR;
		}
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(this.getReferenceClass());
			this.addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(TerminalType.PROP_TYPE, category.getType()));
			List<TerminalType> list = criteria.list();
			if (list != null && list.size() > 0) {
				return list.get(0);
			}
			return null;
		}
	}

	public void saveOrUpdateTerminalTypes(List terminalTypeList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (terminalTypeList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			for (Iterator iterator = terminalTypeList.iterator(); iterator.hasNext();) {
				TerminalType item = (TerminalType) iterator.next();
				TerminalTypeDAO dao = TerminalTypeDAO.getInstance();
				TerminalType existingItem = dao.get(item.getId(), item.getOutletId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getName() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					dao.update(existingItem);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					dao.save(item);
				}

			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public TerminalType get(String id, String outletId) {
		return get(new TerminalType(id, outletId));
	}
}