package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Outlet;
import com.floreantpos.swing.PaginationSupport;

public class BankAccountDAO extends BaseBankAccountDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	@Override
	protected void delete(Object obj, Session session) {
		BankAccount bean = (BankAccount) obj;
		if (bean == null) {
			throw new PosException("Bank account not found!");
		}
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	@Override
	public List<BankAccount> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.addOrder(Order.asc(BankAccount.PROP_BANK_NAME));
			return criteria.list();
		}
	}
	
	public void findAllBankAccounts(String bankName, PaginationSupport tableModel, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BankAccount.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(BankAccount.PROP_ACTIVE, true));
			if (StringUtils.isNotBlank(bankName)) {
				criteria.add(Restrictions.ilike(BankAccount.PROP_BANK_NAME, bankName.trim(), MatchMode.ANYWHERE));
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$
			}
			tableModel.setNumRows(rowCount(criteria));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(BankAccount.PROP_BANK_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public void findAllBankName(String bankName, PaginationSupport tableModel) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BankAccount.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(bankName)) {
				criteria.add(Restrictions.ilike(BankAccount.PROP_BANK_NAME, bankName.trim(), MatchMode.ANYWHERE));
			}
			tableModel.setNumRows(rowCount(criteria));
			criteria.setProjection(Projections.property(BankAccount.PROP_BANK_NAME));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(Order.asc(BankAccount.PROP_BANK_NAME).ignoreCase());
			tableModel.setRows(criteria.list());
		}
	}

	public boolean isAccountExists(String existingId, String bankName, String accountNumber, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BankAccount.class);
			addDeletedFilter(criteria);
			if (StringUtils.isNotBlank(existingId)) {
				criteria.add(Restrictions.ne(BankAccount.PROP_ID, existingId));
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq("outletId", outletId)); //$NON-NLS-1$
			}
			criteria.add(Restrictions.eq(BankAccount.PROP_BANK_NAME, bankName).ignoreCase());
			criteria.add(Restrictions.eq(BankAccount.PROP_ACCOUNT_NO, accountNumber));
			criteria.setMaxResults(1);
			return criteria.uniqueResult() != null;
		}
	}

	public List<BankAccount> findBankAccount(String bankName, String accountName, String accountNo, Outlet outlet) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BankAccount.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(bankName)) {
				criteria.add(Restrictions.eq(BankAccount.PROP_BANK_NAME, bankName).ignoreCase());
			}
			if (StringUtils.isNotBlank(accountName)) {
				criteria.add(Restrictions.ilike(BankAccount.PROP_ACCOUNT_NAME, accountName, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotBlank(accountNo)) {
				criteria.add(Restrictions.eq(BankAccount.PROP_ACCOUNT_NO, accountNo));
			}

			if (outlet != null) {
				criteria.add(Restrictions.eq(CustomPayment.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.addOrder(Order.asc(BankAccount.PROP_BANK_NAME));
			return criteria.list();
		}
	}

	public List<BankAccount> findAllActive(String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(BankAccount.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(BankAccount.PROP_ACTIVE, true));

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(BankAccount.PROP_OUTLET_ID, outletId));
			}

			criteria.addOrder(Order.asc(BankAccount.PROP_BANK_NAME));
			return criteria.list();
		}
	}

}