/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.RecepieItem;
import com.floreantpos.model.RecipeTable;
import com.floreantpos.model.Store;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.ext.InvMapKey;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.BeanTableModel;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class RecepieDAO extends BaseRecepieDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public RecepieDAO() {
	}

	@Override
	protected void delete(Object obj, Session session) {
		Recepie bean = (Recepie) obj;
		if (bean == null) {
			throw new PosException(Messages.getString("RecepieDAO.0")); //$NON-NLS-1$
		}
		//checkForeignRelation(bean);
		bean.setDeleted(Boolean.TRUE);
		update(bean, session);
	}

	//	private void checkForeignRelation(Recepie bean) throws PosException {
	//		try (Session session = createNewSession()) {
	//			List<String> foreignItemNames = GenericDAO.getInstance().getForeignDataListNames(session, MenuItem.class, MenuItem.PROP_DEFAULT_RECIPE_ID,
	//					bean.getId());
	//			if (foreignItemNames != null && foreignItemNames.size() > 0) {
	//				String details = Messages.getString("RecepieDAO.1") + (foreignItemNames.size() > 1 ? Messages.getString("RecepieDAO.2") : ":"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	//				int count = 1;
	//				for (String itemName : foreignItemNames) {
	//					details += "\n" + count + ". " + itemName; //$NON-NLS-1$ //$NON-NLS-2$
	//					count++;
	//				}
	//				throw new PosException(Messages.getString("RecepieDAO.4"), details); //$NON-NLS-1$
	//			}
	//		}
	//	}

	public List<Recepie> findAll() {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			return criteria.list();
		}
	}

	public int rowCount(String name, boolean subRecipeOnly) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(Recepie.class);
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(name)) {
				criteria.add(Restrictions.ilike(Recepie.PROP_NAME, name.trim(), MatchMode.ANYWHERE));
			}
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadRecepies(BeanTableModel<Recepie> model, String name, boolean subRecipeOnly) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(Recepie.class);

			if (StringUtils.isNotEmpty(name)) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.ilike(Recepie.PROP_NAME, name.trim(), MatchMode.ANYWHERE));
				criteria.createAlias(Recepie.PROP_MENU_ITEM, "item"); //$NON-NLS-1$
				disjunction.add(Restrictions.ilike("item.name", name.trim(), MatchMode.START)); //$NON-NLS-1$
				criteria.add(disjunction);
			}
			addDeletedFilter(criteria);
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());
			criteria.setProjection(null);

			criteria.addOrder(Order.asc(Recepie.PROP_NAME));
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			List<Recepie> result = criteria.list();
			model.setRows(result);
		} finally {
			closeSession(session);
		}
	}

	public List<Recepie> findBy(MenuItem menuItem) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Recepie.PROP_MENU_ITEM, menuItem));
			addDeletedFilter(criteria);
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	//deleted filter not used due to unique name
	public Recepie findByName(String name, String id) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Recepie.PROP_NAME, name));
			if (StringUtils.isNotBlank(id)) {
				criteria.add(Restrictions.ne(Recepie.PROP_ID, id));
			}
			criteria.setMaxResults(1);
			return (Recepie) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	public List<String> getForeignDataListNames(MenuItem menuItem) {
		try (Session session = createNewSession()) {
			return getForeignDataListNames(menuItem, session);
		}
	}

	@SuppressWarnings("unchecked")
	public List<String> getForeignDataListNames(MenuItem menuItem, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.setProjection(Projections.distinct(Projections.property(Recepie.PROP_NAME)));
		criteria.createAlias("recepieItems", "items"); //$NON-NLS-1$ //$NON-NLS-2$
		criteria.add(Restrictions.or(Restrictions.eq(Recepie.PROP_MENU_ITEM, menuItem), Restrictions.eq("items." + RecepieItem.PROP_INVENTORY_ITEM, menuItem))); //$NON-NLS-1$
		addDeletedFilter(criteria);
		criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
		return criteria.list();
	}

	public void adjustInventory(Double mainItemQty, List<Recepie> recipes) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			adjustInventory(mainItemQty, recipes, session);
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void adjustInventory(Double mainItemQty, List<Recepie> recipes, Session session) throws Exception {
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		HashMap<InvMapKey, Double> recipeMap = new HashMap<>();
		for (Recepie recepie : recipes) {
			Double actualYieldValue = recepie.getYield();
			Double cookingYield = recepie.getCookingYield();
			//TODO: REVIEW FOLLOWING LINE
			//				recepie.setPortion((cookingYield / actualYieldValue) * recepie.getPortion());

			MenuItem menuItem = recepie.getMenuItem();
			MenuItemDAO.getInstance().initializeUnits(menuItem);
			InventoryUnit unit = menuItem.getUnit();

			InvMapKey key = new InvMapKey(menuItem.getId(), unit == null ? null : unit.getId());
			key.setUnitType(unit == null ? null : unit.getUnitType());
			Double previousValue = itemMap.get(key);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = recepie.getPortion();
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty <= 0)
				continue;
			itemMap.put(key, toBeAdjustQty);
			recepie.populateRecipeItemQuantity(mainItemQty, recipeMap, cookingYield / actualYieldValue);
		}
		InventoryLocation defaultOutInventoryLocation = InventoryLocationDAO.getInstance().getDefaultOutInventoryLocation(session);
		InventoryLocation defaultInInventoryLocation = InventoryLocationDAO.getInstance().getDefaultInInventoryLocation(session);
		adjustInventory(itemMap, InventoryTransactionType.IN, InventoryTransaction.REASON_PREPARE_IN, defaultInInventoryLocation, session);
		adjustInventory(recipeMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_PREPARE_OUT, defaultOutInventoryLocation, session);
	}

	public void adjustRecipeItemsFromInventory(Double mainItemQty, List<Recepie> recipes, Session session) throws Exception {
		adjustRecipeItemsFromInventory(mainItemQty, recipes, session, null);
	}

	public void adjustRecipeItemsFromInventory(Double mainItemQty, List<Recepie> recipes, Session session, TicketItem ticketItem) throws Exception {
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		HashMap<InvMapKey, Double> recipeMap = new HashMap<>();
		for (Recepie recepie : recipes) {
			if (recepie.isDeleted()) {
				continue;
			}
			Double actualYieldValue = recepie.getYield();
			Double cookingYield = recepie.getCookingYield();

			MenuItem menuItem = recepie.getMenuItem();
			if (menuItem == null) {
				continue;
			}

			String unitId = menuItem.getUnit() == null ? "" : menuItem.getUnit().getId(); //$NON-NLS-1$
			InvMapKey menuItemUnit = new InvMapKey(menuItem.getId(), unitId);
			Double previousValue = itemMap.get(menuItemUnit);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = recepie.getPortion();
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty <= 0)
				continue;
			itemMap.put(menuItemUnit, toBeAdjustQty);
			recepie.populateRecipeItemQuantity(mainItemQty, recipeMap, cookingYield / actualYieldValue);
		}
		InventoryLocation defaultOutInventoryLocation = InventoryLocationDAO.getInstance().getDefaultOutInventoryLocation(session);
		adjustInventory(recipeMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_PREPARE_OUT, defaultOutInventoryLocation, session, ticketItem);
	}

	private void adjustInventory(HashMap<InvMapKey, Double> itemMap, InventoryTransactionType transactionType, String reason, InventoryLocation location,
			Session session) throws Exception {
		adjustInventory(itemMap, transactionType, reason, location, session, null);
	}

	private void adjustInventory(HashMap<InvMapKey, Double> itemMap, InventoryTransactionType transactionType, String reason, InventoryLocation location,
			Session session, TicketItem ticketItem) throws Exception {
		Store store = DataProvider.get().getStore();
		boolean isUpdateOnHandBlncForSale = store.isUpdateOnHandBlncForSale();
		boolean isUpdateAvailBlncForSale = store.isUpdateAvlBlncForSale();
		for (InvMapKey menuItemKey : itemMap.keySet()) {
			String menuItemId = menuItemKey.getMenuItemId();
			Double unitQuantity = itemMap.get(menuItemKey);
			MenuItem menuItem = MenuItemDAO.getInstance().getMenuItemWithFields(menuItemId, MenuItem.PROP_NAME, MenuItem.PROP_PRICE, MenuItem.PROP_SKU,
					MenuItem.PROP_BARCODE, MenuItem.PROP_UNIT_ID, MenuItem.PROP_COST, MenuItem.PROP_AVERAGE_UNIT_PURCHASE_PRICE, MenuItem.PROP_AVG_COST);
			if (menuItem == null || !menuItem.isInventoryItem())
				continue;

			IUnit transactionUnit = (IUnit) DataProvider.get().getInventoryUnitById(menuItemKey.getUnitId());
			if (transactionUnit instanceof PackagingUnit) {
				MenuItemDAO.getInstance().initializeUnits(menuItem);
			}
			double baseUnitQuantity = menuItem.getBaseUnitQuantity(menuItemKey.getUnitId());

			menuItem.setId(menuItemId);
			InventoryTransaction trans = new InventoryTransaction();
			if (transactionType == InventoryTransactionType.IN) {
				trans.setToInventoryLocation(location);
			}
			else {
				trans.setFromInventoryLocation(location);
			}
			trans.setReason(reason);
			trans.setTransactionDate(new Date());
			trans.setMenuItem(menuItem);
			trans.setType(transactionType.getType());
			trans.setUser(DataProvider.get().getCurrentUser());
			trans.setUnitPrice(menuItem.getPrice() * baseUnitQuantity);
			trans.setQuantity(unitQuantity);

			if (ticketItem != null) {
				Ticket ticket = ticketItem.getTicket();
				if (ticket != null) {
					trans.setTicketId(ticket.getId());
					trans.setUser(ticket.getOwner());
				}
			}

			if (menuItemKey.getUnitId() != null) {
				trans.setUnit(menuItemKey.getUnitId());
				trans.setUnitType(menuItemKey.getUnitType());
			}
			else {
				InventoryUnit unit = menuItem.getUnit();
				if (unit != null) {
					trans.setUnit(unit.getId());
					trans.setUnitType(unit.getUnitType());
				}
				else {
					trans.setUnitType(UnitType.PACKAGING_UNIT.name());
				}
			}
			trans.setUnitCost(menuItem.getCost());

			//trans.setTotal(trans.getUnitPrice() * trans.getQuantity());
			trans.calculateTotal();
			InventoryTransactionDAO.getInstance().adjustInventoryStock(trans, session, isUpdateAvailBlncForSale, isUpdateOnHandBlncForSale);
			ActionHistoryDAO.logJournalForInventoryTrans(trans);
		}
	}

	public String getMenuItemName(Recepie recepie) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.createAlias("menuItem", "item"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.setProjection(Projections.property("item.name")); //$NON-NLS-1$
			criteria.add(Restrictions.eq(RecipeTable.PROP_ID, recepie.getId()));
			addDeletedFilter(criteria);
			return (String) criteria.uniqueResult();
		} finally {
			closeSession(session);
		}
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		//		Recepie recepie = (Recepie) obj;
		//		if (nameExists(recepie.getId(), recepie.getName())) {
		//			throw new PosException(Messages.getString("RecepieDAO.6")); //$NON-NLS-1$
		//		}
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		//		Recepie recepie = (Recepie) obj;
		//		if (nameExists(recepie.getId(), recepie.getName())) {
		//			throw new PosException(Messages.getString("RecepieDAO.6")); //$NON-NLS-1$
		//		}
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		//		Recepie recepie = (Recepie) obj;
		//		if (nameExists(recepie.getId(), recepie.getName())) {
		//			throw new PosException(Messages.getString("RecepieDAO.6")); //$NON-NLS-1$
		//		}
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public boolean nameExists(String id, String name) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (StringUtils.isNotEmpty(id)) {
				criteria.add(Restrictions.ne(Recepie.PROP_ID, id));
			}
			criteria.add(Restrictions.eq(Recepie.PROP_NAME, name).ignoreCase());
			addDeletedFilter(criteria);
			Number rowCount = (Number) criteria.uniqueResult();
			return rowCount != null && rowCount.intValue() > 0;
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateRecipes(List<Recepie> dataList, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		saveOrUpdateRecipes(dataList, updateLastUpdateTime, updateSyncTime, false, false);
	}

	//download and upload
	public void saveOrUpdateRecipes(List<Recepie> dataList, boolean updateLastUpdateTime, boolean updateSyncTime, boolean saveNewDataOnly, boolean forceUpdate)
			throws Exception {
		if (dataList == null)
			return;

		Map<String, MenuItem> itemMap = new HashMap<>();
		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (Iterator<Recepie> iterator = dataList.iterator(); iterator.hasNext();) {
				Recepie item = (Recepie) iterator.next();
				PosLog.debug(RecepieDAO.class, "Start saving recipe: " + item.getId());
				List<RecepieItem> items = item.getRecepieItems();
				item.setRecepieItems(null);
				MenuItem existingRecipeMenuItem = null;
				if (item.getMenuItem() != null) {
					String recipeItemId = item.getMenuItem().getId();
					existingRecipeMenuItem = itemMap.get(recipeItemId);
					if (existingRecipeMenuItem == null) {
						existingRecipeMenuItem = MenuItemDAO.getInstance().get(recipeItemId);
						if (existingRecipeMenuItem == null) {
							if (saveNewDataOnly) {
								continue;
							}
							throw new Exception("No item found with id " + recipeItemId); //$NON-NLS-1$
						}
						itemMap.put(existingRecipeMenuItem.getId(), existingRecipeMenuItem);
					}
				}
				item.setMenuItem(existingRecipeMenuItem);
				item.setUpdateLastUpdateTime(updateLastUpdateTime);
				item.setUpdateSyncTime(updateSyncTime);

				Recepie existingItem = get(item.getId());
				if (existingItem != null) {
					if (!forceUpdate && (saveNewDataOnly || !BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime()))) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					item.setVersion(version);
				}
				else {
					//item.setVersion(0);
					save(item, session);
				}
				List<RecepieItem> existingItems = null;
				if (existingItem != null && existingItem.getRecepieItems() != null) {
					existingItems = existingItem.getRecepieItems();
				}
				if (existingItems == null) {
					existingItems = new ArrayList<>();
				}
				if (items != null && items.size() > 0) {
					for (RecepieItem recipeItem : items) {
						MenuItem inventoryItem = recipeItem.getInventoryItem();
						if (inventoryItem != null) {
							String itemId = inventoryItem.getId();
							if (StringUtils.isEmpty(itemId)) {
								continue;
							}
							MenuItem existingMenuItem = itemMap.get(itemId);
							if (existingMenuItem == null) {
								existingMenuItem = MenuItemDAO.getInstance().get(itemId);
								if (existingMenuItem == null) {
									throw new Exception("No item found with id " + itemId); //$NON-NLS-1$
								}
								itemMap.put(existingMenuItem.getId(), existingMenuItem);
							}
							recipeItem.setInventoryItem(existingMenuItem);
						}
						recipeItem.setRecepie(item);
						int idx = existingItems.indexOf(recipeItem);
						RecepieItem existingRecipeItem = null;
						if (idx != -1) {
							existingRecipeItem = existingItems.get(idx);
							if (existingRecipeItem == null) {
								recipeItem.setVersion(0);
								RecepieItemDAO.getInstance().save(recipeItem, session);
							}
							else {
								recipeItem.setVersion(existingRecipeItem.getVersion());
							}
						}
						else {
							recipeItem.setVersion(0);
							RecepieItemDAO.getInstance().save(recipeItem, session);
						}
					}
				}
				item.setRecepieItems(items);
				saveOrUpdate(item, session);
			}
			tx.commit();
		} catch (Exception e) {
			//tx.rollback();
			throw e;
		}
	}
}