package com.floreantpos.model.util;

import java.io.File;
import java.io.Serializable;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.WeakHashMap;
import java.util.stream.Collectors;

import javax.swing.ImageIcon;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;

import com.floreantpos.PosLog;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.Currency;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DayPart;
import com.floreantpos.model.Department;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.ImageResource;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryUnit;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemInventoryStatus;
import com.floreantpos.model.MenuShift;
import com.floreantpos.model.Multiplier;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PosPrinters;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PriceShift;
import com.floreantpos.model.PrinterGroup;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.ReportGroup;
import com.floreantpos.model.SalesArea;
import com.floreantpos.model.Shift;
import com.floreantpos.model.Store;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TaxGroup;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.TerminalType;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.User;
import com.floreantpos.model.UserType;
import com.floreantpos.model.dao.CustomPaymentDAO;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.dao.GlobalConfigDAO;
import com.floreantpos.model.dao.ImageResourceDAO;
import com.floreantpos.model.dao.InventoryLocationDAO;
import com.floreantpos.model.dao.InventoryUnitDAO;
import com.floreantpos.model.dao.OrderTypeDAO;
import com.floreantpos.model.dao.OutletDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.model.dao.TicketDAO;
import com.floreantpos.model.dao.UserDAO;
import com.floreantpos.model.dao.UserTypeDAO;
import com.floreantpos.util.DefaultCOADataInserter.DataMode;
import com.orocube.medlogics.Module;

public abstract class DataProvider {
	public final static String DATA_PROVIDER_CLASS_NAME = "data.provider.class"; //$NON-NLS-1$

	private static DataProviderFactory dataProviderFactory;

	private List<CacheRefreshListener> refreshListeners = new ArrayList<>(5);

	private Ticket ticketToBeCreated;
	protected OrderType orderType;

	protected Map<String, UserType> userTypeMap = new WeakHashMap<String, UserType>();

	protected Map<String, InventoryUnit> inventoryUnitMap = new HashMap<String, InventoryUnit>();

	protected Map<Pair<String, String>, InventoryLocation> inventoryLocationMap = new HashMap<Pair<String, String>, InventoryLocation>();

	protected Map<String, Outlet> outletMap = new HashMap<String, Outlet>();

	public abstract void initialize();

	public abstract void initializeFormatByOutlet(String outletId);

	public abstract Store getStore();

	public abstract Outlet getOutlet();

	public void refreshOutlet(Outlet outlet) {

	}

	protected void loadInventoryUnit() {
		inventoryUnitMap.clear();
		List<InventoryUnit> allTestItemUnits = InventoryUnitDAO.getInstance().findAll();
		for (InventoryUnit inventoryUnit : allTestItemUnits) {
			inventoryUnitMap.put(inventoryUnit.getId(), inventoryUnit);
		}
	}

	protected void loadInventoryLocation() {
		inventoryLocationMap.clear();
		List<InventoryLocation> inventoryLocationList = InventoryLocationDAO.getInstance().findAll();
		for (InventoryLocation inventoryLocation : inventoryLocationList) {
			inventoryLocationMap.put(Pair.of(inventoryLocation.getId(), inventoryLocation.getOutletId()), inventoryLocation);
		}
	}

	public List<InventoryLocation> getInventoryLocationByOutletId(String outletId) {
		return inventoryLocationMap.entrySet().stream().filter(entry -> entry.getKey().getRight().equals(outletId)).map(Map.Entry::getValue)
				.collect(Collectors.toList());
	}

	protected void loadAllOutlets() {
		outletMap.clear();
		List<Outlet> outletList = OutletDAO.getInstance().findAll();
		for (Outlet outlet : outletList) {
			outletMap.put(outlet.getId(), outlet);
		}
	}

	public String getOutletId() {
		Outlet outlet = getOutlet();
		if (outlet == null)
			return null;
		return outlet.getId();
	}

	public Outlet getOutletById(String outletId) {
		if (StringUtils.isBlank(outletId)) {
			return null;
		}
		Outlet outlet = outletMap.get(outletId);
		if (outlet == null) {
			outlet = OutletDAO.getInstance().get(outletId);
			outletMap.put(outletId, outlet);
		}
		return outlet;
	};

	public OrderType getOrderType() {
		if (orderType == null) {
			orderType = OrderTypeDAO.getInstance().getOrderType(getOutlet());
		}

		return orderType;
	}

	//	public abstract StoreSession getStoreSession();

	//	public abstract void setStoreSession(StoreSession session);

	public abstract InventoryUnit getInventoryUnitById(String unitId);

	public IUnit getUnitById(String unitId, String unitType) {
		return null;
	};

	public IUnit getUnitByCode(String code) {
		return null;
	}

	public abstract TaxGroup getTaxGroupById(String taxGroupId);

	public abstract ReportGroup getReportGroupById(String reportGroupId);

	public abstract PrinterGroup getPrinterGroupById(String printerGroupId);

	public abstract Department getDepartmentById(String departmentId);

	public abstract OrderType getOrderTypeById(String orderTypeId, String outletId);

	public abstract InventoryLocation getInventoryLocationById(String inventoryLocationId, String outletId);

	public abstract Terminal getTerminalById(Integer terminalId, String outletId);

	public abstract Shift getShiftById(String shiftId, String outletId);

	public abstract Terminal getCurrentTerminal();

	public abstract void setCloudTerminal(Terminal terminal);

	//	public abstract OrderType getOrderType(String orderTypeId, String outletId);

	//	public abstract List<OrderType> getOrderTypes();

	public UserType getUserType(String userTypeId) {
		if (StringUtils.isEmpty(userTypeId))
			return null;

		UserType userType = userTypeMap.get(userTypeId);
		if (userType == null) {
			userType = UserTypeDAO.getInstance().get(userTypeId);
			userTypeMap.put(userTypeId, userType);
		}

		return userType;
	}

	public User getUserById(String userId, String outletId) {
		if (StringUtils.isEmpty(userId))
			return null;

		try (Session session = GenericDAO.getInstance().createNewSession()) {
			return UserDAO.getInstance().get(userId, outletId, session);
		}
	}

	public abstract Currency getCurrency(String currencyId, String outletId);

	public abstract TerminalType getTerminalType(String terminalTypeId, String outletId);

	public abstract Multiplier getMultiplierById(String id);

	public abstract List<Multiplier> getMultiplierList();

	public abstract InventoryLocation getDefaultInLocation();

	public abstract InventoryLocation getDefaultOutLocation();

	public abstract void refreshStore();

	public abstract void refreshCurrentTerminal();

	public abstract SalesArea getSalesArea(String salesAreaId);

	public abstract String getRecipeMenuItemName(Recepie recipe);

	public abstract PosPrinters getPrinters();

	public abstract MenuItemInventoryStatus getMenuItemStockStatus(MenuItem menuItem);

	public abstract Customer getCustomer(String id);

	public abstract Double getPriceFromPriceRule(MenuItem menuItem, OrderType orderType, Department department, SalesArea salesArea, Customer customer,
			IUnit unit);

	public abstract List<DayPart> getDaryPartShifts();

	public abstract List<PriceShift> getPriceShifts();

	public abstract List<MenuShift> getMenuShifts();

	public abstract File getAppConfigFileLocation();

	public abstract Currency getMainCurrency();

	public abstract void loadMainCurrencyAndFormats(String outletId);

	public abstract List<Currency> getCurrencies(String outletId);

	public abstract int getTimeZoneOffset();

	public static DataProvider get() {
		if (dataProviderFactory == null) {
			try {

				Properties properties = new Properties();
				properties.load(DataProvider.class.getResourceAsStream("/application.properties"));

				String className = properties.getProperty(DataProviderFactory.PROP_DATA_PROVIDER_CLASS);

				PosLog.info(DataProvider.class, "Data provider factory: " + className);

				if (StringUtils.isEmpty(className)) {
					throw new RuntimeException("Data provider factory class is needed. Put factory class name in application.properties file"
							+ "as data_provider_class=class_name, and put that file in class path root.");
				}
				dataProviderFactory = (DataProviderFactory) Class.forName(className).newInstance();
			} catch (Exception e) {
				LogFactory.getLog(DataProvider.class).error("Could not initialize data provider", e); //$NON-NLS-1$
			}
		}

		return dataProviderFactory.getDataProvider();
	}

	public Object getObjectOf(Class clazz, Serializable id) {
		try (Session session = StoreDAO.getInstance().createNewSession()) {
			return session.get(clazz, id);
		}
	}

	public ImageIcon getIconFromImageResource(String imageResourceId) {
		try {
			if (imageResourceId == null) {
				return null;
			}
			ImageResource imageResource = ImageResourceDAO.getInstance().findById(imageResourceId);
			if (imageResource != null) {
				return imageResource.getImageIcon();
			}
		} catch (Exception e0) {
			PosLog.error(DataProvider.class, e0);
		}
		return null;
	}

	public ImageIcon getIconFromImageResource(String imageResourceId, int width, int height) {
		try {
			if (StringUtils.isEmpty(imageResourceId)) {
				return null;
			}
			ImageResource imageResource = ImageResourceDAO.getInstance().get(imageResourceId);
			if (imageResource != null) {
				return imageResource.getScaledImage(width, height);
			}
		} catch (Exception e0) {
			PosLog.error(DataProvider.class, e0);
		}
		return null;
	}

	public ImageResource getImageResource(String imageResourceId) {
		try {
			if (StringUtils.isBlank(imageResourceId)) {
				return null;
			}
			ImageResource imageResource = ImageResourceDAO.getInstance().findById(imageResourceId);
			if (imageResource != null) {
				return imageResource;
			}
		} catch (Exception e0) {
			PosLog.error(DataProvider.class, e0);
		}
		return null;
	}

	public void addCacheRefreshListener(CacheRefreshListener listener) {
		refreshListeners.add(listener);
	}

	public void removeCacheRefreshListener(CacheRefreshListener listener) {
		refreshListeners.remove(listener);
	}

	public void fireCacheRefreshed() {
		for (CacheRefreshListener cacheRefreshListener : refreshListeners) {
			cacheRefreshListener.cacheRefreshed();
		}
	}

	/**
	 * This method is here to help re-ordering old ticket using the following path:
	 * Dine IN->Customer selection view->Selected customer history view->Selected ticket order info->Reorder.
	 * 
	 * @return the ticket that should be created if reorder is selected.
	 */
	public Ticket getTicketToBeCreated() {
		return ticketToBeCreated;
	}

	/**
	 * This method is here to help re-ordering old ticket using the following path:
	 * Dine IN->Customer selection view->Selected customer history view->Selected ticket order info->Reorder.
	 * 
	 * @param ticket The ticket that should be created if reorder is selected.
	 */
	public void setTicketToBeCreated(Ticket ticket) {
		this.ticketToBeCreated = ticket;
	}

	public PrinterGroup getDefaultPrinterGroup() {
		return null;
	}

	public boolean isCaching(Class beanClass) {
		return false;
	}

	public abstract String getMqttDeviceId();

	public abstract Date getServerTimestamp();

	public abstract Tax getTaxById(String taxId, String outletId);

	public abstract NumberFormat getDefaultNumberFormat();

	public abstract User getCurrentUser();

	public CustomPayment getCustomPaymentById(String unitId, String outletId) {
		if (StringUtils.isEmpty(unitId))
			return null;
		return CustomPaymentDAO.getInstance().get(unitId, outletId);
	}

	public Date getTransactionsTicketDate(String id, String outletId) {
		return TicketDAO.getInstance().getOrderDate(id, outletId);
	}

	public String getCurrentOutletId() {
		return getOutletId();
	}

	public List<PosTransaction> getTransactionsByEntityId(String entityId) {
		return PosTransactionDAO.getInstance().getTransactionsByEntityId(entityId);
	}

	public TicketType getTicketType() {
		return null;
	}

	public Module getModule() {
		return null;
	}

	//	public OrderType getOrderType(String outletId) {
	//		return null;
	//	}

	public String getTicketIdPrefex() {
		return StringUtils.EMPTY;
	}

	public void clearLabStuffCache() {
	}

	public void clearLabDoctorCache() {
	}

	public void clearTestMethodCache() {
	}

	public String getSenderEmailHost() {
		return null;
	}

	public abstract ChartOfAccounts getCOAFromMap(String code, Session session);

	public abstract ChartOfAccounts getCOAFromMap(String code, Session session, boolean isInsertCOA);

	public BookingInfo getBookingInfoByBookingId(String bookingId) {
		return null;
	}

	public List<Module> getEnabledModules() {
		return GlobalConfigDAO.getInstance().getEnabledModules();
	}

	public void addInventoryUnitToCache(InventoryUnit inventoryUnit) {
		if (StringUtils.isNotBlank(inventoryUnit.getId())) {
			inventoryUnitMap.put(inventoryUnit.getId(), inventoryUnit);
		}
	}

	public List<InventoryUnit> getAllInventoryUnits() {
		return new ArrayList<InventoryUnit>(inventoryUnitMap.values());
	}

	public List<Outlet> getAllOutlets() {
		return new ArrayList<Outlet>(outletMap.values());
	}

	public DataMode getDataMode() {
		return DataMode.MEDLOGICS;
	};
}
