package com.floreantpos.services.report;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Session;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Tax;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.TicketItemTax;
import com.floreantpos.model.dao.GenericDAO;
import com.floreantpos.model.ext.PaperSize;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.ReportConstants;
import com.floreantpos.report.ReportItem;
import com.floreantpos.report.ReportUtil;
import com.floreantpos.report.SalesReportModel;
import com.floreantpos.report.SalesReportUtil;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.POSUtil;
import com.ibm.icu.text.DecimalFormat;

import net.sf.jasperreports.engine.JREmptyDataSource;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRTableModelDataSource;

public class SalesTaxBreakdownReportService {
	private Date startDate;
	private Date toDate;
	private Outlet outlet;

	HashMap<String, ReportItem> reportItemMap = new HashMap<String, ReportItem>();
	private SalesReportModel monthlySalesTaxBreakdownReportModel;

	public SalesTaxBreakdownReportService(Date startDate, Date toDate, Outlet outlet) {
		this.startDate = startDate;
		this.toDate = toDate;
		this.outlet = outlet;
	}

	public JasperPrint createSalesTaxBreakdownReportJasperPrint() throws JRException {
		SalesReportModel monthlySalesTaxBreakdownReportModel = createSalesTaxBreakdownReportModels();
		return createSalesTaxBreakdownReportJasperPrint(monthlySalesTaxBreakdownReportModel);
	}

	public JasperPrint createSalesTaxBreakdownReportJasperPrint(SalesReportModel monthlySalesTaxBreakdownReportModel) throws JRException {

		JasperReport dailySalesReport = ReportUtil.getReport(PaperSize.getReportNameAccording2Size(ReportConstants.SALES_TAX_BREAKDOWN_SUB_REPORT));

		HashMap<String, Object> map = new HashMap<String, Object>();
		Outlet outlet = DataProvider.get().getOutlet();
		ReportUtil.populateRestaurantProperties(map, outlet);
		ReportUtil.populateReportHeader(map, outlet, startDate, toDate, null, true);
		populateReportParameter(map);

		map.put("dailySalesDataSource", new JRTableModelDataSource(monthlySalesTaxBreakdownReportModel)); //$NON-NLS-1$
		map.put("dailySalesReport", dailySalesReport); //$NON-NLS-1$

		JasperReport masterReport = ReportUtil.getReport(PaperSize.getReportNameAccording2Size(ReportConstants.SALES_TAX_BREAKDOWN_REPORT));
		JasperPrint print = JasperFillManager.fillReport(masterReport, map, new JREmptyDataSource());
		return print;
	}

	private void populateReportParameter(HashMap<String, Object> map) {
		String reportTitle = POSConstants.SALES_TAX_REPORT + " (Breakdown)";
		String colName = POSConstants.DATE + "/" + POSConstants.TAX; //$NON-NLS-1$

		map.put("reportTitle", reportTitle); //$NON-NLS-1$ 
		map.put("colName", ReportUtil.reportLabelWithBoldTag(colName));//$NON-NLS-1$
		map.put("currencySymbol", CurrencyUtil.getCurrencySymbolWithBracket()); //$NON-NLS-1$

		map.put("colId", ReportUtil.reportLabelWithBoldTag(POSConstants.ID));//$NON-NLS-1$
		map.put("colQty", ReportUtil.reportLabelWithBoldTag(POSConstants.QTY));//$NON-NLS-1$
		map.put("colUnit", ReportUtil.reportLabelWithBoldTag(POSConstants.UNIT));//$NON-NLS-1$
		map.put("colSales", ReportUtil.reportLabelWithBoldTag(POSConstants.UNIT_PRICE));//$NON-NLS-1$
		map.put("colDiscount", ReportUtil.reportLabelWithBoldTag(POSConstants.DISCOUNT));//$NON-NLS-1$
		map.put("colCost", ReportUtil.reportLabelWithBoldTag(POSConstants.COST));//$NON-NLS-1$
		map.put("colCostPer", ReportUtil.reportLabelWithBoldTag(POSConstants.COST_PERCENTAGE));//$NON-NLS-1$
		map.put("colNetTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.NET_SALES));//$NON-NLS-1$
		map.put("colTax", ReportUtil.reportLabelWithBoldTag(POSConstants.TAX));//$NON-NLS-1$
		map.put("colSC", ReportUtil.reportLabelWithBoldTag(POSConstants.S_C));//$NON-NLS-1$
		map.put("colGrossT", ReportUtil.reportLabelWithBoldTag(POSConstants.TOTAL));//$NON-NLS-1$

		map.put("groupTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GROUP_TOTAL));//$NON-NLS-1$
		map.put("lblItemTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GRAND_TOTAL));//$NON-NLS-1$
		map.put("lblModifierTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.MODIFIER_TOTAL + ": "));//$NON-NLS-1$ //$NON-NLS-2$
		map.put("lblGrandTotal", ReportUtil.reportLabelWithBoldTag(POSConstants.GRAND_TOTAL));//$NON-NLS-1$

		map.put("lblGrossSales", ReportUtil.reportLabelWithBoldTag(POSConstants.GROSS_SALES)); //$NON-NLS-1$
		map.put("lblServiceCharge", ReportUtil.reportLabelWithBoldTag(Messages.getString("SalesReportService.0"))); //$NON-NLS-1$ //$NON-NLS-2$
		map.put("lblReturnAmount", ReportUtil.reportLabelWithBoldTag(POSConstants.RETURN)); //$NON-NLS-1$
		map.put("lblProfit", ReportUtil.reportLabelWithBoldTag(POSConstants.GROSS_PROFIT)); //$NON-NLS-1$
		map.put("lblProfitPer", ReportUtil.reportLabelWithBoldTag(POSConstants.PROFIT + " %")); //$NON-NLS-1$ //$NON-NLS-2$

		map.put("itemSection", POSConstants.ITEMS); //$NON-NLS-1$ 
	}

	public SalesReportModel createSalesTaxBreakdownReportModels() {

		List<Date> betweenDates = POSUtil.getMonthBetweenDates(startDate, toDate);

		GenericDAO dao = new GenericDAO();
		try (Session session = dao.createNewSession();) {
			session.setDefaultReadOnly(true);

			for (Date reportDate : betweenDates) {
				Date startOfDay;
				Date endOfDay;

				startOfDay = DateUtil.startOfMonth(reportDate);
				endOfDay = DateUtil.endOfMonth(reportDate);

				SalesReportUtil salesReportUtil = new SalesReportUtil(startOfDay, endOfDay);
				salesReportUtil.setOutlet(outlet);
				List<TicketItem> ticketItemList = salesReportUtil.getTicketItemList(session);

				for (TicketItem ticketItem : ticketItemList) {
					String outletId = ticketItem.getTicket().getOutletId();
					createReportItemMap(startOfDay, ticketItem, ticketItem.getTaxes(), outletId);

					if (ticketItem.isHasModifiers()) {
						List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
						if (ticketItemModifiers != null) {
							for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
								createReportItemMapFromModifier(startOfDay, ticketItemModifier.getTaxes(), outletId);
							}
						}
					}
				}
			}
		}

		monthlySalesTaxBreakdownReportModel = new SalesReportModel();
		ArrayList<ReportItem> items = new ArrayList<>(reportItemMap.values());
		Comparator<ReportItem> comparator = Comparator.comparing(ReportItem::getDate, Comparator.nullsLast(Comparator.naturalOrder()));
		items.sort(comparator);

		monthlySalesTaxBreakdownReportModel.setItems(items);
		return monthlySalesTaxBreakdownReportModel;
	}

	private void createReportItemMap(Date startOfDay, TicketItem ticketItem, List<TicketItemTax> taxes, String outletId) {
		for (TicketItemTax ticketItemTax : taxes) {

			String taxName = ticketItemTax.getName();
			if (StringUtils.isNotBlank(ticketItemTax.getId())) {
				Tax tax = DataProvider.get().getTaxById(ticketItemTax.getId(), outletId);
				if (tax != null) {
					taxName = tax.getName();
				}
			}

			String mapKey = taxName + "_" + DateUtil.format("MM_yyyy", startOfDay); //$NON-NLS-1$ //$NON-NLS-2$
			ReportItem reportItem = reportItemMap.get(mapKey);
			if (reportItem == null) {
				reportItem = new ReportItem();

				reportItem.setName(DateUtil.format("MMM yyyy", startOfDay)); //$NON-NLS-1$
				reportItem.setTaxName(taxName);
				reportItem.setDate(startOfDay);
				reportItemMap.put(mapKey, reportItem);
			}

			double taxAmount = ticketItemTax.getTaxAmount();
			DecimalFormat decimalFormat = new DecimalFormat("#.##");//$NON-NLS-1$
			taxAmount = Double.valueOf(decimalFormat.format(taxAmount));

			reportItem.setGrossSale(Double.NaN);
			reportItem.setDiscount(Double.NaN);
			reportItem.setReturnAmount(Double.NaN);
			reportItem.setNetTotal(Double.NaN);
			reportItem.setServiceCharge(Double.NaN);
			reportItem.setTaxTotal(reportItem.getTaxTotal() + taxAmount);
			reportItem.setGrossTotal(Double.NaN);
			reportItem.setTotal(Double.NaN);
		}
	}

	private void createReportItemMapFromModifier(Date startOfDay, List<TicketItemTax> modifierTaxes, String outletId) {
		for (TicketItemTax ticketItemTax : modifierTaxes) {

			String taxName = ticketItemTax.getName();
			if (StringUtils.isNotBlank(ticketItemTax.getId())) {
				Tax tax = DataProvider.get().getTaxById(ticketItemTax.getId(), outletId);
				if (tax != null) {
					taxName = tax.getName();
				}
			}

			String mapKey = taxName + "_" + DateUtil.format("MM_yyyy", startOfDay); //$NON-NLS-1$ //$NON-NLS-2$
			ReportItem reportItem = reportItemMap.get(mapKey);
			if (reportItem == null) {
				reportItem = new ReportItem();
				reportItem.setName(DateUtil.format("MMM yyyy", startOfDay)); //$NON-NLS-1$
				reportItem.setTaxName(taxName);
				reportItem.setDate(startOfDay);
				reportItemMap.put(mapKey, reportItem);
			}
			double taxAmount = ticketItemTax.getTaxAmount();
			reportItem.setGrossSale(Double.NaN);
			reportItem.setDiscount(Double.NaN);
			reportItem.setReturnAmount(Double.NaN);
			reportItem.setNetTotal(Double.NaN);
			reportItem.setServiceCharge(Double.NaN);
			reportItem.setTaxTotal(reportItem.getTaxTotal() + taxAmount);
			reportItem.setGrossTotal(Double.NaN);
			reportItem.setTotal(Double.NaN);
		}
	}

}
