package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.PosException;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Project;
import com.floreantpos.model.SalaryAdvanceBackTransaction;
import com.floreantpos.model.SalaryTransaction;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.UserSalaryDeductionModel;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.POSUtil;

public class SalaryTransactionDAO extends BaseSalaryTransactionDAO {

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List getMonthlySalary(Date startOfMonth, Date endOfMonth) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(PosTransaction.class);
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Property.forName("class").eq(SalaryTransaction.class));
			disjunction.add(Property.forName("class").eq(SalaryAdvanceBackTransaction.class));
			criteria.add(disjunction);

			criteria.add(Restrictions.eq(SalaryTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.between(SalaryTransaction.PROP_EVENT_TIME, startOfMonth, endOfMonth));
			criteria.addOrder(Order.asc(PosTransaction.PROP_USER_ID));

			return criteria.list();
		}
	}

	private double getSalaryAdvanceBackAmount(User user, Date startOfMonth, Date endOfMonth) {
		try (Session session = createNewSession()) {

			Criteria criteria = session.createCriteria(SalaryAdvanceBackTransaction.class);

			criteria.add(Restrictions.eq(SalaryAdvanceBackTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.eq(SalaryAdvanceBackTransaction.PROP_USER_ID, user.getId()));
			criteria.add(Restrictions.between(SalaryAdvanceBackTransaction.PROP_EVENT_TIME, startOfMonth, endOfMonth));

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(SalaryAdvanceBackTransaction.PROP_AMOUNT));
			criteria.setProjection(projectionList);

			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	public void loadSalaryPayment(PaginationSupport model, Date startOfMonth, Date endOfMonth, String salaryUserId, String projectId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(SalaryTransaction.class);

			criteria.add(Restrictions.eq(SalaryTransaction.PROP_VOIDED, false));
			criteria.add(Restrictions.between(SalaryTransaction.PROP_EVENT_TIME, startOfMonth, endOfMonth));
			if (StringUtils.isNotBlank(salaryUserId)) {
				criteria.add(Restrictions.eq(SalaryTransaction.PROP_USER_ID, salaryUserId));
			}

			if (StringUtils.isNotBlank(projectId)) {
				criteria.add(Restrictions.eq(SalaryTransaction.PROP_PROJECT_ID, projectId));
			}

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			model.setNumRows(rowCount.intValue());

			criteria.setProjection(null);
			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			List<SalaryTransaction> transactions = criteria.list();
			transactions.sort(Comparator.comparing(SalaryTransaction::getUserName, Comparator.nullsLast(Comparator.naturalOrder())));
			for (SalaryTransaction salaryTransaction : transactions) {
				salaryTransaction.putAdvanceBackAmount(getSalaryAdvanceBackAmount(salaryTransaction.getUser(), startOfMonth, endOfMonth));
			}

			model.setRows(transactions);
		}
	}

	public SalaryTransaction saveSalaryTransaction(Date eventTime, String batchNo, Project project, Object paymentMethod, String paymentRef,
			User accountsManager, User paidToUser, double amount, double salaryAdvancedReturnAmount, String note, String deductionGsonData,
			double totalDeductionAmount, /*BalanceUpdateTransaction balanceUpdateTransaction,*/ Session session) throws Exception {
		return saveSalaryTransaction(eventTime, batchNo, project, paymentMethod, paymentRef, accountsManager, paidToUser, amount, salaryAdvancedReturnAmount,
				note, deductionGsonData, totalDeductionAmount, null, session);
	}

	public SalaryTransaction saveSalaryTransaction(Date eventTime, String batchNo, Project project, Object paymentMethod, String paymentRef,
			User accountsManager, User paidToUser, double amount, double salaryAdvancedReturnAmount, String note, String deductionGsonData,
			double totalDeductionAmount, /*BalanceUpdateTransaction balanceUpdateTransaction,*/ String linkedBankAccountId, Session session) throws Exception {

		SalaryTransaction salaryTransaction = createTransaction(paymentMethod, paymentRef, TransactionType.OUT, amount);
		salaryTransaction.setEventTime(eventTime);
		salaryTransaction.setBatchNo(batchNo);
		salaryTransaction.setProject(project);
		// set AccountNo to Server
		salaryTransaction.setServer(accountsManager);
		salaryTransaction.setAccountManager(accountsManager == null ? null : accountsManager);
		//paid salary to
		salaryTransaction.setUser(paidToUser);
		salaryTransaction.putSalaryPaidTo(paidToUser == null ? null : paidToUser.getId());

		salaryTransaction.setNote(note);
		salaryTransaction.setAccountProcessed(true);
		if (paymentMethod.toString().equalsIgnoreCase("Cash") && accountsManager != null) {
			salaryTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
		}
		else {
			salaryTransaction.setSourceType(paymentMethod.toString());
		}
		//salaryTransaction.putSourceId(balanceUpdateTransaction.getId());
		salaryTransaction.putDeductionGsonData(deductionGsonData);
		salaryTransaction.putTotalDeductionAmount(totalDeductionAmount);
		salaryTransaction.putEmployeeMonthlySalaryAmount(paidToUser.getMonthlySalaryAmount());
		if (StringUtils.isNotBlank(linkedBankAccountId)) {
			salaryTransaction.putLinkedBankAccountId(linkedBankAccountId);
		}

		save(salaryTransaction, session);
		return salaryTransaction;
	}

	public SalaryTransaction createTransaction(Object paymentMethod, String paymentRef, TransactionType transactionType, double amount) {
		SalaryTransaction salaryTransaction = new SalaryTransaction();

		PaymentType paymentType = initPaymentMethod(paymentMethod, paymentRef, salaryTransaction);

		salaryTransaction.setPaymentType(paymentType);
		salaryTransaction.setTransactionType(transactionType.name());
		salaryTransaction.setTransactionTime(DataProvider.get().getServerTimestamp());
		salaryTransaction.setAmount(transactionType == TransactionType.DEBIT ? -Math.abs(amount) : Math.abs(amount));
		salaryTransaction.setOutletId(DataProvider.get().getOutlet().getId());
		return salaryTransaction;
	}

	public static PaymentType initPaymentMethod(Object paymentMethod, String paymentRef, PosTransaction posTransaction) {
		PaymentType paymentType = null;
		if (paymentMethod != null) {
			if (paymentMethod instanceof String) {
				paymentType = PaymentType.CASH;
			}
			if (paymentMethod instanceof CustomPayment) {
				paymentType = PaymentType.CUSTOM_PAYMENT;

				CustomPayment customPayment = (CustomPayment) paymentMethod;
				posTransaction.setCustomPaymentId(customPayment.getId());
				posTransaction.setCustomPaymentName(customPayment.getName());
				posTransaction.setCustomPaymentRef(paymentRef);
			}
			if (paymentMethod instanceof BankAccount) {
				paymentType = PaymentType.BANK_ACCOUNT;

				BankAccount bankAccount = (BankAccount) paymentMethod;
				posTransaction.setBankAccount(bankAccount);
			}
		}
		else {
			paymentType = PaymentType.CASH;
		}
		return paymentType;
	}

	public void saveSalaryExpenses(Date eventTime, String batchNo, Project project, Object paymentMethod, String paymentRef, User accountsManager,
			User paidToUser, double salaryAmount, double salaryAdvancedReturnAmount, String note, String deductionGsonData, double totalDeductionAmount,
			List<UserSalaryDeductionModel> repeatDeductionList, Boolean isAMDepositMethod) throws Exception {
		saveSalaryExpenses(eventTime, batchNo, project, paymentMethod, paymentRef, accountsManager, paidToUser, salaryAmount, salaryAdvancedReturnAmount, note,
				deductionGsonData, totalDeductionAmount, repeatDeductionList, isAMDepositMethod, null);
	}

	public void saveSalaryExpenses(Date eventTime, String batchNo, Project project, Object paymentMethod, String paymentRef, User accountsManager,
			User paidToUser, double salaryAmount, double salaryAdvancedReturnAmount, String note, String deductionGsonData, double totalDeductionAmount,
			List<UserSalaryDeductionModel> repeatDeductionList, Boolean isAMDepositMethod, BankAccount linkedBankAccount) throws Exception {

		boolean isCustomPayment = paymentMethod instanceof CustomPayment;

		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Transaction tx = session.beginTransaction();

			accountsManager = session.get(User.class, accountsManager);
			paidToUser = session.get(User.class, paidToUser);

			if (paymentMethod != null && "Cash".equalsIgnoreCase(paymentMethod.toString())) {
				if (accountsManager.getAccountsManagerAmount() < salaryAmount) {
					throw new PosException("Not enough balance in account manager's account");
				}
			}
			String linkedBankAccountId = "";
			if (linkedBankAccount != null) {
				linkedBankAccountId = linkedBankAccount.getId();
			}
			SalaryAdvanceBackTransaction slaryAdvanceBackTransaction = null;
			if (salaryAdvancedReturnAmount > 0) {
				slaryAdvanceBackTransaction = SalaryAdvanceBackTransactionDAO.getInstance().saveSalaryAdvanceBackTransaction(eventTime, batchNo, project,
						paymentMethod, paymentRef, accountsManager, paidToUser, salaryAdvancedReturnAmount, note, session);
			}

			SalaryTransaction salaryTransaction = saveSalaryTransaction(eventTime, batchNo, project, paymentMethod, paymentRef, accountsManager, paidToUser,
					salaryAmount, salaryAdvancedReturnAmount, note, deductionGsonData, totalDeductionAmount, linkedBankAccountId, session);

			BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveSalaryExpenseTransaction(session, eventTime,
					batchNo, project, paymentMethod, paymentRef, accountsManager, paidToUser, salaryAmount, note, isAMDepositMethod, salaryTransaction, false,
					linkedBankAccountId);

			LedgerEntryDAO.getInstance().saveSalaryLedgerEntry(session, balanceUpdateTransaction, true, false, linkedBankAccountId, false,
					slaryAdvanceBackTransaction);

			tx.commit();
		}

	}

	public void saveSalaryRevertExpenses(SalaryTransaction sourceSalaryTransaction, String note, User revertedByUser) throws Exception {

		PaymentType paymentType = sourceSalaryTransaction.getPaymentType();

		boolean isCustomPayment = false;
		Object paymentMethod = null;
		if (PaymentType.CUSTOM_PAYMENT == paymentType) {
			isCustomPayment = true;
		}
		else if (PaymentType.BANK_ACCOUNT == paymentType) {
			String bankAccountId = sourceSalaryTransaction.getBankAccountId();
			if (StringUtils.isNotBlank(bankAccountId)) {
				paymentMethod = BankAccountDAO.getInstance().get(bankAccountId);
			}
		}

		try (Session session = BalanceUpdateTransactionDAO.getInstance().createNewSession()) {
			Date now = StoreDAO.getServerTimestamp();
			Transaction tx = session.beginTransaction();

			User accountManager = UserDAO.getInstance().get(sourceSalaryTransaction.getAccountManagerId(), sourceSalaryTransaction.getOutletId(), session);
			User paidToUser = UserDAO.getInstance().get(sourceSalaryTransaction.getSalaryPaidTo(), sourceSalaryTransaction.getOutletId(), session);

			Double salaryAmount = sourceSalaryTransaction.getAmount();
			sourceSalaryTransaction.setVoidedByUser(revertedByUser);
			sourceSalaryTransaction.putRevertReason(note);
			sourceSalaryTransaction.setVoided(true);
			sourceSalaryTransaction.setVoidDate(now);

			PosTransactionDAO.getInstance().saveOrUpdate(sourceSalaryTransaction, session);

			String linkedBankAccountId = sourceSalaryTransaction.getLinkedBankAccountId();

			BalanceUpdateTransaction balanceUpdateTransaction = BalanceUpdateTransactionDAO.getInstance().saveSalaryExpenseTransaction(session, now,
					sourceSalaryTransaction.getBatchNo(), sourceSalaryTransaction.getProject(), paymentMethod, null, accountManager, paidToUser, salaryAmount,
					note, true, sourceSalaryTransaction, true, linkedBankAccountId);
			
			LedgerEntryDAO.getInstance().saveSalaryLedgerEntry(session, balanceUpdateTransaction, true, false, linkedBankAccountId, true, null);

			tx.commit();
		}

	}
}