package com.floreantpos.report;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;
import org.hibernate.criterion.Subqueries;
import org.hibernate.transform.Transformers;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosLog;
import com.floreantpos.model.AdvanceAdjustmentTransaction;
import com.floreantpos.model.AdvanceRefundTransaction;
import com.floreantpos.model.AdvanceTransaction;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CategoryBreakOut;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.DischargeRecord;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.OTStatus;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.Shift;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.SurgeryInfo;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketSource;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.dao.BookingInfoDAO;
import com.floreantpos.model.dao.DischargeRecordDAO;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.PosTransactionDAO;
import com.floreantpos.model.dao.SurgeryInfoDAO;
import com.floreantpos.model.dao._RootDAO;
import com.floreantpos.model.ext.CardTypeEnum;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class SalesReportUtil {

	private Date fromDate;
	private Date toDate;
	private User user;
	private String categoryName;
	private StoreSession storeSession;
	private Shift shift;
	private List<String> orderTypeIds;
	private boolean isShowOnlineTicket = Boolean.FALSE;
	private boolean isIncludeAllRole = Boolean.TRUE;
	private Outlet outlet;
	private String projectId;
	private TicketType ticketType;

	public SalesReportUtil(Date fromDate, Date toDate) {
		super();
		this.fromDate = fromDate;
		this.toDate = toDate;
	}

	public SalesReportUtil(Date fromDate, Date toDate, User user) {
		super();
		this.fromDate = fromDate;
		this.toDate = toDate;
		this.user = user;
	}

	public SalesReportUtil(StoreSession storeSession) {
		super();
		this.storeSession = storeSession;
	}

	@Deprecated
	public SalesReportUtil(User user, StoreSession storeSession) {
		super();
		this.user = user;
		this.storeSession = storeSession;
	}

	public double calculateItemCount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.FALSE);
		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(TicketItem.PROP_QUANTITY));
		PosLog.debug(getClass(), "Item sales count query: " + criteria.toString()); //$NON-NLS-1$
		double totalGrossSales = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalGrossSales;
	}

	public double calculateTotalAdmissionCount(Session session) {
		Criteria criteria = session.createCriteria(BookingInfo.class);
		criteria.setProjection(Projections.rowCount());
		if (fromDate != null) {
			criteria.add(Restrictions.ge(BookingInfo.PROP_FROM_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(BookingInfo.PROP_FROM_DATE, toDate));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(BookingInfo.PROP_OUTLET_ID, outlet.getId()));
		}
		BookingInfoDAO.getInstance().addDeletedFilter(criteria);
		PosLog.debug(getClass(), "Admission count query: " + criteria.toString()); //$NON-NLS-1$
		double totalAdmissionCount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalAdmissionCount;
	}

	public double calculateTotalDischargeCount(Session session) {
		Criteria criteria = session.createCriteria(DischargeRecord.class);
		criteria.setProjection(Projections.rowCount());
		if (fromDate != null) {
			criteria.add(Restrictions.ge(DischargeRecord.PROP_DISCHARGE_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(DischargeRecord.PROP_DISCHARGE_DATE, toDate));
		}
		if (outlet != null) {
			DetachedCriteria detachedAdmIdCriteriaForOutlet = DetachedCriteria.forClass(BookingInfo.class);
			detachedAdmIdCriteriaForOutlet.setProjection(Projections.property(BookingInfo.PROP_BOOKING_ID));
			detachedAdmIdCriteriaForOutlet.add(Restrictions.eq(BookingInfo.PROP_OUTLET_ID, outlet.getId()));
			criteria.add(Subqueries.propertiesIn(new String[] { DischargeRecord.PROP_ADMISSION_ID }, detachedAdmIdCriteriaForOutlet));
		}
		DischargeRecordDAO.getInstance().addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(DischargeRecord.PROP_STATUS, PatientBookingStatus.DISCHARGED.name()));
		PosLog.debug(getClass(), "Discharge count query: " + criteria.toString()); //$NON-NLS-1$
		double totalDischargeCount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalDischargeCount;
	}

	public double calculateOTCount(Session session) {
		Criteria criteria = session.createCriteria(SurgeryInfo.class);
		criteria.setProjection(Projections.rowCount());
		if (fromDate != null) {
			criteria.add(Restrictions.ge(SurgeryInfo.PROP_COMPLETED_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(SurgeryInfo.PROP_COMPLETED_DATE, toDate));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(SurgeryInfo.PROP_OUTLET_ID, outlet.getId()));
		}
		SurgeryInfoDAO.getInstance().addDeletedFilter(criteria);
		criteria.add(Restrictions.eq(SurgeryInfo.PROP_SURGERY_STATUS, OTStatus.COMPLETED.name()));
		PosLog.debug(getClass(), "OT count query: " + criteria.toString()); //$NON-NLS-1$
		double totalOTCount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalOTCount;
	}

	public double calculateReturnItemCount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.TRUE);
		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(TicketItem.PROP_QUANTITY));
		PosLog.debug(getClass(), "Return item sales count query: " + criteria.toString()); //$NON-NLS-1$
		double totalReturnItemCount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalReturnItemCount;
	}

	public double calculateTotalCost(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_TOTAL_COST));
		double totalCost = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return totalCost;
	}

	public double calculateGrossSalesWithOutIncludedTax(Session session) {
		return calculateGrossSales(session) - calculateIncludedTax(session);
	}

	public double calculateGrossSalesWithTax(Session session) {
		return calculateGrossSalesWithOutIncludedTax(session) + calculateSalesTaxAmount(session) - calculateReturnAmountWithOutIncludedTax(session);
	}

	public double calculateTotalSales(Session session) {
		double calculateDiscountAmount = calculateDiscountAmount(session);
		double calculateReturnAmount = calculateReturnAmountWithOutIncludedTax(session);
		calculateDiscountAmount = NumberUtil.isZero(calculateDiscountAmount) ? 0.0 : (-1) * calculateDiscountAmount;
		calculateReturnAmount = NumberUtil.isZero(calculateReturnAmount) ? 0.0 : (-1) * calculateReturnAmount;
		double netTotalAmount = calculateGrossSalesWithOutIncludedTax(session) + calculateDiscountAmount + calculateReturnAmount;
		return netTotalAmount + calculateTotalServiceChargeAmount(session) + calculateSalesTaxAmount(session);
	}

	public double calculateOthersSalesAmount(Session session) {
		return calculateTotalReceipt(session) - (calculateCashReceipt(session) + calculateCreditReceipt(session));
	}

	public double calculateCashReceiptWithRefund(Session session) {
		return calculateCashReceipt(session) - calculatePaymentTypeRefund(session, PaymentType.CASH);
	}

	public double calculateCreditReceiptWithRefund(Session session) {
		return calculateCreditReceipt(session) - calculatePaymentTypeRefund(session, PaymentType.CARD);
	}

	public double calculateOthersSalesAmountWithRefund(Session session) {
		return calculateOthersSalesAmount(session) - calculateOtherRefund(session);
	}

	public double calculateItemCountWithoutRetunItem(Session session) {
		return calculateItemCount(session) + calculateReturnItemCount(session);
	}

	private double calculateGrossSales(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.FALSE);
		criteria.setProjection(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT));

		PosLog.debug(getClass(), "Gross sale query: " + criteria.toString()); //$NON-NLS-1$

		double grossSales = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return grossSales;
	}

	public double calculateNetSales(Session session) {
		double labDoctorFeeAmount = calculateLabDoctorFeeAmount(session);
		double agentFeeAmount = calculateTicketReferralCommissionAmount(session);
		double totalRfOnNetSales = calculateTotalRfOnNetSales(session);

		double grossSales = calculateGrossSalesWithOutIncludedTax(session);
		double discountAmount = calculateDiscountAmount(session);
		double refundAmount = calculateReturnAmountWithOutIncludedTax(session);
		discountAmount = NumberUtil.isZero(discountAmount) ? 0.0 : (-1) * discountAmount;
		refundAmount = NumberUtil.isZero(refundAmount) ? 0.0 : (-1) * refundAmount;
		return calculateNetSalesAmount(grossSales, labDoctorFeeAmount, agentFeeAmount, totalRfOnNetSales, discountAmount, refundAmount);
	}

	public double calculateNetSalesAmount(double grossSales, double labDoctorFeeAmount, double agentFeeAmount, double totalRfOnNetSales, double discountAmount,
			double refundAmount) {
		return grossSales - Math.abs(discountAmount) + refundAmount - labDoctorFeeAmount - (agentFeeAmount + totalRfOnNetSales);
	}

	private double calculateIncludedTax(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.FALSE);
		criteria.add(Restrictions.eq(TicketItem.PROP_TAX_INCLUDED, Boolean.TRUE));
		//criteria.setProjection(Projections.sum(TicketItem.PROP_TAX_AMOUNT));
		criteria.setProjection(Projections.sum(TicketItem.PROP_ADJUSTED_TAX));
		PosLog.debug(getClass(), "Included tax sale query: " + criteria.toString()); //$NON-NLS-1$
		double includedTax = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return includedTax;
	}

	public double calculateDiscountAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_ADJUSTED_DISCOUNT));
		double discountSales = POSUtil.getDoubleAmount(criteria.uniqueResult());

		criteria = buildCriteriaForTicketItem(session, true);
		criteria.setProjection(Projections.sum(TicketItem.PROP_DISCOUNT_AMOUNT));
		double returnItemDiscountSum = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return discountSales - returnItemDiscountSum;
	}

	public double calculateReturnAmountWithOutIncludedTax(Session session) {
		return calculateReturnAmount(session) + calculateIncludedTaxForReturn(session);
	}

	private double calculateReturnAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.TRUE);
		criteria.setProjection(Projections.sum(TicketItem.PROP_TOTAL_AMOUNT));
		double returnAmount = Math.abs(POSUtil.getDoubleAmount(criteria.uniqueResult()));
		return returnAmount;
	}

	private double calculateIncludedTaxForReturn(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.TRUE);
		criteria.add(Restrictions.eq(TicketItem.PROP_TAX_INCLUDED, Boolean.TRUE));
		//criteria.setProjection(Projections.sum(TicketItem.PROP_TAX_AMOUNT));
		criteria.setProjection(Projections.sum(TicketItem.PROP_ADJUSTED_TAX));
		double includedTax = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return includedTax;
	}

	public double calculateTotalServiceChargeAmount(Session session) {
		return calculateItemServiceChargeAmount(session) + calculateTicketServiceChargeAmount(session);
	}

	private double calculateItemServiceChargeAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_SERVICE_CHARGE));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return scFromItems;
	}

	private double calculateTicketServiceChargeAmount(Session session) {
		Criteria criteria = buildCriteriaForTicket(session, fromDate, toDate, user, storeSession);
		criteria.setProjection(Projections.sum(Ticket.PROP_TICKET_SERVICE_CHARGE));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return scFromItems;
	}

	public double calculateSalesTaxAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, null);
		criteria.setProjection(Projections.sum(TicketItem.PROP_ADJUSTED_TAX));
		double tax = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return tax;
	}

	public Criteria buildCriteriaForTicketItem(Session session) {
		return buildCriteriaForTicketItem(session, null);
	}

	public Criteria buildCriteriaForTicketItem(Session session, Boolean itemReturned) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "t"); //$NON-NLS-1$
		if (fromDate != null) {
			criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, toDate));
		}
		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		if (itemReturned != null) {
			criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, itemReturned));
		}
		criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
		if (isShowOnlineTicket) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_SOURCE, TicketSource.Online.name())); //$NON-NLS-1$
		}

		if (ticketType != null) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_TYPE, ticketType.getTypeNo())); //$NON-NLS-1$
		}
		if (storeSession != null) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_STORE_SESSION_ID, storeSession.getId())); //$NON-NLS-1$
		}
		if (shift != null) {
			//				criteria.add(Restrictions.eq("t." + Ticket.PROP_SHIFT_ID, shift.getId())); //$NON-NLS-1$
			if (StringUtils.isBlank(shift.getId())) {
				criteria.add(Restrictions.isNull("t." + Ticket.PROP_SHIFT_ID)); //$NON-NLS-1$
			}
			else {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_SHIFT_ID, shift.getId())); //$NON-NLS-1$
			}
		}
		if (orderTypeIds != null && orderTypeIds.size() > 0) {
			criteria.add(Restrictions.in("t." + Ticket.PROP_ORDER_TYPE_ID, orderTypeIds)); //$NON-NLS-1$
		}
		if (!POSUtil.isBlankOrNull(categoryName)) {
			if (categoryName.equalsIgnoreCase(POSConstants.NONE)) {
				criteria.add(Restrictions.isNull(TicketItem.PROP_CATEGORY_NAME));
			}
			else {
				criteria.add(Restrictions.eq(TicketItem.PROP_CATEGORY_NAME, categoryName));
			}
		}

		criteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_PROJECT_ID, projectId)); //$NON-NLS-1$
		}

		_RootDAO.addUserWithAllRoleCriteria(criteria, user, "t." + Ticket.PROP_OWNER_ID, isIncludeAllRole); //$NON-NLS-1$

		//		_RootDAO.addSalesAdminCriteria(session, criteria, "t." + Ticket.PROP_ID, fromDate, toDate, outlet.getId(), projectId);

		return criteria;

	}

	//	public double calculateGratuity(Session session) {
	//		Criteria ticketCriteria = session.createCriteria(Ticket.class, "t").createAlias(Ticket.PROP_GRATUITY, "g"); //$NON-NLS-1$ //$NON-NLS-2$
	//
	//		ticketCriteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
	//		if (fromDate != null) {
	//			ticketCriteria.add(Restrictions.ge("t." + Ticket.PROP_CREATE_DATE, fromDate)); //$NON-NLS-1$
	//		}
	//		if (toDate != null) {
	//			ticketCriteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, toDate)); //$NON-NLS-1$
	//		}
	//		if (storeSession != null) {
	//			ticketCriteria.add(Restrictions.eq("t." + Ticket.PROP_STORE_SESSION_ID, storeSession.getId())); //$NON-NLS-1$
	//		}
	//		if (isShowOnlineTicket) {
	//			ticketCriteria.add(Restrictions.eq("t." + Ticket.PROP_SOURCE, TicketSource.Online.name())); //$NON-NLS-1$
	//		}
	//		ticketCriteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
	//
	//		_RootDAO.addUserWithAllRoleCriteria(ticketCriteria, user, Ticket.PROP_OWNER_ID, isIncludeAllRole);
	//
	//		ticketCriteria.setProjection(Projections.sum("g." + Gratuity.PROP_AMOUNT)); //$NON-NLS-1$
	//
	//		return POSUtil.getDoubleAmount(ticketCriteria.uniqueResult());
	//	}

	@Deprecated
	public double calculateGratuityOld(Session session) {

		DetachedCriteria ticketCriteria = DetachedCriteria.forClass(Ticket.class);
		ticketCriteria.setProjection(Property.forName(Ticket.PROP_ID));

		if (isShowOnlineTicket) {
			ticketCriteria.add(Restrictions.eq(Ticket.PROP_SOURCE, TicketSource.Online.name()));
		}
		if (fromDate != null) {
			ticketCriteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		}
		if (toDate != null) {
			ticketCriteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));
		}
		ticketCriteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		if (storeSession != null) {
			ticketCriteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, storeSession.getId()));
		}
		if (user != null) {
			ticketCriteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, user.getRoleIds()));
		}
		ticketCriteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$

		Criteria criteria = session.createCriteria(Gratuity.class);
		criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(ticketCriteria));

		criteria.setProjection(Projections.sum(Gratuity.PROP_AMOUNT));

		double gratuity = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return gratuity;
	}

	public double calculateDeliveryCharge(Session session) {
		Criteria criteria = buildCriteriaForTicket(session, fromDate, toDate, user, storeSession);
		criteria.setProjection(Projections.sum(Ticket.PROP_DELIVERY_CHARGE));
		double deliveryCharge = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return deliveryCharge;
	}

	private Criteria buildCriteriaForTicket(Session session, Date fromDate, Date toDate, User user, StoreSession storeSession) {
		Criteria criteria = session.createCriteria(Ticket.class);
		if (fromDate != null) {
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));
		}
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		if (isShowOnlineTicket) {
			criteria.add(Restrictions.eq(Ticket.PROP_SOURCE, TicketSource.Online.name()));
		}
		if (storeSession != null) {
			criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, storeSession.getId()));
		}
		criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_PROJECT_ID, projectId)); //$NON-NLS-1$
		}

		_RootDAO.addUserWithAllRoleCriteria(criteria, user, Ticket.PROP_OWNER_ID, isIncludeAllRole);
		//		_RootDAO.addSalesAdminCriteria(session, criteria, Ticket.PROP_ID, fromDate, toDate, outlet.getId(), projectId);

		return criteria;
	}

	public double calculateFeeAmount(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_FEE_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateTotalTips(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateCashTips(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CashTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateChargedTips(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateCashReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CashTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Cash sale query: " + criteria.toString()); //$NON-NLS-1$
		double cashReceipt = POSUtil.getDoubleAmount(criteria.uniqueResult());

		double cashAdvanceReceipt = calculateAdvanceReceipt(session);
		return cashReceipt + cashAdvanceReceipt;
	}

	public double calculateAdvanceRefundReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransactionWithoutTicket(session, fromDate, toDate, user, storeSession, AdvanceRefundTransaction.class);
		criteria.setProjection(Projections.sum(AdvanceRefundTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Cash advance refund query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());

	}

	public double calculateAdvanceReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransactionWithoutTicket(session, fromDate, toDate, user, storeSession, AdvanceTransaction.class);

		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Cash advance query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateAdvanceAdjustmentReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, AdvanceAdjustmentTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Cash advance adjustment query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateDueCollection(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		criteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, fromDate));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Due collection query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateRFAndLDFPayCollection(Session session, Class<? extends PosTransaction> transactionClass) {
		return calculateRFAndLDFPayCollection(session, transactionClass, false);
	}

	public double calculateRFAndLDFPayCollection(Session session, Class<? extends PosTransaction> transactionClass, boolean isDuePayCollection) {
		return calculateRFAndLDFPayCollection(session, transactionClass, isDuePayCollection, null);
	}

	public double calculateRFAndLDFPayCollection(Session session, Class<? extends PosTransaction> transactionClass, boolean isDuePayCollection,
			Boolean isCashCollection) {
		Criteria criteria = buildCriteriaForRFAndLDFTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		if (isCashCollection != null) {
			if (isCashCollection) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
			}
			else {
				criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
			}
		}

		List<PosTransaction> list = criteria.list();
		double totalTransactionAmount = 0;
		for (PosTransaction posTransaction : list) {
			String transactionType = posTransaction.getTransactionType();
			if (TransactionType.IN.name().equals(transactionType)) {
				posTransaction.setAmount((-1) * posTransaction.getAmount());
			}

			String transTicketIds = posTransaction.getTransTicketIdsDisplay();
			if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) {
				continue;
			}

			Date orderDate = DataProvider.get().getTransactionsTicketDate(transTicketIds, posTransaction.getOutletId());
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (isDuePayCollection) {
				if (orderDate.before(startOfDay)) {
					totalTransactionAmount += posTransaction.getAmount();
				}
			}
			else {
				if (orderDate.after(startOfDay)) {
					totalTransactionAmount += posTransaction.getAmount();
				}
			}
		}
		return POSUtil.getDoubleAmount(totalTransactionAmount);
	}

	public double calculateTotalReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Total received query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateNetReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, PosTransaction.class);
		if (fromDate != null) {
			criteria.add(Restrictions.ge("t." + Ticket.PROP_CREATE_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, toDate));
		}
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Net received query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateCreditReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CreditCardTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		PosLog.debug(getClass(), "Credit sale query: " + criteria.toString()); //$NON-NLS-1$
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateDebitReceipt(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, DebitCardTransaction.class);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateMemberPayment(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CustomerAccountTransaction.class);
		//criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateCustomPaymentWithoutPromotion(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CustomPaymentTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculatePromotionPayment(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CustomPaymentTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateGiftCertReceipts(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, GiftCertificateTransaction.class);
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateRefundAmount(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, RefundTransaction.class);
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		double refundAmount = POSUtil.getDoubleAmount(criteria.uniqueResult());

		criteria = buildCriteriaForTransactionWithoutTicket(session, fromDate, toDate, user, storeSession, AdvanceRefundTransaction.class);
		criteria.setProjection(Projections.sum(AdvanceRefundTransaction.PROP_AMOUNT));
		double advanceRefundAmount = POSUtil.getDoubleAmount(criteria.uniqueResult());

		return refundAmount + advanceRefundAmount;
	}

	public double calculatePaymentTypeRefund(Session session, PaymentType paymentType) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, RefundTransaction.class);
		if (paymentType.equals(PaymentType.CARD)) {
			SimpleExpression cr = Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CREDIT_CARD.name());
			SimpleExpression de = Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.DEBIT_CARD.name());
			criteria.add(Restrictions.or(cr, de));
		}
		else {
			criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, paymentType.name()));
		}
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateOtherRefund(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, RefundTransaction.class);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CREDIT_CARD.name()));
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.DEBIT_CARD.name()));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateToleranceAmount(Session session) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, CashTransaction.class);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TOLERANCE_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateVisaCreditCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.VISA.name()).ignoreCase());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateMasterCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(PosTransactionDAO.createMasterCardSearchCriteria());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateAmexSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(PosTransactionDAO.createAmexOrAmericanExpCardSearchCriteria());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateDiscoverySummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.DISCOVER.name()).ignoreCase());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateVisaDebitCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.VISA.name()).ignoreCase());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateMasterDebitCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		//cash receipt
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		criteria.add(PosTransactionDAO.createMasterCardSearchCriteria());
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateRoundingAmount(Session session) {
		Criteria criteria = session.createCriteria(Ticket.class);
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		if (fromDate != null) {
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, toDate));
		}
		if (this.storeSession != null) {
			criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, this.storeSession.getId()));
		}

		_RootDAO.addUserWithAllRoleCriteria(criteria, user, Ticket.PROP_OWNER_ID, isIncludeAllRole);

		criteria.setProjection(Projections.sum(Ticket.PROP_ROUNDED_AMOUNT));
		double roundedAmount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return roundedAmount;
	}

	private Criteria buildCriteriaForTransaction(Session session, Date fromDate, Date toDate, User user, StoreSession storeSession,
			Class<? extends PosTransaction> transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass).createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
		if (transactionClass.equals(RefundTransaction.class) || transactionClass.equals(AdvanceRefundTransaction.class)
				|| transactionClass.equals(PayOutTransaction.class)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		}
		else {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		}
		_RootDAO.addUserWithAllRoleCriteria(criteria, user, PosTransaction.PROP_USER_ID, isIncludeAllRole);
		if (storeSession != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
		}
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		if (fromDate != null) {
			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		}
		criteria.add(Restrictions.eq("t." + PosTransaction.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_PROJECT_ID, projectId)); //$NON-NLS-1$
		}
		criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
		//		_RootDAO.addSalesAdminCriteria(session, criteria, "t." + Ticket.PROP_ID, fromDate, toDate, outlet.getId(), projectId);
		return criteria;
	}

	private Criteria buildCriteriaForTransactionWithoutTicket(Session session, Date fromDate, Date toDate, User user, StoreSession storeSession,
			Class<? extends PosTransaction> transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass); //$NON-NLS-1$
		if (transactionClass.equals(RefundTransaction.class) || transactionClass.equals(AdvanceRefundTransaction.class)
				|| transactionClass.equals(PayOutTransaction.class)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		}
		else {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		}

		_RootDAO.addUserWithAllRoleCriteria(criteria, user, PosTransaction.PROP_USER_ID, isIncludeAllRole);
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		if (fromDate != null) {
			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		}
		criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_PROJECT_ID, projectId)); //$NON-NLS-1$
		}
		return criteria;
	}

	public List<TicketItem> getTicketItemList(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		return criteria.list();
	}

	public Date getFromDate() {
		return fromDate;
	}

	public void setFromDate(Date fromDate) {
		this.fromDate = fromDate;
	}

	public Date getToDate() {
		return toDate;
	}

	public void setToDate(Date toDate) {
		this.toDate = toDate;
	}

	public User getUser() {
		return user;
	}

	public void setUser(User user) {
		this.user = user;
	}

	public String getCategoryName() {
		return categoryName;
	}

	public void setCategoryName(String categoryName) {
		this.categoryName = categoryName;
	}

	public StoreSession getStoreSession() {
		return storeSession;
	}

	public void setStoreSession(StoreSession storeSession) {
		this.storeSession = storeSession;
	}

	public Shift getShift() {
		return shift;
	}

	public void setShift(Shift shift) {
		this.shift = shift;
	}

	public List<String> getOrderTypeIds() {
		return orderTypeIds;
	}

	public void setOrderTypeIds(List<String> orderTypeIds) {
		this.orderTypeIds = orderTypeIds;
	}

	public boolean isShowOnlineTickets() {
		return isShowOnlineTicket;
	}

	public void setShowOnlineTickets(boolean isShowOnlineTicket) {
		this.isShowOnlineTicket = isShowOnlineTicket;
	}

	public void setOutlet(Outlet outlet) {
		this.outlet = outlet;
	}

	public void setProjectId(String projectId) {
		this.projectId = projectId;
	}

	/**
	 * @return the isIncludeAllRole
	 */
	public boolean isIncludeAllRole() {
		return isIncludeAllRole;
	}

	/**
	 * @param isIncludeAllRole the isIncludeAllRole to set
	 */
	public void setIncludeAllRole(boolean isIncludeAllRole) {
		this.isIncludeAllRole = isIncludeAllRole;
	}

	//Category break out
	private List<CategoryBreakOut> calculateCategoryGrossSales(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.FALSE);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_ID), TicketItem.PROP_CATEGORY_ID);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_NAME), TicketItem.PROP_CATEGORY_NAME);
		projectionList.add(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT), CategoryBreakOut.GROSS_SALES);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CategoryBreakOut.class));
		List<CategoryBreakOut> list = criteria.list();
		return list;
	}

	private List<CategoryBreakOut> calculateCategoryIncludedTax(Session session, Boolean isReturn) {
		Criteria criteria = buildCriteriaForTicketItem(session, isReturn);
		criteria.add(Restrictions.eq(TicketItem.PROP_TAX_INCLUDED, Boolean.TRUE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_ID), TicketItem.PROP_CATEGORY_ID);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_NAME), TicketItem.PROP_CATEGORY_NAME);
		projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_TAX), CategoryBreakOut.TAX_INCLUDED_AMOUNT);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CategoryBreakOut.class));
		List<CategoryBreakOut> list = criteria.list();

		return list;
	}

	private List<CategoryBreakOut> calculateCategoryDiscountAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, null);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_ID), TicketItem.PROP_CATEGORY_ID);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_NAME), TicketItem.PROP_CATEGORY_NAME);
		projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_DISCOUNT), CategoryBreakOut.DISCOUNT_AMOUNT);
		criteria.setProjection(projectionList);
		criteria.setResultTransformer(Transformers.aliasToBean(CategoryBreakOut.class));
		List<CategoryBreakOut> list = criteria.list();
		return list;
	}

	private List<CategoryBreakOut> calculateCategoryReturnAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session, Boolean.TRUE);

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_ID), TicketItem.PROP_CATEGORY_ID);
		projectionList.add(Projections.groupProperty(TicketItem.PROP_CATEGORY_NAME), TicketItem.PROP_CATEGORY_NAME);
		projectionList.add(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT), CategoryBreakOut.RETURN_AMOUNT);
		criteria.setProjection(projectionList);
		criteria.setResultTransformer(Transformers.aliasToBean(CategoryBreakOut.class));
		List<CategoryBreakOut> list = criteria.list();
		return list;
	}

	/**
	 * @param session
	 * 
	 * GrossSales = GrossSales -IncludedTax
	 * 
	 * ReturnAmount = ReturnAmount + IncludedTaxForReturn
	 * 
	 * NetSales = GrossSales - DiscountAmount + ReturnAmount
	 * 
	 * @return List of CategoryBreakOut
	 */
	public List<CategoryBreakOut> calculateCategoryBreakout(Session session) {
		Map<String, CategoryBreakOut> categoryBreakOutMap = new HashMap<String, CategoryBreakOut>();

		List<CategoryBreakOut> calculateCategoryGrossSales = calculateCategoryGrossSales(session);
		List<CategoryBreakOut> calculateCategoryIncludedTax = calculateCategoryIncludedTax(session, Boolean.FALSE);
		List<CategoryBreakOut> calculateCategoryDiscountAmount = calculateCategoryDiscountAmount(session);
		List<CategoryBreakOut> calculateCategoryReturnAmount = calculateCategoryReturnAmount(session);
		List<CategoryBreakOut> calculateCategoryReturnIncludedTax = calculateCategoryIncludedTax(session, Boolean.TRUE);

		for (CategoryBreakOut categoryGrossSalesBreakOut : calculateCategoryGrossSales) {
			if (NumberUtil.isZero(categoryGrossSalesBreakOut.getGrossSales())) {
				continue;
			}
			String categoryId = categoryGrossSalesBreakOut.getCategoryId();
			String categoryName = categoryGrossSalesBreakOut.getCategoryName();
			String key = generateMapKey(categoryId, categoryName);

			if (StringUtils.isNotBlank(categoryId)) {
				categoryGrossSalesBreakOut.setCategoryName(MenuCategoryDAO.getInstance().getNameById(categoryId));
			}
			else {
				if (StringUtils.isNotBlank(categoryName)) {
					if (categoryName.equals(Messages.getString("MISC"))) { //$NON-NLS-1$
						categoryGrossSalesBreakOut.setCategoryName(Messages.getString("MISC")); //$NON-NLS-1$
					}
				}
				else {
					categoryGrossSalesBreakOut.setCategoryName(POSConstants.NONE);
				}
			}

			categoryBreakOutMap.put(key, categoryGrossSalesBreakOut);
		}

		for (CategoryBreakOut categoryIncludedTaxBreakOut : calculateCategoryIncludedTax) {
			if (NumberUtil.isZero(categoryIncludedTaxBreakOut.getTaxIncludedAmount())) {
				continue;
			}
			String categoryId = categoryIncludedTaxBreakOut.getCategoryId();
			String categoryName = categoryIncludedTaxBreakOut.getCategoryName();
			String key = generateMapKey(categoryId, categoryName);

			CategoryBreakOut categoryBreakOut = categoryBreakOutMap.get(key);
			if (categoryBreakOut == null) {
				categoryBreakOutMap.put(key, categoryIncludedTaxBreakOut);
			}
			else {
				categoryBreakOut.setTaxIncludedAmount(categoryIncludedTaxBreakOut.getTaxIncludedAmount());
			}
		}

		for (CategoryBreakOut categoryDiscountBreakOut : calculateCategoryDiscountAmount) {
			if (NumberUtil.isZero(categoryDiscountBreakOut.getDiscountAmount())) {
				continue;
			}
			String categoryId = categoryDiscountBreakOut.getCategoryId();
			String categoryName = categoryDiscountBreakOut.getCategoryName();
			String key = generateMapKey(categoryId, categoryName);

			CategoryBreakOut categoryBreakOut = categoryBreakOutMap.get(key);
			if (categoryBreakOut == null) {
				categoryBreakOutMap.put(key, categoryDiscountBreakOut);
			}
			else {
				categoryBreakOut.setDiscountAmount(categoryDiscountBreakOut.getDiscountAmount());
			}
		}

		for (CategoryBreakOut categoryReturnAmountBreakOut : calculateCategoryReturnAmount) {
			if (NumberUtil.isZero(categoryReturnAmountBreakOut.getReturnAmount())) {
				continue;
			}
			String categoryId = categoryReturnAmountBreakOut.getCategoryId();
			String categoryName = categoryReturnAmountBreakOut.getCategoryName();
			String key = generateMapKey(categoryId, categoryName);

			CategoryBreakOut categoryBreakOut = categoryBreakOutMap.get(key);
			if (categoryBreakOut == null) {
				categoryBreakOutMap.put(key, categoryReturnAmountBreakOut);
			}
			else {
				categoryBreakOut.setReturnAmount(categoryReturnAmountBreakOut.getReturnAmount());
			}
		}

		for (CategoryBreakOut categoryReturnIncludedTaxBreakOut : calculateCategoryReturnIncludedTax) {
			if (NumberUtil.isZero(categoryReturnIncludedTaxBreakOut.getTaxIncludedAmount())) {
				continue;
			}
			String categoryId = categoryReturnIncludedTaxBreakOut.getCategoryId();
			String categoryName = categoryReturnIncludedTaxBreakOut.getCategoryName();
			String key = generateMapKey(categoryId, categoryName);

			CategoryBreakOut categoryBreakOut = categoryBreakOutMap.get(key);
			if (categoryBreakOut == null) {
				//categoryBreakOutMap.put(categoryId, categoryReturnIncludedTaxBreakOut);
			}
			else {
				categoryBreakOut.setReturnAmount(categoryBreakOut.getReturnAmount() + categoryReturnIncludedTaxBreakOut.getTaxIncludedAmount());
			}
		}

		ArrayList<CategoryBreakOut> categoryBreakOutList = new ArrayList<>(categoryBreakOutMap.values());
		for (CategoryBreakOut categoryReturnIncludedTaxBreakOut : categoryBreakOutList) {
			categoryReturnIncludedTaxBreakOut.calculateBreakoutAmount();
		}

		Comparator<CategoryBreakOut> comparator = Comparator.comparing(CategoryBreakOut::getCategoryName, Comparator.nullsLast(String.CASE_INSENSITIVE_ORDER));
		categoryBreakOutList.sort(comparator);
		return categoryBreakOutList;
	}

	private String generateMapKey(String categoryId, String categoryName) {
		if (StringUtils.isNotBlank(categoryId)) {
			return categoryId;
		}

		String key = ""; //$NON-NLS-1$
		if (StringUtils.isBlank(categoryName)) {
			key += "_" + POSConstants.NONE; //$NON-NLS-1$
		}
		else {
			if (categoryName.equals(Messages.getString("MISC"))) { //$NON-NLS-1$
				key += "_" + "MISC"; //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		return key;
	}

	public double calculateServiceChargeAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_SERVICE_CHARGE));
		double scFromItems = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return scFromItems;
	}

	public double calculateLabDoctorFeeAmount(Session session) {
		return calculateLabDoctorFeeWithoutComboChildAmount(session) + calculateComboChildLDF(session, null);
	}

	public double calculateLabDoctorFeeWithoutComboChildAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_LAB_DOCTOR_FEE));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateTicketReferralCommissionAmount(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_REPORT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateTotalRfOnNetSales(Session session) {
		Criteria criteria = buildCriteriaForTicketItem(session);
		criteria.setProjection(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_NET_SALES));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculateOthersCardSummery(Session session, Class transactionClass, Date fromDate, Date toDate, User user) {
		Criteria criteria = buildCriteriaForTransaction(session, fromDate, toDate, user, storeSession, transactionClass);
		PosTransactionDAO.createOthersCardSearchCriteria(criteria);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	private Criteria buildCriteriaForRFAndLDFTransaction(Session session, Date fromDate, Date toDate, User user, StoreSession storeSession,
			Class<? extends PosTransaction> transactionClass) {
		Criteria criteria = session.createCriteria(transactionClass); //$NON-NLS-1$
		//criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.OUT.name()));

		_RootDAO.addUserWithAllRoleCriteria(criteria, user, PosTransaction.PROP_USER_ID, isIncludeAllRole);
		if (storeSession != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSession.getId()));
		}
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		if (fromDate != null) {
			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
		}
		if (toDate != null) {
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId())); //$NON-NLS-1$
		}
		return criteria;
	}

	public double calculatePurchaseOrderAmount(Session session) {
		Criteria criteria = buildCommonPurchaseOrderCriteria(session);
		criteria.setProjection(Projections.sum(PurchaseOrder.PROP_TOTAL_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	public double calculatePurchaseOrderDueAmount(Session session) {
		Criteria criteria = buildCommonPurchaseOrderCriteria(session);
		criteria.setProjection(Projections.sum(PurchaseOrder.PROP_DUE_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

	private Criteria buildCommonPurchaseOrderCriteria(Session session) {
		Criteria criteria = session.createCriteria(PurchaseOrder.class);
		if (fromDate != null) {
			criteria.add(Restrictions.ge(PurchaseOrder.PROP_CREATED_DATE, fromDate));
		}

		if (toDate != null) {
			criteria.add(Restrictions.le(PurchaseOrder.PROP_CREATED_DATE, toDate));
		}

		if (outlet != null) {
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_OUTLET_ID, outlet.getId()));
		}

		if (StringUtils.isNotBlank(projectId)) {
			criteria.add(Restrictions.eq(PurchaseOrder.PROP_PROJECT_ID, projectId)); //$NON-NLS-1$
		}

		//		Disjunction disjunction = Restrictions.disjunction();
		//		disjunction.add(Restrictions.eq(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_VERIFIED));
		//		disjunction.add(Restrictions.eq(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_CLOSED));
		//		disjunction.add(Restrictions.eq(PurchaseOrder.PROP_STATUS, PurchaseOrder.ORDER_FULLY_RECEIVED));
		//		criteria.add(disjunction);

		criteria.add(Restrictions.or(Restrictions.isNull(PurchaseOrder.PROP_DELETED), Restrictions.eq(PurchaseOrder.PROP_DELETED, Boolean.FALSE)));

		return criteria;
	}

	public double calculateComboChildLDF(Session session, Boolean itemReturned) {

		StringBuilder sb = new StringBuilder();
		sb.append("SELECT COALESCE(SUM(child." + TicketItem.PROP_LAB_DOCTOR_FEE_COL_NAME + "), 0) ");
		sb.append("FROM ticket_item child ");
		sb.append("JOIN ticket t ON ");
		sb.append("  child.parent_ticket_id = t.id ");
		sb.append("  AND child.parent_ticket_outlet_id = t.outlet_id ");
		sb.append("WHERE child.combo_child = TRUE ");
		sb.append("  AND child.voided = FALSE ");
		sb.append("  AND t.voided = FALSE ");
		
		if (ticketType != null) {
			sb.append("  AND t.type = :ticketType ");
		}
		
		if (fromDate != null) {
			sb.append(" AND child.create_date >= :fromDate ");
		}
		if (toDate != null) {
			sb.append(" AND child.create_date < :toDate ");
		}

		if (outlet != null) {
			sb.append(" AND child.parent_ticket_outlet_id = :outletId ");
		}

		if (storeSession != null) {
			sb.append(" AND t.store_session_id = :storeSessionId ");
		}
		if (shift != null) {
			if (StringUtils.isBlank(shift.getId())) {
				sb.append(" AND t.shift_id IS NULL ");
			}
			else {
				sb.append(" AND t.shift_id = :shiftId ");
			}
		}
		if (orderTypeIds != null && !orderTypeIds.isEmpty()) {
			sb.append(" AND t.order_type_id IN (:orderTypeIds) ");
		}
		if (StringUtils.isNotBlank(projectId)) {
			sb.append(" AND t.project_id = :projectId ");
		}
		if (user != null) {
			if (isIncludeAllRole) {
				sb.append(" AND t.owner_id IN (:ownerIds) ");
			}
			else {
				sb.append(" AND t.owner_id = :userId ");
			}
		}
		if (isShowOnlineTicket) {
			sb.append(" AND t.source = :source ");
		}
		if (itemReturned != null) {
			sb.append(" AND child.item_returned = :itemReturned ");
		}

		SQLQuery query = session.createSQLQuery(sb.toString());
		
		if (ticketType != null) {
			query.setParameter("ticketType", ticketType.getTypeNo());
		}

		if (fromDate != null) {
			query.setParameter("fromDate", fromDate);
		}
		if (toDate != null) {
			query.setParameter("toDate", toDate);
		}
		if (outlet != null) {
			query.setParameter("outletId", outlet.getId());
		}
		if (storeSession != null)
			query.setParameter("storeSessionId", storeSession.getId());
		if (shift != null && StringUtils.isNotBlank(shift.getId())) {
			query.setParameter("shiftId", shift.getId());
		}
		if (orderTypeIds != null && !orderTypeIds.isEmpty()) {
			query.setParameterList("orderTypeIds", orderTypeIds);
		}
		if (StringUtils.isNotBlank(projectId))
			query.setParameter("projectId", projectId);
		if (user != null) {
			if (isIncludeAllRole) {
				query.setParameterList("ownerIds", user.getRoleIds());
			}
			else {
				query.setParameter("userId", user.getId());
			}
		}
		if (isShowOnlineTicket) {
			query.setParameter("source", TicketSource.Online.name());
		}
		if (itemReturned != null) {
			query.setParameter("itemReturned", itemReturned);
		}
		PosLog.debug(getClass(), "LDF query: " + query.getQueryString() + "\n");

		Number result = (Number) query.uniqueResult();
		return result != null ? result.doubleValue() : 0.0;
	}

	public TicketType getTicketType() {
		return ticketType;
	}

	public void setTicketType(TicketType ticketType) {
		this.ticketType = ticketType;
	}

}
