/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.awt.Color;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.SerializationUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.search.annotations.Field;
import org.hibernate.search.annotations.Index;
import org.hibernate.search.annotations.Indexed;
import org.hibernate.search.annotations.TermVector;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.Messages;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.model.base.BaseMenuItem;
import com.floreantpos.model.dao.MenuCategoryDAO;
import com.floreantpos.model.dao.MenuGroupDAO;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.MenuItemSizeDAO;
import com.floreantpos.model.dao.PriceTableItemDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.HibernateSearchUtil;
import com.floreantpos.model.util.InventoryUnitConvertionUtil;
import com.floreantpos.model.util.ReferralCommissionType;
import com.floreantpos.model.util.ServiceChargeType;
import com.floreantpos.model.util.ServiceType;
import com.floreantpos.model.util.SortUtil;
import com.floreantpos.model.util.pricecalc.DataUtilCalcFactory;
import com.floreantpos.services.MenuItemService;
import com.floreantpos.util.CurrencyUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.OrgJsonUtil;
import com.floreantpos.util.POSUtil;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "buttonColor", "textColor", "image", "orderTypeList", "properties", "discounts", "terminals",
		"recepieItems", "units", "sizes", "variantName", "parentMenuItem", "explorerImage", "taxGroup", "reportGroup", "unit", "stockStatus", "availableUnit",
		"unitOnHand", "variants" })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@Entity
@Indexed
public class MenuItem extends BaseMenuItem implements TimedModel {

	public static final String MED_GENERICS_NAME = "medGenerics.name";
	public static final String DOSAGE_FORMS_NAME = "dosageForms.name";
	public static final String MANUFACTURER_NAME = "manufacturer.name";
	public static final String JSON_PROP_SUFFIX_TAXGROUP = ".taxgroup"; //$NON-NLS-1$
	public static final String JSON_PROP_SUFFIX_FOR_HERE = ".for_here"; //$NON-NLS-1$
	public static final String JSON_PROP_SUFFIX_TO_GO = ".to_go"; //$NON-NLS-1$
	public static final String JSON_PROP_OUTDOOR_BOOKING_FEE = "outdoor_booking_fee"; //$NON-NLS-1$

	private static final long serialVersionUID = 1L;
	public static final String TRANSIENT_PROP_VENDOR_NAME = "vendorNames"; //$NON-NLS-1$
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private static final String JSON_PROP_ENABLE_QTY_SELECTION = "enable.qty.selection"; //$NON-NLS-1$
	private static final String JSON_PROP_MOD_PRICE_COMBO_ITEM = "menuitem.modifiable.price.on.comboItem"; //$NON-NLS-1$
	/**
	 * Vendor Names added to show in inventory items which contains vendors.
	 * This property has been set in MenuItemDAO and method name is loadInventoryItems 
	 * 
	 */
	private transient String vendorNames;

	/* [CONSTRUCTOR MARKER BEGIN] */
	public MenuItem() {
	}

	/**
	 * Constructor for primary key
	 */
	public MenuItem(java.lang.String id) {
		super(id);
	}

	/**
	 * Constructor for required fields
	 */
	public MenuItem(java.lang.String id, java.lang.String name, java.lang.Double price) {

		super(id, name, price);
	}

	/* [CONSTRUCTOR MARKER END] */

	private MenuItem parentMenuItem;
	private MenuItemInventoryStatus stockStatus;

	public static final String TYPE_RAW_METARIAL = "Raw Material"; //$NON-NLS-1$
	public static final String TYPE_INVENTORY_ITEM = "Inventory Item"; //$NON-NLS-1$
	public static final String TYPE_MENU_ITEM = "Menu Item"; //$NON-NLS-1$
	private transient JSONObject properties;
	private String outletId;
	private transient MedGenerics medGenerics;
	private transient DosageForms dosageForms;
	private transient Manufacturer manufacture;

	/**
	 * Used in Excel Data import only
	 */
	private transient double stockQty;

	@Override
	public Integer getSortOrder() {
		return super.getSortOrder() == null ? 9999 : super.getSortOrder();
	}

	@Override
	public Integer getButtonColorCode() {
		Integer buttonColorCode = super.getButtonColorCode();
		if (buttonColorCode == null || buttonColorCode == 0 || buttonColorCode == -1316371) {
			buttonColorCode = Color.WHITE.getRGB();
		}
		return buttonColorCode;
	}

	@XmlTransient
	public Color getButtonColor() {
		Integer buttonColorCode = getButtonColorCode();
		if (buttonColorCode == null) {
			return null;
		}

		return new Color(getButtonColorCode());
	}

	public void setButtonColor(Color buttonColor) {
		if (buttonColor != null) {
			setTextColorCode(buttonColor.getRGB());
		}
	}

	@XmlTransient
	public Color getTextColor() {
		if (getTextColorCode() == null) {
			return null;
		}
		return new Color(getTextColorCode());
	}

	public void setTextColor(Color textColor) {
		if (textColor != null) {
			setTextColorCode(textColor.getRGB());
		}
	}

	@XmlTransient
	public String getDisplayName() {
		//		Terminal terminal = DataProvider.get().getCurrentTerminal();
		//		if (((terminal != null && terminal.isShowTranslatedName()) || isVariant()) && StringUtils.isNotEmpty(super.getTranslatedName())) {
		//			return super.getTranslatedName();
		//		}
		if (StringUtils.isNotEmpty(super.getTranslatedName())) {
			return super.getTranslatedName();
		}

		if (ProductType.match(getProductType(), ProductType.MEDICINE)) {
			String strength = StringUtils.isNotBlank(getStrength()) ? getStrength() : ""; //$NON-NLS-1$
			return super.getName() + " " + getDosageFormName() + " " + strength; //$NON-NLS-1$ //$NON-NLS-2$
		}

		return super.getName();
	}

	@XmlTransient
	@JsonIgnore
	public String getServiceItemNameWithPrice() {
		if (POSUtil.getBoolean(getProperty("service.price.included"), false)) { //$NON-NLS-1$
			return super.getName() + " - " + Messages.getString("TicketView.35"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		return getName() + " - " + CurrencyUtil.getCurrencySymbol() + NumberUtil.format(getPrice()); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getUnitNameDisplay() {
		InventoryUnit unit = getUnit();
		if (unit == null) {
			if (isEachUnit(super.getUnitName())) {
				return ""; //$NON-NLS-1$
			}
			return super.getUnitName();
		}
		String displayName = unit.getDisplayName();
		String unitName = StringUtils.isBlank(displayName) ? unit.getCode() : displayName;
		if (StringUtils.isBlank(unitName)) {
			unitName = getUnitName();
		}
		if (isEachUnit(unitName)) {
			return ""; //$NON-NLS-1$
		}
		return unitName;
	}

	private boolean isEachUnit(String unitNameDisplay) {
		return unitNameDisplay == null || "ea".equalsIgnoreCase(unitNameDisplay) || "pc".equalsIgnoreCase(unitNameDisplay) //$NON-NLS-1$//$NON-NLS-2$
				|| ("Each".equalsIgnoreCase(unitNameDisplay)); //$NON-NLS-1$
	}

	public String createVariantName(String name) {
		List<Attribute> attributes = getAttributes();
		if (attributes == null) {
			return name;
		}
		String variantName = " ("; //$NON-NLS-1$
		for (Iterator<Attribute> iterator = attributes.iterator(); iterator.hasNext();) {
			Attribute attribute = (Attribute) iterator.next();
			variantName += attribute.getName();
			if (iterator.hasNext()) {
				variantName += ", "; //$NON-NLS-1$
			}
		}
		variantName += ")"; //$NON-NLS-1$

		if (StringUtils.isNotBlank(name) && (name.length() + variantName.length()) > 255) {
			name = name.substring(0, name.length() - (variantName.length() + 3)) + "..." + variantName; //$NON-NLS-1$
		}
		else {
			name += variantName;
		}
		return name;
	}

	@XmlTransient
	public String getVariantName() {
		String name = super.getName();
		List<Attribute> attributes = getAttributes();
		if (attributes == null) {
			return name;
		}
		name += " ("; //$NON-NLS-1$
		for (Iterator iterator = attributes.iterator(); iterator.hasNext();) {
			Attribute attribute = (Attribute) iterator.next();
			name += attribute.getName();
			if (iterator.hasNext()) {
				name += ", "; //$NON-NLS-1$
			}
		}
		name += ")"; //$NON-NLS-1$
		return name;
	}

	public void setVariantName(String variantName) {

	}

	public void setDisplayName(String displayName) {

	}

	@Override
	public String toString() {
		return getDisplayName();
	}

	public String getUniqueId() {
		return ("menu_item_" + getName() + "_" + getId()).replaceAll("\\s+", "_"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	}

	//	public TicketItem convertToTicketItem() {
	//		return convertToTicketItem(null, 0);
	//	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity) {
		return convertToTicketItem(ticket, itemQuantity, false, null, true, false);
	}

	public TicketItem convertToTicketItemInOnlinePrice(Ticket ticket, double itemQuantity) {
		return convertToTicketItem(ticket, itemQuantity, false, null, true, true);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, IUnit destinationUnit) {
		return convertToTicketItem(ticket, itemQuantity, false, destinationUnit, true);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, boolean hasComboModifiers) {
		return this.convertToTicketItem(ticket, itemQuantity, hasComboModifiers, null, true);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, boolean hasComboModifiers, IUnit destinationUnit, boolean checkEditablePrice) {
		return convertToTicketItem(ticket, itemQuantity, hasComboModifiers, destinationUnit, checkEditablePrice, false);
	}

	public TicketItem convertToTicketItem(Ticket ticket, double itemQuantity, boolean hasComboModifiers, IUnit destinationUnit, boolean checkEditablePrice,
			boolean useOnlinePrice) {
		return DataUtilCalcFactory.getCalc().convertToTicketItem(this, ticket, itemQuantity, hasComboModifiers, destinationUnit, checkEditablePrice,
				useOnlinePrice);
	}

	public void setTicketItemUnitPriceAndCost(TicketItem ticketItem, MenuItem menuItem, IUnit destinationUnit, Ticket ticket) {
		setTicketItemUnitPriceAndCost(ticketItem, menuItem, destinationUnit, ticket, true);
	}

	public void setTicketItemUnitPriceAndCost(TicketItem ticketItem, MenuItem menuItem, IUnit destinationUnit, Ticket ticket, boolean checkEditablePrice) {
		setTicketItemUnitPriceAndCost(ticketItem, menuItem, destinationUnit, ticket, checkEditablePrice, false);
	}

	public void setTicketItemUnitPriceAndCost(TicketItem ticketItem, MenuItem menuItem, IUnit destinationUnit, Ticket ticket, boolean checkEditablePrice,
			boolean useOnlinePrice) {
		DataUtilCalcFactory.getCalc().setTicketItemUnitPriceAndCost(this, ticketItem, destinationUnit, ticket, checkEditablePrice, useOnlinePrice);
	}

	public double getDestinationUnitPrice(MenuItem menuItem, IUnit destinationUnit, Double priceFromPriceRule, double unitQuantity) {
		return DataUtilCalcFactory.getCalc().getDestinationUnitPrice(menuItem, destinationUnit, priceFromPriceRule, unitQuantity);
	}

	public double getPriceFromPriceRule(Ticket ticket) {
		return DataUtilCalcFactory.getCalc().getPriceFromPriceRule(this, ticket);
	}

	public boolean hasModifiers() {
		return super.isHasModifiers();
	}

	public boolean hasMandatoryModifiers() {
		return super.isHasMandatoryModifiers();
	}

	public boolean hasAutoShowModifierGroup() {
		List<MenuItemModifierSpec> modifierSpecs = getMenuItemModiferSpecs();

		if (modifierSpecs == null || modifierSpecs.size() == 0) {
			return false;
		}

		for (MenuItemModifierSpec menuItemModifierSpec : modifierSpecs) {
			if (menuItemModifierSpec.isUseModifierGroupSettings()) {
				ModifierGroup modifierGroup = menuItemModifierSpec.getModifierGroup();
				if (!modifierGroup.isEnable()) {
					continue;
				}
				if (menuItemModifierSpec.isUseModifierGroupSettings()) {
					menuItemModifierSpec.copyModifierGroupProperties();
				}

				if (modifierGroup.isAutoShow() || modifierGroup.getMinQuantity() > 0) {
					return true;
				}
			}
			else {
				if (!menuItemModifierSpec.isEnable()) {
					continue;
				}

				if (menuItemModifierSpec.isAutoShow() || menuItemModifierSpec.getMinQuantity() > 0) {
					return true;
				}
			}
		}

		return false;
	}

	public static void setItemTaxes(TicketItem ticketItem, TaxGroup itemTaxGroup, OrderType orderType) {
		DataUtilCalcFactory.getCalc().setItemTaxes(ticketItem, itemTaxGroup, orderType);
	}

	public String getStringWithUnderScore(String orderType, String additionalString) {

		orderType = orderType.replaceAll(" ", "_");//$NON-NLS-1$ //$NON-NLS-2$

		return orderType + additionalString;
	}

	@JsonIgnore
	public List<IUnit> getUnits() {
		return getUnits(false);
	}

	public List<IUnit> getUnits(boolean includeRecipeUnits) {
		return getUnits(true, includeRecipeUnits);
	}

	public List<IUnit> getUnits(boolean includeStockUnits, boolean includeRecipeUnits) {
		return getUnits(includeStockUnits, includeRecipeUnits, false);
	}

	public List<IUnit> getUnits(boolean includeStockUnits, boolean includeRecipeUnits, boolean skipStockNotCountableUnits) {
		List<IUnit> units = DataUtilCalcFactory.getCalc().getUnits(this, includeStockUnits, includeRecipeUnits, skipStockNotCountableUnits);
		if (units.isEmpty()) {
			units.add(DataProvider.get().getInventoryUnitById("each")); //$NON-NLS-1$
		}
		return units;
	}

	@JsonIgnore
	public List<InventoryStockUnit> getVariantStockUnits() {
		if (isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getStockUnits();
			}
		}
		return super.getStockUnits();
	}

	public Double getCost(IUnit selectedUnit) {
		return DataUtilCalcFactory.getCalc().getCost(this, selectedUnit);
	}

	public Double getCost(String unitId) {
		return DataUtilCalcFactory.getCalc().getCost(this, unitId);
	}

	public double getUnitQuantity(IUnit sourceInventoryUnit, IUnit destinationInventoryUnit) {
		return InventoryUnitConvertionUtil.getUnitQuantity(sourceInventoryUnit, destinationInventoryUnit, this);
	}

	public double getBaseUnitQuantity(String unitId) {
		return InventoryUnitConvertionUtil.getBaseUnitQuantity(unitId, this);
	}

	@Override
	public Double getAverageUnitPurchasePrice() {
		Double avgprice = super.getAverageUnitPurchasePrice();
		if (avgprice == 0)
			return getCost();
		return avgprice;
	}

	@XmlTransient
	@JsonIgnore
	public Double getAvailableUnit() {
		stockStatus = getStockStatus();
		return stockStatus == null ? 0.0 : stockStatus.getAvailableUnit();
	}

	@XmlTransient
	@JsonIgnore
	public Double getUnitOnHand() {
		stockStatus = getStockStatus();
		return stockStatus == null ? 0.0 : stockStatus.getUnitOnHand();
	}

	//	public void setAvailableUnit(Double availableUnit) {
	//		if (stockStatus == null) {
	//			stockStatus = new MenuItemInventoryStatus();
	//			stockStatus.setOutletId(DataProvider.get().getOutlet().getId());
	//			stockStatus.setMenuItemId(getId());
	//		}
	//		stockStatus.setAvailableUnit(availableUnit);
	//	}

	@JsonIgnore
	public Double getRetailPrice() {
		return getPrice();
	}

	@JsonIgnoreProperties
	@XmlTransient
	@Deprecated
	@JsonIgnore
	public MenuGroup getParent() {
		String groupId = getMenuGroupId();
		if (StringUtils.isNotEmpty(groupId)) {
			return MenuGroupDAO.getInstance().get(groupId);
		}
		return null;
	}

	public void setParent(MenuGroup menuGroup) {
		setMenuGroup(menuGroup);
	}

	public void setMenuGroup(MenuGroup menuGroup) {
		if (menuGroup == null) {
			setMenuGroupId(null);
			setMenuGroupName(null);
			setMenuCategoryId(null);
			setMenuCategoryName(null);
			setTicketDiscountApplicable(true);
		}
		else {
			setMenuGroupId(menuGroup.getId());
			setMenuGroupName(menuGroup.getDisplayName());
			setMenuCategoryId(menuGroup.getMenuCategoryId());
			setMenuCategoryName(menuGroup.getMenuCategoryName());

			String categoryId = menuGroup.getMenuCategoryId();
			if (StringUtils.isNotBlank(categoryId)) {
				MenuCategory menuCategory = MenuCategoryDAO.getInstance().get(categoryId);
				if (menuCategory != null) {
					setTicketDiscountApplicable(menuCategory.isTicketDiscountApplicable());
				}
			}
		}
	}

	@XmlTransient
	public PrinterGroup getPrinterGroup() {
		return DataProvider.get().getPrinterGroupById(getPrinterGroupId());
	}

	public void setPrinterGroup(PrinterGroup printerGroup) {
		String printerGroupId = null;
		if (printerGroup != null) {
			printerGroupId = printerGroup.getId();
		}
		super.setPrinterGroupId(printerGroupId);
	}

	@XmlTransient
	public InventoryUnit getUnit() {
		return DataProvider.get().getInventoryUnitById(getUnitId());
	}

	public void setUnit(InventoryUnit unit) {
		String unitId = null;
		if (unit != null) {
			unitId = unit.getId();
			setUnitName(unit.getName());
		}
		super.setUnitId(unitId);
	}

	@Override
	public String getUnitId() {
		String unitId = super.getUnitId();
		if (StringUtils.isBlank(unitId)) {
			return "each"; //$NON-NLS-1$
		}
		return unitId;
	}

	@XmlTransient
	public ReportGroup getReportGroup() {
		return DataProvider.get().getReportGroupById(getReportGroupId());
	}

	public void setReportGroup(ReportGroup reportGroup) {
		String reportGroupId = null;
		if (reportGroup != null) {
			reportGroupId = reportGroup.getId();
		}
		super.setReportGroupId(reportGroupId);
	}

	@XmlTransient
	public TaxGroup getTaxGroup() {
		TaxGroup taxGroup = DataProvider.get().getTaxGroupById(getTaxGroupId());
		if (taxGroup == null && isVariant() && getParentMenuItemId() != null) {
			taxGroup = getParentMenuItem().getTaxGroup();
		}

		return taxGroup;
	}

	public TaxGroup getTaxGroup(Customer customer) {
		return getTaxGroup();
	}

	public void setTaxGroup(TaxGroup taxGroup) {
		String taxGroupId = null;
		if (taxGroup != null) {
			taxGroupId = taxGroup.getId();
		}
		super.setTaxGroupId(taxGroupId);
	}

	//TODO:
	public void setOrderTypeList(List checkedValues) {
	}

	@XmlTransient
	public MenuItem getParentMenuItem() {
		if (parentMenuItem != null) {
			return parentMenuItem;
		}
		String parentMenuItemId = getParentMenuItemId();
		if (StringUtils.isEmpty(parentMenuItemId)) {
			parentMenuItemId = getProperty("variantParentMenuItemId");
			if (StringUtils.isBlank(parentMenuItemId)) {
				return null;
			}
		}
		parentMenuItem = MenuItemDAO.getInstance().get(parentMenuItemId);
		return parentMenuItem;
	}

	public void setParentMenuItem(MenuItem parentMenuItem) {
		this.parentMenuItem = parentMenuItem;

		if (parentMenuItem != null) {
			String parentMenuItemId = parentMenuItem.getId();
			if (StringUtils.isBlank(super.getParentMenuItemId()) && StringUtils.isNotBlank(parentMenuItemId)) {
				super.setParentMenuItemId(parentMenuItemId);
				addProperty("variantParentMenuItemId", parentMenuItemId); //$NON-NLS-1$
			}
		}
	}

	@XmlTransient
	@JsonIgnore
	public MenuItemInventoryStatus getStockStatus() {
		if (stockStatus != null && stockStatus.getOutletId() != null && stockStatus.getOutletId().equals(DataProvider.get().getOutlet().getId())) {
			return stockStatus;
		}
		stockStatus = DataProvider.get().getMenuItemStockStatus(this);
		return stockStatus;
	}

	public void setStockStatus(MenuItemInventoryStatus stockStatus) {
		this.stockStatus = stockStatus;
	}

	@Override
	public void setPropertiesJson(String propertiesJson) {
		super.setPropertiesJson(propertiesJson);
		if (StringUtils.isNotEmpty(propertiesJson)) {
			properties = new JSONObject(propertiesJson);
		}
		else {
			properties = new JSONObject();
		}
	}

	private void buildPropertis() {
		if (properties != null) {
			return;
		}
		String json = getPropertiesJson();
		if (StringUtils.isEmpty(json)) {
			properties = new JSONObject();
			return;
		}
		properties = new JSONObject(json);
	}

	public void addProperty(String key, String value) {
		buildPropertis();
		properties.put(key, value);
		setPropertiesJson(properties.toString());
	}

	public JSONObject getProperties() {
		return properties;
	}

	public String getProperty(String key) {
		buildPropertis();
		if (properties.has(key)) {
			return properties.getString(key);
		}
		return null;
	}

	public void removeProperty(String key) {
		buildPropertis();
		properties.remove(key);
		setPropertiesJson(properties.toString());
	}

	//set orphan collections to null before cloning this menu item.
	//because hibernate gives reference change error if orphan collections are not set to null
	private MenuItem cloneItemConsideringOrphanCollections(MenuItem itemToClone) {
		List<MenuItem> variants = itemToClone.getVariants();

		itemToClone.setVariants(null);

		MenuItem newMenuItem = (MenuItem) SerializationUtils.clone(itemToClone);

		itemToClone.setVariants(variants);

		return newMenuItem;
	}

	public MenuItem clone() {
		MenuItem newMenuItem = cloneItemConsideringOrphanCollections(this);

		String newName = doDuplicateName(getName());
		newMenuItem.setName(newName);
		newMenuItem.setId(null);
		newMenuItem.setDefaultRecipeId(null);

		List<TestItem> testItems = newMenuItem.getTestItems();
		if (testItems != null) {
			for (Iterator iterator = testItems.iterator(); iterator.hasNext();) {
				TestItem testItem = (TestItem) iterator.next();
				if (testItem == null) {
					iterator.remove();
				}
			}
			newMenuItem.setTestItems(new ArrayList<>(testItems));
		}

		List<InventoryStockUnit> stockUnits = newMenuItem.getStockUnits();
		if (stockUnits != null) {
			for (InventoryStockUnit stockUnit : stockUnits) {
				stockUnit.setId(null);
				stockUnit.setMenuItemId(newMenuItem.getId());
			}
		}
		newMenuItem.setStockUnits(new ArrayList<>(stockUnits));

		List<ComboItem> comboItems = newMenuItem.getComboItems();
		newMenuItem.setComboItems(new ArrayList<>());
		if (comboItems != null) {
			for (ComboItem comboItem : comboItems) {
				comboItem.setId(null);
				newMenuItem.getComboItems().add(comboItem);
			}
		}
		List<ComboGroup> comboGroups = newMenuItem.getComboGroups();
		newMenuItem.setComboGroups(new ArrayList<>());
		if (comboGroups != null) {
			for (ComboGroup comboGroup : comboGroups) {
				comboGroup.setId(null);
				List<MenuItem> comboGroupItems = comboGroup.getItems();
				comboGroup.setItems(new ArrayList<>());
				if (comboGroupItems != null) {
					for (MenuItem comboGroupItem : comboGroupItems) {
						comboGroup.getItems().add(comboGroupItem);
					}
				}
				newMenuItem.getComboGroups().add(comboGroup);
			}
		}
		copyModifierSpecsToMenuItem(newMenuItem, newMenuItem.getMenuItemModiferSpecs());

		newMenuItem.setVariants(null);
		List<MenuItem> variants = getVariants();
		if (variants != null) {
			for (MenuItem variantMenuItem : variants) {
				if (variantMenuItem.isDeleted()) {
					continue;
				}
				MenuItem newVariantMenuItem = cloneItemConsideringOrphanCollections(variantMenuItem);
				newVariantMenuItem.setId(null);

				newVariantMenuItem.setMenuItemModiferSpecs(null);
				newVariantMenuItem.setComboGroups(null);
				newVariantMenuItem.setComboItems(null);
				newVariantMenuItem.setVariants(null);

				List<Attribute> newAttributies = new ArrayList<Attribute>();
				List<Attribute> attributes2 = variantMenuItem.getAttributes();
				for (Attribute attribute : attributes2) {
					Attribute newVariantAttribute = (Attribute) SerializationUtils.clone(attribute);
					newAttributies.add(newVariantAttribute);
				}
				newVariantMenuItem.setAttributes(newAttributies);
				newMenuItem.addTovariants(newVariantMenuItem);
			}
		}

		return newMenuItem;
	}

	public static void copyModifierSpecsToMenuItem(MenuItem newMenuItem, List<MenuItemModifierSpec> menuItemModifierSpecs) {
		List<MenuItemModifierSpec> newSpecs = new ArrayList<>();
		if (menuItemModifierSpecs != null) {
			for (MenuItemModifierSpec menuModifierSpec : menuItemModifierSpecs) {
				menuModifierSpec.setId(null);
				Set<MenuItemModifierPage> specPages = menuModifierSpec.getModifierPages();
				Set<MenuItemModifierPage> newPages = new HashSet<>();
				if (specPages != null) {
					for (MenuItemModifierPage menuItemModifierPage : specPages) {
						List<MenuItemModifierPageItem> newPageItems = new ArrayList<>();
						menuItemModifierPage.setId(null);
						menuItemModifierPage.setModifierSpecId(menuModifierSpec.getId());
						List<MenuItemModifierPageItem> pageItems = menuItemModifierPage.getPageItems();
						if (pageItems != null) {
							for (MenuItemModifierPageItem menuItemModifierPageItem : pageItems) {
								menuItemModifierPageItem.setId(null);
								menuItemModifierPageItem.setParentPage(menuItemModifierPage);
								newPageItems.add(menuItemModifierPageItem);
							}
						}
						menuItemModifierPage.setPageItems(newPageItems);
						newPages.add(menuItemModifierPage);
					}
				}
				menuModifierSpec.setModifierPages(newPages);
				newSpecs.add(menuModifierSpec);
			}
		}
		newMenuItem.setMenuItemModiferSpecs(newSpecs);
	}

	private static String doDuplicateName(String existingName) {
		String newName = new String();
		int lastIndexOf = existingName.lastIndexOf(" "); //$NON-NLS-1$
		if (lastIndexOf == -1) {
			newName = existingName + " 1"; //$NON-NLS-1$
		}
		else {
			String itemCountProcessName = existingName.substring(lastIndexOf + 1, existingName.length()).trim();
			if (!StringUtils.isEmpty(itemCountProcessName) && StringUtils.isNumeric(itemCountProcessName)) {
				Integer count = Integer.valueOf(itemCountProcessName);
				count += 1;
				int lastIndex = existingName.length() - itemCountProcessName.length();
				newName = existingName.substring(0, lastIndex) + String.valueOf(count);
			}
			else {
				newName = existingName.trim() + " 1"; //$NON-NLS-1$
			}
		}
		return newName;
	}

	public void setUnitSelection(boolean unitSelection) {
		addProperty(AppConstants.ALLOW_UNIT_SELECTION, String.valueOf(unitSelection));
	}

	public boolean isAllowUnitSelection() {
		if (ProductType.MEDICINE.name().equals(getProductType()) || ProductType.GOODS.name().equals(getProductType())) {
			return Boolean.TRUE;
		}

		if (isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getBooleanProperty(AppConstants.ALLOW_UNIT_SELECTION, Boolean.FALSE);
			}
			return Boolean.FALSE;
		}
		return getBooleanProperty(AppConstants.ALLOW_UNIT_SELECTION, Boolean.FALSE);
	}

	public boolean getBooleanProperty(String key, boolean defaultValue) {
		String string = getProperty(key);
		if (StringUtils.isBlank(string)) {
			return defaultValue;
		}
		return Boolean.valueOf(string).booleanValue();
	}

	public String getVendorNames() {
		return vendorNames;
	}

	public void setVendorNames(String vendorName) {
		this.vendorNames = vendorName;
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@XmlTransient
	public double getTotalTaxRate() {
		TaxGroup taxGroup = getTaxGroup();
		if (taxGroup != null) {
			return taxGroup.getTotalTaxRate();
		}
		return 0.0;
	}

	public void setTotalTaxRate(double totalTaxRate) {
	}

	@Override
	public void addTomenuItemModiferSpecs(MenuItemModifierSpec menuItemModifierSpec) {
		List<MenuItemModifierSpec> specs = getMenuItemModiferSpecs();
		if (specs == null) {
			specs = new ArrayList<>(1);
			setMenuItemModiferSpecs(specs);
		}
		for (MenuItemModifierSpec spec : specs) {
			if (spec.getModifierGroupId().equals(menuItemModifierSpec.getModifierGroupId())) {
				return;
			}
		}
		specs.add(menuItemModifierSpec);
	}

	@Override
	public String getUnitName() {
		String unitName = super.getUnitName();
		if (StringUtils.isNotEmpty(unitName)) {
			return unitName;
		}
		String unitId = getUnitId();
		if (StringUtils.isNotEmpty(unitId)) {
			InventoryUnit unit = DataProvider.get().getInventoryUnitById(unitId);
			if (unit != null) {
				return unit.getName();
			}
		}
		return unitName;
	}

	@XmlTransient
	public void setEnableComboQuantitySelection(boolean isEnableQtySelection) {
		this.addProperty(JSON_PROP_ENABLE_QTY_SELECTION, String.valueOf(isEnableQtySelection));
	}

	public boolean isEnableComboQuantitySelection() {
		return POSUtil.getBoolean(getProperty(JSON_PROP_ENABLE_QTY_SELECTION));
	}

	/**
	 * This method will return all variant which is not deleted.
	 * 
	 */
	@XmlTransient
	@JsonIgnore
	public List<MenuItem> getActiveVariants() {
		List<MenuItem> activeVariants = new ArrayList<>();
		if (getVariants() != null) {
			for (Iterator<MenuItem> iterator = getVariants().iterator(); iterator.hasNext();) {
				MenuItem menuItem = iterator.next();
				if (menuItem.getParentMenuItemId() != null && menuItem.isVisible()) {
					activeVariants.add(menuItem);
				}
			}
		}
		SortUtil.sortVariants(activeVariants);
		return activeVariants;
	}

	public boolean attributesEqual(MenuItem other) {
		List<Attribute> currentAttributeList = this.getAttributes();
		List<Attribute> otherAttributeList = other.getAttributes();
		if (currentAttributeList == null || otherAttributeList == null) {
			return false;
		}

		if (currentAttributeList.size() != otherAttributeList.size()) {
			return false;
		}

		for (Attribute attribute : currentAttributeList) {
			if (!otherAttributeList.contains(attribute)) {
				return false;
			}
		}
		return true;
	}

	public boolean hasSimilerGroupSpec(ModifierGroup selectedGroup) {
		List<MenuItemModifierSpec> menuItemModiferSpecs = this.getMenuItemModiferSpecs();
		if (menuItemModiferSpecs == null || menuItemModiferSpecs.isEmpty())
			return false;
		for (Iterator<MenuItemModifierSpec> iterator = menuItemModiferSpecs.iterator(); iterator.hasNext();) {
			MenuItemModifierSpec menuItemModifierSpec = (MenuItemModifierSpec) iterator.next();
			if (menuItemModifierSpec.getModifierGroup().getName().equals(selectedGroup.getName())) {
				return true;
			}
		}
		return false;
	}

	public boolean hasSimilerItemInOtherComboGroup(ComboGroup selectedcomboGroup, List<MenuItem> menuItems) {
		List<ComboGroup> comboGroups = getComboGroups();
		if (comboGroups == null) {
			return false;
		}

		for (ComboGroup comboGroup : comboGroups) {
			if (selectedcomboGroup == comboGroup) {
				continue;
			}

			if (menuItems == null || menuItems.isEmpty()) {
				return false;
			}
			List<MenuItem> existedItems = comboGroup.getItems();
			if (existedItems == null || existedItems.isEmpty()) {
				return false;
			}
			for (MenuItem existedItem : existedItems) {
				if (existedItem == null || StringUtils.isEmpty(existedItem.getId())) {
					continue;
				}
				for (MenuItem menuItem : menuItems) {
					if (menuItem != null && StringUtils.isNotEmpty(menuItem.getId()) && existedItem.getId().equals(menuItem.getId())) {
						return true;
					}
				}
			}
		}
		return false;
	}

	@XmlTransient
	public Double getAdjustedPrice() {
		if (isVariant()) {
			MenuItem variantParent = getParentMenuItem();
			if (variantParent != null) {
				return variantParent.getPrice() + super.getPrice();
			}
		}

		return super.getPrice();
	}

	@XmlTransient
	@JsonIgnore
	public String getNameAndUnitPriceDisplayString() {
		if (isEditablePrice()) {
			return getDisplayName();
		}
		String sym = CurrencyUtil.getCurrencySymbol();
		String unitNameDisplay = getUnitNameDisplay();
		if (StringUtils.isNotBlank(unitNameDisplay)) {
			unitNameDisplay = "/" + unitNameDisplay; //$NON-NLS-1$
		}
		return getDisplayName() + "   " + sym + NumberUtil.formatNumber(getVariantPrice()) + unitNameDisplay; //$NON-NLS-1$
	}

	@XmlTransient
	public String getOnlineDisplayString() {
		String sym = CurrencyUtil.getCurrencySymbol();
		String unitNameDisplay = getUnitNameDisplay();
		if (StringUtils.isNotBlank(unitNameDisplay)) {
			unitNameDisplay = "/" + unitNameDisplay; //$NON-NLS-1$
		}
		return getDisplayName() + "   " + sym + NumberUtil.formatNumber(getOnlinePrice(outletId)) + unitNameDisplay; //$NON-NLS-1$
	}

	public void setAdjustedPrice(Double adjustedPrice) {

	}

	public Boolean isHasOnlinePrice() {
		PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online", this); //$NON-NLS-1$
		if (priceTableItem != null) {
			return Boolean.TRUE;
		}
		return Boolean.FALSE;
	}

	//This method is used only for showing grid online price.
	@XmlTransient
	public Double getOnlinePrice() {
		//		if (isComboItem() && !isBasePriceForComboItem()) {
		//			return null;
		//		}
		return getOnlinePrice(outletId);
	}

	public Double getOnlinePrice(String outletId) {
		if (isVariant()) {
			PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online_" + outletId, this); //$NON-NLS-1$
			if (priceTableItem != null) {
				return priceTableItem.getPrice();
			}
			return getParentMenuItem() == null ? 0.0 : getParentMenuItem().getOnlinePrice(outletId) + getPrice();
		}
		PriceTableItem priceTableItem = PriceTableItemDAO.getInstance().getItemByPriceTableId("menugreat_online_" + outletId, this); //$NON-NLS-1$
		if (priceTableItem != null) {
			return priceTableItem.getPrice();
		}
		return getPrice();
	}

	public void setOutletId(String outletId) {
		this.outletId = outletId;
	}

	public void setOnlinePrice(Double onlinePrice) {
	}

	@Id
	@Override
	public String getId() {
		return super.getId();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.YES, index = Index.YES)
	@Override
	public String getName() {
		return super.getName();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@Override
	public String getDescription() {
		return super.getDescription();
	}

	@Field(store = org.hibernate.search.annotations.Store.YES, termVector = TermVector.NO, index = Index.YES)
	@XmlTransient
	public String getStoreId() {
		if (StringUtils.isBlank(HibernateSearchUtil.getCurrentStoreId())) {
			return DataProvider.get().getStore().getUuid();
		}

		return HibernateSearchUtil.getCurrentStoreId();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isEnableOnlineOrdering() {
		return super.isEnableOnlineOrdering();
	}

	@Override
	@Field(store = org.hibernate.search.annotations.Store.YES)
	public Boolean isDeleted() {
		return super.isDeleted();
	}

	@XmlTransient
	public Double getVariantPrice() {
		if (isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getPrice() + super.getPrice();
			}
		}
		return super.getPrice();
	}

	public void setVariantPrice(Double price) {
		//no use
	}

	@XmlTransient
	public Double getVariantCost() {
		Double cost = super.getCost();
		if (cost == 0 && isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null) {
				return parentMenuItem.getCost();
			}
		}
		return cost;
	}

	public void setVariantCost(Double price) {
		//no use
	}

	public void setDeafultPizzaSize(String pizzaSizeId) {
		this.addProperty("DeafultPizzaSize", pizzaSizeId); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public MenuItemSize getDeafultPizzaSize() {
		String pizzaSizeId = getProperty("DeafultPizzaSize"); //$NON-NLS-1$
		if (StringUtils.isBlank(pizzaSizeId)) {
			return null;
		}
		return MenuItemSizeDAO.getInstance().get(pizzaSizeId);
	}

	public void setDeafultPizzaCrust(String pizzaCrustId) {
		this.addProperty("DeafultPizzaCrust", pizzaCrustId); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getProfitMargin() {
		double profit = getVariantPrice() - getVariantCost();
		double abs = Math.abs(getVariantPrice());
		if (abs > 0) {
			return NumberUtil.round(profit / abs * 100);
		}
		else {
			return 0d;
		}
	}

	public void setProfitMargin(double profitMargin) {
	}

	public void setModifiablePriceForComboItem(boolean isModifiable) {
		this.addProperty(JSON_PROP_MOD_PRICE_COMBO_ITEM, String.valueOf(isModifiable));
	}

	@XmlTransient
	public boolean isModifiablePriceForComboItem() {
		return POSUtil.getBoolean(getProperty(JSON_PROP_MOD_PRICE_COMBO_ITEM));
	}

	public void setBasePriceForComboItem(boolean value) {
		this.addProperty("base.priced.comboItem", String.valueOf(value)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isBasePriceForComboItem() {
		return POSUtil.getBoolean(getProperty("base.priced.comboItem")); //$NON-NLS-1$
	}

	public void setBaseAndGroupPriceForComboItem(boolean value) {
		this.addProperty("base_and_group.priced.comboItem", String.valueOf(value)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isBaseAndGroupPriceForComboItem() {
		return POSUtil.getBoolean(getProperty("base_and_group.priced.comboItem")); //$NON-NLS-1$
	}

	public void setHighestPricedComboItem(boolean highestPricedComboItem) {
		this.addProperty("highest.priced.comboItem", String.valueOf(highestPricedComboItem)); //$NON-NLS-1$
	}

	@XmlTransient
	public boolean isHighestPricedComboItem() {
		return POSUtil.getBoolean(getProperty("highest.priced.comboItem")); //$NON-NLS-1$
	}

	public boolean hasModifierInSpec(String itemModifierSpecId, MenuModifier itemMenuModifier) {
		List<MenuItemModifierSpec> menuItemModiferSpecs = getMenuItemModiferSpecs();
		if (menuItemModiferSpecs != null) {
			for (MenuItemModifierSpec menuItemModifierSpec : menuItemModiferSpecs) {
				if (menuItemModifierSpec.getId().equals(itemModifierSpecId)) {
					Set<MenuModifier> modifiers = menuItemModifierSpec.getModifiers();
					if (modifiers.contains(itemMenuModifier)) {
						return true;
					}
				}
			}
		}
		return false;
	}

	@JsonIgnore
	@XmlTransient
	public String getAttributeNames() {
		List<Attribute> attributes = getAttributes();
		String names = ""; //$NON-NLS-1$
		if (attributes != null) {
			for (Iterator iterator = attributes.iterator(); iterator.hasNext();) {
				Attribute attribute = (Attribute) iterator.next();
				names += attribute.getName();
				if (iterator.hasNext()) {
					names += " / "; //$NON-NLS-1$
				}
			}
		}
		return names;
	}

	public void setAttributeNames() {
	}

	@Override
	public String getImageId() {
		if (!isVariant()) {
			return super.getImageId();
		}
		if (StringUtils.isBlank(super.getImageId()) && getParentMenuItem() != null) {
			return getParentMenuItem().getImageId();
		}
		return super.getImageId();
	}

	public void setUnitWiseSelection(boolean unitSelection) {
		addProperty(AppConstants.ALLOW_UNIT_WISE_SELECTION, String.valueOf(unitSelection));
	}

	public boolean isAllowUnitWiseSelection() {
		return POSUtil.getBoolean(getProperty(AppConstants.ALLOW_UNIT_WISE_SELECTION), Boolean.FALSE);
	}

	public void setCustomerRequired(boolean customerRequired) {
		addProperty("service.customerRequired", String.valueOf(customerRequired)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isCustomerRequired() {
		return getBooleanProperty("service.customerRequired", false); //$NON-NLS-1$
	}

	public void setPaymentType(ServicePaymentType paymentType) {
		addProperty("service.paymentType", paymentType == null ? null : paymentType.name()); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public ServicePaymentType getPaymentType() {
		String paymentType = getProperty("service.paymentType"); //$NON-NLS-1$
		if (StringUtils.isBlank(paymentType)) {
			return null;
		}
		return ServicePaymentType.valueOf(paymentType);
	}

	@XmlTransient
	@JsonIgnore
	public String getAvailableQtyDisplay() {
		return NumberUtil.format(getAvailableUnit());
	}

	@XmlTransient
	@JsonIgnore
	public String getOnHandQtyDisplay() {
		String unitName = getUnitName();
		if (StringUtils.isNotBlank(unitName)) {
			unitName = " " + unitName; //$NON-NLS-1$
		}
		else {
			unitName = ""; //$NON-NLS-1$
		}
		return NumberUtil.format(getUnitOnHand()) + unitName;
	}

	public void setOnHandQtyDisplay(String s) {
	}

	@XmlTransient
	@JsonIgnore
	public Integer getVariantSortOrder() {
		String sortOrderString = getProperty("variantSortOrder"); //$NON-NLS-1$
		if (StringUtils.isBlank(sortOrderString)) {
			return null;
		}
		return POSUtil.parseInteger(sortOrderString);
	}

	public void setVariantSortOrder(Integer sortOrder) {
		addProperty("variantSortOrder", String.valueOf(sortOrder)); //$NON-NLS-1$
	}

	public Boolean isVariantVisible() {
		if (isVariant()) {
			MenuItem parentMenuItem = getParentMenuItem();
			if (parentMenuItem != null && !parentMenuItem.isVisible()) {
				return false;
			}
		}
		return super.isVisible();
	}

	public double calculateBaseUnitCost(IUnit originalUnit) {
		if (originalUnit == null) {
			return getCost();
		}

		double baseUnitQuantity = getBaseUnitQuantity(originalUnit.getId());
		return baseUnitQuantity * getCost();
	}

	@JsonIgnore
	public double getItemCost() {
		double cost = getCost();
		if (isVariant()) {
			if (cost == 0) {
				cost = getParentMenuItem().getCost();
			}
		}

		if (isComboItem()) {
			cost = 0;
			List<ComboItem> comboItems = getComboItems();
			for (Iterator<ComboItem> iterator = comboItems.iterator(); iterator.hasNext();) {
				ComboItem comboItem = (ComboItem) iterator.next();
				cost += comboItem.getCost();
			}
		}

		return cost;
	}

	@XmlTransient
	@JsonIgnore
	public String getUrgentPreparationTime() {
		String estDeliveryTime = getProperty("est.urgent.preparation.time"); //$NON-NLS-1$
		if (StringUtils.isBlank(estDeliveryTime)) {
			return ""; //$NON-NLS-1$
		}
		return estDeliveryTime;
	}

	/**
	 * * This method is used to put Urgent Preparation Date as string (HH:mm) format.
	 * 
	 * @param String estDeliveryTime (HH:mm) format
	 * 
	 */
	public void putUrgentPreparationTime(String estPreparationTime) {
		addProperty("est.urgent.preparation.time", estPreparationTime); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getPreparationTime() {
		String estDeliveryTime = getProperty("est.preparation.time"); //$NON-NLS-1$
		if (StringUtils.isBlank(estDeliveryTime)) {
			return ""; //$NON-NLS-1$
		}
		return estDeliveryTime;
	}

	/**
	 * * This method is used to put Preparation Date as string (HH:mm) format.
	 * 
	 * @param String estDeliveryTime (HH:mm) format
	 * 
	 */
	public void putPreparationTime(String estPreparationTime) {
		addProperty("est.preparation.time", estPreparationTime); //$NON-NLS-1$
	}

	public void setLabDoctorRequired(boolean labDoctorRequired) {
		addProperty("lab.doctor.required", String.valueOf(labDoctorRequired)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isLabDoctorRequired() {
		return getBooleanProperty("lab.doctor.required", false); //$NON-NLS-1$
	}

	public void putLabDoctorFee(double labDoctorFee) {
		addProperty("lab.doctor.fee", String.valueOf(labDoctorFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isPriceIncludesDoctorFee() {
		//return getBooleanProperty("lab.price_includes_doctor_fee", true); //$NON-NLS-1$
		return true; //$NON-NLS-1$
	}

	public void setPriceIncludesDoctorFee(boolean priceIncludesDoctorFee) {
		//addProperty("lab.price_includes_doctor_fee", String.valueOf(priceIncludesDoctorFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isAllowDrFeeBeforeLabwork() {
		return getBooleanProperty("allow.dr.fee.before.labwork", false); //$NON-NLS-1$
	}

	public void setAllowDrFeeBeforeLabwork(boolean allowDrFeeBeforeLabwork) {
		addProperty("allow.dr.fee.before.labwork", String.valueOf(allowDrFeeBeforeLabwork)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getLabDoctorFee() {
		return NumberUtil.parseOrGetZero(getProperty("lab.doctor.fee")).doubleValue(); //$NON-NLS-1$
	}

	public void putUrgentPrice(double urgentPrice) {
		addProperty("urgent.price", String.valueOf(urgentPrice)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getUrgentPrice() {
		return NumberUtil.parseOrGetZero(getProperty("urgent.price")).doubleValue(); //$NON-NLS-1$
	}

	public void putReportType(String reportType) {
		addProperty("report.type", reportType); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getReportType() {
		String property = getProperty("report.type");
		if (StringUtils.isBlank(property)) {
			return ReportType.TABULAR_REPORT.name();
		}
		return property;
	}

	public String getReportTypeDisplay() {
		ReportType reportType = ReportType.fromString(getReportType());
		if (reportType == ReportType.TABULAR_REPORT) {
			return "3 CT";
		}
		else if (reportType == ReportType.POSITIVE_NEGATIVE) {
			return "2 CT";
		}
		else if (reportType == ReportType.PLAIN_REPORT_SIDE_BY_SIDE) {
			return "PSS";
		}
		return reportType.toString();
	}

	public void putReportDescription(String reportDescription) {
		addProperty("report.description", reportDescription); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getReportDescription() {
		String property = getProperty("report.description");
		if (StringUtils.isBlank(property)) {
			return "";
		}
		return property;
	}

	public void putPlainReportResultTemplate(String plainReportResult) {
		addProperty("plain_report.result_template", plainReportResult); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getPlainReportResultTemplate() {
		String property = getProperty("plain_report.result_template");
		if (StringUtils.isBlank(property)) {
			List<TestItem> testItems = getTestItems();
			if (testItems != null && !testItems.isEmpty()) {
				TestItem testItem = testItems.get(0);
				if (testItem != null) {
					return testItem.getResult();
				}
			}
			return "";
		}
		ReportType reportType = ReportType.fromString(getReportType());
		if (ReportType.PLAIN_REPORT == reportType) {
			return property;
		}
		return "";
	}

	public void putBasePrice(double basePrice) {
		addProperty("base.price", String.valueOf(basePrice)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getBasePrice() {
		return NumberUtil.parseOrGetZero(getProperty("base.price")).doubleValue(); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getB2bPrice() {
		return NumberUtil.parseOrGetZero(getProperty("b2b.price")).doubleValue(); //$NON-NLS-1$
	}

	public void putB2bPrice(double b2bPrice) {
		addProperty("b2b.price", String.valueOf(b2bPrice)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<RecepieItem> getAdditionalItems() {
		List<RecepieItem> additionalItems = new ArrayList<>();
		JSONObject jsonProperties = getProperties();
		if (jsonProperties == null || !jsonProperties.has("additinalItems")) { //$NON-NLS-1$
			return additionalItems;
		}
		JSONArray jsonArray = jsonProperties.getJSONArray("additinalItems"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.length(); i++) {
			JSONObject jsonObject = jsonArray.getJSONObject(i);
			RecepieItem additionalItem = new RecepieItem();
			String itemId = OrgJsonUtil.getString(jsonObject, RecepieItem.PROP_INVENTORY_ITEM);
			if (StringUtils.isBlank(itemId)) {
				continue;
			}
			MenuItem inventoryItem = MenuItemDAO.getInstance().get(itemId);
			if (inventoryItem == null) {
				continue;
			}
			additionalItem.setInventoryItem(inventoryItem);
			additionalItem.setQuantity(OrgJsonUtil.getDouble(jsonObject, RecepieItem.PROP_QUANTITY));
			additionalItem.setUnitCode(OrgJsonUtil.getString(jsonObject, RecepieItem.PROP_UNIT_CODE));
			additionalItems.add(additionalItem);
		}
		return additionalItems;
	}

	public void setAdditionalItems(List<RecepieItem> additionalItems) {
		if (additionalItems == null) {
			additionalItems = new ArrayList<>();
		}
		JSONArray additionalItemsArray = new JSONArray();
		for (RecepieItem recepieItem : additionalItems) {
			JSONObject jsonObject = new JSONObject();
			jsonObject.put(RecepieItem.PROP_INVENTORY_ITEM, recepieItem.getInventoryItem().getId());
			jsonObject.put(RecepieItem.PROP_QUANTITY, recepieItem.getQuantity());
			jsonObject.put(RecepieItem.PROP_UNIT_CODE, recepieItem.getUnitCode());
			additionalItemsArray.put(jsonObject);
		}
		buildPropertis();
		getProperties().put("additinalItems", additionalItemsArray); //$NON-NLS-1$
		setPropertiesJson(properties.toString());
	}

	@XmlTransient
	@JsonIgnore
	public String getLabTestRoomNo() {
		String property = getProperty("lab.test.room.no"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return ""; //$NON-NLS-1$
		}
		return property;
	}

	public void putLabTestRoomNo(String roomNo) {
		addProperty("lab.test.room.no", roomNo); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getLabTestMethod() {
		String property = getProperty("method"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return ""; //$NON-NLS-1$
		}
		return property;
	}

	public void putLabTestMethod(String labTestMethod) {
		addProperty("method", labTestMethod); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getDeliveryTimeType() {
		String property = getProperty("delivery_time_type"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return TimeType.DAY.name();
		}
		return property;
	}

	public void putDeliveryTimeType(String deliveryTimeType) {
		addProperty("delivery_time_type", deliveryTimeType); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getDeliveryTime() {
		String property = getProperty("delivery_time"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return "1"; //$NON-NLS-1$
		}
		return property;
	}

	public void putDeliveryTime(String deliveryTime) {
		addProperty("delivery_time", deliveryTime); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getUrgentDeliveryTimeType() {
		String property = getProperty("urgent_delivery_time_type"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return TimeType.HOUR.name();
		}
		return property;
	}

	public void putUrgentDeliveryTimeType(String urgentDeliveryTimeType) {
		addProperty("urgent_delivery_time_type", urgentDeliveryTimeType); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getUrgentDeliveryTime() {
		String property = getProperty("urgent_delivery_time"); //$NON-NLS-1$
		if (StringUtils.isBlank(property)) {
			return "2"; //$NON-NLS-1$
		}
		return property;
	}

	public void putUrgentDeliveryTime(String urgentDeliveryTime) {
		addProperty("urgent_delivery_time", urgentDeliveryTime); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<ReportTemplate> getReportTemplates() {
		List<ReportTemplate> reportTemplates = new ArrayList<>();
		JSONObject jsonProperties = getProperties();
		if (jsonProperties == null || !jsonProperties.has("report_templates")) { //$NON-NLS-1$
			return reportTemplates;
		}
		JSONArray jsonArray = jsonProperties.getJSONArray("report_templates"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.length(); i++) {
			JSONObject jsonObject = jsonArray.getJSONObject(i);
			ReportTemplate reportTemplate = new ReportTemplate();
			reportTemplate.setName(OrgJsonUtil.getString(jsonObject, ReportTemplate.JSON_PROP_NAME));
			reportTemplate.setValue(OrgJsonUtil.getString(jsonObject, ReportTemplate.JSON_PROP_VALUE));
			reportTemplate.setDefault(OrgJsonUtil.getBoolean(jsonObject, ReportTemplate.JSON_PROP_DEFAULT));
			reportTemplates.add(reportTemplate);
		}
		return reportTemplates;
	}

	public void setReportTemplates(List<ReportTemplate> reportTemplates) {
		if (reportTemplates == null) {
			reportTemplates = new ArrayList<>();
		}
		JSONArray itemsArray = new JSONArray();
		for (ReportTemplate repotTemplate : reportTemplates) {
			JSONObject jsonObject = new JSONObject();
			jsonObject.put(ReportTemplate.JSON_PROP_NAME, repotTemplate.getName());
			jsonObject.put(ReportTemplate.JSON_PROP_VALUE, repotTemplate.getValue());
			jsonObject.put(ReportTemplate.JSON_PROP_DEFAULT, repotTemplate.isDefault());
			itemsArray.put(jsonObject);
		}
		buildPropertis();
		getProperties().put("report_templates", itemsArray); //$NON-NLS-1$
		setPropertiesJson(properties.toString());
	}

	public void addOrUpdateReportTemplate(ReportTemplate reportTemplate) {
		boolean reportTemplateExists = false;
		List<ReportTemplate> reportTemplates = getReportTemplates();
		if (reportTemplate.getName() != null && reportTemplates != null && reportTemplates.size() > 0) {
			for (ReportTemplate rt : reportTemplates) {
				if (rt.getName().equalsIgnoreCase(reportTemplate.getName())) {
					rt.setValue(reportTemplate.getValue());
					reportTemplateExists = true;
					break;
				}
			}
		}
		if (!reportTemplateExists) {
			if (reportTemplates == null) {
				reportTemplates = new ArrayList<>();
			}
			int nextIndex = reportTemplates.size();
			for (ReportTemplate rt : reportTemplates) {
				if (rt.getName() == null) {
					continue;
				}
				int index = POSUtil.parseInteger(rt.getName().replaceAll("[^0-9\\.]", "")); //$NON-NLS-1$ //$NON-NLS-2$
				if (index > nextIndex) {
					nextIndex = index;
				}
			}
			nextIndex += 1;
			reportTemplate.setName("Template " + nextIndex); //$NON-NLS-1$
			reportTemplates.add(reportTemplate);
		}
		setReportTemplates(reportTemplates);
	}

	@XmlTransient
	@JsonIgnore
	public ReportTemplate getDefaultReportTemplate() {
		List<ReportTemplate> reportTemplates = getReportTemplates();
		if (reportTemplates != null && reportTemplates.size() > 0) {
			for (ReportTemplate reportTemplate : reportTemplates) {
				if (reportTemplate.isDefault()) {
					return reportTemplate;
				}
			}
		}
		return null;
	}

	public void setDefaultReportTemplate(ReportTemplate defaultReportTemplate) {
		if (defaultReportTemplate == null) {
			return;
		}
		List<ReportTemplate> reportTemplates = getReportTemplates();
		if (reportTemplates != null && reportTemplates.size() > 0) {
			for (ReportTemplate reportTemplate : reportTemplates) {
				if (defaultReportTemplate.getName().equalsIgnoreCase(reportTemplate.getName())) {
					reportTemplate.setDefault(true);
				}
				else if (reportTemplate.isDefault()) {
					reportTemplate.setDefault(false);
				}
			}
		}
		setReportTemplates(reportTemplates);
	}

	public String getContainerId() {
		return getProperty("container_id"); //$NON-NLS-1$
	}

	public void putContainerId(String containerId) {
		addProperty("container_id", containerId); //$NON-NLS-1$
	}

	public String getSpecimenDisplay() {
		String id = getSpecimenId();
		if (StringUtils.isNotBlank(id)) {
			Specimen specimen = (Specimen) DataProvider.get().getObjectOf(Specimen.class, id);
			if (specimen != null) {
				return specimen.getName();
			}
		}
		return null;
	}

	public String getContainerDisplay() {
		String id = getContainerId();
		if (StringUtils.isNotBlank(id)) {
			Tag container = (Tag) DataProvider.get().getObjectOf(Tag.class, id);
			if (container != null) {
				return container.getName();
			}
		}
		return null;
	}

	public void putFoldType(String foldType) {
		addProperty("fold.type", foldType); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public FoldType getFoldType() {
		String property = getProperty("fold.type");
		if (StringUtils.isBlank(property)) {
			return FoldType.ONE_FOLD;
		}
		return FoldType.fromString(property);
	}

	@XmlTransient
	@JsonIgnore
	public boolean IsShowColumHeader() {
		return getBooleanProperty("show.colum.header", true); //$NON-NLS-1$
	}

	public void putShowColumHeader(boolean isShowColumHeader) {
		addProperty("show.colum.header", String.valueOf(isShowColumHeader)); //$NON-NLS-1$
	}

	public String getHeaderNote() {
		String property = getProperty("header_note");
		if (property == null) {
			property = StringUtils.EMPTY;
		}
		return property; //$NON-NLS-1$
	}

	public void putHeaderNote(String headerNote) {
		addProperty("header_note", headerNote); //$NON-NLS-1$
	}

	public String getMachineName() {
		String property = getProperty("machine_name");
		if (property == null) {
			property = StringUtils.EMPTY;
		}
		return property; //$NON-NLS-1$
	}

	public void putMachineName(String machineName) {
		addProperty("machine_name", machineName); //$NON-NLS-1$
	}

	public String getStrength() {
		return getProperty("strength"); //$NON-NLS-1$
	}

	public void putStrength(String strength) {
		addProperty("strength", strength); //$NON-NLS-1$
	}

	public String getPacksize() {
		return getProperty("packsize"); //$NON-NLS-1$
	}

	public void putPacksize(String packsize) {
		addProperty("packsize", packsize); //$NON-NLS-1$
	}

	public Manufacturer getManufacture() {
		String manufactureId = getManufactureId();
		if (StringUtils.isNotBlank(manufactureId) && manufacture == null) {
			manufacture = (Manufacturer) DataProvider.get().getObjectOf(Manufacturer.class, manufactureId);
		}
		return manufacture;
	}

	public MedGenerics getMedGenerics() {
		String medGenericId = getMedGenericId();
		if (StringUtils.isNotBlank(medGenericId) && medGenerics == null) {
			medGenerics = (MedGenerics) DataProvider.get().getObjectOf(MedGenerics.class, medGenericId);
		}
		return medGenerics;
	}

	public DosageForms getDosageForms() {
		String dosageFormsId = getDosageFormsId();
		if (StringUtils.isNotBlank(dosageFormsId) && dosageForms == null) {
			dosageForms = (DosageForms) DataProvider.get().getObjectOf(DosageForms.class, dosageFormsId);
		}
		return dosageForms;
	}

	public String getManufacturerName() {
		return MenuItemService.getManufacturerName(this);
	}

	public String getMedGenericName() {
		return MenuItemService.getMedGenericsName(this);
	}

	public String getDosageFormName() {
		return MenuItemService.getDosageFormsName(this);
	}

	public double getMrpPrice() {
		return NumberUtil.parseOrGetZero(getProperty("mrp_price")).doubleValue(); //$NON-NLS-1$
	}

	public void putMrpPrice(double mrpPrice) {
		addProperty("mrp_price", String.valueOf(mrpPrice)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<String> getQuantityUnits() {
		List<String> quantityUnits = new ArrayList<>();
		JSONObject jsonProperties = getProperties();
		if (jsonProperties == null || !jsonProperties.has("quantityUnits")) { //$NON-NLS-1$
			return quantityUnits;
		}
		JSONArray jsonArray = jsonProperties.getJSONArray("quantityUnits"); //$NON-NLS-1$
		for (int i = 0; i < jsonArray.length(); i++) {
			Object value = jsonArray.get(i);
			if (value == null) {
				continue;
			}
			quantityUnits.add(value.toString());
		}
		return quantityUnits;
	}

	public void setQuantityUnits(List<String> quantityUnits) {
		if (quantityUnits == null) {
			quantityUnits = new ArrayList<>();
		}
		JSONArray jsonArray = new JSONArray();
		for (Iterator iterator = quantityUnits.iterator(); iterator.hasNext();) {
			String value = (String) iterator.next();
			jsonArray.put(value);
		}
		buildPropertis();
		getProperties().put("quantityUnits", jsonArray); //$NON-NLS-1$
		setPropertiesJson(properties.toString());
	}

	public boolean addOrUpdateQuantityUnit(String addedQuantityUnit) {
		if (StringUtils.isBlank(addedQuantityUnit)) {
			return false;
		}
		List<String> quantityUnits = getQuantityUnits();
		if (addedQuantityUnit != null && quantityUnits != null && quantityUnits.size() > 0) {
			for (String qv : quantityUnits) {
				if (qv.equalsIgnoreCase(addedQuantityUnit)) {
					return false;
				}
			}
		}
		quantityUnits.add(addedQuantityUnit);
		setQuantityUnits(quantityUnits);
		return true;
	}

	public void putRfOnReportType(String chargeType) {
		addProperty("rf.type_on_report", chargeType); //$NON-NLS-1$
	}

	public String getRfOnReportType() {
		String rfType = getProperty("rf.type_on_report");
		if (StringUtils.isNotBlank(rfType)) {
			return rfType;
		}
		return ReferralCommissionType.GROUP.name();
	}

	public void putRfRateOnReport(String commissionValue) {
		addProperty("rf.rate_on_report", commissionValue); //$NON-NLS-1$
	}

	public String getRfRateOnReport() {
		String rfRate = getProperty("rf.rate_on_report");
		if (StringUtils.isNotBlank(rfRate)) {
			return rfRate;
		}
		return "";
	}

	public void setManufacturer(Manufacturer manufacture) {
		if (manufacture != null) {
			setManufactureId(manufacture.getId());
			addProperty(MANUFACTURER_NAME, manufacture.getName());
		}
		else {
			setManufactureId(null);
			removeProperty(MANUFACTURER_NAME);
		}
	}

	public void setDosageForms(DosageForms dosageForms) {
		if (dosageForms != null) {
			setDosageFormsId(dosageForms.getId());
			addProperty(DOSAGE_FORMS_NAME, dosageForms.getName());
		}
		else {
			setDosageFormsId(null);
			removeProperty(DOSAGE_FORMS_NAME);
		}
	}

	public void setMedGenerics(MedGenerics medGenerics) {
		if (medGenerics != null) {
			setMedGenericId(medGenerics.getId());
			addProperty(MED_GENERICS_NAME, medGenerics.getName());
		}
		else {
			setMedGenericId(null);
			removeProperty(MED_GENERICS_NAME);
		}
	}

	public void putWarrantyDuration(int duration) {
		addProperty("warranty.duration", String.valueOf(duration)); //$NON-NLS-1$
	}

	public Integer getWarrantyDuration() {
		String duration = getProperty("warranty.duration");
		if (StringUtils.isNotBlank(duration)) {
			return NumberUtil.parseOrGetZero(duration).intValue();
		}
		return null;
	}

	public void putWarrantyDurationType(String warrantyDurationType) {
		addProperty("warranty.duration.type", warrantyDurationType); //$NON-NLS-1$
	}

	public String getWarrantyDurationType() {
		String durationType = getProperty("warranty.duration.type");
		if (StringUtils.isNotBlank(durationType)) {
			return durationType;
		}
		return "";
	}

	public String getWarrantyDurationDisplay() {
		Integer warrantyDuration = getWarrantyDuration();
		if (warrantyDuration == null) {
			return null;
		}
		String warrantyDurationType = getWarrantyDurationType();
		DurationType durationType = DurationType.fromString(warrantyDurationType);
		return warrantyDuration + " " + durationType.toString();
	}

	/**
	 * Used in Excel Data import only
	 */
	public double getStockQty() {
		return stockQty;
	}

	/**
	 * Used in Excel Data import only
	 */
	public void setStockQty(double stockQty) {
		this.stockQty = stockQty;
	}

	public void putPatientRequired(boolean patientRequired) {
		addProperty("patient_required", String.valueOf(patientRequired)); //$NON-NLS-1$
	}

	public boolean isPatientRequired() {
		return getBooleanProperty("patient_required", false); //$NON-NLS-1$
	}

	public void putDoctorFeeRate(double doctorFeeRate) {
		addProperty("dr.fee_rate", String.valueOf(doctorFeeRate)); //$NON-NLS-1$
	}

	public double getDoctorFeeRate() {
		String property = getProperty("dr.fee_rate");
		if (StringUtils.isBlank(property)) {
			return 0d;
		}
		return PropertyContainer.parseDouble(getProperty("dr.fee_rate")); //$NON-NLS-1$
	}

	public void putDoctorFeeType(String feeType) {
		addProperty("dr.fee_type", feeType); //$NON-NLS-1$
	}

	public String getDoctorFeeType() {
		String feeType = getProperty("dr.fee_type"); //$NON-NLS-1$
		if (StringUtils.isBlank(feeType)) {
			return "";
		}
		return feeType;
	}

	public void calculateDoctorFee() {
		double doctorFeeAmount = 0d;

		String doctorFeeType = getDoctorFeeType();
		double doctorFeeRate = getDoctorFeeRate();
		if (ServiceChargeType.FIXEDAMOUNT == ServiceChargeType.fromName(doctorFeeType)) {
			doctorFeeAmount = doctorFeeRate;
		}
		else if (ServiceChargeType.PERCENTAGE == ServiceChargeType.fromName(doctorFeeType)) {
			doctorFeeAmount = NumberUtil.round(getPrice() * (doctorFeeRate / 100));
		}

		putLabDoctorFee(doctorFeeAmount);
	}

	public String getAnesthesiaServiceType() {
		String menuGroupId = getMenuGroupId();
		if (ServiceType.ANESTHESIA == ServiceType.fromDisplayName(getServiceType()) && StringUtils.isNotBlank(menuGroupId)) {
			ServiceType service = ServiceType.valueOf(menuGroupId);
			if (service != null) {
				return service.getDisplayName();
			}
		}
		return ServiceType.ANESTHESIA_GENERAL.getDisplayName();
	}

	public void setAnesthesiaServiceType(String serviceType) {
	}

	public void putSampleCollectionNeeded(boolean value) {
		addProperty("sample_collection_needed", String.valueOf(value)); //$NON-NLS-1$
	}

	public boolean isSampleCollectionNeeded() {
		return getBooleanProperty("sample_collection_needed", true); //$NON-NLS-1$
	}

}