package com.floreantpos.model.dao;

import java.util.Date;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;

import com.floreantpos.model.OutdoorTicket;
import com.floreantpos.model.Patient;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.ext.OutdoorTicketStatus;
import com.floreantpos.swing.PaginatedListModel;

public class OutdoorTicketDAO extends BaseOutdoorTicketDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public OutdoorTicketDAO() {
	}

	public OutdoorTicket getOutdoorTicket(String id, String outletId) {

		try (Session session = createNewSession()) {
			return getOutdoorTicket(id, outletId, session);
		}

	}

	public OutdoorTicket getOutdoorTicket(String id, String outletId, Session session) {
		Criteria criteria = session.createCriteria(OutdoorTicket.class);

		addDeletedFilter(criteria);

		criteria.add(Restrictions.eq(OutdoorTicket.PROP_ID, id));
		criteria.add(Restrictions.eq(OutdoorTicket.PROP_OUTLET_ID, outletId));

		criteria.addOrder(Order.asc(OutdoorTicket.PROP_CREATE_DATE));
		criteria.setMaxResults(1);

		return (OutdoorTicket) criteria.uniqueResult();
	}

	public void loadOutdoorTickets(PaginatedListModel<OutdoorTicket> dataModel, String orderId, String patientId, String doctorId, String outletId,
			Date startOfDate, Date endOfDate, String ticketOrderStatus) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.like(OutdoorTicket.PROP_ID, orderId, MatchMode.ANYWHERE));
			}

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_CUSTOMER_ID, patientId));
			}

			if (StringUtils.isNotBlank(doctorId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_DOCTOR_ID, doctorId));
			}

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_OUTLET_ID, outletId));
			}

			if (startOfDate != null && endOfDate != null) {
				criteria.add(Restrictions.between(OutdoorTicket.PROP_APPOINTMENT_DATE, startOfDate, endOfDate));
			}

			if (OutdoorTicketStatus.CLOSED.name().equals(ticketOrderStatus)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_STATUS, OutdoorTicketStatus.CLOSED.name()));
			}
			else {
				criteria.add(Restrictions.ne(OutdoorTicket.PROP_STATUS, OutdoorTicketStatus.CLOSED.name()));
			}

			SimpleExpression isNotVoided = Restrictions.ne(OutdoorTicket.PROP_STATUS, OutdoorTicketStatus.VOIDED.name());
			Criterion isNullStatus = Restrictions.isNull(OutdoorTicket.PROP_STATUS);
			criteria.add(Restrictions.or(isNotVoided, isNullStatus));

			dataModel.setNumRows(rowCount(criteria));

			criteria.addOrder(Order.desc(OutdoorTicket.PROP_TOKEN_NO));
			criteria.addOrder(Order.desc(OutdoorTicket.PROP_APPOINTMENT_DATE));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			dataModel.setData(criteria.list());
		}
	}

	public void saveWithPatient(OutdoorTicket outdoorTicket) {
		try (Session session = createNewSession()) {

			Transaction tx = session.beginTransaction();

			if (StringUtils.isBlank(outdoorTicket.getCustomerId())) {
				Patient patient = new Patient();

				patient.setFirstName(outdoorTicket.getCustomerName());
				patient.setName(outdoorTicket.getCustomerName());
				patient.setDateOfBirth(outdoorTicket.getDateOfBirth());
				patient.setMobileNo(outdoorTicket.getCustomerPhone());
				patient.putPatientGender(outdoorTicket.getCustomerGender());
				patient.setOutletId(outdoorTicket.getOutletId());

				PatientDAO.getInstance().save(patient, session);

				outdoorTicket.setCustomerId(patient.getId());
			}

			saveOrUpdate(outdoorTicket, session);

			tx.commit();

		}
	}

	public Ticket saveWithOutdoorOrder(OutdoorTicket outdoorTicket, Ticket ticket) {
		try (Session session = createNewSession()) {

			Transaction tx = session.beginTransaction();

			saveOrUpdate(outdoorTicket, session);

			String source = ticket.getSource();
			if (StringUtils.isBlank(source)) {
				ticket.setSource(outdoorTicket.getSource());
			}

			String outdoorTicketId = outdoorTicket.getId();
			ticket.setOutdoorTicketId(outdoorTicketId);
			if (StringUtils.isBlank(ticket.getId())) {
				ticket.setId(outdoorTicketId);
				TicketDAO.getInstance().save(ticket, session);
			}
			else {
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}

			tx.commit();

			return ticket;
		}
	}

	public OutdoorTicket findByTransactionId(String transactionId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(OutdoorTicket.PROP_TRANSACTION_ID, transactionId));

			criteria.addOrder(Order.asc(OutdoorTicket.PROP_CREATE_DATE));
			criteria.setMaxResults(1);

			return (OutdoorTicket) criteria.uniqueResult();
		}
	}

	//	public boolean isRevisitFeeApplicable(String customerId, String doctorId, String excludedCutrentId, int inDays) {
	//		OutdoorTicket previousOutdoorVisit = getPreviousOutdoorVisit(customerId, doctorId, excludedCutrentId, inDays);
	//		return previousOutdoorVisit != null;
	//	}

	public OutdoorTicket getPreviousOutdoorVisit(String customerId, String doctorId, String excludedCutrentId, String outletId) {
		if (StringUtils.isBlank(customerId)) {
			return null;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(OutdoorTicket.class);
			addDeletedFilter(criteria);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_OUTLET_ID, outletId));
			}

			if (StringUtils.isNotBlank(excludedCutrentId)) {
				criteria.add(Restrictions.ne(OutdoorTicket.PROP_ID, excludedCutrentId));
			}

			criteria.add(Restrictions.eq(OutdoorTicket.PROP_STATUS, OutdoorTicketStatus.PAID.name()));
			criteria.add(Restrictions.eq(OutdoorTicket.PROP_CUSTOMER_ID, customerId));

			if (StringUtils.isNotBlank(doctorId)) {
				criteria.add(Restrictions.eq(OutdoorTicket.PROP_DOCTOR_ID, doctorId));
			}

			criteria.addOrder(Order.desc(OutdoorTicket.PROP_APPOINTMENT_DATE));
			criteria.setMaxResults(1);

			return (OutdoorTicket) criteria.uniqueResult();
		}
	}
}