package pl.allegro.finance.tradukisto.internal.converters;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;

import java.math.BigDecimal;

import com.floreantpos.util.NumberUtil;

import pl.allegro.finance.tradukisto.internal.BigDecimalToStringConverter;
import pl.allegro.finance.tradukisto.internal.IntegerToStringConverter;

public class BigDecimalToBankingMoneyConverter implements BigDecimalToStringConverter {
	private static final String CURRENCY_SYMBOL = "Taka Only";
	private static final String SUB_UNIT_CURRENCY_SYMBOL = "Poysa Only";

	private static final String FORMAT_DOUBLE_VALUE = "%s %s %s %s";
	private static final String FORMAT_INTEGER_VALUE = "%s %s";
	private static final int MAXIMAL_DECIMAL_PLACES_COUNT = 2;

	private final IntegerToStringConverter converter;
	private final String currencySymbol;

	public BigDecimalToBankingMoneyConverter(IntegerToStringConverter converter, String currencySymbol) {
		this.converter = converter;
		this.currencySymbol = currencySymbol;
	}

	@Override
	public String asWords(BigDecimal value) {
		return asWords(value, currencySymbol);
	}

	@Override
	public String asWords(BigDecimal value, String currencySymbol) {
		validate(value);

		Integer units = value.intValue();
		Integer subunits = value.remainder(BigDecimal.ONE).multiply(new BigDecimal(100)).intValue();

		if (NumberUtil.isZero(subunits)) {
			return format(FORMAT_INTEGER_VALUE, converter.asWords(units).replaceAll("-", " "), CURRENCY_SYMBOL);
		}
		return format(FORMAT_DOUBLE_VALUE, converter.asWords(units).replaceAll("-", " "), CURRENCY_SYMBOL.replace("Only", ""),
				converter.asWords(subunits).replaceAll("-", " "), SUB_UNIT_CURRENCY_SYMBOL);
	}

	private void validate(BigDecimal value) {
		checkArgument(value.scale() <= MAXIMAL_DECIMAL_PLACES_COUNT, "can't transform more than %s decimal places for value %s", MAXIMAL_DECIMAL_PLACES_COUNT,
				value);

		checkArgument(valueLessThatIntMax(value), "can't transform numbers greater than Integer.MAX_VALUE for value %s", value);

		checkArgument(valueGreaterThanOrEqualToZero(value), "can't transform negative numbers for value %s", value);
	}

	private boolean valueLessThatIntMax(BigDecimal value) {
		return value.compareTo(new BigDecimal(Integer.MAX_VALUE).add(BigDecimal.ONE)) == -1;
	}

	private boolean valueGreaterThanOrEqualToZero(BigDecimal value) {
		return value.signum() >= 0;
	}
}
