package com.floreantpos.model;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.floreantpos.model.base.BaseDoctorVisit;
import com.floreantpos.model.dao.BookingInfoDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.PatientDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.GsonUtil;
import com.google.gson.Gson;
import com.google.gson.JsonObject;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "booking", "patient", "propertiesContainer", "bed", "doctor" })
@XmlAccessorType(XmlAccessType.FIELD)
@XmlRootElement
public class DoctorVisit extends BaseDoctorVisit implements PropertyContainer2, IVisit {
	private static final long serialVersionUID = 1L;

	private transient Patient patient;
	private transient com.google.gson.JsonObject propertiesContainer;
	private transient Bed bed;
	private transient BookingInfo booking;
	private transient Doctor doctor;

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public DoctorVisit () {
    }

	/**
	 * Constructor for primary key
	 */
	public DoctorVisit (java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@XmlTransient
	@JsonIgnore
	public Patient getPatient() {
		if (patient != null) {
			return patient;
		}
		if (StringUtils.isNotBlank(super.getPatientId())) {
			patient = PatientDAO.getInstance().get(super.getPatientId());
		}
		return patient;
	}

	public void setPatient(Patient patient) {
		this.patient = patient;
		super.setPatientId(patient == null ? null : patient.getId());
		if (patient == null) {
			removeProperty("patientName"); //$NON-NLS-1$
			return;
		}
		addProperty("patientName", patient.getName()); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getPatientName() {
		if (StringUtils.isBlank(super.getPatientId())) {
			return StringUtils.EMPTY;
		}
		String patientName = getProperty("patientName"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(patientName)) {
			return patientName;
		}
		Patient patient = getPatient();
		if (patient != null) {
			return patient.getName();
		}
		return StringUtils.EMPTY;
	}

	public void addToInstructions(String instruction) {
		List<String> instructions = super.getInstructions();
		if (instructions == null) {
			instructions = new ArrayList<>();
		}
		for (String exInstruction : instructions) {
			if (exInstruction.equalsIgnoreCase(instruction)) {
				return;
			}
		}
		instructions.add(instruction);
		super.setInstructions(instructions);
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = null;
	}

	private void initPropertyContainer() {
		if (propertiesContainer == null) {
			if (StringUtils.isBlank(super.getProperties())) {
				propertiesContainer = new com.google.gson.JsonObject();
			}
			else {
				propertiesContainer = new Gson().fromJson(super.getProperties(), com.google.gson.JsonObject.class);
			}
		}
	}

	@Override
	public JsonObject getPropertyStore() {
		initPropertyContainer();
		return propertiesContainer;
	}

	@XmlTransient
	@JsonIgnore
	public Bed getBed() {
		return bed;
	}

	public void setBed(Bed bed) {
		this.bed = bed;
	}

	@XmlTransient
	@JsonIgnore
	public BookingInfo getBooking() {
		if (booking != null) {
			return booking;
		}
		if (StringUtils.isNotBlank(super.getAdmissionId())) {
			return booking = BookingInfoDAO.getInstance().findByBookingId(super.getAdmissionId());
		}
		return booking;
	}

	public void setBooking(BookingInfo booking) {
		this.booking = booking;
		super.setAdmissionId(booking == null ? null : booking.getBookingId());
	}

	@XmlTransient
	@JsonIgnore
	public String getTemparature() {
		return getProperty("temparature"); //$NON-NLS-1$
	}

	public void setTemparature(String temparature) {
		addProperty("temparature", temparature); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getPressureUp() {
		return getProperty("pressure.up"); //$NON-NLS-1$
	}

	public void setPressureUp(String pressureUp) {
		addProperty("pressure.up", pressureUp); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getPressureDown() {
		return getProperty("pressure.down"); //$NON-NLS-1$
	}

	public void setPressureDown(String pressureDown) {
		addProperty("pressure.down", pressureDown); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getHeartBitPerMinute() {
		return getProperty("heartBitPerMinute"); //$NON-NLS-1$
	}

	public void setHeartBitPerMinute(String heartBitPerMinutes) {
		addProperty("heartBitPerMinute", heartBitPerMinutes); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getFollowUpNotes() {
		return getProperty("notes.followup"); //$NON-NLS-1$
	}

	public void setFollowUpNotes(String followUpNotes) {
		addProperty("notes.followup", followUpNotes); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getNotes() {
		return getProperty("inspection.notes", StringUtils.EMPTY); //$NON-NLS-1$
	}

	public void putNotes(String notes) {
		addProperty("inspection.notes", notes); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public Doctor getDoctor() {
		if (doctor != null) {
			return doctor;
		}
		if (StringUtils.isNotBlank(super.getDoctorId())) {
			return doctor = DoctorDAO.getInstance().get(super.getDoctorId());
		}
		return doctor;
	}

	public void setDoctor(Doctor doctor) {
		this.doctor = doctor;
		super.setDoctorId(doctor == null ? null : doctor.getId());
		if (doctor == null) {
			removeProperty("doctorName"); //$NON-NLS-1$
		}
		else {
			addProperty("doctorName", doctor.getName()); //$NON-NLS-1$
		}
	}

	@XmlTransient
	@JsonIgnore
	public String getDoctorVisitMedicineItemsNameDisplay(boolean showInDetailView) {

		String medicineNames = ""; //$NON-NLS-1$
		Boolean showBanglaTextOnPrescription = DataProvider.get().getOutlet().isShowReportsInBangla();

		for (Iterator<DoctorVisitMedicineItem> iterator = getMedicineItems().iterator(); iterator.hasNext();) {
			DoctorVisitMedicineItem dvMedItem = (DoctorVisitMedicineItem) iterator.next();
			medicineNames += dvMedItem.getItemNameDisplayWithInstruction(showInDetailView, showBanglaTextOnPrescription);
		}
		return medicineNames;
	}

	@XmlTransient
	@JsonIgnore
	public List<String> getProblems() {
		return GsonUtil.convertJSONToList(getProperty("problems", "[]"), String.class);
	}

	public void putProblems(List<String> problems) {
		addProperty("problems", GsonUtil.createGson().toJson(problems));
	}

	public void putRoomName(String roomName) {
		if (roomName == null) {
			removeProperty("roomName"); //$NON-NLS-1$
			return;
		}
		addProperty("roomName", roomName); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getRoomName() {
		String roomName = getProperty("roomName"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(roomName)) {
			return roomName;
		}
		BookingInfo booking = getBooking();
		if (booking == null) {
			return StringUtils.EMPTY;
		}
		StringBuilder roomStringBuilder = new StringBuilder();
		List<Bed> beds = booking.getBeds();
		if (beds != null && beds.size() > 0) {
			for (Iterator<Bed> iterator = beds.iterator(); iterator.hasNext();) {
				Bed bed = (Bed) iterator.next();
				if (bed.getRoom() == null) {
					continue;
				}
				roomStringBuilder.append(bed.getRoom().getName());
				if (iterator.hasNext()) {
					roomStringBuilder.append(", "); //$NON-NLS-1$
				}
			}
		}
		return roomStringBuilder.toString();
	}

	public void putBedNumber(String bedNumber) {
		if (bedNumber == null) {
			removeProperty("bedNumber"); //$NON-NLS-1$
			return;
		}
		addProperty("bedNumber", bedNumber); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public String getBedNumber() {
		String bedNumber = getProperty("bedNumber"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(bedNumber)) {
			return bedNumber;
		}
		BookingInfo booking = getBooking();
		if (booking == null) {
			return StringUtils.EMPTY;
		}
		StringBuilder bedStringBuilder = new StringBuilder();
		List<Bed> beds = booking.getBeds();
		if (beds != null && beds.size() > 0) {
			for (Iterator<Bed> iterator = beds.iterator(); iterator.hasNext();) {
				Bed bed = (Bed) iterator.next();
				bedStringBuilder.append(bed.getNumber());
				if (iterator.hasNext()) {
					bedStringBuilder.append(", "); //$NON-NLS-1$
				}
			}
		}
		return bedStringBuilder.toString();
	}

	@XmlTransient
	@JsonIgnore
	public String getDoctorName() {
		if (StringUtils.isBlank(super.getDoctorId())) {
			return StringUtils.EMPTY;
		}
		String doctorName = getProperty("doctorName"); //$NON-NLS-1$
		if (StringUtils.isNotBlank(doctorName)) {
			return doctorName;
		}
		Doctor doctor = getDoctor();
		if (doctor != null) {
			return doctor.getName();
		}
		return StringUtils.EMPTY;
	}
}