package com.floreantpos.services.report;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.LogicalExpression;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;

import com.floreantpos.PosException;
import com.floreantpos.model.AdvanceAdjustmentTransaction;
import com.floreantpos.model.AdvanceRefundTransaction;
import com.floreantpos.model.AdvanceTransaction;
import com.floreantpos.model.BalanceSubType;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashDropTransaction;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.CustomerAccountTransaction;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.GiftCertificateTransaction;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.GratuityPaymentHistory;
import com.floreantpos.model.LdfPayTransaction;
import com.floreantpos.model.PayOutTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.PurchaseRefundTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.RfPayTransaction;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.dao.GratuityDAO;
import com.floreantpos.model.dao.TerminalDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;

public class CashDrawerReportService {
	public final static String HEADERLINE1 = "headerLine1"; //$NON-NLS-1$
	public final static String SSREPORTTITLE = "SSReportTitle"; //$NON-NLS-1$
	public final static String SALESBALANCE = "salesBalance"; //$NON-NLS-1$
	public final static String ROWGROSSSALES = "rowGrossSales"; //$NON-NLS-1$
	public final static String ROWNETSALES = "rowNetSales"; //$NON-NLS-1$
	public final static String TITLECATBREAKOUT = "TitleCatBreakOut"; //$NON-NLS-1$
	public final static String ROWDISCOUNT = "rowDiscount"; //$NON-NLS-1$
	public final static String ROWRETURN = "rowReturn"; //$NON-NLS-1$
	public final static String ROWSALESTAX = "rowSalesTax"; //$NON-NLS-1$
	public final static String ROWSC = "rowSC"; //$NON-NLS-1$
	public final static String ROWDC = "rowDC"; //$NON-NLS-1$
	public final static String ROWTOTAL = "rowTotal"; //$NON-NLS-1$
	public final static String ROWTIPS = "rowTips"; //$NON-NLS-1$
	public final static String ROWFEE = "rowFee"; //$NON-NLS-1$
	public final static String ROWGRECEIVABLE = "rowGReceivable"; //$NON-NLS-1$
	public final static String ROWCRECEIPTS = "rowCReceipts"; //$NON-NLS-1$
	public final static String ROWCCARDS = "rowCCards"; //$NON-NLS-1$
	public final static String ROWDCARDS = "rowDCards"; //$NON-NLS-1$
	public final static String ROWMPAYMENTS = "rowMPayments"; //$NON-NLS-1$
	public final static String ROWCPAYMENTS = "rowCPayments"; //$NON-NLS-1$
	public final static String ROWGCERT = "rowGCert"; //$NON-NLS-1$
	public final static String ROWREFUNDPLUS = "rowRefundPlus"; //$NON-NLS-1$
	public final static String ROWTOLERANCE = "rowTolerance"; //$NON-NLS-1$
	public final static String ROWRECEIPTDIFF = "rowReceiptDiff"; //$NON-NLS-1$
	public final static String ROWCASHTIPS = "rowCashTips"; //$NON-NLS-1$
	public final static String ROWCHARGEDTIPS = "rowChargedTips"; //$NON-NLS-1$
	public final static String USER = "user"; //$NON-NLS-1$
	public final static String DATE = "date"; //$NON-NLS-1$
	public final static String TOTALVOID = "totalVoid"; //$NON-NLS-1$
	public final static String DECLAREDTIPS = "declaredTips"; //$NON-NLS-1$

	public final static String REPORTTITLE = "reportTitle"; //$NON-NLS-1$
	public final static String ROWGROSSRECEIPTS = "rowGrossReceipts"; //$NON-NLS-1$
	public final static String ROWGRETURNS = "rowGReturns"; //$NON-NLS-1$
	public final static String ROWGCCHANGE = "rowGCChange"; //$NON-NLS-1$
	public final static String ROWTIPSPAID = "rowTipsPaid"; //$NON-NLS-1$
	public final static String ROWTIPSPAIDBYOTHERTERMINAL = "rowTipsPaidByOtherTerminal"; //$NON-NLS-1$
	public final static String ROWTIPSDIFF = "rowTipsDiff"; //$NON-NLS-1$
	public final static String CASHBALANCE = "cashBalance"; //$NON-NLS-1$
	public final static String ROWCASH = "rowCash"; //$NON-NLS-1$
	public final static String ROWPAYOUT = "rowPayOut"; //$NON-NLS-1$
	public final static String ROWREFUNDMINUS = "rowRefundMinus"; //$NON-NLS-1$
	public final static String ROWBEGINCASH = "rowBeginCash"; //$NON-NLS-1$
	public final static String ROWADDBALANCE = "rowAddBalance"; //$NON-NLS-1$
	public final static String ROWCASHIN = "rowCashIn"; //$NON-NLS-1$
	public final static String ROWDBLEED = "rowDBleed"; //$NON-NLS-1$
	public final static String ROWDACC = "rowDAcc"; //$NON-NLS-1$
	public final static String ROWDTIPS = "rowDTips"; //$NON-NLS-1$
	public final static String ROWCTODIPO = "rowCToDipo"; //$NON-NLS-1$
	public final static String ROWCBREAKD = "rowCBreakD"; //$NON-NLS-1$
	public final static String ROWCSRAMOUNT = "rowCSRAmount"; //$NON-NLS-1$
	public final static String ROWVREXCEPTIONS = "rowVRExceptions"; //$NON-NLS-1$
	public final static String ROWVRTAX = "rowVRTax"; //$NON-NLS-1$
	public final static String ROWVRAMOUNT = "rowVRAmount"; //$NON-NLS-1$
	public final static String ROWVRTOTAL = "rowVRTotal"; //$NON-NLS-1$
	public final static String STARTTIMENAME = "startTime"; //$NON-NLS-1$
	public final static String CLOSETIMENAME = "closeTime"; //$NON-NLS-1$
	public final static String REPORTUSER = "reportUser"; //$NON-NLS-1$
	public final static String ROWTOTALTIPS = "rowTotalTips"; //$NON-NLS-1$
	public final static String ROWCASHDUEOROWEDTOSERVER = "rowCashDueOrOwedToServer"; //$NON-NLS-1$
	public final static String LABELROUNDING = "lblRounding"; //$NON-NLS-1$
	public final static String ROUNDINGAMOUNT = "roundingAmount"; //$NON-NLS-1$

	private CashDrawer cashDrawer;
	// private Session session;

	public CashDrawerReportService(CashDrawer cashDrawer) {
		super();
		this.cashDrawer = cashDrawer;
	}

	public void populateReport() {
		try (Session session = TerminalDAO.getInstance().createNewSession()) {
			populateReport(session);
		}
	}

	public void populateSummaryOfCashdrawer() {
		try (Session session = TerminalDAO.getInstance().createNewSession()) {
			populateSummaryOfCashdrawer(session);
		}
	}

	public void populateSummaryOfCashdrawer(Session session) {
		calculateNetSales(session);
		calculateRefundAmount(session);
		calculateDrawerBleed(session);
		calculateCashTips(session);
		calculateChargedTips(session);

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() - cashDrawer.getDrawerBleedAmount() + cashDrawer.getSalesTax()
				+ cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue(POSUtil.getDoubleAmount(totalRev));
		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));
	}

	public void populateReport(Session session) {
		calculateNetSales(session);
		calculateRefundAmount(session);
		calculatePurchaseOrderRefundAmount(session);
		calculateDrawerBleed(session);
		calculateCashTips(session);
		calculateChargedTips(session);

		cashDrawer.setTotalTips(cashDrawer.getCashTips() + cashDrawer.getChargedTips());

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() + cashDrawer.getSalesTax() + cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue(POSUtil.getDoubleAmount(totalRev));
		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));

		calculateCashReceipt(session);
		calculateCreditReceipt(session);
		calculateDebitReceipt(session);
		calculateMemberPayment(session);
		calculateAdvancePayment(session);

		calculateCustomPaymentWithoutPromotion(session);
		calculatePromotionAmount(session);
		calculateGiftCertReceipts(session);
		// calculateVoidAmount();
		calculateCashPayout(session);
		calculateCashRFAndLDFPay(RfPayTransaction.class, session);
		calculateCashRFAndLDFPay(LdfPayTransaction.class, session);
		calculateCashBack(session);

		calculateReceiptDiff();

		calculateAddBalance(session);

		calculateDrawerAccountable();
		calculateTicketCount(session);
	}

	public void populateCashBalance() {
		try (Session session = TerminalDAO.getInstance().createNewSession()) {
			populateCashBalance(session);
		}
	}

	public void populateCashBalance(Session session) {
		calculateNetSales(session);
		calculateRefundAmount(session);
		calculateDrawerBleed(session);
		// calculateCashTips();
		// calculateChargedTips();

		double totalRev = cashDrawer.getNetSales() - cashDrawer.getRefundAmount() + cashDrawer.getSalesTax() + cashDrawer.getServiceCharge();
		cashDrawer.setTotalRevenue(POSUtil.getDoubleAmount(totalRev));
		cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(cashDrawer.getTotalRevenue() + cashDrawer.getCashTips() + cashDrawer.getChargedTips()));

		calculateCashReceipt(session);
		calculateAdvancePayment(session);

		calculateCashPayout(session);
		calculateCashRFAndLDFPay(RfPayTransaction.class, session);
		calculateCashRFAndLDFPay(LdfPayTransaction.class, session);
		calculateCashBack(session);
		calculateAddBalance(session);
		
		calculatePurchaseOrderRefundAmount(session);

		calculateDrawerAccountable();
	}

	private void calculateTicketCount(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.setProjection(Projections.countDistinct(PosTransaction.PROP_TICKET));
		Number rowCount = (Number) criteria.uniqueResult();
		int ticketCount = 0;
		if (rowCount != null) {
			ticketCount = rowCount.intValue();
		}
		cashDrawer.setTicketCount(ticketCount);
	}

	private void calculateMemberPayment(Session session) {
		Criteria criteria = getCriteriaForTransaction(CustomerAccountTransaction.class, session);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.MEMBER_ACCOUNT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCustomerPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateDrawerAccountable() {
		double drawerAccountableAmnt = cashDrawer.getCashReceiptAmount() - cashDrawer.getTipsPaid() - cashDrawer.getPayOutAmount() - cashDrawer.getCashBack()
				+ cashDrawer.getBeginCash() + cashDrawer.getAddBalance() + cashDrawer.getPoRefundAmount() - cashDrawer.getDrawerBleedAmount()
				- cashDrawer.getRfAmount() - cashDrawer.getLdfAmount();
		cashDrawer.setDrawerAccountable(POSUtil.getDoubleAmount(drawerAccountableAmnt));
	}

	private void calculateReceiptDiff() {
		double receiptDiff = cashDrawer.getGrossReceipts() - cashDrawer.getCashReceiptAmount() - cashDrawer.getCreditCardReceiptAmount()
				- cashDrawer.getDebitCardReceiptAmount() - cashDrawer.getAdvancePaymentAmount() - cashDrawer.getCustomerPaymentAmount()
				- cashDrawer.getCustomPaymentAmount() - cashDrawer.getPromotionAmount() - cashDrawer.getGiftCertChangeAmount()
				- cashDrawer.getGiftCertReturnAmount() + cashDrawer.getRefundAmount();
		cashDrawer.setReceiptDifferential(NumberUtil.round(receiptDiff));
	}

	private Criteria getCriteriaForTransaction(Class transactionClass, Session session) {
		return getCriteriaForTransaction(transactionClass, null, session);
	}

	private Criteria getCriteriaForTransaction(Class transactionClass, PaymentType paymentType, Session session) {
		Criteria criteria = session.createCriteria(transactionClass);
		if (transactionClass.equals(RefundTransaction.class)) {
			criteria = createRefundTransactionCriteria(session);
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()));
		}
		else if (transactionClass.equals(PayOutTransaction.class) || transactionClass.equals(CashDropTransaction.class)
				|| transactionClass.equals(LdfPayTransaction.class)) {
			LogicalExpression orExpression = Restrictions.or(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.DEBIT.name()),
					Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.OUT.name()));
			criteria.add(orExpression);
		}
		else if (transactionClass.equals(RfPayTransaction.class)) {
			// RfPayTransaction will add all TransactionType
		}
		else {
			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		}

		if (paymentType != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, paymentType.name()));
		}
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
		return criteria;
	}

	private void calculateNetSales(Session session) {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class, session);
		criteria.add(Property.forName("class").ne(AdvanceAdjustmentTransaction.class));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TAX_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		projectionList.add(Projections.sum(PosTransaction.PROP_SERVICE_CHARGE_AMOUNT));
		criteria.setProjection(projectionList);

		List<Object[]> list = criteria.list();
		if (list == null || list.isEmpty()) {
			return;
		}
		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			Object[] objects = (Object[]) iterator.next();
			if (objects[0] == null) {
				return;
			}
			double salesAmnt = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double taxAmnt = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;
			double tipsAmnt = (objects[2] != null) ? ((Number) objects[2]).doubleValue() : 0;
			double serviceChrgAmnt = (objects[3] != null) ? ((Number) objects[3]).doubleValue() : 0;

			double netSales = salesAmnt - taxAmnt - tipsAmnt - serviceChrgAmnt;
			cashDrawer.setNetSales(NumberUtil.round(netSales));
			cashDrawer.setSalesTax(NumberUtil.round(taxAmnt));
			cashDrawer.setServiceCharge(NumberUtil.round(serviceChrgAmnt));
		}
	}

	private void calculateCashTips(Session session) {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class, session);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setCashTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateChargedTips(Session session) {
		Criteria criteria = getCriteriaForTransaction(PosTransaction.class, session);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_TIPS_AMOUNT));
		cashDrawer.setChargedTips(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashReceipt(Session session) {
		Criteria criteria = getCriteriaForTransaction(CashTransaction.class, session);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCashReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()) + getAdvanceDeposit(PaymentType.CASH, session));
	}

	private void calculateAdvancePayment(Session session) {
		Criteria criteria = getCriteriaForTransaction(AdvanceTransaction.class, PaymentType.ADVANCE_ADJUSTMENT, session);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setAdvancePaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	public double getAdvanceDeposit(PaymentType paymentType, Session session) {
		Criteria criteria = getCriteriaForTransaction(AdvanceTransaction.class, paymentType, session);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		double advanceCashReceiptAmount = POSUtil.getDoubleAmount(criteria.uniqueResult());
		return advanceCashReceiptAmount;
	}

	private void calculateCreditReceipt(Session session) {
		Criteria criteria = getCriteriaForTransaction(CreditCardTransaction.class, session);
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		cashDrawer.setCreditCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()) + getAdvanceDeposit(PaymentType.CREDIT_CARD, session));
	}

	private void calculateDebitReceipt(Session session) {
		Criteria criteria = getCriteriaForTransaction(DebitCardTransaction.class, session);
		criteria.setProjection(Projections.sum(CashTransaction.PROP_AMOUNT));
		cashDrawer.setDebitCardReceiptAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	// private void calculateCustomPayment() {
	// Criteria criteria =
	// getCriteriaForTransaction(CustomPaymentTransaction.class);
	// criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
	// cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	// }

	private void calculateCustomPaymentWithoutPromotion(Session session) {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class, session);
		criteria.add(Restrictions.ne(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setCustomPaymentAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()) + getAdvanceDeposit(PaymentType.CUSTOM_PAYMENT, session));
	}

	private void calculatePromotionAmount(Session session) {
		Criteria criteria = getCriteriaForTransaction(CustomPaymentTransaction.class, session);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.PROMOTION.name()));
		criteria.setProjection(Projections.sum(CustomPaymentTransaction.PROP_AMOUNT));
		cashDrawer.setPromotionAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateGiftCertReceipts(Session session) {
		Criteria criteria = getCriteriaForTransaction(GiftCertificateTransaction.class, session);
		criteria.setProjection(Projections.sum(GiftCertificateTransaction.PROP_AMOUNT));
		cashDrawer.setGiftCertChangeAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateRefundAmount(Session session) {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class, session);
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setRefundAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculatePurchaseOrderRefundAmount(Session session) {
		Criteria criteria = getCriteriaForTransaction(PurchaseRefundTransaction.class, session);
		criteria.add(Restrictions.eq(PurchaseRefundTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PurchaseRefundTransaction.PROP_AMOUNT));
		cashDrawer.setPoRefundAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	
	private void calculateVoidAmount(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.TRUE));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCardVoidAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashPayout(Session session) {
		Criteria criteria = getCriteriaForTransaction(PayOutTransaction.class, session);
		// criteria.add(Restrictions.ne(PayOutTransaction.PROP_EXPENSE, true));
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setPayOutAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashRFAndLDFPay(Class transactionClass, Session session) {
		Criteria criteria = getCriteriaForTransaction(transactionClass, session);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));

		List<PosTransaction> list = criteria.list();

		double totalTodaysPay = 0;
		double totalDuePay = 0;
		for (PosTransaction posTransaction : list) {
			String transactionType = posTransaction.getTransactionType();
			if (TransactionType.IN.name().equals(transactionType)) {
				posTransaction.setAmount((-1) * posTransaction.getAmount());
			}

			String transTicketIds = posTransaction.getTransTicketIdsDisplay();
			if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) {
				continue;
			}

			Date orderDate = DataProvider.get().getTransactionsTicketDate(transTicketIds, posTransaction.getOutletId());
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (orderDate.before(startOfDay)) {
				totalDuePay += posTransaction.getAmount();
			}
			else {
				totalTodaysPay += posTransaction.getAmount();
			}
		}

		if (transactionClass.equals(RfPayTransaction.class)) {
			cashDrawer.setRfAmount(POSUtil.getDoubleAmount(totalTodaysPay));
			cashDrawer.setRfDuePay(POSUtil.getDoubleAmount(totalDuePay));
		}
		else if (transactionClass.equals(LdfPayTransaction.class)) {
			cashDrawer.setLdfAmount(POSUtil.getDoubleAmount(totalTodaysPay));
			cashDrawer.setLdfDuePay(POSUtil.getDoubleAmount(totalDuePay));
		}
	}

	private void calculateDrawerBleed(Session session) {
		Criteria criteria = getCriteriaForTransaction(CashDropTransaction.class, session);
		criteria.setProjection(Projections.sum(PayOutTransaction.PROP_AMOUNT));
		cashDrawer.setDrawerBleedAmount(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	private void calculateCashBack(Session session) {
		Criteria criteria = getCriteriaForTransaction(RefundTransaction.class, session);
		criteria.add(Restrictions.eq(PosTransaction.PROP_PAYMENT_TYPE_STRING, PaymentType.CASH.name()));
		criteria.setProjection(Projections.sum(RefundTransaction.PROP_AMOUNT));
		cashDrawer.setCashBack(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	public double findTotalPaidGratuititesForStaffBank(CashDrawer staffBank) throws PosException {
		try (Session session = GratuityDAO.getInstance().createNewSession()) {
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			detachedCriteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, staffBank.getId()));
			detachedCriteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
			detachedCriteria.setProjection(Projections.property("t.id")); //$NON-NLS-1$

			Criteria criteria = session.createCriteria(Gratuity.class);
			criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(detachedCriteria));
			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Gratuity.PROP_AMOUNT));
			projectionList.add(Projections.sum(Gratuity.PROP_TIPS_PAID_AMOUNT));
			criteria.setProjection(projectionList);

			List list = criteria.list();
			double total = 0;
			if (list.size() > 0) {
				Object[] objects = (Object[]) list.get(0);
				if (objects.length > 0) {
					total = POSUtil.getDoubleAmount(objects[0]);
				}
				if (objects.length > 1) {
					total -= POSUtil.getDoubleAmount(objects[1]);
				}

			}
			criteria = session.createCriteria(GratuityPaymentHistory.class);
			criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, staffBank));
			criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
			return total + POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	private void calculateTipsPaidByOtherTerminal(Session session) throws PosException {
		DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
		detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		detachedCriteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
		detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		detachedCriteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
		detachedCriteria.setProjection(Projections.property("t.id")); //$NON-NLS-1$

		Criteria criteria = session.createCriteria(Gratuity.class);
		criteria.add(Property.forName(Gratuity.PROP_TICKET_ID).in(detachedCriteria));

		criteria.setProjection(Projections.sum(Gratuity.PROP_TIPS_PAID_AMOUNT));

		double totalPaidTips = POSUtil.getDoubleAmount(criteria.uniqueResult());

		criteria = session.createCriteria(GratuityPaymentHistory.class);
		criteria.add(Restrictions.eq(GratuityPaymentHistory.PROP_CASH_DRAWER, cashDrawer));
		criteria.setProjection(Projections.sum(GratuityPaymentHistory.PROP_AMOUNT));
		double tipsPaidFromThisBank = POSUtil.getDoubleAmount(criteria.uniqueResult());

		cashDrawer.setTipsPaidByOtherTerminal(totalPaidTips - tipsPaidFromThisBank);
	}

	private void calculateAddBalance(Session session) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, cashDrawer.getId()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.CASH_DRAWER_BALANCE.name()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.ADD_BALANCE.name()));
		criteria.setProjection(Projections.sum(BalanceUpdateTransaction.PROP_AMOUNT));
		cashDrawer.setAddBalance(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	}

	// private void calculateGrossReceipts() {
	// Criteria criteria = getCriteriaForTransaction(CashTransaction.class);
	// criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
	// cashDrawer.setGrossReceipts(POSUtil.getDoubleAmount(criteria.uniqueResult()));
	// }
	//

	@SuppressWarnings("unchecked")
	public DetailsOfCashdrawerDataProvider populateDetailOfCashdrawer() {
		try (Session session = TerminalDAO.getInstance().createNewSession()) {

			DetailsOfCashdrawerDataProvider dataProvider = new DetailsOfCashdrawerDataProvider();
			dataProvider.setDetailTransactionList(getGrossReceiptCollectionList(null, session));
			dataProvider.setAddBalanceList(getAddBalanceTransactionList(session));
			dataProvider.setPayoutList(getPayoutList(session));
			dataProvider.setRfPayList(getReferrerAndLDFPayTransactionList(false, RfPayTransaction.class, session));
			dataProvider.setRfDueList(getReferrerAndLDFPayTransactionList(true, RfPayTransaction.class, session));
			dataProvider.setLdfPayList(getReferrerAndLDFPayTransactionList(false, LdfPayTransaction.class, session));
			dataProvider.setLdfDueList(getReferrerAndLDFPayTransactionList(true, LdfPayTransaction.class, session));
			dataProvider.setRefundTransactionList(getRefundTransactionList(session));
			dataProvider.setDueCollectionList(getGrossReceiptCollectionList(true, session));
			dataProvider.setVoidedCollectionList(getVoidedTransactionList(session));
			return dataProvider;
		}
	}

	private List<CashDrawerDetailReport> getAddBalanceTransactionList(Session session) {
		Criteria criteria = session.createCriteria(BalanceUpdateTransaction.class);
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_ACCOUNT_NUMBER, cashDrawer.getId()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_BALANCE_TYPE_STRING, BalanceType.CASH_DRAWER_BALANCE.name()));
		criteria.add(Restrictions.eq(BalanceUpdateTransaction.PROP_TRANSACTION_SUB_TYPE, BalanceSubType.ADD_BALANCE.name()));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_RECEPIENT_ID), CashDrawerDetailReport.RECEPIENT_ID);
		projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		projectionList.add(Projections.property(BalanceUpdateTransaction.PROP_EXTRA_PROPERTIES), CashDrawerDetailReport.PROPERTIES);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();
		for (CashDrawerDetailReport detailReport : list) {
			detailReport.initDepositFromName();
		}
		return list;
	}

	private List<CashDrawerDetailReport> getGrossReceiptCollectionList(Boolean due, Session session) {
		List<CashDrawerDetailReport> collectionList = getDueAndGrossReceiptCollectionList(due, false, session);
		collectionList.addAll(getDueAndGrossReceiptCollectionList(due, true, session));
		collectionList.sort(Comparator.comparing(CashDrawerDetailReport::getTransTime));
		return collectionList;
	}

	private List<CashDrawerDetailReport> getDueAndGrossReceiptCollectionList(Boolean due, boolean isRefund, Session session) {

		Criteria criteria = getCriteriaForTransaction(isRefund ? RefundTransaction.class : PosTransaction.class, session);
		criteria.add(Property.forName("class").ne(AdvanceAdjustmentTransaction.class));
		List<PosTransaction> list = criteria.list();

		if (due == null) {
			List<CashDrawerDetailReport> totalList = new ArrayList<>();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				PosTransaction transaction = (PosTransaction) iterator.next();

				CashDrawerDetailReport detailReport = new CashDrawerDetailReport(transaction, null);
				if (isRefund) {
					detailReport.setTransactionAmount((-1) * detailReport.getTransactionAmount());
				}
				detailReport.initInvoiceDate();

				totalList.add(detailReport);
			}
			return totalList;
		}

		List<CashDrawerDetailReport> dueList = new ArrayList<>();
		List<CashDrawerDetailReport> todayPayList = new ArrayList<>();

		for (Iterator iterator = list.iterator(); iterator.hasNext();) {
			PosTransaction transaction = (PosTransaction) iterator.next();

			Ticket ticket = transaction.getTicket();
			Date orderDate = ticket == null ? null : ticket.getCreateDate();
			Date startOfDay = DateUtil.startOfDay(transaction.getTransactionTime());

			CashDrawerDetailReport detailReport = new CashDrawerDetailReport(transaction, null);
			if (isRefund) {
				detailReport.setTransactionAmount((-1) * detailReport.getTransactionAmount());
			}
			detailReport.initInvoiceDate();

			if (orderDate != null && orderDate.before(startOfDay)) {
				dueList.add(detailReport);
			}
			else {
				todayPayList.add(detailReport);
			}
		}

		return due ? dueList : todayPayList;
	}

	private List getPayoutList(Session session) {
		Criteria criteria = session.createCriteria(PayOutTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(PayOutTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(PayOutTransaction.PROP_RECEPIENT_ID), CashDrawerDetailReport.RECEPIENT_ID);
		projectionList.add(Projections.property(PayOutTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(PayOutTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();
		for (CashDrawerDetailReport detailReport : list) {
			detailReport.initRecepientName();
		}
		return list;
	}

	private List getReferrerAndLDFPayTransactionList(boolean due, Class transactionClass, Session session) {
		Criteria criteria = getCriteriaForTransaction(transactionClass, session);

		List<PosTransaction> list = criteria.list();
		List<CashDrawerDetailReport> dueList = new ArrayList<>();
		List<CashDrawerDetailReport> todayPayList = new ArrayList<>();

		for (PosTransaction posTransaction : list) {
			String transactionType = posTransaction.getTransactionType();
			if (TransactionType.IN.name().equals(transactionType)) {
				posTransaction.setAmount((-1) * posTransaction.getAmount());
			}

			String transTicketIds = posTransaction.getTransTicketIdsDisplay();
			if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) { //$NON-NLS-1$
				continue;
			}

			Date orderDate = DataProvider.get().getTransactionsTicketDate(transTicketIds, posTransaction.getOutletId());
			Date startOfDay = DateUtil.startOfDay(posTransaction.getTransactionTime());
			if (orderDate.before(startOfDay)) {
				dueList.add(new CashDrawerDetailReport(posTransaction, orderDate));
			}
			else {
				todayPayList.add(new CashDrawerDetailReport(posTransaction, orderDate));
			}
		}
		return due ? dueList : todayPayList;
	}

	private List getRefundTransactionList(Session session) {
		Criteria criteria = createRefundTransactionCriteria(session);

		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

		List<CashDrawerDetailReport> finalList = new ArrayList<CashDrawerDetailReport>();
		List<PosTransaction> list = criteria.list();
		for (PosTransaction transaction : list) {
			CashDrawerDetailReport detailReport = new CashDrawerDetailReport(transaction, null);
			//detailReport.initInvoiceDate();
			finalList.add(detailReport);
		}
		return finalList;
	}

	private Criteria createRefundTransactionCriteria(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.or(Property.forName("class").eq(RefundTransaction.class), Property.forName("class").eq(AdvanceRefundTransaction.class))); //$NON-NLS-1$ //$NON-NLS-2$
		return criteria;
	}

	private List getVoidedTransactionList(Session session) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawer.getId()));
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.TRUE));

		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), CashDrawerDetailReport.TRANS_TIME);
		projectionList.add(Projections.property(PosTransaction.PROP_TICKET), CashDrawerDetailReport.PROP_TICKET);
		projectionList.add(Projections.property(PosTransaction.PROP_AMOUNT), CashDrawerDetailReport.TRANSACTION_AMOUNT);
		projectionList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), CashDrawerDetailReport.PAYMENT_TYPE);
		projectionList.add(Projections.property(PosTransaction.PROP_USER_ID), CashDrawerDetailReport.USER_ID);
		projectionList.add(Projections.property(PosTransaction.PROP_EXTRA_PROPERTIES), CashDrawerDetailReport.PROPERTIES);
		criteria.setProjection(projectionList);

		criteria.setResultTransformer(Transformers.aliasToBean(CashDrawerDetailReport.class));

		List<CashDrawerDetailReport> list = criteria.list();
		for (CashDrawerDetailReport detailReport : list) {
			detailReport.initVoidedByUser();
		}
		return list;
	}

	public class DetailsOfCashdrawerDataProvider {
		private List<CashDrawerDetailReport> addBalanceList;
		private List<CashDrawerDetailReport> detailTransactionList;
		private List<CashDrawerDetailReport> payoutList;
		private List<CashDrawerDetailReport> rfPayList;
		private List<CashDrawerDetailReport> rfDueList;
		private List<CashDrawerDetailReport> ldfPayList;
		private List<CashDrawerDetailReport> ldfDueList;
		private List<CashDrawerDetailReport> refundTransactionList;
		private List<CashDrawerDetailReport> dueCollectionList;
		private List<CashDrawerDetailReport> voidedCollectionList;

		public DetailsOfCashdrawerDataProvider() {
		}

		public List<CashDrawerDetailReport> getAddBalanceList() {
			return addBalanceList;
		}

		public void setAddBalanceList(List<CashDrawerDetailReport> addBalanceTransactionList) {
			this.addBalanceList = addBalanceTransactionList;
		}

		public List<CashDrawerDetailReport> getDetailTransactionList() {
			return detailTransactionList;
		}

		public List<CashDrawerDetailReport> getPayoutList() {
			return payoutList;
		}

		public List<CashDrawerDetailReport> getRfPayList() {
			return rfPayList;
		}

		public List<CashDrawerDetailReport> getRfDueList() {
			return rfDueList;
		}

		public List<CashDrawerDetailReport> getLdfPayList() {
			return ldfPayList;
		}

		public List<CashDrawerDetailReport> getLdfDueList() {
			return ldfDueList;
		}

		public List<CashDrawerDetailReport> getRefundTransactionList() {
			return refundTransactionList;
		}

		public List<CashDrawerDetailReport> getDueCollectionList() {
			return dueCollectionList;
		}

		public List<CashDrawerDetailReport> getVoidedCollectionList() {
			return voidedCollectionList;
		}

		public void setDetailTransactionList(List<CashDrawerDetailReport> detailTransactionList) {
			this.detailTransactionList = detailTransactionList;
		}

		public void setPayoutList(List<CashDrawerDetailReport> payoutList) {
			this.payoutList = payoutList;
		}

		public void setRfPayList(List<CashDrawerDetailReport> rfPayList) {
			this.rfPayList = rfPayList;
		}

		public void setRfDueList(List<CashDrawerDetailReport> rfDueList) {
			this.rfDueList = rfDueList;
		}

		public void setLdfPayList(List<CashDrawerDetailReport> ldfPayList) {
			this.ldfPayList = ldfPayList;
		}

		public void setLdfDueList(List<CashDrawerDetailReport> ldfDueList) {
			this.ldfDueList = ldfDueList;
		}

		public void setRefundTransactionList(List<CashDrawerDetailReport> refundTransactionList) {
			this.refundTransactionList = refundTransactionList;
		}

		public void setDueCollectionList(List<CashDrawerDetailReport> dueCollectionList) {
			this.dueCollectionList = dueCollectionList;
		}

		public void setVoidedCollectionList(List<CashDrawerDetailReport> voidedCollectionList) {
			this.voidedCollectionList = voidedCollectionList;
		}

	}

}
