/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.db.update;

import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;

import com.floreantpos.PosLog;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.dao.BookingInfoDAO;
import com.floreantpos.model.dao.DoctorDAO;
import com.floreantpos.model.dao.TicketItemDAO;

public class UpdateDBTo417 {

	private SessionFactory sessionFactory;
	private String schemaName;

	public UpdateDBTo417(SessionFactory sessionFactory, String schemaName) {
		this.sessionFactory = sessionFactory;
		this.schemaName = schemaName;
	}
	
	public void update() {
		updateAdmissions();
		updateLabDoctors();
	}

	public void updateAdmissions() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = session.createCriteria(BookingInfo.class);
			criteria.setProjection(Projections.rowCount());

			int totalRecords = ((Long) criteria.uniqueResult()).intValue();

			PosLog.info(UpdateDBTo417.class, "Found total admissions: " + totalRecords); //$NON-NLS-1$

			criteria = session.createCriteria(BookingInfo.class);
			criteria.setMaxResults(totalRecords);

			List<BookingInfo> admissions = criteria.list();
			
			Transaction transaction = session.beginTransaction();
			for (BookingInfo admission : admissions) {

				admission.setCustomer(admission.getCustomer());
				if (admission.getDoctorId() != null) {
					admission.setDoctor(DoctorDAO.getInstance().get(admission.getDoctorId()));
				}

				BookingInfoDAO.getInstance().saveOrUpdate(admission, session);

			}
			transaction.commit();
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}
	
	public void updateLabDoctors() {
		PosLog.info(getClass(), "Updating " + schemaName + " schema."); //$NON-NLS-1$ //$NON-NLS-2$

		int totalItems = ticketItemRowCount();
		PosLog.info(UpdateDBTo416.class, "Found total ticket items: " + totalItems); //$NON-NLS-1$

		int increment = 1000;
		int count = 0;
		while (count < totalItems) {
			try (Session session = sessionFactory.openSession()) {
				Transaction transaction = session.beginTransaction();

				Pagination<TicketItem> pagination = new Pagination<TicketItem>(count, increment);
				pagination.setNumRows(totalItems);
				findTicketItems(pagination, session);

				List<TicketItem> items = pagination.getDataList();
				for (TicketItem item : items) {
					String labDoctorId = item.getLabDoctorId();
					if (StringUtils.isBlank(labDoctorId)) {
						continue;
					}
					
					Doctor doctor = DoctorDAO.getInstance().get(labDoctorId);
					item.setLabDoctor(doctor);
					
					TicketItemDAO.getInstance().update(item, session);

				}
				transaction.commit();

				count += increment;
			}
		}

		PosLog.info(getClass(), schemaName + " update completed successfully"); //$NON-NLS-1$
	}
	
	private void findTicketItems(Pagination<TicketItem> pagination, Session session) {
		Criteria criteria = session.createCriteria(TicketItem.class);
		criteria.addOrder(Order.asc(TicketItem.PROP_CREATE_DATE));
		criteria.setFirstResult(pagination.getCurrentRowIndex());
		criteria.setMaxResults(pagination.getPageSize());

		PosLog.info(getClass(),
				String.format("Searching ticket item from: %s to %s", pagination.getCurrentRowIndex(), (pagination.getCurrentRowIndex() + pagination.getPageSize()))); //$NON-NLS-1$

		pagination.setRows(criteria.list());

	}

	public int ticketItemRowCount() {
		try (Session session = sessionFactory.openSession()) {
			Criteria criteria = session.createCriteria(TicketItem.class);
			criteria.setProjection(Projections.rowCount());
			return ((Long) criteria.uniqueResult()).intValue();
		}
	}
}