/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonTypeInfo.Id;
import com.floreantpos.model.base.BasePosTransaction;
import com.floreantpos.model.dao.PayoutReasonDAO;
import com.floreantpos.model.dao.PayoutRecepientDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@XmlSeeAlso({ CashTransaction.class, CustomPaymentTransaction.class, GiftCertificateTransaction.class, CreditCardTransaction.class, DebitCardTransaction.class,
		CashDropTransaction.class, RefundTransaction.class, SalaryTransaction.class, SalaryAdvanceTransaction.class, SalaryAdvanceBackTransaction.class,
		PayOutTransaction.class, VoidTransaction.class, CustomerAccountTransaction.class, ReversalTransaction.class, RfPayTransaction.class,
		LdfPayTransaction.class, PurchaseTransaction.class, ExpenseTransaction.class })
@XmlRootElement
@XmlAccessorType(XmlAccessType.FIELD)
@JsonIgnoreProperties(ignoreUnknown = true, value = { "ticketItems", "terminal", "properties", "cashDrawer" })
@JsonTypeInfo(use = Id.CLASS, include = JsonTypeInfo.As.PROPERTY, property = "classType")
@JsonSubTypes({ @Type(value = CashTransaction.class), @Type(value = CustomPaymentTransaction.class), @Type(value = GiftCertificateTransaction.class),
		@Type(value = CreditCardTransaction.class), @Type(value = DebitCardTransaction.class), @Type(value = CashDropTransaction.class),
		@Type(value = RefundTransaction.class), @Type(value = SalaryTransaction.class), @Type(value = SalaryAdvanceTransaction.class),
		@Type(value = SalaryAdvanceBackTransaction.class), @Type(value = PayOutTransaction.class), @Type(value = VoidTransaction.class),
		@Type(value = CustomerAccountTransaction.class), @Type(value = ReversalTransaction.class), @Type(value = RfPayTransaction.class),
		@Type(value = LdfPayTransaction.class), @Type(value = PurchaseTransaction.class), @Type(value = ExpenseTransaction.class) })
public class PosTransaction extends BasePosTransaction implements TimedModel {

	private static final String BANK_ACCOUNT = "bank_account"; //$NON-NLS-1$
	private static final String ACCOUNT_MANAGER_NAME = "account_manager_name"; //$NON-NLS-1$
	private static final String VENDOR_NAME = "vendor_name"; //$NON-NLS-1$
	private static final String SERVER_NAME = "server_name"; //$NON-NLS-1$
	private static final String PAYOUT_EXPENSE_RECEPIENT_NAME = "payout_expense_recepient_name"; //$NON-NLS-1$
	private static final String PAYOUT_EXPENSE_SUB_REASON = "payout_expense_sub_reason"; //$NON-NLS-1$
	private static final String PAYOUT_EXPENSE_REASON = "payout_expense_reason"; //$NON-NLS-1$
	private static final String USER_NAME = "user_name"; //$NON-NLS-1$
	private static final String VOIDED_BY_USER_NAME = "voidedByUserName"; //$NON-NLS-1$

	public static final String JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO = "gift_card_balance_add_info"; //$NON-NLS-1$
	public static final String JSON_PROP_GIFT_CARD_ADDED_AMOUNT = "addedAmount"; //$NON-NLS-1$
	public static final String JSON_PROP_GIFT_CARD_GIFT_CARD_NO = "giftCardNo"; //$NON-NLS-1$
	public static final String JSON_PROP_GIFT_CARD_TICKET_ITEM_ID = "ticketItemId"; //$NON-NLS-1$
	public static final String JSON_PROP_CUSTOMER_NAME = "CUSTOMER_NAME"; //$NON-NLS-1$

	public final static String CASH = "CASH"; //$NON-NLS-1$
	public final static String GIFT_CERT = "GIFT_CERT"; //$NON-NLS-1$
	public final static String CREDIT_CARD = "CREDIT_CARD"; //$NON-NLS-1$
	public final static String DEBIT_CARD = "DEBIT_CARD"; //$NON-NLS-1$
	public final static String CASH_DROP = "CASH_DROP"; //$NON-NLS-1$
	public final static String REFUND = "REFUND"; //$NON-NLS-1$
	public final static String PAY_OUT = "PAY_OUT"; //$NON-NLS-1$
	public final static String VOID_TRANS = "VOID_TRANS"; //$NON-NLS-1$
	public final static String GIFT_CARD_BALANCE_ADD = "GIFT_CARD_BALANCE_ADD"; //$NON-NLS-1$
	public static final String PROP_CASH_BACK_POSTFIX = "_CASH_BACK"; //$NON-NLS-1$
	public static final String PROP_TENDERED_POSTFIX = "_TENDERED"; //$NON-NLS-1$
	public static final String PROP_PAID_POSTFIX = "_PAID"; //$NON-NLS-1$
	public static final String JSON_PROP_REFUNDED_AMOUNT = "REFUNDED_AMOUNT"; //$NON-NLS-1$

	public static final String TRANSACTION_FEE = "transactionFee"; //$NON-NLS-1$

	private static final String JSON_PROP_GIFT_CARD_NUMBER = "gift_card_number"; //$NON-NLS-1$
	private static final String JSON_PROP_GIFT_CARD_BALANCE_AMOUNT = "gift_card_balance_amount"; //$NON-NLS-1$

	public static final String CURRENCY_ID = "currency.id"; //$NON-NLS-1$
	public static final String CURRENCY_NAME = "currency.name"; //$NON-NLS-1$
	public static final String CURRENCY_CODE = "currency.code"; //$NON-NLS-1$
	public static final String CURRENCY_SYMBOL = "currency.symbol"; //$NON-NLS-1$
	public static final String CURRENCY_RATE = "currency.rate"; //$NON-NLS-1$
	public static final String CURRENCY_TENDERED = "currency.tendered"; //$NON-NLS-1$
	public static final String CURRENCY_CHANGE = "currency.change"; //$NON-NLS-1$
	public static final String CURRENCY_PAID_AMOUNT = "currency.paidAmount"; //$NON-NLS-1$
	public static final String PAYMENT_CURRENCIES = "paymentCurrencies"; //$NON-NLS-1$

	private String cardTrack;
	private String cardNo;
	private String cardExpYear;
	private String cardExpMonth;
	private String cardCVV;
	private transient String ticketId;

	private String classType;
	private static final long serialVersionUID = 1L;

	private transient boolean refunded = false;
	private transient boolean syncEdited;
	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	private transient Project project;
	private CashDrawer cashDrawer;
	private transient com.google.gson.JsonObject propertiesContainer;
	private transient BankAccount linkedBankAccount;

	public PosTransaction() {
		setDataVersion(2);
	}

	/**
	 * Constructor for primary key
	 */
	public PosTransaction(java.lang.String id) {
		super(id);
		setDataVersion(2);
	}

	/**
	 * Constructor for required fields
	 */
	public PosTransaction(java.lang.String id, java.lang.String transactionType, java.lang.String paymentTypeString) {

		super(id, transactionType, paymentTypeString);
		setDataVersion(2);
	}

	@Override
	public String getTransactionType() {
		String type = super.getTransactionType();

		if (StringUtils.isEmpty(type)) {
			return TransactionType.CREDIT.name();
		}

		return type;
	}

	public boolean isCard() {
		return (this instanceof CreditCardTransaction) || (this instanceof DebitCardTransaction);
	}

	public void addProperty(String name, String value) {
		addExtraProperty(name, value);
	}

	public boolean hasProperty(String key) {
		return getExtraProperty(key) != null;
	}

	public String getProperty(String key) {
		return getExtraProperty(key);
	}

	public String getProperty(String key, String defaultValue) {
		if (getExtraProperties() == null) {
			return defaultValue;
		}

		String string = getExtraProperty(key);
		if (StringUtils.isEmpty(string)) {
			return defaultValue;
		}

		return string;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getExtraProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public Double calculateTotalAmount() {
		return getAmount() + getTipsAmount();
	}

	public Double calculateAuthorizeAmount() {
		double advanceTipsPercentage = 0;//CardConfig.getAdvanceTipsPercentage();
		return getTenderAmount() + getTenderAmount() * (advanceTipsPercentage / 100);
	}

	@XmlTransient
	public String getCardTrack() {
		return cardTrack;
	}

	public void setCardTrack(String cardTrack) {
		this.cardTrack = cardTrack;
	}

	public String getCardNo() {
		return cardNo;
	}

	public void setCardNo(String cardNo) {
		this.cardNo = cardNo;
	}

	public String getCardExpYear() {
		return cardExpYear;
	}

	public void setCardExpYear(String expYear) {
		this.cardExpYear = expYear;
	}

	public String getCardExpMonth() {
		return cardExpMonth;
	}

	public void setCardExpMonth(String expMonth) {
		this.cardExpMonth = expMonth;
	}

	public String getCardCVV() {
		return cardCVV;
	}

	public void setCardCVV(String cardCVV) {
		this.cardCVV = cardCVV;
	}

	public String getGlobalId() {
		return super.getId();
	}

	public String getTicketId() {
		if (StringUtils.isNotEmpty(ticketId)) {
			return ticketId;
		}
		Ticket ticket = getTicket();
		if (ticket == null) {
			return ""; //$NON-NLS-1$
		}
		return ticket.getId();
	}

	public void setTicketId(String ticketId) {
		this.ticketId = ticketId;
	}

	public void calculateTaxAmount() {
		if (getAmount() <= 0)
			return;
		Ticket ticket = getTicket();
		if (ticket == null || ticket.getTotalAmount() <= 0 || isVoided() || this instanceof RefundTransaction)
			return;
		setTaxAmount(((getAmountWithTolerance() - getTipsAmount()) * ticket.getTaxAmount() / ticket.getTotalAmount()));
	}

	public void calculateServiceChargeAmount() {
		if (getAmount() <= 0)
			return;
		Ticket ticket = getTicket();
		if (ticket == null || ticket.getServiceCharge() <= 0 || isVoided() || this instanceof RefundTransaction)
			return;
		setServiceChargeAmount(((getAmountWithTolerance() - getTipsAmount()) * ticket.getServiceCharge() / ticket.getTotalAmount()));
	}

	public boolean isRefunded() {
		return refunded;
	}

	public void setRefunded(boolean refunded) {
		this.refunded = refunded;
	}

	@XmlTransient
	public Department getDepartment() {
		return DataProvider.get().getDepartmentById(getDepartmentId());
	}

	public void setDepartment(Department department) {
		String departmentId = null;
		if (department != null) {
			departmentId = department.getId();
		}
		super.setDepartmentId(departmentId);
	}

	@XmlTransient
	public Terminal getTerminal() {
		return DataProvider.get().getTerminalById(getTerminalId(), getOutletId());
	}

	public void setTerminal(Terminal terminal) {
		Integer terminalId = null;
		if (terminal != null) {
			terminalId = terminal.getId();
		}
		super.setTerminalId(terminalId);
	}

	@XmlTransient
	public User getUser() {
		return DataProvider.get().getUserById(getUserId(), getOutletId());
	}

	/**
	 *  @param user : For {@link SalaryTransaction }, the user represents the person to whom the salary is being paid.
	 *   otherwise current user
	 *   
	 * @see #putSalaryPaidTo(String); for how the salary recipient is updated in {@link SalaryTransaction}.
	 */
	public void setUser(User user) {
		String userId = null;
		if (user != null) {
			userId = user.getId();
			addProperty(USER_NAME, user.getFullName());
		}
		else {
			removeProperty(USER_NAME);
		}
		super.setUserId(userId);
	}

	@XmlTransient
	public String getAccountManagerName() {
		String managerId = getAccountManagerId();
		if (StringUtils.isNotBlank(managerId)) {
			if (DataProvider.get().getStore().getId().equals(managerId)) {
				return "Store";
			}

			String accountManagerName = getProperty(ACCOUNT_MANAGER_NAME);
			if (StringUtils.isNotBlank(accountManagerName)) {
				return accountManagerName;
			}

			User user = DataProvider.get().getUserById(managerId, getOutletId());
			if (user != null) {
				return user.getFullName();
			}
		}
		return StringUtils.EMPTY;
	}

	public void setAccountManager(User accountManager) {
		if (accountManager != null) {
			addProperty(ACCOUNT_MANAGER_NAME, accountManager.getFullName());
			super.setAccountManagerId(accountManager.getId());
		}
		else {
			removeProperty(ACCOUNT_MANAGER_NAME);
			super.setAccountManagerId(null);
		}
	}

	@XmlTransient
	public String getUserName() {
		String name = getProperty(USER_NAME);
		if (StringUtils.isNotBlank(name)) {
			return name;
		}
		User user = getUser();
		return user == null ? null : user.getFullName();
	}

	public void setUserName(String userName) {
		//this.userName = userName;
	}

	@XmlTransient
	public String getServerName() {
		String name = getProperty(SERVER_NAME);
		if (StringUtils.isNotBlank(name)) {
			return name;
		}
		User server = getServer();
		return server == null ? null : server.getFullName();
	}

	public void setServerName(String serverName) {
	}

	@XmlTransient
	public User getServer() {
		return DataProvider.get().getUserById(getServerId(), getOutletId());
	}

	/**
	 * @param server : For {@link SalaryTransaction} and {@link ExpenseTransaction} AccountManager is server, otherwise ticket owner is server
	 * 
	 *   @see #putAccountManagerId(String); for Account Manager ID.
	*/
	public void setServer(User server) {
		if (server == null) {
			setServerId(null);
			removeProperty(SERVER_NAME);
		}
		else {
			setServerId(server.getId());
			addProperty(SERVER_NAME, server.getFullName());
		}
	}

	public void setCashDrawer(CashDrawer cashDrawer) {
		this.cashDrawer = cashDrawer;
		if (cashDrawer != null)
			super.setCashDrawerId(cashDrawer.getId());
	}

	@XmlTransient
	public CashDrawer getCashDrawer() {
		if (cashDrawer != null)
			return cashDrawer;
		String cashDrawerId = getCashDrawerId();
		if (StringUtils.isBlank(cashDrawerId)) {
			return null;
		}
		return (CashDrawer) DataProvider.get().getObjectOf(CashDrawer.class, cashDrawerId);
	}

	public boolean isSyncEdited() {
		return syncEdited;
	}

	public void setSyncEdited(boolean syncEdited) {
		this.syncEdited = syncEdited;
	}

	public void setPaymentType(PaymentType paymentType) {
		if (paymentType != null) {
			setPaymentTypeString(paymentType.name());
		}
		else {
			setPaymentTypeString(null);
		}
	}

	@XmlTransient
	public PaymentType getPaymentType() {
		String paymentTypeString = super.getPaymentTypeString();
		if (paymentTypeString == null) {
			return null;
		}
		try {
			return PaymentType.valueOf(paymentTypeString);
		} catch (Exception e) {
			return PaymentType.fromDisplayString(paymentTypeString);
		}
	}

	public double getAmountWithTolerance() {
		return super.getAmount() - super.getToleranceAmount();
	}

	@Override
	public String getExtraProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getExtraProperties();
		if (StringUtils.isBlank(properties)) {
			propertiesContainer = new JsonObject();
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
		return properties;
	}

	@Override
	public void setExtraProperties(String properties) {
		super.setExtraProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, com.google.gson.JsonObject.class);
	}

	public void addExtraProperty(String key, String value) {
		if (StringUtils.isEmpty(value)) {
			return;
		}
		if (propertiesContainer == null) {
			propertiesContainer = new com.google.gson.JsonObject();
		}
		propertiesContainer.addProperty(key, value);
		if (TRANSACTION_FEE == key) {
			setFeeAmount(POSUtil.getDoubleAmount(value));
		}
	}

	public String getExtraProperty(String key) {
		if (propertiesContainer == null) {
			getExtraProperties();
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public String getClassType() {
		if (classType == null) {
			return classType = getClass().getName();
		}
		return classType;
	}

	public void setClassType(String classType) {
		this.classType = classType;
	}

	public double getGiftCardBalanceId() {
		return POSUtil.getDoubleAmount(getProperty(JSON_PROP_GIFT_CARD_NUMBER));
	}

	public void setGiftCardBalanceNumber(String giftCardId) {
		addProperty(JSON_PROP_GIFT_CARD_NUMBER, String.valueOf(giftCardId));
	}

	public double getGiftCardBalanceAmount() {
		return POSUtil.parseDouble(getProperty(JSON_PROP_GIFT_CARD_BALANCE_AMOUNT));
	}

	public void setGiftCardBalanceAmount(double amount) {
		addProperty(JSON_PROP_GIFT_CARD_BALANCE_AMOUNT, String.valueOf(amount));
	}

	public void addGiftCardBalanceAddInfo(String ticketItemId, String giftCardNo, double addedAmount) {
		JSONObject jsonObject = new JSONObject();
		jsonObject.put(JSON_PROP_GIFT_CARD_TICKET_ITEM_ID, ticketItemId);
		jsonObject.put(JSON_PROP_GIFT_CARD_GIFT_CARD_NO, giftCardNo);
		jsonObject.put(JSON_PROP_GIFT_CARD_ADDED_AMOUNT, addedAmount);

		String giftCardInfo = getGiftCardBalanceAddInfo();
		if (StringUtils.isBlank(giftCardInfo)) {
			JSONArray jsonArray = new JSONArray();
			jsonArray.put(jsonObject);
			addProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO, jsonArray.toString());
		}
		else {
			JSONArray jsonArray = new JSONArray(giftCardInfo);
			jsonArray.put(jsonObject);
			addProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO, jsonArray.toString());
		}
	}

	public String getGiftCardBalanceAddInfo() {
		return getProperty(JSON_PROP_GIFT_CARD_GIFT_CARD_BALANCE_ADD_INFO);
	}

	public Double getRefundedAmount() {
		String refundedAmountText = getProperty(JSON_PROP_REFUNDED_AMOUNT);
		if (StringUtils.isNotEmpty(refundedAmountText)) {
			try {
				return NumberUtil.parseDouble(refundedAmountText);
			} catch (Exception e2) {
			}
		}

		return 0.0;
	}

	public Double getRefundableAmount() {
		return getAmount() - getRefundedAmount();
	}

	@XmlTransient
	public Customer getCustomer() {
		String customerId = getCustomerId();
		if (StringUtils.isEmpty(customerId)) {
			return null;
		}
		return DataProvider.get().getCustomer(customerId);
	}

	public void setCustomer(Customer customer) {
		if (customer == null) {
			removeProperty(JSON_PROP_CUSTOMER_NAME);
			return;
		}
		setCustomerId(customer.getId());
		setCustomerName(customer.getName());
	}

	@XmlTransient
	public String getCustomerName() {
		if (getCustomer() != null) {
			return getCustomer().getName();
		}
		if (EntityType.match(getEntityType(), EntityType.PURCHASE)) {
			InventoryVendor vendor = ((PurchaseOrder) DataProvider.get().getObjectOf(PurchaseOrder.class, getEntityId())).getVendor();
			return vendor == null ? StringUtils.EMPTY : vendor.getName();
		}
		return getProperty(JSON_PROP_CUSTOMER_NAME, ""); //$NON-NLS-1$
	}

	public void setCustomerName(String customerName) {
		addProperty(JSON_PROP_CUSTOMER_NAME, customerName);
	}

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	@Override
	public String toString() {
		//return ToStringBuilder.reflectionToString(this);
		return getTransactionType() + ", " + getCardAuthCode();
	}

	public String buildPaymentTypeDisplayName() {
		String paymentTypeString;
		PaymentType paymentType = getPaymentType();
		if (paymentType == null) {
			paymentTypeString = ""; //$NON-NLS-1$
		}

		if (paymentType.equals(PaymentType.CUSTOM_PAYMENT)) {
			String customPaymentName = getCustomPaymentName();
			paymentTypeString = StringUtils.isEmpty(customPaymentName) ? "" : customPaymentName; //$NON-NLS-1$
			String customPaymentRef = getCustomPaymentRef();
			if (StringUtils.isNotBlank(customPaymentRef)) {
				paymentTypeString += " (" + customPaymentRef + ")";
			}
		}
		else if (paymentType.equals(PaymentType.BANK_ACCOUNT)) {
			paymentTypeString = getBankAccountDisplay();
		}
		else {
			paymentTypeString = paymentType.getDisplayString();
		}
		return paymentTypeString;
	}

	public String getBankAccountDisplay() {
		String bank = getProperty(BANK_ACCOUNT);
		if (StringUtils.isNotBlank(bank)) {
			return bank;
		}
		String bankAccountId = getBankAccountId();
		if (StringUtils.isNotBlank(bankAccountId)) {
			BankAccount bankAccount = (BankAccount) DataProvider.get().getObjectOf(BankAccount.class, bankAccountId);
			if (bankAccount != null) {
				return bankAccount.getDisplayString();
			}
		}
		return bankAccountId;

	}

	public String getDateDisplayString() {
		return DateUtil.formatReportDateWithBrowserTimeOffset(getTransactionTime());
	}

	public void setOrderDateDisplayString(String transTicketIds) {
	}

	public String getOrderDateDisplayString() {
		String transTicketIds = getTransTicketIdsDisplay();
		if (StringUtils.isNotBlank(transTicketIds) && transTicketIds.contains(",")) {
			return "Multiple";
		}
		return DateUtil.formatReportDateWithBrowserTimeOffset(DataProvider.get().getTransactionsTicketDate(transTicketIds, getOutletId()));
	}

	public String getPaymentTypeDisplayString() {
		PaymentType paymentType = getPaymentType();
		if (paymentType == PaymentType.CUSTOM_PAYMENT) {
			String customPaymentName = getCustomPaymentName();
			if (StringUtils.isNotBlank(customPaymentName)) {
				return customPaymentName;
			}

			String customPaymnetId = getCustomPaymentId();
			CustomPayment customPayment = DataProvider.get().getCustomPaymentById(customPaymnetId, getOutletId());
			if (customPayment != null) {
				return customPayment.getName();
			}
		}
		else if (paymentType == PaymentType.BANK_ACCOUNT) {
			return getBankAccountDisplay();
		}
		return paymentType.getDisplayString();
	}

	public String getTransTicketIdsDisplay() {
		String transTicketIds = getTransTicketIds();
		if (StringUtils.isNotBlank(transTicketIds)) {
			transTicketIds = transTicketIds.trim();
			return transTicketIds.replaceAll("'", "");
		}
		return transTicketIds;
	}

	public void setRecepientCustomer(Customer customer) {
		if (customer != null) {
			addProperty("recepient_customer_name", customer.getName());
			setRecepientId(customer.getId());
		}
	}

	public String getRecepientCustomerName() {
		String property = getProperty("recepient_customer_name");
		if (StringUtils.isNotBlank(property)) {
			return property;
		}
		String recepientId = getRecepientId();
		if (StringUtils.isBlank(recepientId)) {
			return "";
		}
		Customer customer = DataProvider.get().getCustomer(recepientId);
		if (customer != null) {
			return customer.getName();
		}
		return "";
	}

	public String getRecepientDisplay() {
		String name = getProperty(PAYOUT_EXPENSE_RECEPIENT_NAME);
		if (StringUtils.isNotBlank(name)) {
			return name;
		}
		PayoutRecepient payoutRecepient = getRecepient();
		if (payoutRecepient != null) {
			return payoutRecepient.getName();
		}

		if (payoutRecepient == null && this instanceof ExpenseTransaction || this instanceof PurchaseTransaction) {
			String vendorName = getProperty(VENDOR_NAME);
			if (StringUtils.isNotBlank(vendorName)) {
				return vendorName;
			}
			String vendorId = getVendorId();
			if (StringUtils.isNotBlank(vendorId)) {
				InventoryVendor vendor = (InventoryVendor) DataProvider.get().getObjectOf(InventoryVendor.class, vendorId);
				return vendor != null ? vendor.getName() : StringUtils.EMPTY;
			}
		}
		if (this instanceof PurchaseRefundTransaction) {
			return getAccountManagerName();
		}

		return StringUtils.EMPTY;

	}

	public void putOriginPurchaseOrderId(String id) {
		addProperty("origin.po.id", id); //$NON-NLS-1$
	}

	public String getOriginPurchaseOrderId() {
		return getProperty("origin.po.id");
	}

	private void putCustomPaymentRef(String customPaymentRefJson) {
		addProperty("custom.payment.ref", customPaymentRefJson); //$NON-NLS-1$
	}

	public List getCustomPaymentRefJsonList() {
		String customPaymentRef = getProperty("custom.payment.ref", "[]"); //$NON-NLS-1$ //$NON-NLS-2$
		List list = new Gson().fromJson(customPaymentRef, List.class);
		if (list.size() > 0) {
			return list;
		}
		String oldCustomPaymentRef = super.getCustomPaymentRef();
		if (StringUtils.isNotBlank(oldCustomPaymentRef)) {
			Map<String, String> map = new HashMap<>();
			map.put("Old.Custom.Payment.Ref", oldCustomPaymentRef);
			list.add(map);
		}
		return list;
	}

	@Override
	public String getCustomPaymentRef() {
		return null;
	}

	@Override
	public void setCustomPaymentRef(String customPaymentRef) {
		if (StringUtils.isBlank(customPaymentRef)) {
			return;
		}
		List customPaymentRefJsonList = getCustomPaymentRefJsonList();
		customPaymentRefJsonList.add(customPaymentRef);
		putCustomPaymentRef(new Gson().toJson(customPaymentRefJsonList));
	}

	public void setCustomPaymentRefDisplay(String customPaymentRefJson) {

	}

	public String getCustomPaymentRefDisplay() {
		List jsonList = getCustomPaymentRefJsonList();
		if (jsonList.isEmpty()) {
			return StringUtils.EMPTY;
		}

		String paymentRefStr = jsonList.toString().replace("\"", StringUtils.EMPTY).replace(",", ", ").replace(":", " : ");

		if (paymentRefStr.startsWith("[{") && paymentRefStr.endsWith("}]")) {
			return paymentRefStr.substring(2, paymentRefStr.length() - 2);
		}
		else if (paymentRefStr.startsWith("[") && paymentRefStr.endsWith("]")) {
			return paymentRefStr.substring(1, paymentRefStr.length() - 1);
		}

		return paymentRefStr;
	}

	public String getEntityNoDisplay() {
		Ticket ticket = getTicket();
		return ticket == null ? getEntityNo() == null ? StringUtils.EMPTY : getEntityNo() : ticket.getId();
	}

	public String getReasonDisplay() {
		//		if (this instanceof ExpenseTransaction || this instanceof PurchaseTransaction || this instanceof PurchaseRefundTransaction) {
		//			if (TransactionType.CREDIT.name().equals(this.getTransactionType())) {
		//				return "Cash in"; //$NON-NLS-1$
		//			}
		//		}

		if (this instanceof SummaryExpenseTransaction) {
			return "";
		}

		String reasoneDiplay = getProperty(PAYOUT_EXPENSE_REASON, ""); //$NON-NLS-1$

		if (StringUtils.isNotBlank(reasoneDiplay)) {
			return reasoneDiplay;
		}
		PayoutReason reason = getReason();
		return reason == null ? "" : reason.getReason(); //$NON-NLS-1$
	}

	public Double getExpenseAmount() {
		if (this instanceof ExpenseTransaction || this instanceof PurchaseTransaction || this instanceof PurchaseRefundTransaction) {
			if (getAmount() == 0) {
				return 0d;
			}
			if (getTransactionType().equals(TransactionType.CREDIT.name())) {
				return getAmount() * (-1);
			}
			return Math.abs(getAmount());
		}
		return 0d;
	}

	public PayoutReason getReason() {
		String payoutReason = getReasonId();
		if (StringUtils.isNotEmpty(payoutReason)) {
			return PayoutReasonDAO.getInstance().get(payoutReason);
		}
		return null;
	}

	public void setReason(PayoutReason payoutReason) {
		if (payoutReason != null) {
			super.setReasonId(payoutReason.getId());
			addProperty(PAYOUT_EXPENSE_REASON, payoutReason.getReason());
		}
		else {
			super.setReasonId(null);
			removeProperty(PAYOUT_EXPENSE_REASON);
		}
	}

	public void setSubReason(PayoutSubReason subReason) {
		if (subReason != null) {
			super.setSubReasonId(subReason.getId());
			addProperty(PAYOUT_EXPENSE_SUB_REASON, subReason.getReason());
		}
		else {
			super.setSubReasonId(null);
			removeProperty(PAYOUT_EXPENSE_SUB_REASON);
		}
	}

	public String getSubReason() {
		String subReason = getProperty(PAYOUT_EXPENSE_SUB_REASON);
		if (StringUtils.isNotBlank(subReason)) {
			return subReason;
		}
		String subReasonId = getSubReasonId();
		if (StringUtils.isNotBlank(subReasonId)) {
			PayoutSubReason payoutReason = (PayoutSubReason) DataProvider.get().getObjectOf(PayoutSubReason.class, subReasonId);
			if (payoutReason != null) {
				return payoutReason.getReason();
			}
		}

		return "";
	}

	public void setVendor(InventoryVendor inventoryVendor) {
		if (inventoryVendor != null) {
			super.setVendorId(inventoryVendor.getId());
			addProperty(VENDOR_NAME, inventoryVendor.getName());
		}
		else {
			removeProperty(VENDOR_NAME);
			super.setVendorId(null);
		}

	}

	public PayoutRecepient getRecepient() {
		String payoutRecepientId = getRecepientId();
		if (StringUtils.isNotEmpty(payoutRecepientId)) {
			return PayoutRecepientDAO.getInstance().get(payoutRecepientId);
		}
		return null;
	}

	public void setRecepient(PayoutRecepient payoutPayoutRecepient) {
		if (payoutPayoutRecepient != null) {
			addProperty(PAYOUT_EXPENSE_RECEPIENT_NAME, payoutPayoutRecepient.getName());
			super.setRecepientId(payoutPayoutRecepient.getId());
		}
		else {
			removeProperty(PAYOUT_EXPENSE_RECEPIENT_NAME);
			super.setRecepientId(null);
		}
	}

	public String getProjectDescription() {
		Ticket ticket = getTicket();
		if (ticket != null) {
			return ticket.getId();
		}

		String note = getNote();
		if (StringUtils.isNotBlank(note)) {
			return note;
		}

		return StringUtils.isBlank(getEntityNo()) ? StringUtils.EMPTY : getEntityNo();
	}

	public void setProject(Project project) {
		this.project = project;
		if (project == null) {
			removeProperty("project_name");
			setProjectId(null);
		}
		else {
			addProperty("project_name", project.getName());
			setProjectId(project.getId());
		}
	}

	public Project getProject() {
		if (this.project != null) {
			return this.project;
		}
		String projectId = getProjectId();
		if (StringUtils.isNotBlank(projectId)) {
			this.project = (Project) DataProvider.get().getObjectOf(Project.class, projectId);
		}

		return this.project;
	}

	public String getProjectNameDisplay() {
		String projectName = getProperty("project_name");
		if (StringUtils.isNotBlank(projectName)) {
			return projectName;
		}
		Project project = getProject();
		if (project != null) {
			return project.getName();
		}
		return StringUtils.EMPTY;
	}

	public String getClassTypeDislay() {
		if (this instanceof CashTransaction || this instanceof CreditCardTransaction || this instanceof DebitCardTransaction
				|| this instanceof CustomPaymentTransaction) {
			return "Sales";
		}

		if (this instanceof RefundTransaction) {
			return "Refund";
		}

		if (this instanceof PurchaseTransaction) {
			return "Purchase";
		}

		if (this instanceof ExpenseTransaction) {
			return "Expense";
		}

		return "";
	}

	public String getAmountDisplay() {
		Double amount = getAmount();
		if (this instanceof CashTransaction || this instanceof CreditCardTransaction || this instanceof DebitCardTransaction
				|| this instanceof CustomPaymentTransaction) {
			return NumberUtil.formatAmount(amount);
		}

		if (this instanceof RefundTransaction || this instanceof PurchaseTransaction || this instanceof ExpenseTransaction) {
			return "(" + NumberUtil.formatAmount(amount) + ")";
		}

		return "";
	}

	public String getEventTimeDisplay() {
		Date eventTime = getEventTime();
		return eventTime == null ? StringUtils.EMPTY : DateUtil.formatReportShortDateWithBrowserTimeOffset(eventTime);
	}

	public String getSalaryPaidTo() {
		return getProperty("user.salary_paid_to", "");
	}

	public void putSalaryPaidTo(String paidTo) {
		addProperty("user.salary_paid_to", paidTo);
	}

	public double getAmountWithoutRoundingAmount() {
		return (getAmount() - getRoundedAmount());
	}

	public void setAmountWithoutRoundingAmount() {
	}

	public BankAccount getLinkedBankAccount() {
		return linkedBankAccount;
	}

	public void setLinkedBankAccount(BankAccount bankAccountTo) {
		this.linkedBankAccount = bankAccountTo;
		if (bankAccountTo != null) {
			putLinkedBankAccountId(bankAccountTo.getId());
		}
	}

	public String getLinkedBankAccountId() {
		return getProperty("linked_bank_account", "");
	}

	public void putLinkedBankAccountId(String linkedBankAccountId) {
		addProperty("linked_bank_account", linkedBankAccountId);
	}

	public String getVoidedByUserId() {
		return getProperty("voidedByUser", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putVoidedByUserId(String voidedByUser) {
		addProperty("voidedByUser", voidedByUser); //$NON-NLS-1$
	}

	public void setVoidedByUser(User user) {
		if (user != null) {
			putVoidedByUserId(user.getId());
			addProperty(VOIDED_BY_USER_NAME, user.getFullName());
		}
		else {
			removeProperty(VOIDED_BY_USER_NAME);
			putVoidedByUserId(null);
		}

	}

	public String getVoidedByUser() {
		String userName = getProperty(VOIDED_BY_USER_NAME);
		if (StringUtils.isNotBlank(userName)) {
			return userName;
		}
		String voidedByUserId = getVoidedByUserId();
		if (StringUtils.isNotBlank(voidedByUserId)) {
			User user = DataProvider.get().getUserById(voidedByUserId, DataProvider.get().getCurrentOutletId());
			return user.getFullName();
		}
		else {
			return StringUtils.EMPTY;
		}
	}

	public String getBalanceUpdateTransId() {
		return getProperty("balanceUpdateTransId", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putBalanceUpdateTransId(String balanceUpdateTransId) {
		addProperty("balanceUpdateTransId", balanceUpdateTransId); //$NON-NLS-1$
	}

	public void setBankAccount(BankAccount bankAccount) {
		if (bankAccount != null) {
			setBankAccountId(bankAccount.getId());
			addProperty(BANK_ACCOUNT, bankAccount.getDisplayString());
		}
		else {
			setBankAccountId(null);
			removeProperty(BANK_ACCOUNT);
		}
	}

	public String getType() {
		if (this instanceof SalaryTransaction) {
			return "Salary"; //$NON-NLS-1$
		}
		else if (this instanceof SalaryAdvanceTransaction) {
			return "Loan/Advance"; //$NON-NLS-1$
		}
		else if (this instanceof ExpenseTransaction) {
			return "Expense"; //$NON-NLS-1$
		}
		return StringUtils.EMPTY;
	}

	public String getRevertReason() {
		return getProperty("revert_reason", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRevertReason(String revertReason) {
		addProperty("revert_reason", revertReason); //$NON-NLS-1$
	}

	public boolean isSSLTransaction() {
		if (hasProperty("sslREsponseGson")) {
			return true;
		}
		return false;
	}

	public String getSSLResponse() {
		return getProperty("sslREsponseGson", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putSSLResponse(String sslREsponseGson) {
		addProperty("sslREsponseGson", sslREsponseGson); //$NON-NLS-1$
	}

	public String getVendorDisplay() {
		String name = getProperty(VENDOR_NAME);
		return StringUtils.isBlank(name) ? StringUtils.EMPTY : name;
	}

	public boolean isServiceOrderTransaction() {
		return POSUtil.getBoolean(getProperty("isServiceOrderTransaction"), false); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putServiceOrderTransaction(boolean isServiceOrderTransaction) {
		addProperty("isServiceOrderTransaction", String.valueOf(isServiceOrderTransaction)); //$NON-NLS-1$
	}

	public void putAdvanceTransactionTicketId(String ticketId) {
		addProperty("ticketId", ticketId); //$NON-NLS-1$
	}

	public String getAdvanceTransactionTicketId() {
		return getProperty("ticketId"); //$NON-NLS-1$
	}

	public void setChartOfAccounts(ChartOfAccounts chartOfAccounts) {
		if (chartOfAccounts == null) {
			setCoaId(null);
			setCoaCode(null);
		}
		else {
			setCoaId(chartOfAccounts.getId());
			setCoaCode(chartOfAccounts.getAccountCode());
		}
	}

	public String getChartOfAccountsName() {
		String coaCode = getCoaCode();
		if (StringUtils.isNotBlank(coaCode)) {
			ChartOfAccounts coaAccount = DataProvider.get().getCOAFromMap(coaCode);
			return coaAccount.getNameDisplay();
		}
		return "";
	}
	
}