package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;

import com.floreantpos.model.BalanceUpdateTransaction;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.ChartOfAccounts;
import com.floreantpos.model.CustomPayment;
import com.floreantpos.model.DateTypeOption;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PayoutReason;
import com.floreantpos.model.PayoutRecepient;
import com.floreantpos.model.PayoutSubReason;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Project;
import com.floreantpos.model.Store;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.POSUtil;

public class ExpenseTransactionDAO extends BaseExpenseTransactionDAO {

	public ExpenseTransactionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public ExpenseTransaction saveExpensesTransaction(User currentUser, Date eventTime, User accountsManager, String batchNo, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note,
			BalanceUpdateTransaction balanceUpdateTransaction, Session session) throws Exception {
		return saveExpensesTransaction(currentUser, eventTime, accountsManager, batchNo, project, recepient, reason, subReason, expensesAmount, note,
				balanceUpdateTransaction, null, session);
	}

	public ExpenseTransaction saveExpensesTransaction(User currentUser, Date eventTime, User accountsManager, String batchNo, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note,
			BalanceUpdateTransaction balanceUpdateTransaction, ChartOfAccounts chartOfAccounts, Session session) throws Exception {

		return saveExpensesTransaction(currentUser, eventTime, accountsManager, batchNo, project, recepient, reason, subReason, expensesAmount, note,
				balanceUpdateTransaction, chartOfAccounts, session, null, "", null);
	}

	public ExpenseTransaction saveExpensesTransaction(User currentUser, Date eventTime, User accountsManager, String batchNo, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note,
			BalanceUpdateTransaction balanceUpdateTransaction, ChartOfAccounts chartOfAccounts, Session session, Object paymentMethod, String paymentRef,
			BankAccount linkedBankAccount) throws Exception {

		ExpenseTransaction expenseTransaction = null;
		if (paymentMethod != null) {
			expenseTransaction = createTransactionForByz(paymentMethod, paymentRef, TransactionType.OUT, expensesAmount);
			expenseTransaction.setSourceType(paymentMethod.toString());
			if (linkedBankAccount != null) {
				expenseTransaction.setBankAccountId(linkedBankAccount.getId());
			}

		}
		else {
			expenseTransaction = createTransaction(PaymentType.CASH, TransactionType.OUT, expensesAmount);
			expenseTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
		}
		expenseTransaction.setEventTime(eventTime);
		expenseTransaction.setBatchNo(batchNo);
		expenseTransaction.setRecepient(recepient);
		expenseTransaction.setReason(reason);
		expenseTransaction.setMemoNo(balanceUpdateTransaction.getMemoNo()); //$NON-NLS-1$
		expenseTransaction.addProperty("memo_image_id", balanceUpdateTransaction.getMemoImageId()); //$NON-NLS-1$
		expenseTransaction.setNote(note);
		expenseTransaction.setUser(currentUser);
		// set AccountNo to Server
		expenseTransaction.setServer(accountsManager);
		//		expenseTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_ACM);
		expenseTransaction.setAccountManager(accountsManager == null ? null : accountsManager);
		expenseTransaction.setProject(project);
		expenseTransaction.setSubReason(subReason);

		expenseTransaction.setAccountProcessed(true);
		expenseTransaction.putBalanceUpdateTransId(balanceUpdateTransaction.getId());
		expenseTransaction.setChartOfAccounts(chartOfAccounts);

		save(expenseTransaction, session);
		return expenseTransaction;
	}

	public PaymentType initPaymentMethod(Object paymentMethod, String paymentRef, PosTransaction posTransaction) {
		PaymentType paymentType = null;
		if (paymentMethod != null) {
			if (paymentMethod instanceof String) {
				paymentType = PaymentType.CASH;
			}
			if (paymentMethod instanceof CustomPayment) {
				paymentType = PaymentType.CUSTOM_PAYMENT;

				CustomPayment customPayment = (CustomPayment) paymentMethod;
				posTransaction.setCustomPaymentId(customPayment.getId());
				posTransaction.setCustomPaymentName(customPayment.getName());
				posTransaction.setCustomPaymentRef(paymentRef);
			}
			if (paymentMethod instanceof BankAccount) {
				paymentType = PaymentType.BANK_ACCOUNT;

				BankAccount bankAccount = (BankAccount) paymentMethod;
				posTransaction.setBankAccount(bankAccount);
			}
		}
		else {
			paymentType = PaymentType.CASH;
		}
		return paymentType;
	}

	public ExpenseTransaction saveStoreExpensesTransaction(User currentUser, Date eventTime, Store store, String batchNo, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note,
			BalanceUpdateTransaction balanceUpdateTransaction, Session session) throws Exception {
		return saveStoreExpensesTransaction(currentUser, eventTime, store, batchNo, project, recepient, reason, subReason, expensesAmount, note,
				balanceUpdateTransaction, null, null, session);
	}

	public ExpenseTransaction saveStoreExpensesTransaction(User currentUser, Date eventTime, Store store, String batchNo, Project project,
			PayoutRecepient recepient, PayoutReason reason, PayoutSubReason subReason, double expensesAmount, String note,
			BalanceUpdateTransaction balanceUpdateTransaction, String memoNo, ChartOfAccounts chartOfAccounts, Session session) throws Exception {

		ExpenseTransaction expenseTransaction = createTransaction(PaymentType.CASH, TransactionType.OUT, expensesAmount);
		expenseTransaction.setEventTime(eventTime);
		expenseTransaction.setBatchNo(batchNo);
		expenseTransaction.setRecepient(recepient);
		expenseTransaction.setReason(reason);
		expenseTransaction.setNote(note);
		expenseTransaction.setUser(currentUser);
		// set AccountNo to Server
		expenseTransaction.setServerId(store.getId());
		expenseTransaction.setAccountManagerId(store == null ? null : store.getId());

		expenseTransaction.setProject(project);
		expenseTransaction.setSubReason(subReason);

		expenseTransaction.setAccountProcessed(true);
		expenseTransaction.setSourceType(ExpenseTransaction.EXPENSE_FROM_STORE);
		expenseTransaction.putBalanceUpdateTransId(balanceUpdateTransaction.getId());
		expenseTransaction.setChartOfAccounts(chartOfAccounts);

		if (StringUtils.isNotBlank(memoNo)) {
			expenseTransaction.setMemoNo(memoNo);
		}

		save(expenseTransaction, session);
		return expenseTransaction;

	}

	public ExpenseTransaction createTransaction(PaymentType paymentType, TransactionType transactionType, double amount) {
		ExpenseTransaction expenseTransaction = new ExpenseTransaction();
		expenseTransaction.setPaymentType(paymentType);
		expenseTransaction.setTransactionType(transactionType.name());
		Date now = DataProvider.get().getServerTimestamp();
		expenseTransaction.setTransactionTime(now);
		expenseTransaction.setAmount(transactionType == TransactionType.DEBIT ? -Math.abs(amount) : Math.abs(amount));
		expenseTransaction.setOutletId(DataProvider.get().getOutlet().getId());
		return expenseTransaction;
	}

	public ExpenseTransaction createTransactionForByz(Object paymentMethod, String paymentRef, TransactionType transactionType, double amount) {
		ExpenseTransaction expenseTransaction = new ExpenseTransaction();

		PaymentType paymentType = initPaymentMethod(paymentMethod, paymentRef, expenseTransaction);

		expenseTransaction.setPaymentType(paymentType);
		expenseTransaction.setTransactionType(transactionType.name());
		Date now = DataProvider.get().getServerTimestamp();
		expenseTransaction.setTransactionTime(now);
		expenseTransaction.setAmount(transactionType == TransactionType.DEBIT ? -Math.abs(amount) : Math.abs(amount));
		expenseTransaction.setOutletId(DataProvider.get().getOutlet().getId());
		return expenseTransaction;
	}

	public void loadAllExpenses(PaginationSupport model, Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo, User user,
			String outletId) {
		loadAllExpenses(model, fromDate, toDate, accountNo, batchNo, memoNo, user, outletId, DateTypeOption.CREATE_DATE);
	}

	public void loadAllExpenses(PaginationSupport model, Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo, User user,
			String outletId, DateTypeOption dateTypeOption) {
		try (Session session = createNewSession()) {
			Criteria criteria = getCriteriaForExpense(dateTypeOption, fromDate, toDate, accountNo, batchNo, memoNo, user, outletId, session);

			model.setNumRows(rowCount(criteria));

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			criteria.addOrder(Order.desc(ExpenseTransaction.PROP_TRANSACTION_TIME));
			model.setRows(criteria.list());
		}
	}

	private Criteria getCriteriaForExpense(Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo, User user, String outletId,
			Session session) {
		return getCriteriaForExpense(DateTypeOption.CREATE_DATE, fromDate, toDate, accountNo, batchNo, memoNo, user, outletId, session);
	}

	private Criteria getCriteriaForExpense(DateTypeOption dateTypeOption, Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo,
			User user, String outletId, Session session) {
		Criteria criteria = session.createCriteria(ExpenseTransaction.class);

		criteria.add(Restrictions.eq(ExpenseTransaction.PROP_VOIDED, false));

		if (StringUtils.isNotBlank(batchNo)) {
			criteria.add(Restrictions.eq(ExpenseTransaction.PROP_BATCH_NO, batchNo));
		}

		if (StringUtils.isNotBlank(memoNo)) {
			criteria.add(Restrictions.eq(ExpenseTransaction.PROP_MEMO_NO, memoNo));
		}

		if (fromDate != null && toDate != null) {
			criteria.add(
					Restrictions.between(dateTypeOption != null ? dateTypeOption.getPropString() : PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
		}

		if (StringUtils.isNotBlank(accountNo)) {
			criteria.add(Restrictions.eq(ExpenseTransaction.PROP_ACCOUNT_MANAGER_ID, accountNo));
		}

		if (user != null) {
			criteria.add(Restrictions.eq(ExpenseTransaction.PROP_USER_ID, user.getId()));
		}

		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(ExpenseTransaction.PROP_OUTLET_ID, outletId));
		}

		return criteria;
	}

	public double calculateTotalCashExpense(Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo) {
		return calculateTotalCashExpense(fromDate, toDate, accountNo, batchNo, memoNo, null);
	}

	public double calculateTotalCashExpense(Date fromDate, Date toDate, String accountNo, String batchNo, String memoNo, String outletId) {
		try (Session session = createNewSession()) {
			Criteria criteria = getCriteriaForExpense(fromDate, toDate, accountNo, batchNo, memoNo, null, outletId, session);

			criteria.setProjection(Projections.sum(ExpenseTransaction.PROP_AMOUNT));
			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}
	}

	public List<ExpenseTransaction> getExpenseTransactionsByIds(List<String> ids) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(ExpenseTransaction.class);
			criteria.add(Restrictions.in(ExpenseTransaction.PROP_ID, ids));
			return criteria.list();
		}
	}

}