/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.time.LocalDate;
import java.time.Period;
import java.time.ZoneId;
import java.util.Date;

import javax.xml.bind.annotation.XmlRootElement;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.model.base.BaseInventoryVendor;
import com.floreantpos.model.dao.StoreDAO;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

@XmlRootElement
public class InventoryVendor extends BaseInventoryVendor implements IdContainer, TimedModel {
	private static final long serialVersionUID = 1L;
	private transient JsonObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public InventoryVendor() {
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for primary key
	 */
	public InventoryVendor(java.lang.String id) {
		super(id);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/**
	 * Constructor for required fields
	 */
	public InventoryVendor(java.lang.String id, java.lang.String name) {

		super(id, name);
		setCreateDate(StoreDAO.getServerTimestamp());
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public Boolean isVisible() {
		return super.isVisible() == null ? true : super.isVisible();
	}

	@Override
	public String toString() {
		return getName();
	}

	@Override
	public String getProperties() {
		if (propertiesContainer != null) {
			return propertiesContainer.toString();
		}

		String properties = super.getProperties();
		if (StringUtils.isEmpty(properties)) {
			return null;
		}

		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
		return properties;
	}

	@Override
	public void setProperties(String properties) {
		super.setProperties(properties);
		propertiesContainer = new Gson().fromJson(properties, JsonObject.class);
	}

	public void addProperty(String key, String value) {
		if (propertiesContainer == null) {
			propertiesContainer = new JsonObject();
		}
		propertiesContainer.addProperty(key, value);
	}

	public String getProperty(String key) {
		if (propertiesContainer == null) {
			return null;
		}
		if (propertiesContainer.has(key)) {
			JsonElement jsonElement = propertiesContainer.get(key);
			if (!jsonElement.isJsonNull()) {
				return jsonElement.getAsString();
			}
		}
		return null;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void removeProperty(String propertyName) {
		if (propertiesContainer != null) {
			propertiesContainer.remove(propertyName);
		}
	}

	public String getApproxAge() {
		Date dateOfBirth = getDateOfBirth();
		if (dateOfBirth != null) {
			LocalDate birthdate = dateOfBirth.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
			LocalDate currentDate = LocalDate.now();
			Period agePeriod = Period.between(birthdate, currentDate);
			int years = agePeriod.getYears();
			int months = agePeriod.getMonths();
			int days = agePeriod.getDays();
			if (years > 0) {
				return String.format("%d Y", years);
			}
			if (months > 0) {
				return String.format("%d M", months);
			}

			return String.format("%d days", days);
		}
		return null;
	}

	public String getOccupation() {
		String property = getProperty("occupation");
		if (property == null) {
			return StringUtils.EMPTY;
		}
		return property;
	}

	public void putOccupation(String value) {
		addProperty("occupation", value);
	}

	@Override
	public String getName() {
		String fName = getFirstName();
		String lName = getLastName();
		String name = super.getName();
		if (StringUtils.isNotBlank(fName)) {
			name = fName;
		}
		if (StringUtils.isNotBlank(lName)) {
			name += " " + lName;
		}
		return name;
	}

	@Override
	public String getVendorType() {
		String vendorType = super.getVendorType();
		return StringUtils.isBlank(vendorType) ? VendorType.BUSINESS.name() : vendorType;
	}

	public enum VendorType {
		BUSINESS("Business"), BLOOD_DONOR("Blood donor");

		private final String displayName;

		VendorType(String displayName) {
			this.displayName = displayName;
		}

		public String getNameDisplay() {
			return displayName;
		}

		public static VendorType fromString(String name) {
			if (StringUtils.isBlank(name)) {
				return BUSINESS;
			}

			VendorType[] values = values();
			for (VendorType status : values) {
				if (name.equals(status.name())) {
					return status;
				}
			}

			return BUSINESS;
		}
	}
}