package com.floreantpos.model.util;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.model.ComboGroup;
import com.floreantpos.model.ComboItem;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.MenuItemModifierSpec;
import com.floreantpos.model.MenuModifier;
import com.floreantpos.model.ModifierGroup;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketItemModifier;
import com.floreantpos.model.dao.MenuItemDAO;
import com.floreantpos.model.dao.MenuItemModifierSpecDAO;
import com.floreantpos.model.dao.MenuModifierDAO;
import com.floreantpos.model.dao.OnlineItemDAO;
import com.floreantpos.util.POSUtil;

public class TicketItemUtil {

	public static void validateTicketItem(TicketItem ticketItem, Ticket oldTicket) {
		validateTicketItem(ticketItem, oldTicket, Messages.getString("IS_NOT_AVAILABLE_FOR_REORDER")); //$NON-NLS-1$
	}

	public static void validateTicketItem(TicketItem ticketItem, Ticket oldTicket, String validationMessage) {

		MenuItem menuItem = MenuItemDAO.getInstance().get(ticketItem.getMenuItemId());
		isItemQuantityAvailable(ticketItem, oldTicket, menuItem, ticketItem.getTicket().getOutletId(), 0);

		if (menuItem.isHasVariant()) {
			throw new PosException(ticketItem.getName() + validationMessage);
		}

		if (!ticketItem.isComboItem().equals(menuItem.isComboItem())) {
			throw new PosException(ticketItem.getName() + validationMessage);
		}

		verifyModifiers(ticketItem, validationMessage);
		verifyComboItem(ticketItem, validationMessage);
	}

	private static void verifyModifiers(TicketItem originalTicketItem, String validationMessage) {
		List<TicketItemModifier> ticketItemModifiers = originalTicketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				String menuModifierId = ticketItemModifier.getItemId();
				if (!ticketItemModifier.isInfoOnly()) {
					MenuModifier ticketItemMenuModifier = MenuModifierDAO.getInstance().get(menuModifierId);
					// if detach from group	
					String modifierSpecId = ticketItemModifier.getGroupId();
					MenuItemModifierSpec modifierSpec = MenuItemModifierSpecDAO.getInstance().get(modifierSpecId);
					if (modifierSpec != null) {
						if (modifierSpec.isUseModifierGroupSettings()) {
							ModifierGroup modifierGroup = modifierSpec.getModifierGroup();

							List<MenuModifier> modifiers = modifierGroup.getModifiers();
							checkModifiers(ticketItemMenuModifier, modifiers, ticketItemModifier.getName(), validationMessage);
						}
						else {
							MenuItemModifierSpecDAO.getInstance().initialize(modifierSpec);
							List<MenuModifier> modifiers = new ArrayList<>(modifierSpec.getModifiers());
							checkModifiers(ticketItemMenuModifier, modifiers, ticketItemModifier.getName(), validationMessage);
						}
					}
					else {
						//modifier spec has been deleted.
						throw new PosException(ticketItemModifier.getName() + validationMessage);
					}

					//if deleted 
					if (ticketItemMenuModifier != null && ticketItemMenuModifier.isDeleted()) {
						throw new PosException(ticketItemMenuModifier.getName() + validationMessage);
					}
				}
			}
		}
	}

	private static void checkModifiers(MenuModifier ticketItemMenuModifier, List<MenuModifier> modifiers, String ticketItemModifierName,
			String validationMessage) {
		if (modifiers != null) {
			List<String> menuModifierIds = POSUtil.getStringIds(modifiers, MenuModifier.class);
			if (!menuModifierIds.contains(ticketItemMenuModifier.getId()) || !ticketItemMenuModifier.isEnableOnlineOrdering()
					|| !ticketItemMenuModifier.isEnable()) {
				throw new PosException(ticketItemModifierName + validationMessage);
			}
		}
	}

	private static void verifyComboItem(TicketItem comboTicketItem, String validationMessage) {
		if (!comboTicketItem.isComboItem()) {
			return;
		}

		MenuItem comboMenuItem = MenuItemDAO.getInstance().loadInitialized(comboTicketItem.getMenuItemId());
		List<String> currentMenuItemKeyIds = new ArrayList<String>();
		List<ComboItem> recentComboItems = comboMenuItem.getComboItems();
		if (recentComboItems != null) {
			for (ComboItem comboItem : recentComboItems) {
				currentMenuItemKeyIds.add(comboItem.getItemId());
			}
		}

		List<ComboGroup> comboGroups = comboMenuItem.getComboGroups();
		if (comboGroups != null) {
			for (ComboGroup comboGroup : comboGroups) {
				List<MenuItem> comboGroupItems = comboGroup.getItems();
				if (comboGroupItems != null) {
					for (MenuItem comboGroupItem : comboGroupItems) {
						currentMenuItemKeyIds.add(comboGroupItem.getId() + "_" + comboGroup.getId()); //$NON-NLS-1$
					}
				}
			}
		}

		List<TicketItem> comboItems = comboTicketItem.getComboItems();
		if (comboItems != null) {
			for (TicketItem ticketItem : comboItems) {

				MenuItem menuItem = MenuItemDAO.getInstance().loadInitialized(ticketItem.getMenuItemId());
				isItemQuantityAvailable(comboTicketItem, menuItem, comboTicketItem.getTicket().getOutletId(), 0);

				String key = ticketItem.getMenuItemId();
				if (StringUtils.isNotBlank(ticketItem.getGroupId())) {
					key += "_" + ticketItem.getGroupId(); //$NON-NLS-1$
				}
				if (!currentMenuItemKeyIds.contains(key)) {
					throw new PosException(ticketItem.getName() + validationMessage);
				}

				verifyModifiers(ticketItem, validationMessage);
			}
		}
	}

	public static void updateTicketItem(TicketItem ticketItem) {

		updateModifiers(ticketItem);

		if (ticketItem.isComboItem()) {
			MenuItem updatedComboItem = MenuItemDAO.getInstance().get(ticketItem.getMenuItemId());
			ticketItem.setName(updatedComboItem.getName());
			List<TicketItem> comboItems = ticketItem.getComboItems();
			if (comboItems != null) {
				for (TicketItem comboTicketItem : comboItems) {
					MenuItem updatedMenuItem = MenuItemDAO.getInstance().get(comboTicketItem.getMenuItemId());
					comboTicketItem.setName(updatedMenuItem.getName());

					updateModifiers(comboTicketItem);
				}
			}
		}
	}

	private static void updateModifiers(TicketItem ticketItem) {

		List<TicketItemModifier> ticketItemModifiers = ticketItem.getTicketItemModifiers();
		if (ticketItemModifiers != null && ticketItemModifiers.size() > 0) {
			for (TicketItemModifier ticketItemModifier : ticketItemModifiers) {
				String menuModifierId = ticketItemModifier.getItemId();
				String itemModifierName = ticketItemModifier.getName();
				if (!ticketItemModifier.isInfoOnly()) {
					MenuModifier ticketItemMenuModifier = MenuModifierDAO.getInstance().get(menuModifierId);
					itemModifierName = ticketItemMenuModifier.getDisplayName();
				}

				ticketItemModifier.setName(itemModifierName);
			}
		}
	}

	private static void isItemQuantityAvailable(TicketItem ticketItem, MenuItem menuItem, String outletId, double selectedQty) {
		isItemQuantityAvailable(ticketItem, ticketItem.getTicket(), menuItem, outletId, selectedQty);
	}

	private static void isItemQuantityAvailable(TicketItem ticketItem, Ticket oldTicket, MenuItem menuItem, String outletId, double selectedQty) {
		if (menuItem.isDeleted() || !menuItem.isVisible()) {
			throw new PosException(menuItem.getName() + Messages.getString("TicketItemUtil.0")); //$NON-NLS-1$
		}
		if (menuItem.isDisableWhenStockAmountIsZero() && !TicketUtil.hasAvailableStock(ticketItem, oldTicket, menuItem, selectedQty)) {
			throw new PosException(menuItem.getName() + Messages.getString("TicketItemUtil.1")); //$NON-NLS-1$
		}

		boolean isEnableOnlineOrdering = Boolean.FALSE;
		if (menuItem.isVariant()) {
			if (menuItem.getParentMenuItem() != null) {
				isEnableOnlineOrdering = OnlineItemDAO.getInstance().isEnabledOnlineOrdering(menuItem.getParentMenuItem(), new Outlet(outletId));
			}
		}
		else {
			isEnableOnlineOrdering = OnlineItemDAO.getInstance().isEnabledOnlineOrdering(menuItem, new Outlet(outletId));
		}
		if (!isEnableOnlineOrdering) {
			throw new PosException(menuItem.getName() + Messages.getString("TicketItemUtil.0")); //$NON-NLS-1$
		}
	}
}
