/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang.RandomStringUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Conjunction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.LogicalExpression;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;
import org.hibernate.criterion.Subqueries;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;
import org.json.JSONArray;
import org.json.JSONObject;

import com.floreantpos.Messages;
import com.floreantpos.POSConstants;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.config.CardConfig;
import com.floreantpos.config.TerminalConfig;
import com.floreantpos.extension.PaymentGatewayPlugin;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.AdvanceAdjustmentTransaction;
import com.floreantpos.model.Agent;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.Appointment;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.BloodInventoryTransaction;
import com.floreantpos.model.BookingInfo;
import com.floreantpos.model.CardReader;
import com.floreantpos.model.ComboTicketItem;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.Doctor;
import com.floreantpos.model.GiftCard;
import com.floreantpos.model.Gratuity;
import com.floreantpos.model.IUnit;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryStock;
import com.floreantpos.model.InventoryStockUnit;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.LdfPayTransaction;
import com.floreantpos.model.LedgerEntry;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.OutdoorTicket;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.PackagingUnit;
import com.floreantpos.model.Patient;
import com.floreantpos.model.PatientBookingStatus;
import com.floreantpos.model.PaymentStatusFilter;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.ProductType;
import com.floreantpos.model.Project;
import com.floreantpos.model.Recepie;
import com.floreantpos.model.RfPayTransaction;
import com.floreantpos.model.SharedCalcItemModel;
import com.floreantpos.model.Shift;
import com.floreantpos.model.ShipmentStatus;
import com.floreantpos.model.Store;
import com.floreantpos.model.StoreSession;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketPaymentAllocation;
import com.floreantpos.model.TicketSource;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.UnitType;
import com.floreantpos.model.User;
import com.floreantpos.model.UserPermission;
import com.floreantpos.model.UserType;
import com.floreantpos.model.VoidItem;
import com.floreantpos.model.ext.InvMapKey;
import com.floreantpos.model.ext.OutdoorTicketStatus;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.TransactionUserNullException;
import com.floreantpos.payment.PaymentPlugin;
import com.floreantpos.report.DeliverySummaryReportData;
import com.floreantpos.report.DueTicketReportData;
import com.floreantpos.report.EndOfDayReportData;
import com.floreantpos.report.ReferralCommissionReportData;
import com.floreantpos.report.SalesDetailsReportData;
import com.floreantpos.services.PosTransactionService;
import com.floreantpos.swing.PaginatedTableModel;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.ui.views.payment.CardProcessor;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.TicketIdGenerator;
import com.floreantpos.util.TokenNumberGenerator;
import com.google.gson.Gson;

public class TicketDAO extends BaseTicketDAO {
	private final static TicketDAO instance = new TicketDAO();

	/**
	 * Default constructor. Can be used in place of getInstance()
	 */
	public TicketDAO() {
	}

	@Override
	public Order getDefaultOrder() {
		return Order.desc(Ticket.PROP_CREATE_DATE);
	}

	@Override
	protected Serializable save(Object obj, Session session) {
		Ticket ticket = (Ticket) obj;

		performPreSaveOperations(ticket);
		Serializable save = super.save(obj, session);
		performPostSaveOperations(session, ticket);

		return save;
	}

	@Override
	protected void update(Object obj, Session session) {
		Ticket ticket = (Ticket) obj;

		performPreSaveOperations(ticket);
		super.update(obj, session);
		performPostSaveOperations(session, ticket);
	}

	@Override
	public synchronized void saveOrUpdate(Ticket ticket) {
		performPreSaveOperations(ticket);

		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			saveOrUpdate(ticket, session);

			tx.commit();
		}
	}

	public synchronized void saveOrUpdateTicketAndLedgerEntry(Ticket ticket) {
		performPreSaveOperations(ticket);

		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			saveOrUpdate(ticket, session);

			LedgerEntryDAO.getInstance().saveOrderLedgerEntry(ticket, session);
			saveOrUpdateBookingInfoIfNeeded(ticket, session);
			tx.commit();
		}
	}

	public void saveOrUpdateBookingInfoIfNeeded(Ticket ticket, Session session) {
		if (ticket.getTicketType() == null || ticket.getTicketType() != TicketType.IPD || StringUtils.isBlank(ticket.getAdmissionId())) {
			return;
		}
		String ticketDoctorId = ticket.getDoctorId();
		if (StringUtils.isBlank(ticketDoctorId)) {
			return;
		}
		BookingInfo booking = BookingInfoDAO.getInstance().findBooking(ticket.getAdmissionId(), session);
		if (booking == null) {
			return;
		}
		String bookingDoctorId = booking.getDoctorId();
		if (StringUtils.isBlank(bookingDoctorId) || !bookingDoctorId.equals(ticketDoctorId)) {
			Doctor doctor = DoctorDAO.getInstance().get(ticketDoctorId);
			if (doctor != null) {
				booking.setDoctor(doctor);
			}
			BookingInfoDAO.getInstance().saveOrUpdate(booking, session);
		}
	}

	public synchronized void saveOrUpdateTicketAndProject(Ticket ticket, Project project) {
		performPreSaveOperations(ticket);

		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			if (project != null) {
				if (StringUtils.isNotBlank(project.getName()) || StringUtils.isNotBlank(project.getNote())) {

					if (StringUtils.isBlank(project.getId())) {
						ProjectDAO.getInstance().save(project, session);
					}
					else {
						ProjectDAO.getInstance().update(project, session);
					}
				}
				ticket.setProject(project);
			}

			saveOrUpdate(ticket, session);

			if (project != null && StringUtils.isBlank(project.getName())) {
				project.setName(ticket.getId());
				session.saveOrUpdate(project);
			}

			tx.commit();
		}
	}

	@Override
	public void saveOrUpdate(final Ticket ticket, Session session) {
		saveOrUpdate(ticket, session, true);
	}

	public void saveOrUpdate(final Ticket ticket, Session session, boolean updateTableStatus) {
		updateTime(ticket);
		boolean newTicket = StringUtils.isEmpty(ticket.getId()) ? true : isNewTicket(ticket.getId());
		if (newTicket) {
			if (StringUtils.isEmpty(ticket.getId())) {
				ticket.setId(TicketIdGenerator.generateTicketId(ticket));
			}
			if (StringUtils.isEmpty(ticket.getShortId())) {
				ticket.setShortId(RandomStringUtils.randomNumeric(7));
			}
			if (ticket.getTokenNo() == 0) {
				ticket.setTokenNo(TokenNumberGenerator.generateTokenNo());
			}
		}

		ticket.setActiveDate(StoreDAO.getServerTimestamp());

		boolean isNewTicketItemAdded = ticket.getTicketItems().stream().filter(t -> StringUtils.isBlank(t.getId())).findFirst().isPresent();
		if (ticket.isPaid() && ticket.getDueAmount() > 0 || isNewTicketItemAdded) {
			ticket.setPaid(false);
			ticket.setAccountProcessed(false);
			ticket.setClosed(false);
		}
		else if (ticket.getTicketType() != TicketType.IPD && NumberUtil.isZero(ticket.getDueAmount()) && NumberUtil.isZero(ticket.getReferrerFeeDueAmount())
				&& NumberUtil.isZero(ticket.getLabDoctorFeeDueAmount())) {
			ticket.setClosed(true);
		}

		if (newTicket) {
			save(ticket, session);
		}
		else {
			update(ticket, session);
		}
	}

	@Override
	protected void delete(Object obj, Session s) {
		super.delete(obj, s);
	}

	public void saveOrUpdateTemporaryTicket(Ticket ticket) {
		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			ticket.setActiveDate(StoreDAO.getServerTimestamp());
			boolean newTicket = StringUtils.isEmpty(ticket.getId()) ? true : isNewTicket(ticket.getId());
			if (newTicket) {
				if (StringUtils.isEmpty(ticket.getId())) {
					ticket.setId(TicketIdGenerator.generateTicketId(ticket));
				}
				if (StringUtils.isEmpty(ticket.getShortId())) {
					ticket.setShortId(RandomStringUtils.randomNumeric(7));
				}
				if (ticket.getTokenNo() == 0) {
					ticket.setTokenNo(TokenNumberGenerator.generateTokenNo());
				}
				updateTime(ticket);
				session.save(ticket);
			}
			else {
				updateTime(ticket);
				session.update(ticket);
			}

			tx.commit();
		}
	}

	private void performPreSaveOperations(Ticket ticket) {
		//ticket create date
		Date currentTime = StoreDAO.getServerTimestamp();
		if (ticket.getCreateDate() == null) {
			ticket.setCreateDate(currentTime);
		}

		List<TicketItem> ticketItems = ticket.getTicketItems();
		boolean hasNewItemAdded = false;
		String ticketId = ticket.getId();
		String outletId = ticket.getOutletId();
		if (ticketItems != null) {
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem == null) {
					throw new PosException("Ticket item is null for ticket " + ticketId);
				}

				if (ticketItem.getCreateDate() == null) {
					ticketItem.setCreateDate(currentTime);
				}

				if (ticketItem.getId() == null && !ticketItem.isVoided() && !ticketItem.isReturned() && !ticketItem.isDeleted()) {
					hasNewItemAdded = true;
				}

				if (StringUtils.isBlank(ticketItem.getParentTicketId())) {
					ticketItem.setParentTicketId(ticketId);
				}
				if (StringUtils.isBlank(ticketItem.getParentTicketOutletId())) {
					ticketItem.setParentTicketOutletId(outletId);
				}

				if (ticketItem instanceof ComboTicketItem) {
					ComboTicketItem comboTicketItem = (ComboTicketItem) ticketItem;
					List<TicketItem> comboItems = comboTicketItem.getComboItems();
					if (comboItems != null && !comboItems.isEmpty()) {
						for (TicketItem comboChildItem : comboItems) {
							if (StringUtils.isBlank(comboChildItem.getParentTicketId())) {
								comboChildItem.setParentTicketId(ticketId);
							}
							if (StringUtils.isBlank(comboChildItem.getParentTicketOutletId())) {
								comboChildItem.setParentTicketOutletId(outletId);
							}
						}

					}

				}
			}
		}

		if (hasNewItemAdded && ticket.getShipmentStatus() != null) {
			ticket.setShipmentStatus(ShipmentStatus.PARTIAL_CHALAN_CREATED.name());
		}

		//		if (StringUtils.isEmpty(ticket.getStoreSessionId())) {
		//			StoreSession storeSession = DataProvider.get().getStoreSession();
		//			if (storeSession != null) {
		//				ticket.setStoreSessionId(storeSession.getId());
		//			}
		//		}

		try {
			TicketType ticketType = ticket.getTicketType();
			Set<PosTransaction> transactions = ticket.getTransactions();
			if (transactions != null && !transactions.isEmpty()) {
				for (PosTransaction posTransaction : transactions) {
					if (!(TicketType.OUTDOOR == ticketType) && StringUtils.isBlank(posTransaction.getUserId())) {
						throw new TransactionUserNullException("User id is null for transaction " + posTransaction.getId());
					}
					posTransaction.setTicketType(ticket.getType());
					posTransaction.setProject(ticket.getProject());
				}
			}
		} catch (TransactionUserNullException e) {
			PosLog.error(getClass(), e);
		}

	}

	private void performPostSaveOperations(Session session, Ticket ticket) {
		updateStock(ticket, session);
		releaseBookingBedIfNeeded(session, ticket);
		ticket.setShouldUpdateStock(false);
		ticket.clearDeletedItems();
	}

	private void releaseBookingBedIfNeeded(Session session, Ticket ticket) {
		if (ticket == null || StringUtils.isBlank(ticket.getAdmissionId())) {
			return;
		}
		List<TicketItem> ticketItems = ticket.getTicketItems();
		for (TicketItem ticketItem : ticketItems) {
			if (!ticketItem.isBedItem() || !ticketItem.isShouldReleaseBed()) {
				continue;
			}
			BedDAO.getInstance().releaseBed(ticketItem, ticket.getAdmissionId(), session);
		}
	}

	public synchronized void saveOrUpdateSplitTickets(List<Ticket> tickets, List<Integer> tableNumbers) {
		try (Session session = this.createNewSession()) {
			Transaction tx = session.beginTransaction();
			for (Ticket ticket : tickets) {
				ticket.setUpdateLastUpdateTime(true);
				saveOrUpdate(ticket, session, false);
			}
			tx.commit();
		}
	}

	public boolean isNewTicket(String id) {
		Session session = null;
		try {
			session = createNewSession();
			return isNewTicket(id, session);
		} finally {
			closeSession(session);
		}
	}

	private boolean isNewTicket(String id, Session session) {
		Criteria criteria = session.createCriteria(getReferenceClass());
		criteria.setProjection(Projections.rowCount());
		criteria.add(Restrictions.eq(Ticket.PROP_ID, id));
		Number number = (Number) criteria.uniqueResult();
		return number == null || number.intValue() == 0;
	}

	private void clearVoidedItems(Ticket ticket) {
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			ticketItem.setVoidItem(null);
		}
	}

	public synchronized void saveKitchenPrintStatus(Ticket ticket, KitchenTicket kitchenTicket) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			List<KitchenTicketItem> ticketItems = kitchenTicket.getTicketItems();

			KitchenTicketItem parentItem = null;
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) iterator.next();

				if (kitchenTicketItem.isModifierItem() || kitchenTicketItem.isCookingInstruction()) {
					if (parentItem != null) {
						String nameDisplay = parentItem.getMenuItemName().concat("\n" + kitchenTicketItem.getMenuItemName());
						parentItem.setMenuItemName(nameDisplay);
						iterator.remove();
					}
				}
				else {
					parentItem = kitchenTicketItem;
				}
			}

			//Check voided items
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				KitchenTicketItem kitchenTicketItem = (KitchenTicketItem) iterator.next();
				if (kitchenTicketItem.isVoided()) {
					String ticketItemId = kitchenTicketItem.getVoidedItemId();
					KitchenTicketItemDAO.getInstance().markVoided(ticketItemId, kitchenTicketItem.getQuantity(), session);
					iterator.remove();
				}
			}

			//if kitchen ticket contained voided item only, then do not save the ticket
			if (ticketItems.size() > 0) {
				KitchenTicketDAO.getInstance().saveOrUpdate(kitchenTicket, session);
			}

			ticket.clearDeletedItems();
			if (ticket.getTokenNo() == 0) {
				ticket.setTokenNo(TokenNumberGenerator.generateTokenNo());
			}
			saveOrUpdate(ticket, session);
			tx.commit();
		} catch (Exception e) {
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public void voidTicket(Ticket ticket) throws Exception {
		Transaction tx = null;
		try (Session session = createNewSession()) {
			ticket.setShouldUpdateStock(true);
			ticket.setDiscounts(null);
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				if (ticketItem == null) {
					continue;
				}
				boolean inventoryAdjusted = ticketItem.getInventoryAdjustQty().doubleValue() >= ticketItem.getQuantity().doubleValue();
				if (!ticketItem.isVoided()) {
					ticket.voidItem(ticketItem, ticket.getVoidReason(), ticket.isWasted(), Math.abs(ticketItem.getQuantity()));
					if (!inventoryAdjusted) {
						ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
					}
				}
				else if (ticketItem.isReturned() && !ticketItem.isReturnedItemVoided()) {
					ticket.voidReturnedItem(ticketItem, ticket.getVoidReason(), ticket.isWasted());
					if (!inventoryAdjusted) {
						ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
					}
				}
			}
			ticket.calculatePrice();

			tx = session.beginTransaction();

			List<KitchenTicket> kitchenTickets = KitchenTicketDAO.getInstance().findByParentId(ticket.getId());
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			if (kitchenTickets != null) {
				for (KitchenTicket kitchenTicket : kitchenTickets) {
					kitchenTicket.setCreateDate(serverTimestamp);
					kitchenTicket.setVoided(true);
					session.saveOrUpdate(kitchenTicket);
				}
			}
			ticket.setVoided(true);
			ticket.setClosed(true);
			ticket.setClosingDate(StoreDAO.getServerTimestamp());
			//			ticket.setLastUpdateTime(new Date());
			TicketDAO.getInstance().saveOrUpdate(ticket, session);
			LedgerEntryDAO.getInstance().saveOrderLedgerEntry(ticket, session);

			if (ticket.getTicketType() == TicketType.OUTDOOR) {
				String outdoorTicketId = ticket.getOutdoorTicketId();
				if (StringUtils.isNotBlank(outdoorTicketId)) {
					OutdoorTicketDAO outdoorTicketDAO = OutdoorTicketDAO.getInstance();
					OutdoorTicket outdoorTicket = outdoorTicketDAO.getOutdoorTicket(outdoorTicketId, ticket.getOutletId());
					outdoorTicket.setStatus(OutdoorTicketStatus.VOIDED.name());
					outdoorTicketDAO.saveOrUpdate(outdoorTicket, session);

					String appointmentId = outdoorTicket.getAppointmentEntityId();
					if (StringUtils.isNotBlank(appointmentId)) {
						Appointment appointment = AppointmentDAO.getInstance().get(appointmentId, session);
						if (appointment != null) {
							appointment.setDeleted(true);
							AppointmentDAO.getInstance().saveOrUpdate(appointment, session);
						}
					}
				}
			}
			if (ticket.getTicketType() == TicketType.IPD && StringUtils.isNotBlank(ticket.getAdmissionId())) {
				BookingInfo booking = BookingInfoDAO.getInstance().findBooking(ticket.getAdmissionId(), session);
				booking.setClosed(true);
				booking.setStatus(PatientBookingStatus.VOIDED.name());
				BookingInfoDAO.getInstance().saveOrUpdateBooking(booking, false, session);
			}
			tx.commit();

			journalLog(ticket);
		} catch (Exception x) {
			throw x;
		} finally {
			ticket.setShouldUpdateStock(false);
		}
	}

	private void journalLog(Ticket ticket) {
		String actionDescription = "Order is voided. Total: " + NumberUtil.formatNumber(ticket.getTotalAmountWithTips()); //$NON-NLS-1$
		ActionHistoryDAO.saveHistory(ticket, ActionHistory.VOID_CHECK, actionDescription);
	}

	@Deprecated
	private void populateVoidItems(Ticket ticket) {
		Map<String, Double> voidedItemQuantityMap = new HashMap();
		for (TicketItem ticketItem : ticket.getTicketItems()) {
			if (!ticketItem.isVoided())
				continue;
			Double previousValue = voidedItemQuantityMap.get(ticketItem.getMenuItemId());
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double voidedQuantity = 0;
			voidedQuantity = Math.abs(ticketItem.getQuantity());
			voidedQuantity += previousValue;
			if (voidedQuantity == 0) {
				continue;
			}
			voidedItemQuantityMap.put(ticketItem.getMenuItemId(), voidedQuantity);
		}

		Map<TicketItem, VoidItem> toBeVoidedItemsMap = new HashMap<>();

		for (Iterator iterator = ticket.getTicketItems().iterator(); iterator.hasNext();) {
			TicketItem ticketItem = (TicketItem) iterator.next();
			if (ticketItem.getId() == null || ticketItem.isVoided() || ticketItem.getVoidItem() != null) {
				continue;
			}

			Double voidedQuantity = voidedItemQuantityMap.get(ticketItem.getMenuItemId());
			double toBeVoidQuantity = ticketItem.getQuantity();
			if (voidedQuantity != null && voidedQuantity > 0) {
				if (voidedQuantity >= toBeVoidQuantity) {
					voidedItemQuantityMap.put(ticketItem.getMenuItemId(), voidedQuantity - toBeVoidQuantity);
					continue;
				}
				else {
					toBeVoidQuantity -= voidedQuantity;
				}
			}
			toBeVoidedItemsMap.put(ticketItem, new VoidItem(ticket.getVoidReason(), ticket.isWasted(), toBeVoidQuantity));
		}
		Set<TicketItem> keys = toBeVoidedItemsMap.keySet();
		for (TicketItem ticketItem : keys) {
			VoidItem voidItem = toBeVoidedItemsMap.get(ticketItem);
			ticket.voidItem(ticketItem, voidItem.getVoidReason(), voidItem.isItemWasted(), voidItem.getQuantity());
		}
	}

	public void loadFullTicket(Ticket ticket) {
		if (ticket.getId() == null) {
			return;
		}
		if (Hibernate.isInitialized(ticket.getTicketItems()) && Hibernate.isInitialized(ticket.getTransactions())
				&& Hibernate.isInitialized(ticket.getTicketPaymentAllocations())) {
			return;
		}
		Session session = null;
		try {
			session = createNewSession();
			session.refresh(ticket);

			initialize(ticket, session);
		} finally {
			closeSession(session);
		}
	}

	public void initialize(Ticket ticket, Session session) {
		Hibernate.initialize(ticket.getTicketItems());
		Hibernate.initialize(ticket.getTransactions());
		Hibernate.initialize(ticket.getTicketPaymentAllocations());
		cleanupTicketItems(ticket);
	}

	public Ticket loadFullTicket(String id, String outletId) {
		if (StringUtils.isBlank(id)) {
			return null;
		}
		try (Session session = createNewSession()) {
			Ticket ticket = (Ticket) get(getReferenceClass(), new Ticket(id, outletId), session);
			if (ticket == null)
				return null;

			initialize(ticket, session);

			return ticket;
		}
	}

	public Ticket loadFullTicket(String id, String outletId, Session session) {
		Ticket ticket = (Ticket) get(getReferenceClass(), new Ticket(id, outletId), session);
		if (ticket == null)
			return null;

		initialize(ticket, session);

		return ticket;
	}

	private void cleanupTicketItems(Ticket ticket) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		if (ticketItems != null) {
			for (Iterator iterator = ticketItems.iterator(); iterator.hasNext();) {
				TicketItem ticketItem = (TicketItem) iterator.next();
				if (ticketItem == null) {
					iterator.remove();
				}
			}
		}
	}

	public Ticket loadCouponsAndTransactions(String ticketId, String outletId) {
		try (Session session = createNewSession()) {

			Ticket ticket = (Ticket) TicketDAO.getInstance().get(ticketId, outletId, session);
			if (ticket == null) {
				return null;
			}
			Hibernate.initialize(ticket.getTransactions());

			return ticket;
		}
	}

	public List<Ticket> findOpenTickets() {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForGroupSettle(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_DUE_AMOUNT, Double.valueOf(0)));
			criteria.add(Restrictions.isEmpty(Ticket.PROP_TRANSACTIONS));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			for (Iterator iterator = list.iterator(); iterator.hasNext();) {
				Ticket ticket = (Ticket) iterator.next();
				OrderType orderType = ticket.getOrderType();
				if (orderType != null && orderType.isBarTab()) {
					iterator.remove();
				}
			}
			return list;
		}
	}

	public List<Ticket> findMergableTickets(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_DUE_AMOUNT, Double.valueOf(0)));
			criteria.add(Restrictions.isEmpty(Ticket.PROP_TRANSACTIONS));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		}
	}

	public List<Ticket> findOpenTickets(User user, boolean allowOtherUserTicket) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));

			if (!allowOtherUserTicket && user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}
			else if (user != null && !user.hasPermission(UserPermission.EDIT_OTHER_USERS_TICKETS)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		}
	}

	public List<Ticket> findOpenTickets(Integer customerId) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTickets(Terminal terminal, UserType userType) {
		return findOpenTickets(terminal, userType, null, null, null);
	}

	public List<Ticket> findOpenTickets(Terminal terminal, UserType userType, String orderTypeId, Date startDate, Date endDate) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, startDate, endDate));
			}
			if (userType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_TYPE_ID, userType.getId()));
			}
			if (terminal != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (StringUtils.isNotBlank(orderTypeId) && orderTypeId != POSConstants.ALL) {
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderTypeId));
			}
			criteria.addOrder(getDefaultOrder());
			return criteria.list();
		}
	}

	@Deprecated
	public int getNumTickets() {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			updateCriteriaFilters(criteria);

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadTickets(PaginatedTableModel tableModel) {
		try (Session session = this.createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			updateCriteriaFilters(criteria);

			tableModel.setNumRows(rowCount(criteria));

			criteria.addOrder(getDefaultOrder());
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());

			tableModel.setRows(criteria.list());
		}
	}

	@SuppressWarnings("unchecked")
	public List<Ticket> findCustomerTickets(String customerId, PaginationSupport tableModel) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			tableModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			criteria.addOrder(getDefaultOrder());

			return criteria.list();
		}
	}

	public List<Ticket> findNextCustomerTickets(Integer customerId, PaginatedTableModel tableModel, String filter) {
		Session session = null;
		Criteria criteria = null;
		PaymentStatusFilter statusFilter = PaymentStatusFilter.fromString(filter);
		try {
			int nextIndex = tableModel.getNextRowIndex();

			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			if (statusFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}

			criteria.setFirstResult(nextIndex);
			criteria.setMaxResults(tableModel.getPageSize());

			List ticketList = criteria.list();

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				tableModel.setNumRows(rowCount.intValue());

			}

			tableModel.setCurrentRowIndex(nextIndex);

			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findPreviousCustomerTickets(Integer customerId, PaginatedTableModel tableModel, String filter) {
		Session session = null;
		Criteria criteria = null;
		PaymentStatusFilter statusFilter = PaymentStatusFilter.fromString(filter);
		try {

			int previousIndex = tableModel.getPreviousRowIndex();

			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			if (statusFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}

			criteria.setFirstResult(previousIndex);
			criteria.setMaxResults(tableModel.getPageSize());

			List ticketList = criteria.list();

			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				tableModel.setNumRows(rowCount.intValue());

			}

			tableModel.setCurrentRowIndex(previousIndex);

			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketByCustomer(Integer customerId) {
		Session session = null;
		Criteria criteria = null;

		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			List ticketList = criteria.list();
			return ticketList;

		} finally {
			closeSession(session);
		}
	}

	/**
	 * Retrieves all tickets for a specific customer.
	 */
	public List<Ticket> findUnpaidTicketsForCustomerAppendSource(Ticket sourceTicket) {
		return findUnpaidTicketsForCustomerAppendSource(sourceTicket, true);
	}

	public List<Ticket> findUnpaidTicketsForCustomerAppendSource(Ticket sourceTicket, boolean includeSourceTicket) {
		String customerId = sourceTicket.getCustomerId();
		if (StringUtils.isBlank(customerId)) {
			return Collections.emptyList();
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			criteria.add(Restrictions.eq(Ticket.PROP_PAID, false));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, false));
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, sourceTicket.getOutletId()));
			criteria.add(Restrictions.ne(Ticket.PROP_TOTAL_AMOUNT, 0.0));

			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));

			String excludeTicketId = sourceTicket.getId();
			if (StringUtils.isNotBlank(excludeTicketId)) {
				criteria.add(Restrictions.ne(Ticket.PROP_ID, excludeTicketId));
			}

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(Ticket.PROP_ID), Ticket.PROP_ID);
			pList.add(Projections.property(Ticket.PROP_OUTLET_ID), Ticket.PROP_OUTLET_ID);
			criteria.setProjection(pList);

			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			criteria.setResultTransformer(Transformers.aliasToBean(Ticket.class));

			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));

			List<Ticket> ticketList = criteria.list();

			if (includeSourceTicket) {
				//add source ticket to the last
				ticketList.add(sourceTicket);
			}

			return loadFullTickets(ticketList, session);
		}
	}

	public List<Ticket> loadFullTickets(List<Ticket> sourceTickets, Session session) {
		List<Ticket> finalList = new ArrayList<Ticket>();
		if (sourceTickets == null || sourceTickets.isEmpty()) {
			return finalList;
		}
		for (Ticket ticket : sourceTickets) {
			ticket = (Ticket) get(getReferenceClass(), new Ticket(ticket.getId(), ticket.getOutletId()), session);
			if (ticket == null) {
				continue;
			}

			initialize(ticket, session);

			finalList.add(ticket);
		}

		return finalList;
	}

	public List<Ticket> findTickets(PaymentStatusFilter psFilter, OrderType orderType) {
		return findTicketsForUser(psFilter, orderType, null);
	}

	public List<Ticket> findTicketsForUser(PaymentStatusFilter psFilter, OrderType orderType, User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			if (psFilter == PaymentStatusFilter.OPEN) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.FALSE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			else if (psFilter == PaymentStatusFilter.PAID) {
				criteria.add(Restrictions.eq(Ticket.PROP_PAID, Boolean.TRUE));
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			else if (psFilter == PaymentStatusFilter.CLOSED) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));

				Calendar currentTime = Calendar.getInstance();
				currentTime.add(Calendar.HOUR_OF_DAY, -24);

				criteria.add(Restrictions.ge(Ticket.PROP_CLOSING_DATE, currentTime.getTime()));
			}

			if (orderType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
			}

			if (user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public boolean hasOpenTickets(User user) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (user != null)
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			Number result = (Number) criteria.uniqueResult();
			if (result != null) {
				return result.intValue() > 0;
			}
			return false;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTicketsForUser(User user) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			//criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			//criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user == null ? null : user.getId()));
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsByOutletId(Date lastUpdateTime, String outletId, boolean includeClosedTickets) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			if (!includeClosedTickets) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			if (lastUpdateTime != null)
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findCashDrawerTickets(Date lastUpdateTime, String cashDrawerId, Integer terminalId) {
		try (Session session = getSession()) {
			DetachedCriteria detachedCriteria = DetachedCriteria.forClass(PosTransaction.class);
			detachedCriteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerId));
			detachedCriteria.setProjection(Projections.property(PosTransaction.PROP_TICKET));

			Criteria criteria = session.createCriteria(Ticket.class);
			Criterion propertiesIn = Subqueries.propertiesIn(new String[] { Ticket.PROP_ID, Ticket.PROP_OUTLET_ID }, detachedCriteria);
			//@formatter:off
			Conjunction conjunction = Restrictions.and(
					Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE), 
					Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE),
					Restrictions.eq(Ticket.PROP_TERMINAL_ID, terminalId)
					);
			criteria.add(Restrictions.or(propertiesIn, conjunction));
			//@formatter:on
			if (lastUpdateTime != null) {
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			}
			List list = criteria.list();
			return list;
		}
	}

	public List<Ticket> findClosedTicketsByOpenTicketIds(Date lastUpdateTime, List<String> openTicketIds) {
		if (openTicketIds == null || openTicketIds.isEmpty()) {
			return null;
		}
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.in(Ticket.PROP_ID, openTicketIds));
			if (lastUpdateTime != null)
				criteria.add(Restrictions.or(Restrictions.isNull("lastUpdateTime"), Restrictions.gt("lastUpdateTime", lastUpdateTime))); //$NON-NLS-1$ //$NON-NLS-2$
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findOpenTickets(Date startDate, Date endDate) {
		Session session = null;

		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findClosedTickets(Date startDate, Date endDate, String outletId) {
		return findClosedOnlineTickets(startDate, endDate, "", "");
	}

	public List<Ticket> findClosedOnlineTickets(Date startDate, Date endDate, String outletId, String ticketId) {
		Session session = null;

		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Ticket.PROP_SOURCE, TicketSource.Online.name()));
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			if (StringUtils.isNotBlank(ticketId)) {
				criteria.add(Restrictions.ilike(Ticket.PROP_ID, ticketId, MatchMode.ANYWHERE));
			}
			if (startDate != null && endDate != null) {
				SimpleExpression startDateOnCreate = Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate);
				SimpleExpression endDateOnCreate = Restrictions.le(Ticket.PROP_CREATE_DATE, endDate);
				SimpleExpression startDateOnDelivery = Restrictions.ge(Ticket.PROP_DELIVERY_DATE, startDate);
				SimpleExpression endDateOnDelivery = Restrictions.le(Ticket.PROP_DELIVERY_DATE, endDate);
				LogicalExpression createDateAnd = Restrictions.and(startDateOnCreate, endDateOnCreate);
				LogicalExpression deliveryDateAnd = Restrictions.and(startDateOnDelivery, endDateOnDelivery);
				criteria.add(Restrictions.or(createDateAnd, deliveryDateAnd));
			}
			else {
				if (StringUtils.isBlank(ticketId)) {
					Calendar calendar = Calendar.getInstance();
					calendar.add(Calendar.DAY_OF_MONTH, -1);
					Date now = calendar.getTime();
					criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, DateUtil.startOfDay(now)));
				}
			}
			List list = criteria.list();
			return list;
		} finally {
			closeSession(session);
		}
	}

	public void closeOrder(Ticket ticket) {

		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			saveOrUpdate(ticket);

			User driver = ticket.getAssignedDriver();
			if (driver != null) {
				driver.setAvailableForDelivery(true);
				UserDAO.getInstance().saveOrUpdate(driver);
			}

			tx.commit();
		} catch (Exception e) {
			LogFactory.getLog(TicketDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTickets(Date startDate, Date endDate, boolean closed, Terminal terminal) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

			if (terminal != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TERMINAL_ID, terminal.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForLaborHour(Date startDate, Date endDate, int hour, Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_ACTIVE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_ACTIVE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_CREATION_HOUR, Integer.valueOf(hour)));
			//criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			//criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> findTicketsForShift(Date startDate, Date endDate, Shift shit, Outlet outlet) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			criteria.add(Restrictions.eq(Ticket.PROP_SHIFT_ID, shit == null ? null : shit.getId()));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq(Ticket.PROP_REFUNDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public static TicketDAO getInstance() {
		return instance;
	}

	private void updateCriteriaFilters(Criteria criteria) {
		User user = DataProvider.get().getCurrentUser();
		PaymentStatusFilter paymentStatusFilter = TerminalConfig.getPaymentStatusFilter();

		//orderTypeFilter is either order type id, or 'ALL'
		String orderTypeFilter = TerminalConfig.getOrderTypeFilter();
		boolean filterByMyTicket = TerminalConfig.isFilterByOwner();
		OrderType orderType = null;
		//:FIXME
		//if (!POSConstants.ALL.equals(orderTypeFilter)) {
		//	orderType = DataProvider.get().getOrderType(orderTypeFilter);
		//}
		if (paymentStatusFilter == PaymentStatusFilter.OPEN) {
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (!user.canViewAllOpenTickets() || filterByMyTicket) {
				criteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, user.getRoleIds()));
			}
		}
		else if (paymentStatusFilter == PaymentStatusFilter.CLOSED) {
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			//			StoreSessionControl currentStoreOperation = StoreUtil.getCurrentStoreOperation();
			//			if (currentStoreOperation != null && currentStoreOperation.getCurrentData() != null) {
			//				criteria.add(Restrictions.eq(Ticket.PROP_STORE_SESSION_ID, currentStoreOperation.getCurrentData().getId()));
			//			}
			if (!user.canViewAllCloseTickets() || filterByMyTicket) {
				criteria.add(Restrictions.in(Ticket.PROP_OWNER_ID, user.getRoleIds()));
			}
		}

		if (!orderTypeFilter.equals(POSConstants.ALL)) {
			if (orderType != null)
				criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
		}

	}

	public void deleteTickets(List<Ticket> tickets) {
		deleteTickets(tickets, false);
	}

	public void deleteTickets(List<Ticket> tickets, boolean updateStock) {
		Session session = null;
		Transaction tx = null;

		try {
			session = createNewSession();
			tx = session.beginTransaction();

			deleteTickets(session, tickets, updateStock);

			tx.commit();
		} catch (Exception e) {
			LogFactory.getLog(TicketDAO.class).error(e);
			throw new RuntimeException(e);
		} finally {
			closeSession(session);
		}
	}

	public void deleteTickets(Session session, List<Ticket> tickets) throws Exception {
		deleteTickets(session, tickets, false);
	}

	public void deleteTickets(Session session, List<Ticket> tickets, boolean updateStock) throws Exception {
		for (Ticket ticket : tickets) {
			ticket = (Ticket) session.merge(ticket);

			deleteLabDoctorPayments(ticket, session);
			deleteRFPayments(ticket, session);
			deleteLedgerEntry(ticket, session);

			super.delete(ticket, session);
		}
	}

	private void deleteLabDoctorPayments(Ticket ticket, Session session) {
		Double ldfPaidAmount = ticket.getLabDoctorFeePaidAmount();
		if (ldfPaidAmount <= 0) {
			return;
		}
		List<LdfPayTransaction> payments = PosTransactionDAO.getInstance().findPaidTransactionsByTicketId(ticket.getId(), LdfPayTransaction.class, session);
		for (LdfPayTransaction payTransaction : payments) {

			Double amount = payTransaction.getAmount();
			if (amount < ldfPaidAmount) {
				payTransaction.setAmount(amount);
			}
			else {
				payTransaction.setAmount(amount - ldfPaidAmount);
			}

			deleteOrUpdateTransaction(ticket, session, payTransaction);
		}

	}

	private void deleteRFPayments(Ticket ticket, Session session) {
		Double rfPaidAmount = ticket.getReferrerFeePaidAmount();
		if (rfPaidAmount <= 0) {
			return;
		}

		List<RfPayTransaction> payments = PosTransactionDAO.getInstance().findPaidTransactionsByTicketId(ticket.getId(), RfPayTransaction.class, session);
		for (RfPayTransaction payTransaction : payments) {

			Double amount = payTransaction.getAmount();

			if (amount < rfPaidAmount) {
				payTransaction.setAmount(amount);
			}
			else {
				payTransaction.setAmount(amount - rfPaidAmount);
			}

			deleteOrUpdateTransaction(ticket, session, payTransaction);
		}
	}

	private void deleteOrUpdateTransaction(Ticket ticket, Session session, PosTransaction payTransaction) {
		String ticketIdStrings = payTransaction.getTransTicketIds();
		if (StringUtils.isNotBlank(ticketIdStrings)) {
			ticketIdStrings = ticketIdStrings.substring(1, ticketIdStrings.length() - 1);
			List<String> ticketIdList = Stream.of(ticketIdStrings.split(",")).map(String::trim).collect(Collectors.toList());
			String ticketId = "'" + ticket.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
			if (ticketIdList.contains(ticketId)) {
				ticketIdList.remove(ticketId);
			}

			if (ticketIdList.isEmpty()) {
				PosTransactionDAO.getInstance().delete(payTransaction, session);
			}
			else {
				String finalTicketIdStrings = "";
				for (Iterator<String> iterator = ticketIdList.iterator(); iterator.hasNext();) {
					finalTicketIdStrings += iterator.next();
					if (iterator.hasNext()) {
						finalTicketIdStrings += ",";
					}
				}
				payTransaction.setTransTicketIds(finalTicketIdStrings);
				PosTransactionDAO.getInstance().saveOrUpdate(payTransaction, session);
			}
		}
	}

	private void deleteLedgerEntry(Ticket ticket, Session session) {
		LedgerEntryDAO instance = LedgerEntryDAO.getInstance();
		List<LedgerEntry> ledgerEntries = instance.getLedgerEntry(ticket.getId(), session);
		if (ledgerEntries != null) {
			for (LedgerEntry ledgerEntry : ledgerEntries) {
				instance.delete(ledgerEntry, session);
			}
		}
	}

	//	private void adjustInventory(Ticket ticket, Session session) {
	//		List<TicketItem> items = ticket.getTicketItems();
	//		Map<String, Double> itemStockCountMap = getStockCountAdjustedMap(items);
	//
	//		if (ticket.getId() != null) {
	//			List<Object> deletedItems = ticket.getDeletedItems();
	//			if (deletedItems != null) {
	//				for (Object item : deletedItems) {
	//					if (item instanceof TicketItem) {
	//						TicketItem ticketItem = (TicketItem) item;
	//						if (ticketItem.isPrintedToKitchen())
	//							continue;
	//						MenuItemDAO.getInstance().updateStockQuantity(ticketItem.getMenuItemId(), ticketItem.getQuantity(), true, false, session);
	//					}
	//				}
	//			}
	//		}
	//		if (itemStockCountMap.isEmpty()) {
	//			return;
	//		}
	//
	//		for (String menuItemId : itemStockCountMap.keySet()) {
	//			Double usedAmount = itemStockCountMap.get(menuItemId);
	//			Query query = null;
	//			query = session.createQuery(String.format("update MenuItem set availableUnit = (availableUnit - %s) where id = '%s'", usedAmount, menuItemId));
	//			if (query == null) {
	//				return;
	//			}
	//			query.executeUpdate();
	//		}
	//	}

	//	private Map<String, Double> getStockCountAdjustedMap(List<TicketItem> items) {
	//		HashMap<String, Double> itemStockCountMap = new LinkedHashMap<String, Double>();
	//		for (TicketItem ticketItem : items) {
	//			MenuItem menuItem = ticketItem.getMenuItem();
	//			if (menuItem == null) {
	//				continue;
	//			}
	//			if (!ticketItem.isInventoryItem()) {
	//				continue;
	//			}
	//			if (ticketItem.isInventoryAdjusted()) {
	//				continue;
	//			}
	//
	//			Double previousValue = itemStockCountMap.get(ticketItem.getMenuItemId());
	//
	//			if (previousValue == null) {
	//				previousValue = 0.0;
	//			}
	//			itemStockCountMap.put(ticketItem.getMenuItemId(), ticketItem.getQuantity() + previousValue);
	//
	//			for (RecepieItem recepieItem : menuItem.getRecepieItems()) {
	//				MenuItem inventoryItem = recepieItem.getInventoryItem();
	//				Double itemValue = itemStockCountMap.get(inventoryItem.getId());
	//				Double percentage = recepieItem.getPercentage() / 100.0;
	//				if (itemValue == null)
	//					itemValue = 0.0;
	//
	//				itemValue += (ticketItem.getQuantity() * percentage);
	//
	//				itemStockCountMap.put(inventoryItem.getId(), itemValue);
	//			}
	//			ticketItem.setInventoryAdjusted(true);
	//		}
	//		return itemStockCountMap;
	//	}

	protected void updateStock(Ticket ticket, Session session) {
		try {
			if (shouldUpdateStock(ticket) == false) {
				return;
			}

			InventoryLocation defaultOutInventoryLocation = null;
			InventoryLocation defaultInInventoryLocation = null;
			defaultOutInventoryLocation = DataProvider.get().getDefaultOutLocation();
			HashMap<InvMapKey, Double> itemMap = buildItemMapForInventoryAdjustment(ticket);
			HashMap<InvMapKey, Double> comboChildReturnMap = buildComboChildReturnItemMapForInventoryAdjustment(ticket);
			HashMap<InvMapKey, Double> voidedItemsMap = buildItemMapForVoidItems(ticket, session);
			HashMap<InvMapKey, Double> returnedVoidedItemsMap = buildItemMapForReturnedVoidItems(ticket, session);
			adjustInventory(ticket, itemMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_TICKET_SALES, defaultOutInventoryLocation, session);
			adjustInventory(ticket, comboChildReturnMap, InventoryTransactionType.IN, InventoryTransaction.REASON_RETURN, defaultOutInventoryLocation, session);
			if (voidedItemsMap != null && voidedItemsMap.size() > 0) {
				defaultInInventoryLocation = DataProvider.get().getDefaultInLocation();
				adjustInventory(ticket, voidedItemsMap, InventoryTransactionType.IN, InventoryTransaction.REASON_VOID, defaultInInventoryLocation, session);
				adjustBloodInventory(ticket, voidedItemsMap, InventoryTransactionType.IN, InventoryTransaction.REASON_VOID, defaultInInventoryLocation,
						session);
			}
			if (returnedVoidedItemsMap != null && returnedVoidedItemsMap.size() > 0) {
				adjustInventory(ticket, returnedVoidedItemsMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_VOID, defaultOutInventoryLocation,
						session);
			}
			clearVoidedItems(ticket);
		} catch (Exception e) {
			PosLog.error(getClass(), "Failed to update stock balance for ticket: " + ticket.getId(), e);
		}
	}

	private void adjustBloodInventory(Ticket ticket, HashMap<InvMapKey, Double> voidedItemsMap, InventoryTransactionType in, String reason,
			InventoryLocation defaultInInventoryLocation, Session session) {
		if (ticket.getTicketType() == TicketType.BLOOD_BANK_ISSUE) {
			for (TicketItem ticketItem : ticket.getTicketItems()) {
				if (ProductType.BLOOD.name().equals(ticketItem.getProductType())) {
					VoidItem voidItem = ticketItem.getVoidItem();
					Double toBeAdjustQty = voidItem.getQuantity();
					if (toBeAdjustQty <= 0) {
						continue;
					}
					BloodInventoryTransaction bloodInventoryTransaction = null;
					String bitId = ticketItem.getBloodInventoryTransactionId();
					String bloodBagNumber = ticketItem.getBloodBagNumber();
					String donorId = ticketItem.getDonorId();
					if (StringUtils.isNotBlank(bitId)) {
						bloodInventoryTransaction = BloodInventoryTransactionDAO.getInstance().get(bitId);
					}
					else {
						bloodInventoryTransaction = BloodInventoryTransactionDAO.getInstance().findByBagNoAndDonorId(bloodBagNumber, donorId);
					}

					BloodInventoryTransaction inventoryTransaction = new BloodInventoryTransaction();
					inventoryTransaction.setType(in.getType());
					inventoryTransaction.setBloodGroup(bloodInventoryTransaction.getBloodGroup());
					inventoryTransaction.setBloodType(bloodInventoryTransaction.getBloodType());
					inventoryTransaction.setBagNumber(bloodBagNumber);
					inventoryTransaction.setVendorId(bloodInventoryTransaction.getVendorId());
					inventoryTransaction.putDonorName(bloodInventoryTransaction.getDonorName());
					inventoryTransaction.putDonorPhone(bloodInventoryTransaction.getDonorMobile());
					inventoryTransaction.setQuantity(toBeAdjustQty.intValue());
					inventoryTransaction.setUnit(bloodInventoryTransaction.getUnit());
					inventoryTransaction.setVolume(bloodInventoryTransaction.getVolume());
					inventoryTransaction.setInventoryType(reason);
					inventoryTransaction.setOutletId(bloodInventoryTransaction.getOutletId());
					inventoryTransaction.setTransactionDate(StoreDAO.getServerTimestamp());
					inventoryTransaction.setTicketId(ticket.getId());
					inventoryTransaction.setPatient((Patient) ticket.getCustomer());
					User owner = ticket.getOwner();
					inventoryTransaction.setUserId(owner == null ? DataProvider.get().getCurrentUser().getId() : owner.getId());
					BloodInventoryTransactionDAO.getInstance().adjustBloodInventoryStock(inventoryTransaction, session);
				}
			}
		}
	}

	protected void adjustInventory(Ticket ticket, HashMap<InvMapKey, Double> itemMap, InventoryTransactionType transactionType, String reason,
			InventoryLocation location, Session session) throws Exception {
		Store store = DataProvider.get().getStore();
		boolean isUpdateOnHandBlncForSale = store.isUpdateOnHandBlncForSale();
		boolean isUpdateAvailBlncForSale = store.isUpdateAvlBlncForSale();
		for (InvMapKey mapKey : itemMap.keySet()) {
			Double unitQuantity = itemMap.get(mapKey);
			MenuItem menuItem = MenuItemDAO.getInstance().getMenuItemWithFields(session, mapKey.getMenuItemId(), MenuItem.PROP_NAME, MenuItem.PROP_PRICE,
					MenuItem.PROP_SKU, MenuItem.PROP_BARCODE, MenuItem.PROP_UNIT_ID, MenuItem.PROP_COST, MenuItem.PROP_AVERAGE_UNIT_PURCHASE_PRICE,
					MenuItem.PROP_AVG_COST, MenuItem.PROP_DEFAULT_RECIPE_ID, MenuItem.PROP_INVENTORY_ITEM);
			if (menuItem == null) {
				continue;
			}
			TicketItem ticketItem = mapKey.getTicketItem();
			Double remainingUnitQuantity = unitQuantity;
			if (menuItem.isInventoryItem()) {
				Map<String, Double> salesUnitByStockId = ticketItem == null ? Collections.emptyMap() : ticketItem.getSalesUnitByStockId();
				Iterator<Entry<String, Double>> salesUnitByStockIdIterator = salesUnitByStockId.entrySet().iterator();
				do {
					InventoryTransaction invTransaction = new InventoryTransaction();
					if (mapKey.isItemReturned() && InventoryTransaction.REASON_VOID.equals(reason)) {
						reason = InventoryTransaction.REASON_RETURN;
					}
					if (transactionType == InventoryTransactionType.IN) {
						invTransaction.setToInventoryLocation(location);
					}
					else {
						invTransaction.setFromInventoryLocation(location);
					}

					invTransaction.setReason(reason);
					invTransaction.setTransactionDate(new Date());
					invTransaction.setMenuItem(menuItem);
					invTransaction.setType(transactionType.getType());
					invTransaction.setTicketId(ticket.getId());
					invTransaction.setUser(ticket.getOwner());
					invTransaction.setUnitCost(menuItem.getCost());
					IUnit transactionUnit = null;
					if (UnitType.match(mapKey.getUnitType(), UnitType.PACKAGING_UNIT)) {
						transactionUnit = (IUnit) DataProvider.get().getObjectOf(InventoryStockUnit.class, mapKey.getUnitId());
						InventoryStockUnit stockUnit = (InventoryStockUnit) transactionUnit;
						invTransaction.setUnitPrice(stockUnit.getPrice());
						Double cost = stockUnit.getCost();
						if (cost == 0) {
							cost = menuItem.getCost() * stockUnit.getConversionRate();
							stockUnit.setCost(cost);
							InventoryStockUnitDAO.getInstance().saveOrUpdate(stockUnit, session);
						}
						invTransaction.setUnitCost(cost);
					}
					else {
						transactionUnit = (IUnit) DataProvider.get().getUnitById(mapKey.getUnitId(), mapKey.getUnitType());
					}
					if (transactionUnit instanceof PackagingUnit || transactionUnit instanceof InventoryStockUnit) {
						session.refresh(menuItem);
						if (!Hibernate.isInitialized(menuItem.getStockUnits())) {
							Hibernate.initialize(menuItem.getStockUnits());
						}
					}
					invTransaction.setDisplayUnit(transactionUnit, unitQuantity, menuItem);
					if (ticketItem != null) {
						invTransaction.setUnitPrice(ticketItem.getAdjustedUnitPrice());
					}
					invTransaction.setUnit(mapKey.getUnitId());
					invTransaction.setUnitType(mapKey.getUnitType());

					Entry<String, Double> next = salesUnitByStockIdIterator.hasNext() ? salesUnitByStockIdIterator.next() : null;
					if (next != null) {
						String key = next.getKey();
						InventoryStock inventoryStock = session.get(InventoryStock.class, key);
						if (inventoryStock != null) {
							session.evict(inventoryStock);
							invTransaction.setBatchNumber(inventoryStock.getBatchNumber());
							invTransaction.putExpiryDate(inventoryStock.getExpireDate());
							invTransaction.setFromLocationId(inventoryStock.getLocationId());
							Double mapValue = next.getValue();
							if (inventoryStock.getUnit() != null && !inventoryStock.getUnit().equals(mapKey.getUnitId())) {
								IUnit sourceUnit = (IUnit) DataProvider.get().getObjectOf(InventoryStockUnit.class, inventoryStock.getUnit());
								mapValue = ticketItem.getMenuItem().getUnitQuantity(sourceUnit, transactionUnit);
							}
							if (unitQuantity > mapValue) {
								unitQuantity = mapValue;
							}
						}
					}
					else {
						unitQuantity = remainingUnitQuantity;
					}
					remainingUnitQuantity -= unitQuantity;

					invTransaction.setQuantity(unitQuantity);
					invTransaction.calculateTotal();
					InventoryTransactionDAO.getInstance().adjustInventoryStock(invTransaction, session, isUpdateAvailBlncForSale, isUpdateOnHandBlncForSale,
							false);
				} while (remainingUnitQuantity > 0);
			}
			//adjust inventory of recipe items
			if (StringUtils.isNotEmpty(menuItem.getDefaultRecipeId())) {
				Recepie recepie = RecepieDAO.getInstance().get(menuItem.getDefaultRecipeId(), session);
				RecepieDAO.getInstance().adjustRecipeItemsFromInventory(unitQuantity, Arrays.asList(recepie), session);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForInventoryAdjustment(Ticket ticket) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		this.populateItemToMap(ticketItems, itemMap);
		return itemMap;
	}

	private void populateItemToMap(List<TicketItem> ticketItems, HashMap<InvMapKey, Double> itemMap) {
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem == null) {
				continue;
			}
			if (ticketItem.isLabTest() || ticketItem.getMenuItemId() == null || ticketItem.isVoided() || ticketItem.isItemReturned()
					|| ticketItem.isInventoryAdjusted() || (ticketItem.isRequiredShipping() && !ticketItem.isComboChild()) || ticketItem.isSharedItem()) {
				if (ticketItem.isComboItem() && ticketItem.getComboItems() != null) {
					this.populateItemToMap(ticketItem.getComboItems(), itemMap);
				}
				continue;
			}
			InvMapKey key = new InvMapKey(ticketItem);

			Double previousValue = itemMap.get(key);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = ticketItem.getQuantity();
			toBeAdjustQty -= ticketItem.getInventoryAdjustQty();
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty <= 0) {
				continue;
			}
			itemMap.put(key, toBeAdjustQty);
			ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
			if (ticketItem.isComboItem() && ticketItem.getComboItems() != null) {
				this.populateItemToMap(ticketItem.getComboItems(), itemMap);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildComboChildReturnItemMapForInventoryAdjustment(Ticket ticket) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> itemMap = new HashMap<>();
		for (TicketItem ticketItem : ticketItems) {
			if (!ticketItem.isComboItem()) {
				continue;
			}
			if (ticketItem.getComboItems() != null && !ticket.isVoided()) {
				this.populateComboChildReturnItemToMap(ticket.isVoided(), ticketItem.getComboItems(), itemMap, false);

			}
			if (ticketItem.getDeletedComboChildItems() != null) {
				this.populateComboChildReturnItemToMap(ticket.isVoided(), ticketItem.getDeletedComboChildItems(), itemMap, true);
			}
		}
		return itemMap;
	}

	private void populateComboChildReturnItemToMap(Boolean isTicketVoided, List<TicketItem> ticketItems, HashMap<InvMapKey, Double> itemMap,
			boolean isDeletedItem) {
		for (TicketItem ticketItem : ticketItems) {
			if (ticketItem == null || ticketItem.isSharedItem() || !ticketItem.isInventoryItem()) {
				continue;
			}
			InvMapKey key = new InvMapKey(ticketItem);

			Double previousValue = itemMap.get(key);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			Double toBeAdjustQty = isTicketVoided ? 0.0 : ticketItem.getQuantity();
			if (isDeletedItem) {
				toBeAdjustQty = -1 * ticketItem.getInventoryAdjustQty();
			}
			else {
				toBeAdjustQty -= ticketItem.getInventoryAdjustQty();
			}
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty >= 0) {
				continue;
			}
			itemMap.put(key, NumberUtil.round(Math.abs(toBeAdjustQty)));
			ticketItem.setInventoryAdjustQty(ticketItem.getQuantity());
			if (ticketItem.isComboItem() && ticketItem.getComboItems() != null) {
				this.populateItemToMap(ticketItem.getComboItems(), itemMap);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForVoidItems(Ticket ticket, Session session) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> voidItemMap = new HashMap<>();
		this.populateVoidItemToMap(ticket, session, ticketItems, voidItemMap);
		return voidItemMap;
	}

	private void populateVoidItemToMap(Ticket ticket, Session session, List<TicketItem> ticketItems, HashMap<InvMapKey, Double> voidItemMap) {
		for (TicketItem voidTicketItem : ticketItems) {
			VoidItem voidItem = voidTicketItem.getVoidItem();

			if (voidItem == null || (!voidTicketItem.isVoided() && !voidTicketItem.isItemReturned()) || voidTicketItem.isReturnedItemVoided()
					|| voidTicketItem.isSharedItem()) {
				continue;
			}

			InvMapKey mapKey = new InvMapKey(voidTicketItem);
			mapKey.setItemReturned(voidTicketItem.isItemReturned());

			Double previousValue = voidItemMap.get(mapKey);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double toBeAdjustQty = 0;

			TicketItem sourceItem = ticketItems.stream().filter(t -> t.getId().equals(voidTicketItem.getVoidedItemId())).findFirst().orElse(null);
			if (sourceItem != null) {
				double voidItemQuantity = voidItem.getQuantity();
				boolean itemReturnedInDifferentUnit = sourceItem.getUnitName() != null && !sourceItem.getUnitName().equals(voidTicketItem.getUnitName());
				if (itemReturnedInDifferentUnit) {
					double voidItemSourceQuantity = voidTicketItem.getDoubleProperty("sourceUnitVoidQuantity"); //$NON-NLS-1$
					if (voidItemSourceQuantity > 0) {
						voidItemQuantity = voidItemSourceQuantity;
					}
				}
				double previousInvAdjQty = sourceItem.getInventoryAdjustQty();
				toBeAdjustQty = voidItemQuantity < previousInvAdjQty ? voidItemQuantity : previousInvAdjQty;

				toBeAdjustQty += previousValue;
				if (toBeAdjustQty == 0) {
					continue;
				}

				sourceItem.setInventoryAdjustQty(sourceItem.getInventoryAdjustQty() - toBeAdjustQty);
				if (itemReturnedInDifferentUnit) {
					toBeAdjustQty = voidItem.getQuantity();
				}
			}

			if (!voidItem.isItemWasted() && !voidTicketItem.isInventoryAdjusted()) {
				voidItemMap.put(mapKey, toBeAdjustQty);
				voidTicketItem.setInventoryAdjustQty(voidTicketItem.getQuantity());
			}
			Boolean isPrintedToKitchen = voidTicketItem.isPrintedToKitchen();
			VoidItemDAO.getInstance().saveAndSentToKitchenIfNeeded(voidItem, isPrintedToKitchen, ticket, session);
			voidTicketItem.addProperty(TicketItem.JSON_PROP_VOID_ID, voidItem.getId());

			List<VoidItem> voidedModifiers = voidItem.getVoidedModifiers();
			String propModifierVoidIds = ""; //$NON-NLS-1$
			if (voidedModifiers != null && voidedModifiers.size() > 0) {
				for (Iterator<VoidItem> iterator = voidedModifiers.iterator(); iterator.hasNext();) {
					VoidItem voidedModifier = (VoidItem) iterator.next();
					VoidItemDAO.getInstance().saveAndSentToKitchenIfNeeded(voidedModifier, isPrintedToKitchen, ticket, session);
					propModifierVoidIds += voidedModifier.getId();
					if (iterator.hasNext()) {
						propModifierVoidIds += ","; //$NON-NLS-1$
					}
				}
			}
			if (voidTicketItem.isReturned() && StringUtils.isNotBlank(propModifierVoidIds)) {
				voidTicketItem.addProperty(TicketItem.JSON_PROP_MODIIFIER_VOID_IDS, propModifierVoidIds);
			}
			if (voidTicketItem.isComboItem()) {
				this.populateVoidItemToMap(ticket, session, voidTicketItem.getComboItems(), voidItemMap);
			}
		}
	}

	private HashMap<InvMapKey, Double> buildItemMapForReturnedVoidItems(Ticket ticket, Session session) {
		List<TicketItem> ticketItems = ticket.getTicketItems();
		HashMap<InvMapKey, Double> voidItemMap = new HashMap<>();
		this.populateReturnedVoidItemToMap(ticket, session, ticketItems, voidItemMap);
		return voidItemMap;
	}

	private void populateReturnedVoidItemToMap(Ticket ticket, Session session, List<TicketItem> ticketItems, HashMap<InvMapKey, Double> voidItemMap) {
		for (TicketItem voidTicketItem : ticketItems) {
			if (!voidTicketItem.isReturnedItemVoided() || voidTicketItem.isSharedItem()) {
				continue;
			}
			String voidId = voidTicketItem.getProperty(TicketItem.JSON_PROP_VOID_ID);
			if (StringUtils.isBlank(voidId)) {
				continue;
			}
			VoidItem voidItem = VoidItemDAO.getInstance().get(voidId, ticket.getOutletId());
			if (voidItem == null) {
				continue;
			}
			String menuItemId = voidTicketItem.getMenuItemId();
			InvMapKey mapKey = new InvMapKey(menuItemId, voidTicketItem.getUnitName(), voidTicketItem.getUnitType());
			mapKey.setItemReturned(voidTicketItem.isItemReturned());

			Double previousValue = voidItemMap.get(mapKey);
			if (previousValue == null) {
				previousValue = 0.0;
			}
			double toBeAdjustQty = 0;
			toBeAdjustQty = Math.abs(voidItem.getQuantity());
			toBeAdjustQty += previousValue;
			if (toBeAdjustQty == 0) {
				continue;
			}
			if (!voidItem.isItemWasted() && !voidTicketItem.isInventoryAdjusted()) {
				voidItemMap.put(mapKey, toBeAdjustQty);
				voidTicketItem.setInventoryAdjustQty(voidTicketItem.getQuantity());
			}
			Boolean isPrintedToKitchen = voidTicketItem.isPrintedToKitchen();
			VoidItemDAO.getInstance().deleteAndSentToKitchenIfNeeded(voidItem, isPrintedToKitchen, ticket, session);
			if (voidTicketItem.isHasModifiers()) {
				String propModifierVoidIds = voidTicketItem.getProperty(TicketItem.JSON_PROP_MODIIFIER_VOID_IDS);
				if (StringUtils.isNotBlank(propModifierVoidIds)) {
					String[] modifierVoidIds = propModifierVoidIds.split(","); //$NON-NLS-1$
					if (modifierVoidIds.length > 0) {
						for (String modifierVoidId : modifierVoidIds) {
							if (StringUtils.isBlank(modifierVoidId)) {
								continue;
							}
							VoidItem existingVoid = VoidItemDAO.getInstance().get(modifierVoidId, ticket.getOutletId());
							if (existingVoid == null) {
								continue;
							}
							VoidItemDAO.getInstance().delete(existingVoid, session);
						}
					}
				}
			}
			if (voidTicketItem.isComboItem()) {
				this.populateReturnedVoidItemToMap(ticket, session, voidTicketItem.getComboItems(), voidItemMap);
			}
		}
	}

	public int getNumTickets(Date start, Date end) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			//updateCriteriaFilters(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (start != null)
				criteria.add(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, start));

			if (end != null)
				criteria.add(Restrictions.le(Ticket.PROP_DELIVERY_DATE, end));

			criteria.add(Restrictions.isNotNull(Ticket.PROP_DELIVERY_DATE));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();
			}
			return 0;
		} finally {
			closeSession(session);
		}
	}

	public void loadTickets(PaginatedTableModel tableModel, Date beginingDeliveryDate, Date endingDeliveryDate) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(getDefaultOrder());
			//updateCriteriaFilters(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			if (beginingDeliveryDate != null)
				criteria.add(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, beginingDeliveryDate));

			if (endingDeliveryDate != null)
				criteria.add(Restrictions.le(Ticket.PROP_DELIVERY_DATE, endingDeliveryDate));

			//			criteria.add(Restrictions.isNotNull(Ticket.PROP_DELIVERY_DATE));
			criteria.setFirstResult(tableModel.getCurrentRowIndex());
			criteria.setMaxResults(tableModel.getPageSize());
			tableModel.setRows(criteria.list());
			return;

		} finally {
			closeSession(session);
		}
	}

	public List<Ticket> getTicketsWithSpecificFields(String... fields) {
		return getTicketsWithSpecificFields(true, fields);
	}

	public List<Ticket> getTicketsWithSpecificFields(boolean isOnlyOpenTickets, String... fields) {
		Session session = null;
		Criteria criteria = null;
		User currentUser = DataProvider.get().getCurrentUser();
		boolean filterUser = !currentUser.isAdministrator() || !currentUser.isManager();
		try {
			session = createNewSession();
			criteria = session.createCriteria(Ticket.class);
			if (isOnlyOpenTickets) {
				criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));
			}
			if (filterUser) {
				//criteria.createAlias(Ticket.PROP_OWNER, "u");
				//criteria.add(Restrictions.eq("u.id", currentUser.getId()));
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, currentUser.getId()));
			}

			ProjectionList projectionList = Projections.projectionList();
			for (String field : fields) {
				projectionList.add(Projections.property(field), field);
			}

			criteria.setProjection(projectionList);
			criteria.setResultTransformer(Transformers.aliasToBean(Ticket.class));
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public boolean hasTicketByOnlineOrderId(String onlineOrderId) {
		Session session = null;
		try {
			session = createNewSession();
			String sql = "select * from TICKET_PROPERTIES where PROPERTY_VALUE='%s' and PROPERTY_NAME='%s'"; //$NON-NLS-1$
			sql = String.format(sql, onlineOrderId, Ticket.PROPERTY_ONLINE_ID);
			SQLQuery sqlQuery = session.createSQLQuery(sql);
			List list = sqlQuery.list();
			return list.size() > 0;
		} finally {
			closeSession(session);
		}
	}

	public boolean hasTicketByReservationId(String reservationId) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_ID, reservationId));
			criteria.add(Restrictions.eq(Ticket.PROP_SOURCE, TicketSource.Reservation));
			criteria.setProjection(Projections.rowCount());
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue() > 0;

			}
		} catch (Exception e) {
			PosLog.error(getReferenceClass(), e);
		} finally {
			closeSession(session);
		}
		return false;
	}

	//	public List<Ticket> findOnlineTickets(PaginatedTableModel tableModel) {
	//		Session session = null;
	//
	//		try {
	//			session = createNewSession();
	//			String sql = "select TICKET_ID from TICKET_PROPERTIES where PROPERTY_VALUE='online' and PROPERTY_NAME='source'"; //$NON-NLS-1$
	//			SQLQuery sqlQuery = session.createSQLQuery(sql);
	//			List<String> list = sqlQuery.list();
	//			List<Ticket> ticketList = new ArrayList<Ticket>();
	//			for (String ticketId : list) {
	//				if (ticketId == null) {
	//					continue;
	//				}
	//				Ticket ticket = get(ticketId);
	//				ticketList.add(ticket);
	//			}
	//
	//			return ticketList;
	//
	//		} finally {
	//			closeSession(session);
	//		}
	//	}

	public void findTicketForDeliveryDispath(PaginatedTableModel model, OrderType orderType, String customerId, Date startDate, Date endDate,
			boolean onlineOrderOnly, boolean isPickupOnly, boolean isDeliveryOnly, boolean filterUnassigned) {
		User user = DataProvider.get().getCurrentUser();
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			criteria.addOrder(getDefaultOrder());
			criteria.add(Restrictions.eq(Ticket.PROP_ORDER_TYPE_ID, orderType.getId()));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE));

			if (StringUtils.isNotEmpty(customerId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			}

			if (!user.canViewAllOpenTickets()) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			if (startDate != null && endDate != null) {
				criteria.add(Restrictions.and(Restrictions.ge(Ticket.PROP_DELIVERY_DATE, startDate), Restrictions.le(Ticket.PROP_DELIVERY_DATE, endDate)));
			}
			if (isPickupOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.TRUE));
			}
			if (isDeliveryOnly) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_WILL_PICKUP, Boolean.FALSE));
			}
			if (filterUnassigned) {
				criteria.add(Restrictions.isNull(Ticket.PROP_ASSIGNED_DRIVER_ID));
			}
			int currentRowIndex = model.getCurrentRowIndex();
			criteria.setFirstResult(currentRowIndex);
			criteria.setMaxResults(model.getPageSize());

			List<Ticket> list = criteria.list();
			if (onlineOrderOnly) {
				for (Iterator iterator = list.iterator(); iterator.hasNext();) {
					Ticket ticket = (Ticket) iterator.next();
					if (!ticket.isSourceOnline()) {
						iterator.remove();
					}
				}
			}

			model.setRows(list);
		} finally {
			closeSession(session);
		}
	}

	public synchronized static void closeOrders(Ticket... tickets) {
		Session session = getInstance().createNewSession();
		Transaction transaction = null;
		if (tickets == null) {
			return;
		}
		try {
			transaction = session.beginTransaction();
			Date serverTimestamp = StoreDAO.getServerTimestamp();
			for (Ticket ticket : tickets) {
				ticket = getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId());
				ticket.setClosed(true);
				ticket.setClosingDate(serverTimestamp);
				//				ticket.setLastUpdateTime(new Date());

				getInstance().saveOrUpdate(ticket, session);
			}
			transaction.commit();
		} catch (Exception e) {
			PosLog.error(TicketDAO.class, e);

			if (transaction != null)
				transaction.rollback();
		} finally {
			session.close();
		}
	}

	public List<Ticket> findVoidTicketByDate(Date startDate, Date endDate, Outlet outlet) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, true));
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CLOSING_DATE, endDate));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}
			return criteria.list();
		} finally {
			closeSession(session);
		}
	}

	public User findOwner(String ticketId) {
		if (ticketId == null) {
			return null;
		}
		Session session = getSession();
		Criteria criteria = session.createCriteria(Ticket.class);

		criteria.add(Restrictions.eq(Ticket.PROP_ID, ticketId));

		ProjectionList projectionList = Projections.projectionList();
		//projectionList.add(Projections.property(Ticket.PROP_OWNER));
		projectionList.add(Projections.property(Ticket.PROP_OWNER_ID));
		criteria.setProjection(projectionList);
		if (criteria.list().isEmpty()) {
			return null;
		}
		//TODO:
		User user = UserDAO.getInstance().get((String) criteria.list().get(0), null);
		return user;
	}

	public Ticket findByCustomerAndDeliveryDate(String customerId, Date deliveryDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Ticket.class);

			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.between(Ticket.PROP_DELIVERY_DATE, DateUtil.startOfDay(deliveryDate), DateUtil.endOfDay(deliveryDate)));
			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, false));
			List list = criteria.list();
			if (!list.isEmpty()) {
				return (Ticket) list.get(0);
			}
			return null;
		} finally {
			closeSession(session);
		}

	}

	public void saveMergedTickets(List<Ticket> tickets, Ticket rootTicket) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			rootTicket.setShouldUpdateTableStatus(true);
			rootTicket.setUpdateLastUpdateTime(true);

			session = createNewSession();
			tx = session.beginTransaction();

			//for (Ticket ticket : tickets) {
			//update(ticket, session);
			//}

			deleteTickets(session, tickets, true);
			saveOrUpdate(rootTicket, session, true);

			tx.commit();
		} finally {
			closeSession(session);
		}
	}

	public void saveOrUpdateTransferedTicketsList(List<Ticket> ticketsToDelete, List<Ticket> ticketsToUpdate) throws Exception {
		Session session = null;
		try {
			session = createNewSession();
			Transaction tx = session.beginTransaction();

			if (ticketsToDelete != null || ticketsToDelete.size() > 0) {
				deleteTickets(session, ticketsToDelete, true);
			}

			for (Ticket ticket : ticketsToUpdate) {
				saveOrUpdate(ticket, session);
			}

			tx.commit();
		} finally {
			closeSession(session);
		}
	}

	public void reversePayment(Ticket ticket, PosTransaction transaction, User user) throws Exception {
		if (transaction instanceof CreditCardTransaction || transaction instanceof DebitCardTransaction) {
			CardReader transactionCardReader = CardReader.fromString(transaction.getCardReader());
			if (StringUtils.isBlank(transaction.getCardMerchantGateway()) || transactionCardReader == CardReader.EXTERNAL_TERMINAL) {
				transaction.setVoided(true);
			}
			else {
				PaymentGatewayPlugin paymentGatewayByName = CardConfig.getPaymentGatewayByName(transaction.getCardMerchantGateway());
				if (paymentGatewayByName == null) {
					throw new PosException(Messages.getString("TicketDAO.1")); //$NON-NLS-1$
				}
				CardProcessor cardProcessor = paymentGatewayByName.getProcessor();
				cardProcessor.voidTransaction(transaction);
			}
			transaction.setVoided(true);
			transaction.setVoidDate(StoreDAO.getServerTimestamp());
			transaction.setVoidedByUser(user);

			ticket.calculatePrice();
			ticket.setClosed(false);
			ticket.setPaid(false);
			ticket.setAccountProcessed(false);

			Session session = null;
			try {
				session = createNewSession();
				Transaction tx = session.beginTransaction();

				if (StringUtils.isNotBlank(ticket.getCustomerId())) {
					Customer customer = CustomerDAO.getInstance().get(ticket.getCustomerId(), session);
					new PosTransactionService().deductCustomerLoyaltyPoint(customer, transaction.getAmount(), transaction, session);
				}
				if (transaction instanceof AdvanceAdjustmentTransaction) {
					BalanceUpdateTransactionDAO.getInstance().saveAdvanceBalanceUpdateTransaction(transaction, session);
				}

				doVoidTicketPaymentAllocation(ticket, transaction.getId(), session);
				update(ticket, session);
				updateOutdoorTicketPaymentStatus(ticket, session);

				LedgerEntryDAO.getInstance().saveLedgerEntryForVoidTransaction(ticket, transaction, session);

				tx.commit();
			} finally {
				closeSession(session);
			}
		}
		else {
			try (Session session = createNewSession()) {
				Transaction tx = session.beginTransaction();
				PaymentType paymentType = transaction.getPaymentType();
				PaymentPlugin paymentPlugin = paymentType.getPaymentPlugin();
				transaction.setVoided(true);
				transaction.setVoidDate(StoreDAO.getServerTimestamp());
				transaction.setVoidedByUser(user);
				if (paymentPlugin != null) {
					paymentPlugin.voidPayment(transaction, session);
				}
				//				if (paymentType == PaymentType.CASH) {
				//					PosTransactionService.getInstance().adjustMulticurrencyBalance(session, DataProvider.get().getCurrentTerminal(),
				//							Application.getCurrentUser().getActiveDrawerPullReport(), null, transaction);
				//				}
				ticket.calculatePrice();
				ticket.setClosed(false);
				ticket.setPaid(false);

				if (StringUtils.isNotBlank(ticket.getCustomerId())) {
					Customer customer = CustomerDAO.getInstance().get(ticket.getCustomerId(), session);
					new PosTransactionService().deductCustomerLoyaltyPoint(customer, transaction.getAmount(), transaction, session);
				}
				if (transaction instanceof AdvanceAdjustmentTransaction) {
					BalanceUpdateTransactionDAO.getInstance().saveAdvanceBalanceUpdateTransaction(transaction, session);
				}

				doVoidTicketPaymentAllocation(ticket, transaction.getId(), session);
				update(ticket, session);
				updateOutdoorTicketPaymentStatus(ticket, session);

				String linkedBankAccountId = transaction.getLinkedBankAccountId();
				if (StringUtils.isNotBlank(linkedBankAccountId) && transaction.isConfirmPayment()) {
					BankAccount bankAccount = (BankAccount) DataProvider.get().getObjectOf(BankAccount.class, linkedBankAccountId);
					List customPaymentRefJsonList = transaction.getCustomPaymentRefJsonList();

					BalanceUpdateTransactionDAO.getInstance().saveBankAccountTransaction(bankAccount, transaction.getPaymentType(), transaction,
							transaction.getAmount(), null, session, customPaymentRefJsonList.isEmpty() ? null : customPaymentRefJsonList.toString());

				}

				LedgerEntryDAO.getInstance().saveLedgerEntryForVoidTransaction(ticket, transaction, session);

				tx.commit();
			}
		}
	}

	private void updateOutdoorTicketPaymentStatus(Ticket ticket, Session session) {
		if (ticket.getTicketType() == null || TicketType.OUTDOOR != ticket.getTicketType()) {
			return;
		}
		if (StringUtils.isNotBlank(ticket.getOutdoorTicketId()) && !ticket.isPaid()) {
			OutdoorTicket outdoorTicket = OutdoorTicketDAO.getInstance().get(ticket.getOutdoorTicketId(), session);
			if (OutdoorTicketStatus.PENDING.name().equals(outdoorTicket.getStatus())) {
				return;
			}
			outdoorTicket.setStatus(OutdoorTicketStatus.PENDING.name());
			session.update(outdoorTicket);
		}
	}

	private void doVoidTicketPaymentAllocation(Ticket ticket, String voidedTransactionId, Session session) {
		Set<TicketPaymentAllocation> paymentAllocations = ticket.getTicketPaymentAllocations();

		Optional<TicketPaymentAllocation> match = Optional.ofNullable(paymentAllocations).orElse(Collections.emptySet()).stream()
				.filter(a -> voidedTransactionId.equals(a.getTransactionId())).findFirst();

		if (match.isPresent()) {
			TicketPaymentAllocation allocation = match.get();
			allocation.setVoided(true);
			allocation.setVoidDate(StoreDAO.getServerTimestamp());
		}
		else {
			TicketPaymentAllocationDAO allocationDAO = TicketPaymentAllocationDAO.getInstance();
			TicketPaymentAllocation allocation = allocationDAO.findTicketPaymentAllocationBy(voidedTransactionId);
			if (allocation != null) {
				allocation.setVoided(true);
				allocation.setVoidDate(StoreDAO.getServerTimestamp());

				allocationDAO.saveOrUpdate(allocation, session);
			}
		}
	}

	public void reversePayment(Ticket ticket, PosTransaction transaction, boolean reverseGiftCardBalance) throws Exception {
		if (transaction instanceof CreditCardTransaction || transaction instanceof DebitCardTransaction) {
			CardReader transactionCardReader = CardReader.fromString(transaction.getCardReader());
			if (StringUtils.isBlank(transaction.getCardMerchantGateway()) || transactionCardReader == CardReader.EXTERNAL_TERMINAL) {
				transaction.setVoided(true);
			}
			else {
				CardProcessor cardProcessor = CardConfig.getPaymentGatewayByName(transaction.getCardMerchantGateway()).getProcessor();
				cardProcessor.voidTransaction(transaction);
			}
			transaction.setVoided(true);
			transaction.setVoidDate(StoreDAO.getServerTimestamp());
			ticket.calculatePrice();
			ticket.setClosed(false);
			ticket.setPaid(false);
			ticket.setAccountProcessed(false);

			Session session = null;
			try {
				session = createNewSession();
				Transaction tx = session.beginTransaction();

				if (reverseGiftCardBalance) {
					doReverseGiftCardBalance(ticket, transaction, session);
				}
				if (transaction instanceof AdvanceAdjustmentTransaction) {
					BalanceUpdateTransactionDAO.getInstance().saveAdvanceBalanceUpdateTransaction(transaction, session);
				}
				update(ticket, session);

				tx.commit();
			} finally {
				closeSession(session);
			}
		}
		else {
			try (Session session = createNewSession()) {
				Transaction tx = session.beginTransaction();
				PaymentType paymentType = transaction.getPaymentType();
				PaymentPlugin paymentPlugin = paymentType.getPaymentPlugin();
				if (paymentPlugin != null) {
					paymentPlugin.voidPayment(transaction, session);
				}
				transaction.setVoided(true);
				//				if (paymentType == PaymentType.CASH) {
				//					PosTransactionService.getInstance().adjustMulticurrencyBalance(session, DataProvider.get().getCurrentTerminal(),
				//							Application.getCurrentUser().getActiveDrawerPullReport(), null, transaction);
				//				}
				ticket.calculatePrice();
				ticket.setClosed(false);
				ticket.setPaid(false);

				if (reverseGiftCardBalance) {
					doReverseGiftCardBalance(ticket, transaction, session);
				}
				if (transaction instanceof AdvanceAdjustmentTransaction) {
					BalanceUpdateTransactionDAO.getInstance().saveAdvanceBalanceUpdateTransaction(transaction, session);
				}
				update(ticket, session);
				tx.commit();
			}
		}

	}

	public void doReverseGiftCardBalance(Ticket ticket, PosTransaction transaction, Session session) {
		String giftCardInfo = transaction.getGiftCardBalanceAddInfo();
		if (StringUtils.isEmpty(giftCardInfo)) {
			return;
		}

		JSONArray jsonArray = new JSONArray(giftCardInfo);
		for (Object object : jsonArray) {
			JSONObject jsonObject = (JSONObject) object;
			String ticketItemId = jsonObject.getString(PosTransaction.JSON_PROP_GIFT_CARD_TICKET_ITEM_ID);
			String giftCardNumber = jsonObject.getString(PosTransaction.JSON_PROP_GIFT_CARD_GIFT_CARD_NO);
			double addedAmount = jsonObject.getDouble(PosTransaction.JSON_PROP_GIFT_CARD_ADDED_AMOUNT);

			GiftCard giftCard = GiftCardDAO.getInstance().findByCardNumber(session, giftCardNumber);
			if (giftCard != null) {
				giftCard.setBalance(giftCard.getBalance() - addedAmount);
				GiftCardDAO.getInstance().saveOrUpdate(giftCard, session);
			}

			List<TicketItem> ticketItems = ticket.getTicketItems();
			for (TicketItem ticketItem : ticketItems) {
				if (ticketItem.getId().equals(ticketItemId)) {
					ticketItem.setGiftCardPaidAmount(0.0);
				}
			}
		}
	}

	public void loadTicketsForUser(User user, Date from, Date to, PaginatedTableModel listModel) {
		Session session = null;
		try {
			session = getSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			if (user != null)
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			if (from != null)
				criteria.add(Restrictions.or(Restrictions.ge(Ticket.PROP_CREATE_DATE, from), Restrictions.eq(Ticket.PROP_CLOSED, false)));
			if (to != null)
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, to));

			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				listModel.setNumRows(rowCount.intValue());
			}
			else {
				listModel.setNumRows(0);
			}
			criteria.setProjection(null);
			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			listModel.setRows(criteria.list());
		} finally {
			closeSession(session);
		}
	}

	private boolean shouldUpdateStock(Ticket ticket) {
		//		if (ticket.isSourceOnline()) {
		//			return false;
		//		}
		if (ticket.isReservation()) {
			return true;
		}
		return ticket.isShouldUpdateStock();
	}

	//	/**
	//	 * This method returns all those ticket which have Transaction in this session.
	//	 */
	//	public List<Ticket> getTicketsOfCurrentSession(StoreSession storeSession) {
	//		Session session = null;
	//		Criteria criteria = null;
	//
	//		try {
	//			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeSession);
	//			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
	//				return null;
	//
	//			session = createNewSession();
	//			criteria = session.createCriteria(getReferenceClass());
	//			criteria.add(Restrictions.eq(Ticket.PROP_CLOUD_SYNCED, Boolean.FALSE));
	//
	//			criteria.createAlias("transactions", "tx");
	//			criteria.add(Restrictions.in("tx." + PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
	//			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
	//			return criteria.list();
	//		} finally {
	//			closeSession(session);
	//		}
	//	}
	/**
	 * This method returns all those Unsynced ticket which have Transaction in this session.
	 */
	public List findAllUnSyncTicket(StoreSession storeSession) {

		Session session = null;
		try {

			List<String> cashDrawerIds = CashDrawerDAO.getInstance().getCashDrawerIds(storeSession);
			if (cashDrawerIds == null || cashDrawerIds.isEmpty())
				return null;

			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());

			Criterion nullUpdateTime = Restrictions.isNull(Ticket.PROP_LAST_UPDATE_TIME);
			Criterion nullSyncTime = Restrictions.isNull(Ticket.PROP_LAST_SYNC_TIME);
			Criterion gtQuery = Restrictions.gtProperty(Ticket.PROP_LAST_UPDATE_TIME, Ticket.PROP_LAST_SYNC_TIME);
			criteria.add(Restrictions.or(nullUpdateTime, nullSyncTime, gtQuery));

			//			criteria.createAlias("transactions", "tx");
			//			criteria.add(Restrictions.in("tx." + PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerIds));
			//			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

			return criteria.list();
		} finally {
			if (session != null) {
				closeSession(session);
			}
		}
	}

	/**
	 * Updating Tickets and TicketItems Cloud Sync ...
	 */
	public void updateTicketAndTicketItemSync(List<Ticket> ticketOrgin, boolean synced) {
		if (ticketOrgin != null) {
			for (Ticket ticket : ticketOrgin) {
				loadFullTicket(ticket);
				if (!ticket.isCloudSynced()) {
					if (ticket.isClosed()) {
						ticket.setCloudSynced(synced);
						TicketDAO.getInstance().update(ticket);
					}
				}
				List<TicketItem> ticketItems = ticket.getTicketItems();
				if (ticketItems != null) {
					for (TicketItem ticketItem : ticketItems) {
						if (!ticketItem.isCloudSynced()) {
							//if (ticket.isClosed()) {
							ticketItem.setCloudSynced(synced);
							TicketItemDAO.getInstance().update(ticketItem);
							//}
						}
					}
				}
			}
		}
	}

	public void loadClosedTicket(PaginationSupport model, String txtNameOrId, Date fromDate, Date toDate) {
		loadClosedTicket(model, txtNameOrId, fromDate, toDate, null);
	}

	public void loadClosedTicket(PaginationSupport model, String txtNameOrId, Date fromDate, Date toDate, User user) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(Customer.class);

			List<String> customerIds = null;
			if (StringUtils.isNotEmpty(txtNameOrId)) {
				criteria.setProjection(Projections.property(Customer.PROP_ID));
				criteria.add(Restrictions.ilike(Customer.PROP_NAME, txtNameOrId, MatchMode.ANYWHERE));
				customerIds = criteria.list();
			}

			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());

			Disjunction ticketIdOrCustomerNameCriteria = Restrictions.disjunction();
			if (StringUtils.isNotEmpty(txtNameOrId)) {
				ticketIdOrCustomerNameCriteria.add(Restrictions.ilike(Ticket.PROP_ID, txtNameOrId, MatchMode.ANYWHERE));
			}
			if (customerIds != null && customerIds.size() > 0) {
				ticketIdOrCustomerNameCriteria.add(Restrictions.in(Ticket.PROP_CUSTOMER_ID, customerIds));
			}
			criteria.add(ticketIdOrCustomerNameCriteria);

			if (user != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OWNER_ID, user.getId()));
			}

			criteria.add(Restrictions.eq(Ticket.PROP_CLOSED, Boolean.TRUE));
			if (fromDate != null && toDate != null) {
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			}

			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());
			List<Ticket> result = criteria.list();
			model.setRows(result);
		} finally {
			closeSession(session);
		}
	}

	public List<Date> findTicketWithinDate(String outletId, Date beginDate, Date currentDate, Session session) {
		try {
			Criteria criteria = session.createCriteria(Ticket.class);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
			criteria.setProjection(Projections.property(Ticket.PROP_CREATE_DATE));
			List list = criteria.list();
			if (list != null && list.size() > 0)
				return list;
		} catch (Exception e0) {
			PosLog.error(this.getReferenceClass(), e0);
		}
		return null;
	}

	public List<DeliverySummaryReportData> findTicketsForDeliverySummaryReport(Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			//@formatter:off
			String as = " as "; //$NON-NLS-1$
			String hql = "select " //$NON-NLS-1$
			        		+ "t."+ Ticket.PROP_OWNER_ID + as + DeliverySummaryReportData.PROP_EMPLYOYEE_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CUSTOMER_ID+ as + DeliverySummaryReportData.PROP_CUSTOMER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CUSTOMER_NAME + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_ID+ as + DeliverySummaryReportData.PROP_TICKET_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SUBTOTAL_AMOUNT+ as + DeliverySummaryReportData.PROP_NET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TAX_AMOUNT+ as + DeliverySummaryReportData.PROP_TAX_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TOTAL_AMOUNT+ as + DeliverySummaryReportData.PROP_TOTAL_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SERVICE_CHARGE+ as + DeliverySummaryReportData.PROP_SERVICE_CHARGE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DISCOUNT_AMOUNT+ as + DeliverySummaryReportData.PROP_DISCOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CREATE_DATE+ as + DeliverySummaryReportData.PROP_CREATE_DATE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DELIVERY_DATE+ as + DeliverySummaryReportData.PROP_DELIVERY_DATE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "g."+ Gratuity.PROP_AMOUNT+ as + DeliverySummaryReportData.PROP_GRATUITY_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CASH_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CREDIT_CARD_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_CHARGE_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_OTHERS_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CASH_PAYMENT_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_CREDIT_CARD_PAYMENT_DEBIT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_MEMBER_CHARGE_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + DeliverySummaryReportData.PROP_OTHERS_PAYMENT_DEBIT  //$NON-NLS-1$
			        		+ " from Ticket as t left outer join t.gratuity as g"; //$NON-NLS-1$

			 String transSql = String.format("select sum(tr.%s) from PosTransaction as tr" //$NON-NLS-1$
			 		+ " where t=tr.%s and tr.%s='%s' and tr.%s=%s",  //$NON-NLS-1$
			 		PosTransaction.PROP_AMOUNT, 
			 		PosTransaction.PROP_TICKET,
			 		PosTransaction.PROP_TRANSACTION_TYPE,
			 		"%s", //$NON-NLS-1$
					PosTransaction.PROP_VOIDED, false);
			 
			 
			String commonPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + "='%s')"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String othersPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + " not in('%s','%s','%s'))"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String customerQuery = "(select c.%s from Customer as c where t.%s=c.%s)"; //$NON-NLS-1$
			
			hql = String.format(hql, 
					String.format(customerQuery, Customer.PROP_NAME, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(customerQuery, Customer.PROP_MEMBER_ID, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()),
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()));

			hql += " where t." + Ticket.PROP_DELIVERY_DATE + " between :fromDate and :toDate"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " and t." + Ticket.PROP_VOIDED + " =:voided"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " order by t." + Ticket.PROP_OWNER_ID + " asc"; //$NON-NLS-1$ //$NON-NLS-2$

			//@formatter:on
			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", fromDate); //$NON-NLS-1$
			query.setTimestamp("toDate", toDate); //$NON-NLS-1$
			query.setBoolean("voided", Boolean.FALSE); //$NON-NLS-1$
			query.setResultTransformer(Transformers.aliasToBean(DeliverySummaryReportData.class));
			return query.list();
		}
	}

	@Deprecated
	public List<EndOfDayReportData> findTicketsGroupedByEmployee(Date fromDate, Date toDate, List<OrderType> orderTypes) {
		try (Session session = createNewSession()) {
			//@formatter:off
			String as = " as "; //$NON-NLS-1$
			String hql = "select " //$NON-NLS-1$
			        		+ "t."+ Ticket.PROP_OWNER_ID + as + EndOfDayReportData.PROP_EMPLYOYEE_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_CUSTOMER_ID+ as + EndOfDayReportData.PROP_CUSTOMER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CUSTOMER_NAME + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_ID+ as + EndOfDayReportData.PROP_TICKET_ID + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SUBTOTAL_AMOUNT+ as + EndOfDayReportData.PROP_NET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TAX_AMOUNT+ as + EndOfDayReportData.PROP_TAX_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_TOTAL_AMOUNT+ as + EndOfDayReportData.PROP_TOTAL_TICKET_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_SERVICE_CHARGE+ as + EndOfDayReportData.PROP_SERVICE_CHARGE + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "t."+ Ticket.PROP_DISCOUNT_AMOUNT+ as + EndOfDayReportData.PROP_DISCOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "g."+ Gratuity.PROP_AMOUNT+ as + EndOfDayReportData.PROP_GRATUITY_AMOUNT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CASH_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CREDIT_CARD_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_CHARGE_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_OTHERS_PAYMENT_CREDIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CASH_PAYMENT_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_CREDIT_CARD_PAYMENT_DEBIT + "," //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_MEMBER_CHARGE_DEBIT + ","  //$NON-NLS-1$ //$NON-NLS-2$
			        		+ "%s"+ as + EndOfDayReportData.PROP_OTHERS_PAYMENT_DEBIT  //$NON-NLS-1$
			        		+ " from Ticket as t left outer join t.gratuity as g"; //$NON-NLS-1$

			 String transSql = String.format("select sum(tr.%s) from PosTransaction as tr" //$NON-NLS-1$
			 		+ " where t=tr.%s and tr.%s='%s' and tr.%s=%s",  //$NON-NLS-1$
			 		PosTransaction.PROP_AMOUNT, 
			 		PosTransaction.PROP_TICKET,
			 		PosTransaction.PROP_TRANSACTION_TYPE,
			 		"%s", //$NON-NLS-1$
					PosTransaction.PROP_VOIDED, false);
			 
			 
			String commonPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + "='%s')"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String othersPaymentQuery = "(" + transSql + " and tr." + PosTransaction.PROP_PAYMENT_TYPE_STRING + " not in('%s','%s','%s'))"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			String customerQuery = "(select c.%s from Customer as c where t.%s=c.%s)"; //$NON-NLS-1$
			
			hql = String.format(hql, 
					String.format(customerQuery, Customer.PROP_NAME, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(customerQuery, Customer.PROP_MEMBER_ID, Ticket.PROP_CUSTOMER_ID, Customer.PROP_ID),
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.CREDIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.CREDIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()),
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(), PaymentType.CASH.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CREDIT_CARD.name()), 
					String.format(commonPaymentQuery,TransactionType.DEBIT.name(),PaymentType.MEMBER_ACCOUNT.name()), 
					String.format(othersPaymentQuery,TransactionType.DEBIT.name(),PaymentType.CASH.name(),PaymentType.CREDIT_CARD.name(), PaymentType.MEMBER_ACCOUNT.name()));

			hql += " where t." + Ticket.PROP_CREATE_DATE + " >= :fromDate and t." + Ticket.PROP_CREATE_DATE + " < :toDate"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			if (orderTypes != null) {
				String whereCondition = "("; //$NON-NLS-1$
				for (Iterator<OrderType> iterator = orderTypes.iterator(); iterator.hasNext();) {
					OrderType orderType = (OrderType) iterator.next();
					whereCondition += "'" + orderType.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
					if (iterator.hasNext())
						whereCondition += ","; //$NON-NLS-1$
				}
				whereCondition += ")"; //$NON-NLS-1$
				hql += " and t." + Ticket.PROP_ORDER_TYPE_ID + " in " + whereCondition ; //$NON-NLS-1$ //$NON-NLS-2$
			}
			hql += " and t." + Ticket.PROP_VOIDED + " =:voided"; //$NON-NLS-1$ //$NON-NLS-2$
			hql += " order by t." + Ticket.PROP_OWNER_ID + " asc"; //$NON-NLS-1$ //$NON-NLS-2$

			//@formatter:on
			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", DateUtil.toUTC(fromDate)); //$NON-NLS-1$
			query.setTimestamp("toDate", DateUtil.toUTC(toDate)); //$NON-NLS-1$
			query.setBoolean("voided", Boolean.FALSE); //$NON-NLS-1$
			query.setResultTransformer(Transformers.aliasToBean(EndOfDayReportData.class));
			return query.list();
		}
	}

	public int getNumberOfTickets(Date startDate, Date endDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, startDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, endDate));
			Number rowCount = (Number) criteria.uniqueResult();
			if (rowCount != null) {
				return rowCount.intValue();

			}
		} finally {
			closeSession(session);
		}
		return 0;
	}

	/**
	 * find online Tickets By Customer.
	 */
	public List<Ticket> findTicketsByCustomer(String customerId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customerId));
			criteria.add(Restrictions.eq(Ticket.PROP_SOURCE, TicketSource.Online.name()));
			return criteria.list();
		}
	}

	public Ticket get(String ticketId, String outletId) {
		if (StringUtils.isBlank(ticketId)) {
			return null;
		}

		return super.get(new Ticket(ticketId, outletId));
	}

	public Ticket get(String ticketId, String outletId, Session session) {
		if (StringUtils.isBlank(ticketId)) {
			return null;
		}

		return super.get(new Ticket(ticketId, outletId), session);
	}

	public List<Ticket> findDoctorOrders(String outletId, Date beginDate, Date currentDate, Doctor doctor) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, DateUtil.startOfDay(beginDate), DateUtil.endOfDay(currentDate)));
			if (doctor != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_DOCTOR_ID, doctor.getId()));
			}
			else {
				criteria.add(Restrictions.isNotNull(Ticket.PROP_DOCTOR_ID));
			}

			criteria.addOrder(getDefaultOrder());

			List list = criteria.list();
			if (list != null && list.size() > 0)
				return list;
		}
		return null;
	}

	public List<ReferralCommissionReportData> findReferralCommissionOrders(String outletId, Date beginDate, Date currentDate, Customer referrer) {
		return findReferralCommissionOrders(outletId, beginDate, currentDate, referrer, true, true, null, false);
	}

	public List<ReferralCommissionReportData> findReferralCommissionOrders(String outletId, Date beginDate, Date currentDate, Customer referrer,
			boolean includeZeroReferral, boolean groupByReferralType, AgentTypeEnum agentType, Boolean excludePaidOrder) {
		return findReferralCommissionOrders(outletId, beginDate, currentDate, referrer, includeZeroReferral, groupByReferralType, agentType, excludePaidOrder,
				"");
	}

	public List<ReferralCommissionReportData> findReferralCommissionOrders(String outletId, Date beginDate, Date currentDate, Customer referrer,
			boolean includeZeroReferral, boolean groupByReferralType, AgentTypeEnum agentType, Boolean excludePaidOrder, String orderId) {
		return findReferralCommissionOrders(outletId, beginDate, currentDate, referrer, includeZeroReferral, groupByReferralType, agentType, excludePaidOrder,
				orderId, "");
	}

	public List<ReferralCommissionReportData> findReferralCommissionOrders(String outletId, Date beginDate, Date currentDate, Customer referrer,
			boolean includeZeroReferral, boolean groupByReferralType, AgentTypeEnum agentType, Boolean excludePaidOrder, String orderId,
			String patientIdOrName) {
		List<ReferralCommissionReportData> datas = new ArrayList<ReferralCommissionReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_ID, orderId));
			}
			else {
				if (StringUtils.isNotBlank(patientIdOrName)) {
					DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Patient.class);
					detachedCriteria.setProjection(Projections.property(Customer.PROP_ID));

					detachedCriteria.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, patientIdOrName, MatchMode.ANYWHERE),
							Restrictions.eq(Customer.PROP_ID, patientIdOrName)));
					criteria.add(Property.forName(Ticket.PROP_CUSTOMER_ID).in(detachedCriteria));
				}

				if (referrer == null && agentType != null) {
					DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Customer.class);

					if (agentType == AgentTypeEnum.INDOOR_DOCTOR) {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, true));
					}
					else if (agentType == AgentTypeEnum.OUTDOOR_DOCTOR) {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, false));
					}
					else {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_AGENT_TYPE, agentType.name()));
					}
					detachedCriteria.setProjection(Projections.property(Customer.PROP_ID));

					criteria.add(Property.forName(Ticket.PROP_REFERRER_ID).in(detachedCriteria));
				}
				TicketType ticketType = DataProvider.get().getTicketType();
				if (ticketType != null) {
					criteria.add(Restrictions.eq(Ticket.PROP_TYPE, ticketType.getTypeNo()));
				}

				if (beginDate != null) {
					criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, beginDate));
				}
				if (currentDate != null) {
					criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, currentDate));
				}
				if (excludePaidOrder != null && excludePaidOrder) {
					criteria.add(Restrictions.gt(Ticket.PROP_DUE_AMOUNT, 0.0));
				}
				else {
					//criteria.add(Restrictions.isNotNull(Ticket.PROP_REFERRER_ID));
				}
				if (referrer != null) {
					criteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, referrer.getId()));
				}
			}
			if (!groupByReferralType) {
				criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			}

			List<Ticket> list = criteria.list();
			PosLog.info(getClass(), list.size() + " Ticket found. startTime: " + beginDate + " ,endTime: " + currentDate); //$NON-NLS-1$ //$NON-NLS-2$

			if (list != null && list.size() > 0) {
				for (Ticket ticket : list) {
					if (!includeZeroReferral && NumberUtil.isZero(ticket.getTotalReferrerFee())) {
						PosLog.debug(getClass(), "Ignoring ticket due to no referrer fee: " + ticket.getId()); //$NON-NLS-1$
						continue;
					}
					ReferralCommissionReportData data = new ReferralCommissionReportData(ticket);
					//					if (referrer == null) {
					//						if (!includeZeroReferral && NumberUtil.isZero(data.getRfFeeDue())) {
					//							PosLog.debug(getClass(), "Ignore ticket : " + ticket.getId()); //$NON-NLS-1$ //$NON-NLS-2$
					//							continue;
					//						}
					//					}
					datas.add(data);
				}

				if (groupByReferralType) {
					Comparator<ReferralCommissionReportData> comparator = Comparator.comparing(ReferralCommissionReportData::getReferralType,
							Comparator.nullsLast(Comparator.naturalOrder()));
					Comparator<ReferralCommissionReportData> comparator2 = Comparator.comparing(ReferralCommissionReportData::getReferralBy,
							Comparator.nullsLast(Comparator.naturalOrder()));
					datas.sort(comparator.thenComparing(comparator2));
				}
			}

			PosLog.debug(getClass(), datas.size() + " RF report data found"); //$NON-NLS-1$
		}
		return datas;
	}

	public List<DueTicketReportData> findDueOrders(String outletId, Date beginDate, Date currentDate, Customer referrer, AgentTypeEnum referrerType,
			String orderId) {
		List<DueTicketReportData> datas = new ArrayList<DueTicketReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, false));

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}

			criteria.add(Restrictions.ne(Ticket.PROP_DUE_AMOUNT, 0d));

			if (StringUtils.isNotBlank(orderId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_ID, orderId));
			}
			else {
				if (referrer == null && referrerType != null) {
					DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Customer.class);

					if (referrerType == AgentTypeEnum.INDOOR_DOCTOR) {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, true));
					}
					else if (referrerType == AgentTypeEnum.OUTDOOR_DOCTOR) {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, false));
					}
					else {
						detachedCriteria.add(Restrictions.eq(Customer.PROP_AGENT_TYPE, referrerType.name()));
					}
					detachedCriteria.setProjection(Projections.property(Customer.PROP_ID));

					criteria.add(Property.forName(Ticket.PROP_REFERRER_ID).in(detachedCriteria));
				}
				TicketType ticketType = DataProvider.get().getTicketType();
				if (ticketType != null) {
					criteria.add(Restrictions.eq(Ticket.PROP_TYPE, ticketType.getTypeNo()));
				}

				if (beginDate != null) {
					criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, beginDate));
				}
				if (currentDate != null) {
					criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, currentDate));
				}

				if (referrer != null) {
					criteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, referrer.getId()));
				}
			}

			List<Ticket> list = criteria.list();
			PosLog.info(getClass(), list.size() + " Due Ticket found. startTime: " + beginDate + " ,endTime: " + currentDate); //$NON-NLS-1$ //$NON-NLS-2$

			if (list != null && list.size() > 0) {
				for (Ticket ticket : list) {
					DueTicketReportData data = new DueTicketReportData(ticket);
					datas.add(data);
				}
			}

			PosLog.debug(getClass(), datas.size() + " Due ticket data found"); //$NON-NLS-1$
		}
		return datas;
	}

	public List<ReferralCommissionReportData> findDueReport(String outletId, Date beginDate, Date currentDate, String nameOrPhone) {
		List<ReferralCommissionReportData> datas = new ArrayList<ReferralCommissionReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			if (StringUtils.isNotBlank(nameOrPhone)) {

				DetachedCriteria customerCriteria = DetachedCriteria.forClass(Customer.class);
				customerCriteria.setProjection(Property.forName(Customer.PROP_ID));

				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.or(Restrictions.ilike(Customer.PROP_NAME, nameOrPhone, MatchMode.ANYWHERE),
						Restrictions.ilike(Customer.PROP_MOBILE_NO, nameOrPhone, MatchMode.ANYWHERE)));
				customerCriteria.add(disjunction);

				criteria.add(Property.forName(Ticket.PROP_CUSTOMER_ID).in(customerCriteria));
			}

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}

			if (beginDate != null) {
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, beginDate));
			}
			if (currentDate != null) {
				criteria.add(Restrictions.lt(Ticket.PROP_CREATE_DATE, currentDate));
			}

			List<Ticket> list = criteria.list();
			PosLog.info(getClass(), list.size() + " Ticket found. startTime: " + beginDate + " ,endTime: " + currentDate); //$NON-NLS-1$ //$NON-NLS-2$

			if (list != null && list.size() > 0) {
				for (Ticket ticket : list) {
					ReferralCommissionReportData data = new ReferralCommissionReportData(ticket);
					datas.add(data);
				}
			}
		}
		return datas;
	}

	//	public List<ItemwiseRfReportData> findItemWiseRFReportOrders(String outletId, Date beginDate, Date currentDate) {
	//		return findItemWiseRFReportOrders(outletId, beginDate, currentDate, "");
	//	}
	//
	//	public List<ItemwiseRfReportData> findItemWiseRFReportOrders(String outletId, Date beginDate, Date currentDate, String orderId) {
	//		List<ItemwiseRfReportData> datas = new ArrayList<ItemwiseRfReportData>();
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(Ticket.class);
	//			if (StringUtils.isNotBlank(outletId)) {
	//				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
	//			}
	//
	//			if (StringUtils.isNotBlank(orderId)) {
	//				criteria.add(Restrictions.eq(Ticket.PROP_ID, orderId));
	//			}
	//			else {
	//				criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, beginDate, currentDate));
	//				criteria.add(Restrictions.isNotNull(Ticket.PROP_REFERRER_ID));
	//			}
	//
	//			criteria.addOrder(getDefaultOrder());
	//
	//			List<Ticket> list = criteria.list();
	//
	//			double grandTotalItemPrice = 0;
	//			double grandTotalItemDiscount = 0;
	//			double grandTotalItemTax = 0;
	//			double grandTotalRFAmount = 0;
	//			double grandTotalNonRFAmount = 0;
	//			double grandTotalLDFee = 0;
	//			double grandTotalNetSales = 0;
	//
	//			if (list != null && list.size() > 0) {
	//				for (Ticket ticket : list) {
	//					//PosLog.debug(getClass(), "Processing ticket " + ticket.getId());
	//
	//					boolean firstItem = true;
	//					boolean hasRfItemInTicket = false;
	//					List<TicketItem> ticketItems = ticket.getTicketItems();
	//					int pathologyItem = 0;
	//					if (ticketItems != null) {
	//						double totalItemPrice = 0;
	//						double totalItemDiscount = 0;
	//						double totalItemTax = 0;
	//						double totalRFAmount = 0;
	//						double totalNonRFAmount = 0;
	//						double totalLDFee = 0;
	//						double totalNetSales = 0;
	//
	//						for (TicketItem ticketItem : ticketItems) {
	//
	//							if (ticketItem == null) {
	//								PosLog.error(getClass(), "Found null ticket item for ticket " + ticket.getId());
	//								continue;
	//							}
	//							if (ticketItem.isReturnedSource() || ticketItem.isItemReturned() || ticketItem.isVoided()) {
	//								continue;
	//							}
	//
	//							//PosLog.debug(getClass(), "Processing ticket item " + ticketItem.getId());
	//
	//							if (ProductType.match(ticketItem.getProductType(), ProductType.PATHOLOGY)
	//									|| ProductType.match(ticketItem.getProductType(), ProductType.SERVICES)) {
	//								ItemwiseRfReportData ticketItemData = new ItemwiseRfReportData();
	//								ticketItemData.populateTicketItemData(ticketItem);
	//								if (firstItem) {
	//									ticketItemData.setOrderId(ticket.getId());
	//									firstItem = false;
	//								}
	//								datas.add(ticketItemData);
	//
	//								totalItemPrice += ticketItemData.getItemPrice();
	//								totalItemDiscount += ticketItemData.getItemDiscount();
	//								totalItemTax += ticketItemData.getItemTax();
	//								totalRFAmount += ticketItemData.getRf();
	//								totalNonRFAmount += ticketItemData.getNonRf();
	//								totalLDFee += ticketItemData.getLabDoctorFee();
	//								totalNetSales += ticketItemData.getNet();
	//
	//								grandTotalItemPrice += ticketItemData.getItemPrice();
	//								grandTotalItemDiscount += ticketItemData.getItemDiscount();
	//								grandTotalItemTax += ticketItemData.getItemTax();
	//								grandTotalRFAmount += ticketItemData.getRf();
	//								grandTotalNonRFAmount += ticketItemData.getNonRf();
	//								grandTotalLDFee += ticketItemData.getLabDoctorFee();
	//								grandTotalNetSales += ticketItemData.getNet();
	//
	//								hasRfItemInTicket = true;
	//								pathologyItem++;
	//							}
	//						}
	//						if (hasRfItemInTicket) {
	//							if (pathologyItem > 0) {
	//								String netSalesType = ticket.getRfOnNetSalesType();
	//								String rateOnNetSales = ticket.getRfRateOnNetSales();
	//								double commissionOnNetSales = ticket.getReferrerFeeOnNetSales();
	//
	//								if (commissionOnNetSales != 0) {
	//
	//									ItemwiseRfReportData netSalesTicketData = new ItemwiseRfReportData();
	//									netSalesTicketData.setItemName(ReportUtil.getBoldText("Net sales"));
	//									netSalesTicketData.setRfPercentage(NumberUtil.parseDouble(ticket.getRfRateOnNetSales()));
	//									netSalesTicketData.setRfPercentageType(netSalesType);
	//									netSalesTicketData.setRf(Math.round(commissionOnNetSales));
	//									netSalesTicketData.setNonRf(Math.round((-1) * commissionOnNetSales));
	//									netSalesTicketData.setNet(Math.round((-1) * commissionOnNetSales));
	//									datas.add(netSalesTicketData);
	//
	//									totalRFAmount += netSalesTicketData.getRf();
	//									totalNonRFAmount += netSalesTicketData.getNonRf();
	//									totalNetSales += netSalesTicketData.getNet();
	//
	//									grandTotalRFAmount += netSalesTicketData.getRf();
	//									grandTotalNonRFAmount += netSalesTicketData.getNonRf();
	//									grandTotalNetSales += netSalesTicketData.getNet();
	//								}
	//
	//								ItemwiseRfReportData ticketData = new ItemwiseRfReportData();
	//								ticketData.setItemName(ReportUtil.getBoldText(POSConstants.TOTAL));
	//								ticketData.setItemPrice(totalItemPrice);
	//								ticketData.setItemDiscount(totalItemDiscount);
	//								ticketData.setItemTax(totalItemTax);
	//								ticketData.setRf(Math.round(totalRFAmount));
	//								ticketData.setNonRf(Math.round(totalNonRFAmount));
	//								ticketData.setNet(Math.round(totalNetSales));
	//								ticketData.setLabDoctorFee(totalLDFee);
	//
	//								ticketData.setRfOnNetPercentageType(netSalesType);
	//								ticketData.setRfOnNetPercentage(rateOnNetSales);
	//								ticketData.setRfOnNetSales(commissionOnNetSales);
	//								double finalNet = totalNetSales - commissionOnNetSales;
	//								ticketData.setFinalNet(finalNet);
	//								datas.add(ticketData);
	//							}
	//						}
	//					}
	//				}
	//
	//				ItemwiseRfReportData geandTotalData = new ItemwiseRfReportData();
	//				geandTotalData.setItemName(ReportUtil.getBoldText(POSConstants.GRAND_TOTAL));
	//				geandTotalData.setItemPrice(grandTotalItemPrice);
	//				geandTotalData.setItemDiscount(grandTotalItemDiscount);
	//				geandTotalData.setItemTax(grandTotalItemTax);
	//				geandTotalData.setRf(Math.round(grandTotalRFAmount));
	//				geandTotalData.setNonRf(Math.round(grandTotalNonRFAmount));
	//				geandTotalData.setNet(Math.round(grandTotalNetSales));
	//				geandTotalData.setLabDoctorFee(grandTotalLDFee);
	//				datas.add(geandTotalData);
	//			}
	//		}
	//		return datas;
	//	}

	//	public List<SalesDetailsReportData> findSalesDetailReportOrders(String outletId, Date beginDate, Date endDate, Customer agent) {
	//		List<SalesDetailsReportData> datas = new ArrayList<SalesDetailsReportData>();
	//		try (Session session = createNewSession()) {
	//			Criteria criteria = session.createCriteria(Ticket.class);
	//
	//			addDeletedFilter(criteria);
	//			criteria.add(Restrictions.or(Restrictions.isNull(Ticket.PROP_VOIDED), Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE)));
	//
	//			if (StringUtils.isNotBlank(outletId)) {
	//				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
	//			}
	//			criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, beginDate, endDate));
	//
	//			if (agent != null) {
	//				criteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, agent.getId()));
	//			}
	//			criteria.addOrder(getDefaultOrder());
	//
	//			List<Ticket> list = criteria.list();
	//
	//			if (list != null && list.size() > 0) {
	//				for (Ticket ticket : list) {
	//					SalesDetailsReportData data = new SalesDetailsReportData(ticket);
	//					datas.add(data);
	//				}
	//			}
	//		}
	//		return datas;
	//	}

	public List<SalesDetailsReportData> findSalesDetailReportOrders(String outletId, Date beginDate, Date endDate, Customer agent) {
		return findSalesDetailReportOrders(outletId, beginDate, endDate, agent, "");
	}

	public List<SalesDetailsReportData> findSalesDetailReportOrders(String outletId, Date beginDate, Date endDate, Customer agent, String orderId) {
		return findSalesDetailReportOrders(outletId, beginDate, endDate, agent, orderId, null);
	}

	public List<SalesDetailsReportData> findSalesDetailReportOrders(String outletId, Date beginDate, Date endDate, Customer agent, String orderId,
			Customer customer) {
		Map<String, String> referrerNameMap = new HashMap<String, String>();
		List<SalesDetailsReportData> resultLists = new ArrayList<SalesDetailsReportData>();
		try (Session session = createNewSession()) {
			{
				Criteria criteria = createTicketCriteria(orderId, outletId, beginDate, endDate, agent, false, session, customer);
				ProjectionList projectionList = createGroupProjection();
				projectionList.add(Projections.sum(TicketItem.PROP_SUBTOTAL_AMOUNT), SalesDetailsReportData.JSON_PROP_GROSS_SALES);
				projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_DISCOUNT), SalesDetailsReportData.JSON_PROP_DISCOUNT);
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(SalesDetailsReportData.class));
				resultLists.addAll(criteria.list());
			}
			{
				Criteria criteria = createTicketCriteria(orderId, outletId, beginDate, endDate, agent, true, session, customer);
				ProjectionList projectionList = createGroupProjection();
				projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_TOTAL), SalesDetailsReportData.JSON_PROP_REFUND);
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(SalesDetailsReportData.class));
				resultLists.addAll(criteria.list());
			}
			{
				Criteria criteria = createTicketCriteria(orderId, outletId, beginDate, endDate, agent, null, session, customer);
				ProjectionList projectionList = createGroupProjection();
				//projectionList.add(Projections.groupProperty(Ticket.PROP_REFERRER_FEE_ON_REPORT), SalesDetailsReportData.JSON_PROP_AGENT_FEE);
				projectionList.add(Projections.groupProperty("t." + Ticket.PROP_DUE_AMOUNT), SalesDetailsReportData.JSON_PROP_DUE_AMOUNT);
				//projectionList.add(Projections.sum(TicketItem.PROP_ADJUSTED_DISCOUNT), SalesDetailsReportData.JSON_PROP_DISCOUNT);
				//projectionList.add(Projections.sum(TicketItem.PROP_LAB_DOCTOR_FEE), SalesDetailsReportData.JSON_PROP_LAB_DOCTOR_FEE);

				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(SalesDetailsReportData.class));
				resultLists.addAll(criteria.list());
			}
			{
				resultLists.addAll(calculateLDF(orderId, outletId, beginDate, endDate, agent, session, customer));
			}
			{
				Criteria criteria = createTicketCriteria(orderId, outletId, beginDate, endDate, agent, null, session, customer);
				ProjectionList projectionList = createGroupProjection();
				projectionList.add(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_REPORT), SalesDetailsReportData.JSON_PROP_AGENT_FEE);
				projectionList.add(Projections.sum(TicketItem.PROP_REFERRER_FEE_ON_NET_SALES), SalesDetailsReportData.JSON_PROP_RF_ON_NET_SALES);
				criteria.setProjection(projectionList);
				criteria.setResultTransformer(Transformers.aliasToBean(SalesDetailsReportData.class));
				resultLists.addAll(criteria.list());
			}

			Map<String, List<SalesDetailsReportData>> resultMap = resultLists.stream().collect(Collectors.groupingBy(SalesDetailsReportData::getOrderId));
			List<SalesDetailsReportData> datas = new ArrayList<SalesDetailsReportData>();
			for (String ticketId : resultMap.keySet()) {
				SalesDetailsReportData data = new SalesDetailsReportData();
				List<SalesDetailsReportData> reportDatas = resultMap.get(ticketId);
				for (Iterator<SalesDetailsReportData> iterator = reportDatas.iterator(); iterator.hasNext();) {
					SalesDetailsReportData sData = (SalesDetailsReportData) iterator.next();

					String referrerId = sData.getAgentId();
					if (StringUtils.isNotBlank(referrerId)) {
						data.setAgentId(referrerId);
						String referrerName = referrerNameMap.get(referrerId);
						if (StringUtils.isBlank(referrerName)) {
							referrerName = CustomerDAO.getInstance().findCustomerDisplayName(referrerId);
							referrerNameMap.put(referrerId, referrerName);
						}
						data.setAgentName(referrerName);
						data.populateAgentDisplay();
					}
					data.setOrderId(sData.getOrderId());
					data.setOutletId(sData.getOutletId());
					data.setOrderDate(sData.getOrderDate());
					data.setPatientId(sData.getPatientId());

					String patientId = sData.getPatientId();
					if (StringUtils.isNotBlank(patientId)) {
						data.setCustomerId(patientId);

						String patientName = sData.getPatientName();
						if (StringUtils.isBlank(patientName)) {
							patientName = referrerNameMap.get(patientId);
							if (StringUtils.isBlank(patientName)) {
								patientName = CustomerDAO.getInstance().findCustomerDisplayName(patientId);
								referrerNameMap.put(patientId, patientName);
							}
						}
						data.setCustomerName(patientName);
					}

					data.setGrossSales(data.getGrossSales() + sData.getGrossSales());
					data.setDiscount(data.getDiscount() + sData.getDiscount());
					data.setRefund(data.getRefund() + sData.getRefund());
					data.setAgentFee(data.getAgentFee() + sData.getAgentFee() + sData.getReturnItemRF());
					data.setrFOnNetSales(data.getrFOnNetSales() + sData.getrFOnNetSales());
					data.setlabDoctorFee(data.getlabDoctorFee() + sData.getlabDoctorFee());

					if (data.getDueAmount() == 0) {
						data.setDueAmount(sData.getDueAmount());
					}
				}
				data.calculateNetSales();
				datas.add(data);
			}
			datas.sort(Comparator.comparing(SalesDetailsReportData::getOrderDate, Comparator.nullsLast(Comparator.reverseOrder())));

			AtomicInteger counter = new AtomicInteger(1);
			datas = datas.stream().peek(data -> data.setSerialNumber(counter.getAndIncrement())).collect(Collectors.toList());

			return datas;
		}
	}

	private List<SalesDetailsReportData> calculateLDF(String orderId, String outletId, Date beginDate, Date endDate, Customer referrer, Session session,
			Customer customer) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "t"); //$NON-NLS-1$

		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.eq(TicketItem.PROP_PARENT_TICKET_OUTLET_ID, outletId));

		// Optional filters example:
		if (StringUtils.isNotBlank(orderId)) {
			criteria.add(Restrictions.eq(TicketItem.PROP_PARENT_TICKET_ID, orderId));
		}
		else {
			if (beginDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, beginDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			}
			if (referrer != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_REFERRER_ID, referrer.getId()));
			}
			if (customer != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_CUSTOMER_ID, customer.getId()));
			}
		}

		// Projection list with grouping
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(TicketItem.PROP_PARENT_TICKET_ID), SalesDetailsReportData.JSON_PROP_ORDER_ID);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_REFERRER_ID), SalesDetailsReportData.JSON_PROP_AGENT_ID);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_CREATE_DATE), SalesDetailsReportData.JSON_PROP_ORDER_DATE);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_CUSTOMER_ID), SalesDetailsReportData.JSON_PROP_PATIENT_ID);

		//@formatter:off
		String sql = String.format("("
				+ "select "
				+ "coalesce(sum(ti.%s), 0) "
				+ "from ticket_item ti "
				+ "where "
				+ "ti.parent_ticket_id = {alias}.parent_ticket_id "
				+ "and ti.voided = false "
				+ "and ti.create_date >= '%s' "
				+ "and ti.create_date < '%s' "
				+ ") as totalLabDoctorFee", TicketItem.PROP_LAB_DOCTOR_FEE_COL_NAME, beginDate, endDate);
		Projection sqlProjection = Projections.sqlProjection(sql, new String[] { "totalLabDoctorFee" }, new Type[] { DoubleType.INSTANCE });
		projectionList.add(sqlProjection, SalesDetailsReportData.JSON_PROP_LAB_DOCTOR_FEE);
		//@formatter:on

		criteria.setProjection(projectionList);
		criteria.setResultTransformer(Transformers.aliasToBean(SalesDetailsReportData.class));
		return criteria.list();

	}

	private ProjectionList createGroupProjection() {
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_ID), SalesDetailsReportData.JSON_PROP_ORDER_ID);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_REFERRER_ID), SalesDetailsReportData.JSON_PROP_AGENT_ID);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_CREATE_DATE), SalesDetailsReportData.JSON_PROP_ORDER_DATE);
		projectionList.add(Projections.groupProperty("t." + Ticket.PROP_CUSTOMER_ID), SalesDetailsReportData.JSON_PROP_PATIENT_ID);
		return projectionList;
	}

	private Criteria createTicketCriteria(String orderId, String outletId, Date beginDate, Date endDate, Customer agent, Boolean itemReturned, Session session,
			Customer customer) {
		Criteria criteria = session.createCriteria(TicketItem.class).createAlias(TicketItem.PROP_TICKET, "t"); //$NON-NLS-1$

		criteria.add(Restrictions.eq(TicketItem.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.or(Restrictions.isNull("t." + Ticket.PROP_VOIDED), Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)));
		criteria.add(Restrictions.eq("t." + Ticket.PROP_OUTLET_ID, outletId)); //$NON-NLS-1$

		if (StringUtils.isNotBlank(orderId)) {
			criteria.add(Restrictions.eq("t." + Ticket.PROP_ID, orderId)); //$NON-NLS-1$
		}
		else {
			if (beginDate != null) {
				criteria.add(Restrictions.ge(TicketItem.PROP_CREATE_DATE, beginDate));
			}
			if (endDate != null) {
				criteria.add(Restrictions.lt(TicketItem.PROP_CREATE_DATE, endDate));
			}
			if (agent != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_REFERRER_ID, agent.getId()));
			}
			if (customer != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_CUSTOMER_ID, customer.getId()));
			}
		}
		if (itemReturned != null) {
			criteria.add(Restrictions.eq(TicketItem.PROP_ITEM_RETURNED, itemReturned));
		}

		return criteria;
	}

	public double getUnpaidRFAmount(String outletId, Date beginDate, Date currentDate, Customer referrer) {
		try (Session session = createNewSession()) {
			Criteria criteria = createCommonCriteria(outletId, beginDate, currentDate, referrer, false, session);

			ProjectionList projectionList = Projections.projectionList();
			projectionList.add(Projections.sum(Ticket.PROP_RF_PAYABLE_AMOUNT));
			projectionList.add(Projections.sum(Ticket.PROP_REFERRER_FEE_PAID_AMOUNT));
			criteria.setProjection(projectionList);

			Object[] objects = (Object[]) criteria.uniqueResult();
			if (objects[0] == null) {
				return 0;
			}
			double payableAmount = (objects[0] != null) ? ((Number) objects[0]).doubleValue() : 0;
			double paidAmount = (objects[1] != null) ? ((Number) objects[1]).doubleValue() : 0;

			return NumberUtil.round(payableAmount - paidAmount);
		}
	}

	public List<Ticket> getUnpaidRFTicket(String outletId, Date beginDate, Date currentDate, Customer referrer) {
		try (Session session = createNewSession()) {
			Criteria criteria = createCommonCriteria(outletId, beginDate, currentDate, referrer, false, session);

			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));
			List<Ticket> list = criteria.list();
			return list;
		}
	}

	private Criteria createCommonCriteria(String outletId, Date beginDate, Date currentDate, Customer referrer, boolean isLabDoctor, Session session) {
		Criteria criteria = session.createCriteria(Ticket.class);

		if (isLabDoctor) {
			criteria.add(Restrictions.or(Restrictions.eq(Ticket.PROP_LAB_DOCTOR_FEE_PAID, false), Restrictions.isNull(Ticket.PROP_LAB_DOCTOR_FEE_PAID)));
		}
		else {
			criteria.add(Restrictions.or(Restrictions.eq(Ticket.PROP_REFERRER_FEE_PAID, false), Restrictions.isNull(Ticket.PROP_REFERRER_FEE_PAID)));
		}

		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
		}
		criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, beginDate, currentDate));

		if (referrer != null) {
			criteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, referrer.getId()));
		}

		return criteria;
	}

	public void findTicketsByDoctorOrAgentOrPatient(PaginationSupport listModel, Customer customer) {
		findTicketsByDoctorOrAgentOrPatient(listModel, customer, null);
	}

	public void findTicketsByDoctorOrAgentOrPatient(PaginationSupport listModel, Customer customer, List<ProductType> productTypes) {
		findTicketsByDoctorOrAgentOrPatient(listModel, customer, productTypes, null, false, false);
	}

	public void findTicketsByDoctorOrAgentOrPatient(PaginationSupport listModel, Customer customer, List<ProductType> productTypes, BookingInfo bookingInfo,
			boolean isShowAscendingOrder, boolean showOpenTicketsOnly) {
		findTicketsByDoctorOrAgentOrPatient(listModel, customer, productTypes, bookingInfo, isShowAscendingOrder, showOpenTicketsOnly, null);
	}

	public void findTicketsByDoctorOrAgentOrPatient(PaginationSupport listModel, Customer customer, List<ProductType> productTypes, BookingInfo bookingInfo,
			boolean isShowAscendingOrder, boolean showOpenTicketsOnly, String outletId) {
		if (listModel == null || customer == null) {
			return;
		}

		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			if (productTypes != null && !productTypes.isEmpty()) {
				criteria.createAlias(Ticket.PROP_TICKET_ITEMS, "items");
				Disjunction disjunction = Restrictions.disjunction();
				for (ProductType productType : productTypes) {
					disjunction.add(Restrictions.eq("items." + TicketItem.PROP_PRODUCT_TYPE, productType.name()));
				}
				criteria.add(disjunction);
			}

			if (showOpenTicketsOnly) {
				criteria.add(Restrictions.or(Restrictions.gt(Ticket.PROP_DUE_AMOUNT, new Double(0)),
						Restrictions.and(Restrictions.eq(Ticket.PROP_TYPE, TicketType.IPD.getTypeNo()), Restrictions.eq(Ticket.PROP_CLOSED, Boolean.FALSE))));
			}
			if (customer instanceof Doctor) {
				Disjunction disjunction = Restrictions.disjunction();
				disjunction.add(Restrictions.eq(Ticket.PROP_DOCTOR_ID, customer.getId()));
				disjunction.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, customer.getId()));
				criteria.add(disjunction);
			}
			else if (customer instanceof Agent) {
				criteria.add(Restrictions.eq(Ticket.PROP_REFERRER_ID, customer.getId()));
			}
			else {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customer.getId()));
			}
			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			}
			if (bookingInfo != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_ADMISSION_ID, bookingInfo.getBookingId()));
			}
			addDeletedFilter(criteria);

			listModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(listModel.getCurrentRowIndex());
			criteria.setMaxResults(listModel.getPageSize());
			if (isShowAscendingOrder) {
				criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));
			}
			else {
				criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			}
			criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);
			listModel.setRows(criteria.list());

		}
	}

	@SuppressWarnings("unchecked")
	public List<Ticket> findTopSellers(Session session, String outletId, Date fromDate, Date toDate, Integer maxResult) {
		List<String> referrerIds = CustomerDAO.getInstance().findReferrerIds();
		if (referrerIds == null || referrerIds.isEmpty()) {
			return new ArrayList<>();
		}
		Criteria criteria = session.createCriteria(Ticket.class);
		addDeletedFilter(criteria);
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.sum(Ticket.PROP_TOTAL_AMOUNT), Ticket.PROP_TOTAL_AMOUNT);
		projectionList.add(Projections.groupProperty(Ticket.PROP_REFERRER_ID), Ticket.PROP_REFERRER_ID);
		criteria.setProjection(projectionList);
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.in(Ticket.PROP_REFERRER_ID, referrerIds));
		criteria.add(Restrictions.isNotNull(Ticket.PROP_REFERRER_ID));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
		}
		if (fromDate != null && toDate != null) {
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		}
		criteria.setMaxResults(maxResult);
		criteria.addOrder(Order.desc(Ticket.PROP_TOTAL_AMOUNT));
		criteria.setResultTransformer(Transformers.aliasToBean(Ticket.class));
		return criteria.list();
	}

	public String getTicketIds(String patientId, String invoiceNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.isNull(Ticket.PROP_VOIDED), Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE)));

			if (StringUtils.isNotBlank(patientId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, patientId));
			}
			if (StringUtils.isNotBlank(invoiceNo)) {
				criteria.add(Restrictions.ilike(Ticket.PROP_ID, invoiceNo, MatchMode.END));
			}
			criteria.addOrder(getDefaultOrder());
			criteria.setProjection(Projections.property(Ticket.PROP_ID));

			return (String) criteria.list().stream().collect(Collectors.joining(", "));
		}
	}

	public Ticket getOutdoorTicket(String outdoorTicketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.or(Restrictions.isNull(Ticket.PROP_VOIDED), Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE)));
			if (StringUtils.isNotBlank(outdoorTicketId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTDOOR_TICKET_ID, outdoorTicketId));
			}
			criteria.setMaxResults(1);

			Ticket ticket = (Ticket) criteria.uniqueResult();
			if (ticket == null) {
				return null;
			}

			TicketDAO.getInstance().loadFullTicket(ticket);
			return ticket;
		}
	}

	public List<Ticket> findTopReferrerSalesCount(Session session, String outletId, Date fromDate, Date toDate, Integer maxResult) {
		List<String> referrerIds = CustomerDAO.getInstance().findReferrerIds();
		if (referrerIds == null || referrerIds.isEmpty()) {
			return new ArrayList<>();
		}
		Criteria criteria = session.createCriteria(Ticket.class);
		addDeletedFilter(criteria);
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.count(Ticket.PROP_ID), "count"); //$NON-NLS-1$
		projectionList.add(Projections.groupProperty(Ticket.PROP_REFERRER_ID));
		criteria.setProjection(projectionList);
		criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
		criteria.add(Restrictions.in(Ticket.PROP_REFERRER_ID, referrerIds));
		criteria.add(Restrictions.isNotNull(Ticket.PROP_REFERRER_ID));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
		}
		if (fromDate != null && toDate != null) {
			criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
		}
		criteria.addOrder(Order.desc("count")); //$NON-NLS-1$
		if (maxResult > 0) {
			criteria.setMaxResults(maxResult);
		}
		List list = criteria.list();

		List<Ticket> tickets = new ArrayList<>();
		if (list == null || list.size() == 0)
			return tickets;

		for (int i = 0; i < list.size(); i++) {
			Object[] o = (Object[]) list.get(i);
			int index = 0;

			Ticket ticket = new Ticket();
			ticket.setTotalAmount((double) HibernateProjectionsUtil.getInt(o, index++));
			ticket.setReferrerId((String) o[index++]);
			tickets.add(ticket);
		}
		return tickets;
	}

	public Date getOrderDate(String ticketId, String outletId) {
		try (Session session = createNewSession()) {
			return getOrderDate(ticketId, outletId, session);
		}
	}

	public Date getOrderDate(String ticketId, String outletId, Session session) {
		Criteria criteria = session.createCriteria(Ticket.class);

		if (StringUtils.isNotBlank(ticketId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_ID, ticketId));
		}

		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
		}

		criteria.setProjection(Projections.property(Ticket.PROP_CREATE_DATE));
		return (Date) criteria.uniqueResult();
	}

	public List<Ticket> getUnpaidTickets(Customer customer, String outletId) {
		if (customer == null) {
			return Collections.emptyList();
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customer.getId()));
			criteria.add(Restrictions.eq(Ticket.PROP_PAID, false));
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, false));
			criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outletId));
			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));
			List<Ticket> list = criteria.list();
			for (Ticket ticket : list) {
				initialize(ticket, session);
			}
			return list;
		}
	}

	public List<Ticket> findByAdmissionId(String bookingId, TicketType ticketType) {
		Session session = null;
		Criteria criteria = null;
		try {
			session = createNewSession();
			criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_ADMISSION_ID, bookingId));
			if (ticketType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TYPE, ticketType.getTypeNo()));
			}
			criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			List ticketList = criteria.list();
			return ticketList;
		} finally {
			closeSession(session);
		}
	}

	public Ticket findLastTicket(Customer customer) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, customer.getId()));
			criteria.setMaxResults(1);
			criteria.addOrder(Order.desc(Ticket.PROP_CREATE_DATE));
			return (Ticket) criteria.uniqueResult();
		}
	}

	public List<Ticket> findTicketsByProject(Outlet outlet, Date from, Date to, Project project) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, outlet.getId()));
			}

			if (from != null && to != null) {
				criteria.add(Restrictions.between(Ticket.PROP_CREATE_DATE, from, to));
			}

			if (project != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_PROJECT_ID, project.getId()));
			}

			criteria.addOrder(Order.asc(Ticket.PROP_CREATE_DATE));

			return criteria.list();
		}
	}

	public List<Ticket> getTicketListByIds(List<String> ids) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.in(Ticket.PROP_ID, ids));
			return criteria.list();
		}
	}

	public List<Ticket> getSurgeyTickets(String selectedOutletId, Date fromDate, Date toDate, TicketType ticketType) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));

			if (StringUtils.isNotBlank(selectedOutletId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_OUTLET_ID, selectedOutletId));
			}
			if (fromDate != null) {
				criteria.add(Restrictions.ge(Ticket.PROP_CREATE_DATE, fromDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.le(Ticket.PROP_CREATE_DATE, toDate));
			}
			if (ticketType != null) {
				criteria.add(Restrictions.eq(Ticket.PROP_TYPE, ticketType.getTypeNo()));
			}

			return criteria.list();
		}
	}

	public double getTotalDueByPatientId(String patientId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);
			addDeletedFilter(criteria);
			criteria.add(Restrictions.eq(Ticket.PROP_CUSTOMER_ID, patientId));
			criteria.setProjection(Projections.sum(Ticket.PROP_DUE_AMOUNT));
			Object uniqueResult = criteria.uniqueResult();
			if (uniqueResult instanceof Double) {
				Double dueAmount = (Double) uniqueResult;
				return dueAmount;
			}
			else
				return 0.0;
		}
	}

	public void doCalculateSharedItems(Collection<SharedCalcItemModel> items, Date startDate, Date endDate, String outletId) throws PosException {
		if (items.isEmpty()) {
			return;
		}

		try (Session session = createNewSession()) {
			Transaction beginTransaction = session.beginTransaction();
			Gson gson = new Gson();

			for (SharedCalcItemModel sharedCalcItemModel : items) {
				List<TicketItem> ticketItems = TicketItemDAO.getInstance().findTicketItemsByMenuItemId(outletId, startDate, endDate,
						sharedCalcItemModel.getMenuItemId(), session);

				Ticket inventoryAdjustmentTicket = null;
				TicketItem inventoryAdjustmentTicketItem = null;

				double distributedQty = 0d;
				for (Iterator<TicketItem> iterator = ticketItems.iterator(); iterator.hasNext();) {
					TicketItem tItem = iterator.next();

					Ticket ticket = loadFullTicket(tItem.getParentTicketId(), tItem.getParentTicketOutletId(), session);
					TicketItem ticketItem = ticket.getTicketItem(tItem.getId(), true);

					if (ticketItem != null
							/*&& !ticketItem.isSharedItemCalculated()*/ && sharedCalcItemModel.getMenuItemId().equals(ticketItem.getMenuItemId())) {
						double sharedQty;
						if (!iterator.hasNext()) {
							sharedQty = NumberUtil.round(sharedCalcItemModel.getQuantity().doubleValue() - distributedQty);
						}
						else {
							sharedQty = NumberUtil.round(sharedCalcItemModel.getQuantity().doubleValue() / sharedCalcItemModel.getUsedQuantity().doubleValue());
							distributedQty += sharedQty;
						}

						ticketItem.setQuantity(sharedQty);
						ticketItem.setInventoryAdjustQty(sharedQty);

						IUnit unit = DataProvider.get().getUnitById(ticketItem.getUnitName(), ticketItem.getUnitType());
						ticketItem.getMenuItem().setTicketItemUnitPriceAndCost(ticketItem, ticketItem.getMenuItem(), unit, ticket);
						ticketItem.calculatePrice();

						ticketItem.putSharedItemCalculated(true);
						ticket.calculatePrice();

						if (inventoryAdjustmentTicket == null) {
							inventoryAdjustmentTicket = ticket;
							inventoryAdjustmentTicketItem = ticketItem;
						}
					}

					saveOrUpdate(ticket, session);
				}

				double unAdjustQuantity = sharedCalcItemModel.getQuantity().doubleValue() - sharedCalcItemModel.getInventoryAdjustedQty().doubleValue();

				if (inventoryAdjustmentTicket != null && inventoryAdjustmentTicketItem != null && unAdjustQuantity > 0) {
					HashMap<InvMapKey, Double> itemMap = new HashMap<>();
					InvMapKey key = new InvMapKey(inventoryAdjustmentTicketItem);
					itemMap.put(key, NumberUtil.round(unAdjustQuantity));

					adjustInventory(inventoryAdjustmentTicket, itemMap, InventoryTransactionType.OUT, InventoryTransaction.REASON_TICKET_SALES,
							DataProvider.get().getDefaultOutLocation(), session);
				}

			}

			ActionHistory history = new ActionHistory();
			history.setActionName(ActionHistory.ActionType.SHARED_ITEM_CALCULATION.getDisplayName());
			history.setActionType(ActionHistory.ActionType.SHARED_ITEM_CALCULATION.name());
			history.setDescription(gson.toJson(items));
			history.setPerformer(DataProvider.get().getCurrentUser());
			history.setActionTime(StoreDAO.getServerTimestamp());
			history.setOutletId(DataProvider.get().getOutletId());
			history.setEventTime(startDate);
			ActionHistoryDAO.getInstance().save(history, session);

			beginTransaction.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), "Failed to update stock balance during shared item calculation.", e);
			throw new PosException("Shared item calculation failed.");
		}
	}

	public List<Ticket> findAllTicketsByAdmissionId(String addmissionId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(Ticket.class);

			addDeletedFilter(criteria);

			criteria.add(Restrictions.eq(Ticket.PROP_VOIDED, Boolean.FALSE));
			if (StringUtils.isNotBlank(addmissionId)) {
				criteria.add(Restrictions.eq(Ticket.PROP_ADMISSION_ID, addmissionId));
			}

			return loadFullTickets(criteria.list(), session);
		}

	}
}