package com.floreantpos.model.util;

import java.util.Calendar;
import java.util.Date;

import com.floreantpos.PosLog;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.RentalDurationType;
import com.floreantpos.model.RentalStatus;
import com.floreantpos.model.TicketItem;
import com.floreantpos.util.NumberUtil;

public class RentalPriceCalculator {

	/**
	* Always round up fractional packages to charge full package price.
	* Example: 1.2 × 300 = 360 → 2 × 300 = 600
	*/
	private static double roundUpToNearestPackage(double basePrice, double durationCount) {
		double fullPackages = durationCount;
		return basePrice * fullPackages;
	}

	/**
	* Converts a duration value from one type to another (e.g. 3 DAYS → 72 HOURS → 1.5 DAYS).
	*/
	public static double convertDuration(RentalDurationType fromType, RentalDurationType toType, double fromValue) {
		double totalHours = fromType.getHoursPerUnit() * fromValue;
		return totalHours / toType.getHoursPerUnit();
	}

	/**
	* Calculates duration between two dates in a given type (rounded up).
	*/
	public static double calculateDurationBetween(Date start, Date end, RentalDurationType type) {
		boolean hourly = type == RentalDurationType.HOUR;
		Date startDate = hourly ? start : DateUtil.startOfDay(start);
		Date endDate = hourly ? end : DateUtil.startOfDay(end);
		long diffMillis = endDate.getTime() - startDate.getTime();
		double diffHours = diffMillis / (1000.0 * 60 * 60);
		double duration = diffHours / type.getHoursPerUnit();
		return ceil(duration);
	}

	public static double ceil(double duration) {
		if (duration <= 0) {
			return 1;
		}
		return Math.ceil(duration);
	}

	public void calculatePrice(TicketItem ticketItem) {
		MenuItem menuItem = ticketItem.getMenuItem();
		calculatePrice(menuItem.getPrice(), RentalDurationType.valueOf(menuItem.getRentalDurationType()), menuItem.getRentalDuration(), ticketItem);
	}

	public void calculatePrice(Double unitPrice, RentalDurationType baseType, Integer baseDuration, TicketItem ticketItem) {
		double totalPrice = calculateRentalPrice(unitPrice, baseType, baseDuration, RentalDurationType.valueOf(ticketItem.getRentalDurationType()),
				ticketItem.getRentalDuration());
		ticketItem.setUnitPrice(totalPrice);
	}

	public double calculateRentalPrice(double basePricePerUnit, RentalDurationType baseType, Integer baseDuration, RentalDurationType selectedType,
			int selectedDuration) {
		double baseHours = baseType.getHoursPerUnit() * baseDuration;
		double selectedHours = selectedType.getHoursPerUnit() * selectedDuration;
		// How many base rental periods fit into selected duration
		double unitCount = selectedHours / baseHours;
		// Always round up to next full package
		return roundUpToNearestPackage(basePricePerUnit, unitCount);
	}

	public boolean reCalculateRentalPriceIfNeeded(TicketItem ticketItem) {
		if (ticketItem.getRentalStatus() != null && RentalStatus.valueOf(ticketItem.getRentalStatus()) == RentalStatus.CLOSED) {
			return false;
		}
		String rentalDurationType = ticketItem.getRentalDurationType();
		boolean hourly = RentalDurationType.valueOf(rentalDurationType) == RentalDurationType.HOUR;

		Date startDate = hourly ? ticketItem.getRentalStartDate() : DateUtil.startOfDay(ticketItem.getRentalStartDate());
		Date endDate = calculateRentalEndDate(startDate, ticketItem.getRentalDuration(), rentalDurationType);

		PosLog.debug(getClass(), String.format("Rent start date: %s, Duration: %s %s, Rent end date: %s, Price: %s", startDate, ticketItem.getRentalDuration(),
				rentalDurationType, endDate, NumberUtil.format(ticketItem.getUnitPrice())));
		Date currentDate = hourly ? new Date() : DateUtil.startOfDay(new Date());
		if (!currentDate.after(endDate)) {
			PosLog.debug(getClass(), "Skipped auto calculation.");
			return false;
		}
		Double oldUnitPrice = ticketItem.getUnitPrice();
		PosLog.debug(getClass(), "====Starting rental item price auto calculation.====");
		int calculatedRentalDuration = calculateRentalDuration(startDate, currentDate, rentalDurationType);
		ticketItem.setRentalDuration(calculatedRentalDuration);
		MenuItem menuItem = ticketItem.getMenuItem();
		ticketItem.setUnitPrice(calculateRentalPrice(menuItem.getPrice(), RentalDurationType.valueOf(menuItem.getRentalDurationType()),
				menuItem.getRentalDuration(), RentalDurationType.valueOf(rentalDurationType), calculatedRentalDuration));
		PosLog.debug(getClass(), String.format("After auto calculation: Start date: %s, Duration: %s %s, End date: %s, Price: %s", startDate,
				ticketItem.getRentalDuration(), rentalDurationType, currentDate, NumberUtil.format(ticketItem.getUnitPrice())));

		return oldUnitPrice != ticketItem.getUnitPrice();
	}

	public static int calculateRentalDuration(Date startDate, Date endDate, String durationType) {
		if (startDate == null || endDate == null || durationType == null) {
			return 0;
		}

		long diffMillis = endDate.getTime() - startDate.getTime();
		if (diffMillis <= 0) {
			return 0;
		}

		double diffHours = diffMillis / (1000.0 * 60 * 60); // total hours
		double units = 0;

		switch (RentalDurationType.valueOf(durationType)) {
			case HOUR:
				units = diffHours;
				break;

			case DAY:
				units = diffHours / 24.0;
				break;

			case WEEK:
				units = diffHours / (24.0 * 7);
				break;

			case MONTH:
				units = diffHours / (24.0 * 30); // approx
				break;

			default:
				throw new IllegalArgumentException("Unknown duration type: " + durationType);
		}

		// If you charge for partial usage as a full unit
		return (int) ceil(units); // round up fractional duration
	}

	public Date calculateRentalEndDate(Date startDate, int duration, String durationType) {
		Calendar cal = Calendar.getInstance();
		cal.setTime(startDate);

		switch (RentalDurationType.valueOf(durationType)) {
			case HOUR:
				cal.add(Calendar.HOUR_OF_DAY, duration);
				break;
			case DAY:
				cal.add(Calendar.DAY_OF_MONTH, duration);
				break;
			case WEEK:
				cal.add(Calendar.WEEK_OF_YEAR, duration);
				break;
			case MONTH:
				cal.add(Calendar.MONTH, duration);
				break;
			default:
				throw new IllegalArgumentException("Invalid rental duration type: " + durationType);
		}

		return cal.getTime();
	}

	public static RentalPriceCalculator get() {
		return new RentalPriceCalculator();
	}
}
