/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model;

import java.time.LocalDate;
import java.time.Period;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.swing.ImageIcon;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.apache.commons.lang.StringUtils;
import org.json.JSONObject;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.floreantpos.PosLog;
import com.floreantpos.model.base.BaseCustomer;
import com.floreantpos.model.dao.CustomerGroupDAO;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.report.PatientContactData;
import com.floreantpos.util.AESencrp;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

@JsonIgnoreProperties(ignoreUnknown = true, value = { "deliveryAddresses", "deliveryInstructions" })
@XmlSeeAlso({ Patient.class, Agent.class, Doctor.class })
@JsonSubTypes({ @Type(value = Patient.class), @Type(value = Agent.class), @Type(value = Doctor.class) })
@XmlRootElement
public class Customer extends BaseCustomer implements TimedModel {
	private static final long serialVersionUID = 1L;
	public final static String APPETIZER_PREF = "appetizerPref";//$NON-NLS-1$
	public final static String BILLINGSTATEMENT_PREF = "billingStatementPref";//$NON-NLS-1$
	public final static String COCKTAIL_PREF = "cocktailPref";//$NON-NLS-1$
	public final static String DESSERT_PREF = "dessertPref";//$NON-NLS-1$
	public final static String ENTREE_PREF = "entreePref";//$NON-NLS-1$
	public final static String GOLFBALL_PREF = "golfBallPref";//$NON-NLS-1$
	public final static String MEAL_PREF = "mealPref";//$NON-NLS-1$
	public final static String PAYMENT_PREF = "paymentPref";//$NON-NLS-1$
	public final static String PILLOW_PREF = "pillowPref";//$NON-NLS-1$
	public final static String ROLLAWAYCRIB_PREF = "rollawayCribPref";//$NON-NLS-1$

	public final static String SEATING_PREF = "seatingPref";//$NON-NLS-1$
	public final static String SMOKING_PREF = "smokingPref";//$NON-NLS-1$
	public final static String WINE_PREF = "winePref";//$NON-NLS-1$
	public final static String ROOM_ACCESS_PREF = "roomAccessPref";//$NON-NLS-1$
	public final static String ROOM_LOC_PREF = "roomLocPref";//$NON-NLS-1$
	public final static String ROOM_NUMBER_PREF = "roomNumberPref";//$NON-NLS-1$
	public final static String REMAINING_CURRENT_QUARTER = "club62__Remaining_Current_Quarter_F_B_Spends__c";//$NON-NLS-1$
	public final static String STATUS = "club62__Status__c";//$NON-NLS-1$
	public final static String STREET_1 = "street_1";//$NON-NLS-1$
	public final static String STREET_2 = "street_2";//$NON-NLS-1$
	public final static String STREET_3 = "street_3";//$NON-NLS-1$

	private Boolean selected;
	private transient JSONObject propertiesContainer;

	private boolean updateLastUpdateTime = true;
	private boolean updateSyncTime = false;

	public boolean isUpdateSyncTime() {
		return updateSyncTime;
	}

	public void setUpdateSyncTime(boolean shouldUpdateSyncTime) {
		this.updateSyncTime = shouldUpdateSyncTime;
	}

	public boolean isUpdateLastUpdateTime() {
		return updateLastUpdateTime;
	}

	public void setUpdateLastUpdateTime(boolean shouldUpdateUpdateTime) {
		this.updateLastUpdateTime = shouldUpdateUpdateTime;
	}

	/*[CONSTRUCTOR MARKER BEGIN]*/
	public Customer() {
	}

	/**
	 * Constructor for primary key
	 */
	public Customer(java.lang.String id) {
		super(id);
	}

	/*[CONSTRUCTOR MARKER END]*/

	@Override
	public String toString() {
		String fName = getName();
		if (fName == null) {
			fName = getEmail();
		}
		return fName;
	}

	public enum MemberType {
		//@formatter:off
		MEMBER(0, "Member"),//$NON-NLS-1$
		GUEST(1, "Guest"),//$NON-NLS-1$
		EMPLOYEE(2, "Employee"),//$NON-NLS-1$
		STORE_OWNER(3, "StoreOwner")//$NON-NLS-1$
		;
		//@formatter:on

		private int typeInt;
		private String name;

		private MemberType(int type, String name) {
			this.typeInt = type;
			this.name = name;
		}

		public int getTypeInt() {
			return typeInt;
		}

		public String getName() {
			return name;
		}

		@Override
		public String toString() {
			return name;
		}

		public static MemberType fromInt(int type) {
			MemberType[] values = values();
			for (MemberType memberType : values) {
				if (memberType.getTypeInt() == type) {
					return memberType;
				}
			}

			return MemberType.MEMBER;
		}
	}

	public ImageIcon getImage() {
		return null;
	}

	public String getName() {
		String name = super.getFirstName();
		if (StringUtils.isNotEmpty(super.getLastName())) {
			name += " " + super.getLastName(); //$NON-NLS-1$
		}
		return name;
	}

	@Override
	public void setMobileNo(String mobileNo) {
		super.setMobileNo(mobileNo);
	}

	@Override
	public void setWorkPhoneNo(String workPhoneNo) {
		super.setWorkPhoneNo(workPhoneNo);
	}

	@Override
	public void setHomePhoneNo(String homePhoneNo) {
		super.setHomePhoneNo(homePhoneNo);
	}

	public Boolean isSelected() {
		return selected == null ? Boolean.FALSE : selected;
	}

	public void setSelected(Boolean enable) {
		this.selected = enable;
	}

	@XmlTransient
	@JsonIgnore
	public CustomerGroup getCustomerGroup() {
		if (StringUtils.isNotEmpty(getCustomerGroupId())) {
			return CustomerGroupDAO.getInstance().get(getCustomerGroupId());
		}
		return null;
	}

	public void setCustomerGroup(CustomerGroup customerGroup) {
		String customerGroupId = null;
		if (customerGroup != null) {
			customerGroupId = customerGroup.getId();
		}
		super.setCustomerGroupId(customerGroupId);
	}

	public void setEncryptedSecretCode(String secretCode) {
		try {
			setSecretCode(AESencrp.encrypt(secretCode));
		} catch (Exception e) {
			setSecretCode(secretCode);
		}
	}

	public String getUnencryptedSecretCode() {
		String secretKey = getSecretCode();
		if (StringUtils.isNotEmpty(secretKey)) {
			try {
				secretKey = AESencrp.decrypt(secretKey);
			} catch (Exception e) {
			}
		}
		return secretKey;
	}

	public void setEncryptedPin(String pin) {
		try {
			setPin(AESencrp.encrypt(pin));
		} catch (Exception e) {
			setPin(pin);
		}
	}

	public String getUnencryptedPin() {
		String pin = getPin();
		if (StringUtils.isNotEmpty(pin)) {
			try {
				pin = AESencrp.decrypt(pin);
			} catch (Exception e) {
			}
		}
		return pin;
	}

	public void setEncryptedSignatureApprovalCode(String code) {
		try {
			setSignatureApprovalCode(AESencrp.encrypt(code));
		} catch (Exception e) {
			setSignatureApprovalCode(code);
		}
	}

	public String getUnencryptedSignatureApprovalCode() {
		String code = getSignatureApprovalCode();
		if (StringUtils.isNotEmpty(code)) {
			try {
				code = AESencrp.decrypt(code);
			} catch (Exception e) {
			}
		}
		return code;
	}

	protected void buildPropertis() {
		if (propertiesContainer != null) {
			return;
		}
		String json = super.getProperties();
		if (StringUtils.isEmpty(json)) {
			propertiesContainer = new JSONObject();
			return;
		}
		propertiesContainer = new JSONObject(json);
	}

	public void addProperty(String key, String value) {
		buildPropertis();
		propertiesContainer.put(key, value);
		setProperties(propertiesContainer.toString());
	}

	public JSONObject getPropertiesContainer() {
		return propertiesContainer;
	}

	public void removeProperty(String key) {
		buildPropertis();
		propertiesContainer.remove(key);
		setProperties(propertiesContainer.toString());
	}

	@Override
	public void setProperties(String propertiesJson) {
		super.setProperties(propertiesJson);
		if (StringUtils.isNotEmpty(propertiesJson)) {
			propertiesContainer = new JSONObject(propertiesJson);
		}
		else {
			propertiesContainer = new JSONObject();
		}
	}

	public String getProperty(String key) {
		return getProperty(key, null);
	}

	public String getProperty(String key, String defaultValue) {
		buildPropertis();
		try {
			if (propertiesContainer.has(key)) {
				return propertiesContainer.getString(key);
			}
		} catch (Exception e) {
		}
		return defaultValue;
	}

	public boolean hasProperty(String key) {
		return getProperty(key) != null;
	}

	public boolean isPropertyValueTrue(String propertyName) {
		String property = getProperty(propertyName);

		return POSUtil.getBoolean(property);
	}

	public void putPatientAddressLine(String patientAddress) {
		addProperty("patient.address.line", patientAddress); //$NON-NLS-1$
	}

	public String getPatientAddressLine() {
		return getProperty("patient.address.line", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientGender(String patientGender) {
		addProperty("patient.gender", patientGender); //$NON-NLS-1$
	}

	public String getPatientGender() {
		return getProperty("patient.gender", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientRegistrationReason(String patientRegistrationReason) {
		addProperty("patient.registration.reason", patientRegistrationReason); //$NON-NLS-1$
	}

	public String getPatientRegistrationReason() {
		return getProperty("patient.registration.reason", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putPatientHealthHistory(String patientHealthHistory) {
		addProperty("patient.health.history", patientHealthHistory); //$NON-NLS-1$
	}

	public String getPatientHealthHistory() {
		return getProperty("patient.health.history", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void addPatientEmergencyContacts(String patientEmergencyContacts) {
		addProperty("patient.emergency.contact", patientEmergencyContacts); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<PatientContactData> getPatientEmergencyContacts() {
		List<PatientContactData> patientEmergencyContactList = new ArrayList<PatientContactData>();
		try {
			String emergencyContact = getProperty("patient.emergency.contact"); //$NON-NLS-1$
			if (StringUtils.isEmpty(emergencyContact)) {
				return patientEmergencyContactList;
			}
			patientEmergencyContactList = new Gson().fromJson(emergencyContact, new TypeToken<ArrayList<PatientContactData>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return patientEmergencyContactList;
	}

	public void addPatientDisclosureContacts(String patientDisclosureContacts) {
		addProperty("patient.disclosure.contact", patientDisclosureContacts); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public List<PatientContactData> getPatientDisclosureContacts() {
		List<PatientContactData> patientDisclosureContactList = new ArrayList<PatientContactData>();
		try {
			String disclosureContact = getProperty("patient.disclosure.contact"); //$NON-NLS-1$
			if (StringUtils.isEmpty(disclosureContact)) {
				return patientDisclosureContactList;
			}

			patientDisclosureContactList = new Gson().fromJson(disclosureContact, new TypeToken<ArrayList<PatientContactData>>() {
			}.getType());
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
		return patientDisclosureContactList;
	}

	public void putBloodGroup(String bloodGroup) {
		addProperty("customer.blood.group", bloodGroup); //$NON-NLS-1$
	}

	public String getBloodGroup() {
		return getProperty("customer.blood.group", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfOnReportType(String chargeType) {
		addProperty("rf.type_on_report", chargeType); //$NON-NLS-1$
	}

	public String getRfOnReportType() {
		return getProperty("rf.type_on_report", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfRateOnReport(String commissionValue) {
		addProperty("rf.rate_on_report", commissionValue); //$NON-NLS-1$
	}

	public String getRfRateOnReport() {
		return getProperty("rf.rate_on_report", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfOnNetSalesType(String chargeType) {
		addProperty("rf.type_on_net_sales", chargeType); //$NON-NLS-1$
	}

	public String getRfOnNetSalesType() {
		return getProperty("rf.type_on_net_sales", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putRfRateOnNetSales(String commissionValue) {
		addProperty("rf.rate_on_net_sales", commissionValue); //$NON-NLS-1$
	}

	public String getRfRateOnNetSales() {
		return getProperty("rf.rate_on_net_sales", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putAgentSalary(double salary) {
		addProperty("agent.salary", String.valueOf(salary)); //$NON-NLS-1$
	}

	public double getAgentSalary() {
		return PropertyContainer.parseDouble(getProperty("agent.salary", "")); //$NON-NLS-1$ //$NON-NLS-2$
	}

//	public void putAgentType(String agentTypeName) {
//		addProperty("agent.type", agentTypeName); //$NON-NLS-1$
//	}
//
//	public String getAgentType() {
//		return getProperty("agent.type", ""); //$NON-NLS-1$ //$NON-NLS-2$
//	}

	public String getAgentTypeDisplayName() {
		AgentTypeEnum agentType = AgentTypeEnum.fromString(getAgentType());
		if (agentType != null) {
			return agentType.getDisplayString();
		}
		return "";
	}

	public String getCustomerDateOfBirthDisplay() {
		Date dateOfBirth = getDateOfBirth();
		if (dateOfBirth != null) {
			return DateUtil.formatDate(dateOfBirth);
		}
		return "";
	}

//	public void putDoctorAgentId(String agentId) {
//		addProperty("doctor.agent.id", agentId); //$NON-NLS-1$
//	}

	@XmlTransient
	@JsonIgnore
	public Agent getDoctorAgent() {
		String doctorAgentId = getDoctorAgentId();
		if (StringUtils.isBlank(doctorAgentId)) {
			return null;
		}
		Agent agent = (Agent) DataProvider.get().getObjectOf(Agent.class, doctorAgentId);
		if (agent != null && !agent.isDeleted()) {
			return agent;
		}
		return null;
	}

//	/**
//	 * Do not use this method as the agent may be deleted
//	 * 
//	 * @see #getDoctorAgent 
//	 */
//	@Deprecated
//	public String getDoctorAgentId() {
//		return getProperty("doctor.agent.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
//	}

	public void putDoctorCommissionPercentage(String commissionPercentage) {
		addProperty("doctor.commission_percentage", commissionPercentage); //$NON-NLS-1$
	}

	public String getDoctorCommissionPercentage() {
		return getProperty("doctor.commission_percentage", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorDesignation(String doctorDesignation) {
		addProperty("doctor.designation", doctorDesignation); //$NON-NLS-1$
	}

	public String getDoctorDesignation() {
		return getProperty("doctor.designation", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorOrganization(String doctorOrganization) {
		addProperty("doctor.organization", doctorOrganization); //$NON-NLS-1$
	}

	public String getDoctorOrganization() {
		return getProperty("doctor.organization", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

//	public void putDoctorDepartmentId(String doctorDepartmentId) {
//		addProperty("doctor.department.id", doctorDepartmentId); //$NON-NLS-1$
//	}
//
//	public String getDoctorDepartmentId() {
//		return getProperty("doctor.department.id", ""); //$NON-NLS-1$ //$NON-NLS-2$
//	}

	public void putDoctorDegreesAndTraining(String degreesAndTraining) {
		addProperty("doctor.degrees.and.training", degreesAndTraining); //$NON-NLS-1$
	}

	public String getDoctorDegreesAndTraining() {
		return getProperty("doctor.degrees.and.training", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorSpecialistName(String specialistName) {
		addProperty("doctor.specialist.name", specialistName); //$NON-NLS-1$
	}

	public String getDoctorSpecialistName() {
		return getProperty("doctor.specialist.name", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorScheduleToVisit(String scheduleToVisit) {
		addProperty("doctor.schedule.to.visit", scheduleToVisit); //$NON-NLS-1$
	}

	public String getDoctorScheduleToVisit() {
		return getProperty("doctor.schedule.to.visit", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorAddressLine(String doctorAddress) {
		addProperty("doctor.address.line", doctorAddress); //$NON-NLS-1$
	}

	public String getDoctorAddressLine() {
		return getProperty("doctor.address.line", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

//	public void putIndorDoctor(boolean doctor) {
//		addProperty("doctor.indor", String.valueOf(doctor)); //$NON-NLS-1$
//	}
//
//	public Boolean getIndorDoctor() {
//		return POSUtil.getBoolean(getProperty("doctor.indor"), false); //$NON-NLS-1$
//	}
//
//	public void putLabDoctor(boolean doctor) {
//		addProperty("doctor.lab", String.valueOf(doctor)); //$NON-NLS-1$
//	}
//
//	public Boolean getLabDoctor() {
//		return POSUtil.getBoolean(getProperty("doctor.lab"), false); //$NON-NLS-1$
//	}

	public void setReceiveReferralFee(boolean isReceiveReferralFee) {
		addProperty("doctor.receive.referral.fee", String.valueOf(isReceiveReferralFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public boolean isReceiveReferralFee() {
		return POSUtil.getBoolean(getProperty("doctor.receive.referral.fee"), false); //$NON-NLS-1$
	}

	public void putDoctorRfRateOnReport(double doctorReferralFee) {
		addProperty("doctor.rf.rate_on_report", String.valueOf(doctorReferralFee)); //$NON-NLS-1$
	}

	@XmlTransient
	@JsonIgnore
	public double getDoctorRfRateOnReport() { //rf.rate_on_report
		return NumberUtil.parseOrGetZero(getProperty("doctor.rf.rate_on_report")).doubleValue(); //$NON-NLS-1$
	}

	public void setDepartmentName(String departmentName) {

	}

	@XmlTransient
	@JsonIgnore
	public String getDepartmentName() {
		String departmentName = "";
		String departmentId = getDoctorDepartmentId();
		Department department = DataProvider.get().getDepartmentById(departmentId);
		if (department != null) {
			departmentName = department.getName();
		}
		return departmentName;
	}

	public void setAgentName(String agentName) {

	}

	@XmlTransient
	@JsonIgnore
	public String getAgentName() {
		String agentName = "";
		Agent agent = getDoctorAgent();
		if (agent != null) {
			agentName = agent.getName();
		}
		return agentName;
	}

	public void putOccupation(String occupation) {
		addProperty("OCCUPATION", occupation); //$NON-NLS-1$
	}

	public String getOccupation() {
		return getProperty("OCCUPATION"); //$NON-NLS-1$
	}

	public void putNationality(String nationality) {
		addProperty("NATIONALITY", nationality); //$NON-NLS-1$
	}

	public String getNationality() {
		return getProperty("NATIONALITY"); //$NON-NLS-1$
	}

	public void putRefBy(String refId) {
		addProperty("V_REF_BY", refId); //$NON-NLS-1$
	}

	public String getRefBy() {
		return getProperty("V_REF_BY"); //$NON-NLS-1$
	}

	public void putCompanyAddressId(String addressId) {
		addProperty("company_address_id", addressId); //$NON-NLS-1$
	}

	public String getCompanyAddressId() {
		return getProperty("company_address_id"); //$NON-NLS-1$
	}

	public void putApplyDiscountOnReferralCommission(boolean apply) {
		addProperty("discount_on_referral_commission", String.valueOf(apply)); //$NON-NLS-1$
	}

	public boolean isApplyDiscountOnReferralCommission() {
		boolean hasKey = propertiesContainer.has("discount_on_referral_commission");
		if (hasKey) {
			return POSUtil.getBoolean(getProperty("discount_on_referral_commission"), false);
		}
		Outlet outlet = (Outlet) DataProvider.get().getOutlet();
		return outlet.isApplyDiscountOnReferralCommission();
	}

	public String getAge() {
		Date dateOfBirth = getDateOfBirth();
		if (dateOfBirth != null) {
			LocalDate birthdate = dateOfBirth.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
			LocalDate currentDate = LocalDate.now();
			Period agePeriod = Period.between(birthdate, currentDate);
			int years = agePeriod.getYears();
			int months = agePeriod.getMonths();
			int days = agePeriod.getDays();
			return String.format("Y%d M%d D%d", years, months, days);
		}
		return null;
	}

	public String getApproxAge() {
		Date dateOfBirth = getDateOfBirth();
		if (dateOfBirth != null) {
			LocalDate birthdate = dateOfBirth.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
			LocalDate currentDate = LocalDate.now();
			Period agePeriod = Period.between(birthdate, currentDate);
			int years = agePeriod.getYears();
			int months = agePeriod.getMonths();
			int days = agePeriod.getDays();
			if (years > 0) {
				return String.format("%d Y", years);
			}
			if (months > 0) {
				return String.format("%d M", months);
			}

			return String.format("%d days", days);
		}
		return null;
	}

	public void putHeightInCM(String height) {
		addProperty("height.cm", height); //$NON-NLS-1$
	}

	public String getHeightInCM() {
		return getProperty("height.cm", ""); //$NON-NLS-1$
	}

	public void putWeight(String weight) {
		addProperty("weight", weight); //$NON-NLS-1$
	}

	public String getWeight() {
		return getProperty("weight", ""); //$NON-NLS-1$
	}

	public void putPrivateVisitFee(double privateVisitFee) {
		addProperty("doctor.private_visit_fee", String.valueOf(privateVisitFee)); //$NON-NLS-1$
	}

	public double getPrivateVisitFee() {
		return NumberUtil.parseOrGetZero(getProperty("doctor.private_visit_fee")).doubleValue(); //$NON-NLS-1$
	}

	public void putHospitalRoundFee(double hospitalRoundFee) {
		addProperty("doctor.hospital_round_fee", String.valueOf(hospitalRoundFee)); //$NON-NLS-1$
	}

	public double getHospitalRoundFee() {
		return NumberUtil.parseOrGetZero(getProperty("doctor.hospital_round_fee")).doubleValue(); //$NON-NLS-1$
	}

	public void putEmergencyVisitFee(double emergencyVisitFee) {
		addProperty("doctor.emergency_visit_fee", String.valueOf(emergencyVisitFee)); //$NON-NLS-1$
	}

	public double getEmergencyVisitFee() {
		return NumberUtil.parseOrGetZero(getProperty("doctor.emergency_visit_fee")).doubleValue(); //$NON-NLS-1$
	}

	public void putOtCharge(double otCharge) {
		addProperty("doctor.ot_charge", String.valueOf(otCharge)); //$NON-NLS-1$
	}

	public double getOtCharge() {
		return NumberUtil.parseOrGetZero(getProperty("doctor.ot_charge")).doubleValue(); //$NON-NLS-1$
	}
	
	public String getDoctorSpecializedInField() {
		return getProperty("specializedInField", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	public void putDoctorSpecializedInField(Tag tag) {
		if (tag == null) {
			setSpecializedIn(null);
			removeProperty("specializedInField");
			return;
		}
		setSpecializedIn(tag.getId());
		addProperty("specializedInField", tag.getName()); //$NON-NLS-1$
	}

}