/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.report;

import java.util.Collections;
import java.util.Comparator;

import org.apache.commons.lang.StringUtils;

import com.floreantpos.Messages;
import com.floreantpos.constants.AppConstants;
import com.floreantpos.main.Application;
import com.floreantpos.model.KitchenTicket;
import com.floreantpos.model.KitchenTicketItem;
import com.floreantpos.model.OrderType;
import com.floreantpos.model.Store;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.util.NumberUtil;

public class KitchenTicketDataSource extends AbstractReportDataSource {

	private Store store;

	public KitchenTicketDataSource(KitchenTicket ticket) {
		super(new String[] { "groupId", "groupName", "itemNo", "itemName", "itemQty", "colorCode", "isVoidItem", "isModifier", "isCookingInstruction" }); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$ //$NON-NLS-7$ //$NON-NLS-8$ //$NON-NLS-9$
		store = DataProvider.get().getStore();
		setTicket(ticket);
	}

	private void setTicket(KitchenTicket kitchenTicket) {
		OrderType orderType = kitchenTicket.getOrderType();
		if (orderType != null && !orderType.isAllowSeatBasedOrder()) {
			Terminal terminal = Application.getInstance().getTerminal();
			if (terminal != null && terminal.isGroupByCatagoryKitReceipt()) {
				Collections.sort(kitchenTicket.getTicketItems(), new Comparator<KitchenTicketItem>() {
					public int compare(KitchenTicketItem o1, KitchenTicketItem o2) {
						return (o1.getMenuItemGroupName().compareTo(o2.getMenuItemGroupName()));
					}
				});
			}
		}
		setRows(kitchenTicket.getTicketItems());
	}

	public Object getValueAt(int rowIndex, int columnIndex) {
		KitchenTicketItem item = (KitchenTicketItem) rows.get(rowIndex);

		switch (columnIndex) {

			case 0:
				return item.getMenuItemGroupName();

			case 1:
				String menuItemGroupName = item.getMenuItemGroupName();
				if (StringUtils.isBlank(menuItemGroupName)) {
					menuItemGroupName = Messages.getString("Ungrouped");//$NON-NLS-1$
				}
				return menuItemGroupName;

			case 2:
				return item.getMenuItemCode();

			case 3:
				return item.getMenuItemNameDisplay();

			case 4:
				return NumberUtil.trimDecilamIfNotNeeded(item.getQuantity());

			case 5:
				Ticket ticket = item.getKitchenTicket().getParentTicket();
				String orderTypeId = ""; //$NON-NLS-1$
				if (ticket != null && ticket.getOrderType() != null) {
					orderTypeId = ticket.getOrderType().getId();
				}
				return getColorCode(item, orderTypeId);
			case 6:
				return item.isVoided();

			case 7:
				return item.isModifierItem();

			case 8:
				return item.isCookingInstruction();
		}
		return null;
	}

	public String getColorCode(KitchenTicketItem item, String orderTypeId) {
		if (StringUtils.isNotEmpty(orderTypeId)) {
			orderTypeId += "."; //$NON-NLS-1$
		}
		if (item.isModifierItem()) {
			return store.getProperty(orderTypeId + AppConstants.PROP_KITCHEN_TICKET_MODIFIER_COLOR);
		}
		else if (item.isCookingInstruction()) {
			return store.getProperty(orderTypeId + AppConstants.PROP_KITCHEN_TICKET_INSTRUCTION_COLOR);
		}
		else if (item.getMenuItemName().contains("Seat**")) { //$NON-NLS-1$
			return store.getProperty(orderTypeId + AppConstants.PROP_KITCHEN_TICKET_SEAT_COLOR);
		}

		return Messages.getString("KitchenTicketDataSource.7"); //$NON-NLS-1$
	}
}
