/**
 * ************************************************************************
 * * The contents of this file are subject to the MRPL 1.2
 * * (the  "License"),  being   the  Mozilla   Public  License
 * * Version 1.1  with a permitted attribution clause; you may not  use this
 * * file except in compliance with the License. You  may  obtain  a copy of
 * * the License at http://www.floreantpos.org/license.html
 * * Software distributed under the License  is  distributed  on  an "AS IS"
 * * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * * License for the specific  language  governing  rights  and  limitations
 * * under the License.
 * * The Original Code is FLOREANT POS.
 * * The Initial Developer of the Original Code is OROCUBE LLC
 * * All portions are Copyright (C) 2015 OROCUBE LLC
 * * All Rights Reserved.
 * ************************************************************************
 */
package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.DetachedCriteria;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Projection;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Property;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.Subqueries;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.StringType;
import org.hibernate.type.Type;

import com.floreantpos.Messages;
import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.ActionHistory;
import com.floreantpos.model.AdvanceAdjustmentTransaction;
import com.floreantpos.model.AdvanceTransaction;
import com.floreantpos.model.AgentTypeEnum;
import com.floreantpos.model.BalanceType;
import com.floreantpos.model.BankAccount;
import com.floreantpos.model.CashDrawer;
import com.floreantpos.model.CashTransaction;
import com.floreantpos.model.CreditCardTransaction;
import com.floreantpos.model.CustomPaymentTransaction;
import com.floreantpos.model.Customer;
import com.floreantpos.model.DateTypeOption;
import com.floreantpos.model.DebitCardTransaction;
import com.floreantpos.model.EntityType;
import com.floreantpos.model.ExpenseTransaction;
import com.floreantpos.model.Outlet;
import com.floreantpos.model.Pagination;
import com.floreantpos.model.PaymentType;
import com.floreantpos.model.PosTransaction;
import com.floreantpos.model.Project;
import com.floreantpos.model.PurchaseOrder;
import com.floreantpos.model.PurchaseOrderType;
import com.floreantpos.model.PurchaseRefundTransaction;
import com.floreantpos.model.PurchaseTransaction;
import com.floreantpos.model.RefundTransaction;
import com.floreantpos.model.ReversalTransaction;
import com.floreantpos.model.Terminal;
import com.floreantpos.model.Ticket;
import com.floreantpos.model.TicketItem;
import com.floreantpos.model.TicketType;
import com.floreantpos.model.TransactionReason;
import com.floreantpos.model.TransactionSubType;
import com.floreantpos.model.TransactionType;
import com.floreantpos.model.User;
import com.floreantpos.model.ext.CardTypeEnum;
import com.floreantpos.model.util.DataProvider;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.model.util.FeePaymentMapper;
import com.floreantpos.report.GrandTotalCollectionReportData;
import com.floreantpos.report.UserWiseCollectionReportData;
import com.floreantpos.report.model.CustomerAccountTransactionItem;
import com.floreantpos.report.model.PaymentReceivedReportData;
import com.floreantpos.swing.PaginationSupport;
import com.floreantpos.util.NumberUtil;
import com.floreantpos.util.POSUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class PosTransactionDAO extends BasePosTransactionDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public PosTransactionDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		//		PosTransaction transaction = (PosTransaction) obj;
		//		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
		//			StoreSession storeSession = DataProvider.get().getStoreSession();
		//			if (storeSession != null) {
		//				transaction.setStoreSessionId(storeSession.getId());
		//			}
		//
		//		}

		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		//		PosTransaction transaction = (PosTransaction) obj;
		//		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
		//			StoreSession storeSession = DataProvider.get().getStoreSession();
		//			if (storeSession != null) {
		//				transaction.setStoreSessionId(storeSession.getId());
		//			}
		//		}

		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		//		PosTransaction transaction = (PosTransaction) obj;
		//		if (StringUtils.isEmpty(transaction.getStoreSessionId())) {
		//			StoreSession storeSession = DataProvider.get().getStoreSession();
		//			if (storeSession != null) {
		//				transaction.setStoreSessionId(storeSession.getId());
		//			}
		//		}

		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public List<PosTransaction> findReceived(Terminal terminal, Outlet outlet, Date from, Date to) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));

			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public List<RefundTransaction> findRefundTransactions(Terminal terminal, Outlet outlet, Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(RefundTransaction.class);
			criteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (terminal != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_TERMINAL_ID, terminal.getId()));
			}
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$

			return criteria.list();
		}
	}

	public void saveReversalTransaction(Ticket ticket, PosTransaction transaction, ReversalTransaction reversalTransaction) {
		Transaction tx = null;

		try (Session session = createNewSession()) {
			tx = session.beginTransaction();

			delete(transaction, session);

			saveOrUpdate(reversalTransaction, session);
			TicketDAO.getInstance().update(ticket, session);

			tx.commit();
		} catch (Exception e) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(PosTransactionDAO.class, x);
			}
			throw e;
		}
	}

	private void addMultiUserFilter(User user, Criteria criteria) {
		if (user != null) {
			PosLog.info(getClass(), Messages.getString("PosTransactionDAO.5") + user.getFullName() + Messages.getString("PosTransactionDAO.6") + user.getId()); //$NON-NLS-1$ //$NON-NLS-2$
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Restrictions.eq(PosTransaction.PROP_USER_ID, user == null ? null : user.getId()));
			List<User> linkedUsers = user.getLinkedUser();
			if (linkedUsers != null) {
				for (User linkedUser : linkedUsers) {
					if (linkedUser.getId().equals(user.getId())) {
						continue;
					}
					PosLog.info(getClass(), Messages.getString("PosTransactionDAO.7") + linkedUser.getFullName() + Messages.getString("PosTransactionDAO.8") //$NON-NLS-1$//$NON-NLS-2$
							+ linkedUser.getId());
					disjunction.add(Restrictions.eq(PosTransaction.PROP_USER_ID, linkedUser.getId()));
				}
			}
			criteria.add(disjunction);
		}
	}

	public List<CustomerAccountTransactionItem> findCustomerAccountTransactions(Date fromDate, Date toDate, String customerIdOrName) {
		try (Session session = createNewSession()) {

			String hql = "select t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, t.%s, c.%s" //$NON-NLS-1$
					+ " from PosTransaction as t, Customer as c" //$NON-NLS-1$
					+ " where t.%s=c.%s and t.%s between :fromDate and :toDate and t.%s=false and t.%s is not null" //$NON-NLS-2$
					+ " order by c.%s"; //$NON-NLS-2$
			hql = String.format(hql, PosTransaction.PROP_ID, PosTransaction.PROP_TICKET, PosTransaction.PROP_TRANSACTION_TIME, PosTransaction.PROP_TIPS_AMOUNT,
					PosTransaction.PROP_AMOUNT, PosTransaction.PROP_CUSTOMER_ID, PosTransaction.PROP_TRANSACTION_TYPE, PosTransaction.PROP_PAYMENT_TYPE_STRING,
					PosTransaction.PROP_CUSTOM_PAYMENT_NAME, Customer.PROP_NAME, PosTransaction.PROP_CUSTOMER_ID, Customer.PROP_ID,
					PosTransaction.PROP_TRANSACTION_TIME, PosTransaction.PROP_VOIDED, PosTransaction.PROP_TICKET, Customer.PROP_NAME);

			if (StringUtils.isNotBlank(customerIdOrName)) {
				hql += " and ( LOWER(c.name) like '%" + customerIdOrName.toLowerCase() + "%'"; //$NON-NLS-1$ //$NON-NLS-2$
				hql += " or "; //$NON-NLS-1$
				hql += " LOWER(c.id) like '%" + customerIdOrName.toLowerCase() + "%' )"; //$NON-NLS-1$ //$NON-NLS-2$
			}

			Query query = session.createQuery(hql);
			query.setTimestamp("fromDate", DateUtil.toUTC(fromDate)); //$NON-NLS-1$
			query.setTimestamp("toDate", DateUtil.toUTC(toDate)); //$NON-NLS-1$

			List<Object[]> list = query.list();
			List<CustomerAccountTransactionItem> items = new ArrayList<CustomerAccountTransactionItem>();
			for (Object[] object : list) {
				CustomerAccountTransactionItem accountTransactionItem = new CustomerAccountTransactionItem();
				accountTransactionItem.setTransactionNo(String.valueOf(object[0]));
				Ticket ticket = (Ticket) object[1];
				accountTransactionItem.setTicketNo(ticket.getId());
				accountTransactionItem.setDate((Date) object[2]);
				accountTransactionItem.setTips((double) object[3]);
				accountTransactionItem.setTotalAmount((double) object[4]);
				accountTransactionItem.setCustomerId(String.valueOf(object[5]));
				accountTransactionItem.setCustomerName(String.valueOf(object[6]));
				items.add(accountTransactionItem);
			}

			return items;

		}
	}

	public List<PosTransaction> findTransactionsForCashDrawer(String cashDrawerId, boolean voided) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass(), "t"); //$NON-NLS-1$
			criteria.add(Restrictions.eq(PosTransaction.PROP_CASH_DRAWER_ID, cashDrawerId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, voided));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public List<PosTransaction> findTransactionsForServer(String storeSessionId, String serverId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.eq(PosTransaction.PROP_USER_ID, serverId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_STORE_SESSION_ID, storeSessionId));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public List<PosTransaction> getTransactionsByOutlet(Date from, Date to, Outlet outlet, Pagination pagination) {
		Criteria criteria = null;
		try (Session session = createNewSession()) {
			criteria = session.createCriteria(getReferenceClass());
			if (pagination != null) {
				criteria.setFirstResult(pagination.getCurrentRowIndex());
				criteria.setMaxResults(pagination.getPageSize());
			}
			updateCriteria(criteria, from, to, outlet);
			List list = criteria.list();

			criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.rowCount());
			updateCriteria(criteria, from, to, outlet);
			if (pagination != null) {
				Number uniqueResult = (Number) criteria.uniqueResult();
				pagination.setNumRows(uniqueResult.intValue());
			}
			pagination.setRows(list);
			return list;
		}
	}

	private void updateCriteria(Criteria criteria, Date from, Date to, Outlet outlet) {

		if (from != null && to != null) {
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));
		}
		if (outlet != null) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
		}
	}

	public void savePosTransaction(PosTransaction transaction, List<PosTransaction> existingPosTransactions, Session session) {
		if (existingPosTransactions == null || existingPosTransactions.size() == 0)
			return;
		PosTransaction existingPosTransaction = null;
		int idx = existingPosTransactions.indexOf(transaction);
		if (idx != -1) {
			existingPosTransaction = existingPosTransactions.get(idx);
			if (existingPosTransaction == null) {
				save(transaction, session);
			}
			else {
				transaction.setVersion(existingPosTransaction.getVersion());
				//update(transaction, session);
			}
		}
		else {
			save(transaction, session);
		}
	}

	public void saveOrUpdatePosTransactions(List<PosTransaction> posTransactions) throws Exception {
		saveOrUpdatePosTransactions(posTransactions, false, false);
	}

	public void saveOrUpdatePosTransactions(List<PosTransaction> posTransactions, boolean updateLastUpdateTime, boolean updateSyncTime) throws Exception {
		if (posTransactions == null)
			return;

		Transaction tx = null;
		try (Session session = createNewSession()) {
			tx = session.beginTransaction();
			for (PosTransaction posTransaction : posTransactions) {
				PosLog.debug(PosTransactionDAO.class, "Start saving transaction: " + posTransaction.getId());
				PosTransaction existingPosTransaction = get(posTransaction.getId(), session);
				if (existingPosTransaction != null) {
					if (!BaseDataServiceDao.get().shouldSave(posTransaction.getLastUpdateTime(), existingPosTransaction.getLastUpdateTime())) {
						PosLog.info(getClass(), posTransaction.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					final String id = existingPosTransaction.getId();
					long version = existingPosTransaction.getVersion();
					PropertyUtils.copyProperties(existingPosTransaction, posTransaction);
					existingPosTransaction.setId(id);
					existingPosTransaction.setVersion(version);
					existingPosTransaction.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingPosTransaction.setUpdateSyncTime(updateSyncTime);
					update(existingPosTransaction, session);
				}
				else {
					posTransaction.setUpdateLastUpdateTime(updateLastUpdateTime);
					posTransaction.setUpdateSyncTime(updateSyncTime);
					save(posTransaction, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			PosLog.error(getClass(), e);
			throw e;
		}
	}

	//	public void savePosTransactions(List<PosTransaction> transactions) throws Exception {
	//		if (transactions == null)
	//			return;
	//		Transaction tx = null;
	//		Session session = null;
	//		try {
	//			session = createNewSession();
	//			tx = session.beginTransaction();
	//			for (PosTransaction posTransaction : transactions) {
	//
	//				PaymentType paymentType = posTransaction.getPaymentType();
	//
	//				PosTransaction existPosTransaction = get(posTransaction.getId());
	//				if (existPosTransaction == null) {
	//					existPosTransaction = paymentType.createTransaction();
	//					PropertyUtils.copyProperties(existPosTransaction, posTransaction);
	//					if (StringUtils.isEmpty(existPosTransaction.getStoreSessionId())) {
	//						existPosTransaction.setStoreSessionId(StoreUtil.getCurrentStoreSession().getId());
	//					}
	//
	//					save(existPosTransaction, session);
	//				}
	//				performPreSaveOperations(posTransaction);
	//
	//			}
	//			tx.commit();
	//		} catch (Exception e) {
	//			tx.rollback();
	//			throw e;
	//		} finally {
	//			session.close();
	//		}
	//	}

	public List<PosTransaction> findTopByPaymentTypes(Session session, String outletId, Date fromDate, Date toDate, Integer maxResult) {
		Criteria criteria = session.createCriteria(PosTransaction.class);
		criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
		if (StringUtils.isNotBlank(outletId)) {
			criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outletId));
		}
		criteria.add(Restrictions.or(Restrictions.isNull(PosTransaction.PROP_VOIDED), Restrictions.eq(PosTransaction.PROP_VOIDED, false)));
		criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, DateUtil.getUTCStartOfDay(fromDate), DateUtil.getUTCEndOfDay(toDate)));
		ProjectionList projectionList = Projections.projectionList();
		projectionList.add(Projections.groupProperty(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
		projectionList.add(Projections.sum(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
		criteria.setProjection(projectionList);
		criteria.setMaxResults(maxResult == null ? 10 : maxResult);
		criteria.addOrder(Order.desc(PosTransaction.PROP_AMOUNT));
		criteria.setResultTransformer(Transformers.aliasToBean(PosTransaction.class));
		return criteria.list();
	}

	public static Disjunction createMasterCardSearchCriteria() {
		Criterion masterCard = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name()).ignoreCase();
		Criterion masterCard_ = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name().replaceAll("_", "")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion masterCardSpace = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.MASTER_CARD.name().replaceAll("_", " ")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$

		return Restrictions.or(masterCard, masterCard_, masterCardSpace);
	}

	public static Disjunction createAmexOrAmericanExpCardSearchCriteria() {
		Criterion americanExpress = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name()).ignoreCase();
		Criterion americanExpress_ = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", "")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion americanExprSpace = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", " ")).ignoreCase(); //$NON-NLS-1$//$NON-NLS-2$
		Criterion amex = Restrictions.eq(PosTransaction.PROP_CARD_TYPE, "AMEX").ignoreCase(); //$NON-NLS-1$

		return Restrictions.or(americanExpress, americanExpress_, americanExprSpace, amex);
	}

	@SuppressWarnings("unchecked")
	public List<UserWiseCollectionReportData> findUserWiseCollectionReport(String outletId, Date fromDate, Date toDate, User user, String orderId) {
		List<UserWiseCollectionReportData> datas = new ArrayList<UserWiseCollectionReportData>();
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			//criteria.createAlias("ticket", "t"); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.in(PosTransaction.PROP_TRANSACTION_REASON,
					Arrays.asList(TransactionReason.SALES.name(), TransactionReason.SALES_ADVANCE.name(), TransactionReason.SALES_REFUND.name())));

			if (StringUtils.isNotBlank(orderId)) {
				Ticket ticket = TicketDAO.getInstance().get(orderId, outletId);
				if (ticket != null && TicketType.OUTDOOR == ticket.getTicketType()) {
					criteria.add(Restrictions.eq("t." + Ticket.PROP_OUTDOOR_TICKET_ID, orderId));
				}
				else {
					criteria.add(Restrictions.eq("t." + Ticket.PROP_ID, orderId));
				}
			}

			if (StringUtils.isNotBlank(outletId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outletId));
			}

			if (fromDate != null) {
				criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			}
			if (toDate != null) {
				criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
			}

			addMultiUserFilter(user, criteria);
			List<PosTransaction> list = criteria.list();

			Map<String, UserWiseCollectionReportData> dataMap = new LinkedHashMap<>();
			if (list != null && list.size() > 0) {
				for (PosTransaction posTransaction : list) {
					Ticket ticket = posTransaction.getTicket();
					if (/*ticket == null || */posTransaction.isVoided())
						continue;

					String userId = posTransaction.getUserId();
					String outdoorTicketId = ticket != null ? ticket.getOutdoorTicketId() : "";
					String ticketId = ticket != null ? (StringUtils.isBlank(outdoorTicketId) ? ticket.getId() : outdoorTicketId) : "";
					String key = userId + "_" + ticketId;
					Date ticketDate = ticket != null ? ticket.getCreateDate() : posTransaction.getTransactionTime();

					UserWiseCollectionReportData data = dataMap.computeIfAbsent(key, k -> {
						UserWiseCollectionReportData d = new UserWiseCollectionReportData();
						d.setUserId(userId);
						d.setUserName(posTransaction.getUserName());
						d.setOrderId(ticketId);
						d.setOrderCreateDate(ticketDate);
						d.setOrderDate(DateUtil.formatDateWithTime(DateUtil.convertServerTimeToBrowserTime(ticketDate)));
						d.setPatientName(ticket == null ? posTransaction.getCustomerName() : ticket.getCustomerName());
						d.setSubTotalAmount(ticket != null ? ticket.getSubtotalAmount() : 0.0);
						d.setVatAmount(ticket == null ? 0.0 : ticket.getTaxAmount());
						d.setDiscount(ticket == null ? 0.0 : ticket.getDiscountAmount());
						d.setTotalAmount(ticket != null ? ticket.getTotalAmount() : 0.0);
						return d;
					});

					Date txDayStart = DateUtil.startOfDay(posTransaction.getTransactionTime());
					double amount = posTransaction instanceof RefundTransaction
							|| TransactionReason.SALES_REFUND.name().equals(posTransaction.getTransactionReason()) ? -posTransaction.getAmount()
									: posTransaction.getAmount();

					if (ticketDate.before(txDayStart)) {
						data.setDueColl(data.getDueColl() + amount);
					}
					else {
						data.setAdvancedAmount(data.getAdvancedAmount() + amount);
					}

					data.setTotalReceived(data.getDueColl() + data.getAdvancedAmount());
				}

				datas.addAll(dataMap.values());
				datas.sort(Comparator.comparing(UserWiseCollectionReportData::getOrderCreateDate).thenComparing(UserWiseCollectionReportData::getOrderId));

				int serialNumber = 1;
				for (UserWiseCollectionReportData data : datas) {
					data.setSerialNumber(serialNumber++);
				}
			}
		}
		return datas;

	}

	public List<PosTransaction> findTransactionPaymentsByUser(Date startTime, Date endTime, Outlet outlet, User user, String orderNo,
			boolean isShowOnlyVoidedPayment) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {

			criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, isShowOnlyVoidedPayment));

			if (StringUtils.isNotBlank(orderNo)) {
				criteria.createAlias(PosTransaction.PROP_TICKET, "t");
				criteria.add(Restrictions.eq("t." + Ticket.PROP_ID, orderNo));
			}

			if (isShowOnlyVoidedPayment) {
				criteria.add(Restrictions.between(PosTransaction.PROP_VOID_DATE, startTime, endTime));
			}
			else {
				criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, startTime, endTime));
			}

			//criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));
			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}
			addMultiUserFilter(user, criteria);

			criteria.addOrder(Order.asc(PosTransaction.PROP_USER_ID));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			if (isShowOnlyVoidedPayment) {
				criteria.addOrder(Order.desc(PosTransaction.PROP_VOID_DATE));
			}
			else {
				criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));
			}
			List<PosTransaction> list = criteria.list();
			return list;
		}
	}

	public double findDue(String agentId, Date fromDate, Date toDate, Outlet outlet) {

		Criteria criteria = null;

		try (Session session = createNewSession()) {
			criteria = session.createCriteria(PosTransaction.class);
			criteria.createAlias(PosTransaction.PROP_TICKET, "t"); //$NON-NLS-1$

			criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));

			criteria.add(Restrictions.eq("t." + Ticket.PROP_VOIDED, Boolean.FALSE)); //$NON-NLS-1$
			criteria.add(Restrictions.lt("t." + Ticket.PROP_CREATE_DATE, fromDate)); //$NON-NLS-1$
			criteria.add(Restrictions.eq("t." + Ticket.PROP_REFERRER_ID, agentId)); //$NON-NLS-1$

			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

			return POSUtil.getDoubleAmount(criteria.uniqueResult());
		}

	}

	public Double doPayFeePayment(FeePaymentMapper paymentMapper) throws Exception {
		return doPayFeePayment(paymentMapper, null);
	}

	public Double doPayFeePayment(FeePaymentMapper paymentMapper, String batchNo) throws Exception {

		Session session = null;
		Transaction tx = null;
		try {
			session = GenericDAO.getInstance().createNewSession();
			tx = session.beginTransaction();

			boolean isCalledFromRfPaymentEntry = StringUtils.isNotBlank(batchNo);
			boolean isRefundPayment = paymentMapper.isRefundPayment();
			Terminal terminal = paymentMapper.getTerminal();
			User user = paymentMapper.getUser();
			double tenderAmount = paymentMapper.getTenderAmount();
			if (tenderAmount <= 0) {
				throw new PosException("Tender amount should be greater than 0");
			}

			boolean isDoctorPayment = paymentMapper.isDoctorPayment();
			double unpaidAmount = paymentMapper.getUnpaidAmount();
			if ((isDoctorPayment || !isCalledFromRfPaymentEntry) && paymentMapper.getCustomPayment() != null && tenderAmount > unpaidAmount) {
				throw new PosException("Tender amount must not be greater than the due amount");
			}
			double totalAmount = 0;
			Map<Ticket, Double> ticketsMap = new HashMap<Ticket, Double>();
			//CashDrawer currentCashDrawer = terminal.getActiveCurrentCashDrawer();
			CashDrawer currentCashDrawer = user.getCurrentCashDrawer();
			Customer referrer = paymentMapper.getReferrer();

			double payOutAmount = tenderAmount;
			PosTransaction transaction = null;
			Integer ticketTypeNo = null;
			if (isDoctorPayment) {
				List<TicketItem> unpaidTicketItems = paymentMapper.getUnpaidTicketItems();
				if (unpaidTicketItems != null && !unpaidTicketItems.isEmpty()) {
					for (Iterator iterator = unpaidTicketItems.iterator(); iterator.hasNext();) {
						TicketItem unpaidTicketItem = (TicketItem) iterator.next();
						double itemTotalAmount = 0;
						double labDoctorFeePaidAmount = 0;
						Ticket fullTicket = TicketDAO.getInstance().loadFullTicket(unpaidTicketItem.getParentTicketId(),
								unpaidTicketItem.getParentTicketOutletId(), session);
						if (fullTicket == null) {
							continue;
						}

						if (ticketTypeNo == null) {
							ticketTypeNo = fullTicket.getType();
						}

						double paidTicketLDFAmount = 0;
						TicketItem labDoctorFeeTicketItem = fullTicket.getTicketItem(unpaidTicketItem.getId(), true);
						if (labDoctorFeeTicketItem.getId().equals(unpaidTicketItem.getId())) {
							if (labDoctorFeeTicketItem.isLabDoctorFeePaid()) {
								continue;
							}

							double labDoctorFee = labDoctorFeeTicketItem.getLabDoctorFee();
							Double beforeBalance = labDoctorFeeTicketItem.getLabDoctorFeePaidAmount();
							Double partialPayAmount = labDoctorFee - beforeBalance;
							if (payOutAmount >= partialPayAmount) {
								labDoctorFeeTicketItem.setLabDoctorFeePaidAmount(NumberUtil.round(labDoctorFee));
								labDoctorFeeTicketItem.setLabDoctorFeePaid(true);

								paidTicketLDFAmount += partialPayAmount;
								itemTotalAmount += partialPayAmount;
								payOutAmount -= partialPayAmount;
							}
							else {
								labDoctorFeeTicketItem.setLabDoctorFeePaidAmount(NumberUtil.round(beforeBalance + payOutAmount));
								paidTicketLDFAmount += payOutAmount;
								itemTotalAmount += payOutAmount;
								payOutAmount = 0d;
							}
						}

						labDoctorFeePaidAmount = fullTicket.getLabDoctorFeePaidAmount() + paidTicketLDFAmount;

						fullTicket.setLabDoctorFeePaidAmount(NumberUtil.round(labDoctorFeePaidAmount));
						double dueAmount = fullTicket.getLabDoctorFee() - labDoctorFeePaidAmount;
						fullTicket.setLabDoctorFeePaid(dueAmount <= 0);

						if (!NumberUtil.isZero(paidTicketLDFAmount)) {
							Double amount = ticketsMap.get(fullTicket);
							if (amount == null) {
								ticketsMap.put(fullTicket, paidTicketLDFAmount);
							}
							else {
								ticketsMap.put(fullTicket, amount + paidTicketLDFAmount);
							}
						}

						totalAmount += itemTotalAmount;
//						if (NumberUtil.isZero(fullTicket.getDueAmount()) && fullTicket.getReferrerFeePaid() && fullTicket.getLabDoctorFeePaid()) {
//							fullTicket.setClosed(true);
//						}
						TicketDAO.getInstance().saveOrUpdate(fullTicket, session);
						String ticketId = "'" + fullTicket.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
						if (!NumberUtil.isZero(itemTotalAmount)) {
							FeePaymentMapper ldfPaymentMapper = new FeePaymentMapper(ticketId, referrer, itemTotalAmount,
									iterator.hasNext() ? itemTotalAmount : tenderAmount, paymentMapper.getCustomPayment(), paymentMapper.getCustomPaymnetRef(),
									user, terminal, currentCashDrawer, ticketTypeNo, false);

							transaction = LdfPayTransactionDAO.getInstance().createLdfPayTransaction(ldfPaymentMapper, batchNo, session);

							String transactionId = transaction == null ? "" : transaction.getId();
							LedgerEntryDAO.getInstance().saveLDFPayLedgerEntry(ticketsMap, transactionId, session);
							tenderAmount -= itemTotalAmount;
						}

						if (payOutAmount <= 0) {
							break;
						}
					}
				}
			}
			else {

				List<Ticket> unpaidTickets = paymentMapper.getUnpaidTickets();
				if (unpaidTickets != null && !unpaidTickets.isEmpty()) {
					for (Iterator iterator = unpaidTickets.iterator(); iterator.hasNext();) {
						Ticket ticket = (Ticket) iterator.next();

						Ticket fullTicket = TicketDAO.getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId());
						//	if (fullTicket.isReferrerFeePaid()) {
						//		continue;
						//	}

						if (ticketTypeNo == null) {
							ticketTypeNo = fullTicket.getType();
						}
						double rfTotalAmount = 0;
						double payableRFAmount = fullTicket.getRfPayableAmount();
						Double beforeBalance = fullTicket.getReferrerFeePaidAmount();
						Double partialPayAmount = payableRFAmount - beforeBalance;
						if (isRefundPayment) {
							partialPayAmount = Math.abs(partialPayAmount);
						}
						if (payOutAmount >= partialPayAmount) {
							fullTicket.setReferrerFeePaidAmount(NumberUtil.round(payableRFAmount));
							fullTicket.setReferrerFeePaid(true);
							if (!NumberUtil.isZero(partialPayAmount)) {
								if (isRefundPayment) {
									ticketsMap.put(ticket, (-1) * partialPayAmount);
								}
								else {
									ticketsMap.put(ticket, partialPayAmount);
								}
							}
							rfTotalAmount += partialPayAmount;
							payOutAmount -= partialPayAmount;
						}
						else {
							if (isRefundPayment) {
								fullTicket.setReferrerFeePaidAmount(NumberUtil.round(beforeBalance - payOutAmount));
							}
							else {
								fullTicket.setReferrerFeePaidAmount(NumberUtil.round(beforeBalance + payOutAmount));
							}

							double paidRFAmount = NumberUtil.round(payOutAmount);
							if (!NumberUtil.isZero(paidRFAmount)) {
								if (isRefundPayment) {
									ticketsMap.put(ticket, (-1) * paidRFAmount);
								}
								else {
									ticketsMap.put(ticket, paidRFAmount);
								}
							}
							rfTotalAmount += payOutAmount;
							payOutAmount = 0d;
						}

						if (isCalledFromRfPaymentEntry && payOutAmount > 0) {
							fullTicket.putRfExtraPayment(payOutAmount);
						}
						totalAmount += rfTotalAmount;
//						if (NumberUtil.isZero(fullTicket.getDueAmount()) && fullTicket.getReferrerFeePaid() && fullTicket.getLabDoctorFeePaid()) {
//							fullTicket.setClosed(true);
//						}
						TicketDAO.getInstance().saveOrUpdate(fullTicket, session);

						String ticketId = "'" + fullTicket.getId() + "'"; //$NON-NLS-1$ //$NON-NLS-2$
						if (!NumberUtil.isZero(rfTotalAmount)) {
							FeePaymentMapper rfPaymentMapper = new FeePaymentMapper(ticketId, referrer, rfTotalAmount,
									iterator.hasNext() ? rfTotalAmount : tenderAmount, paymentMapper.getCustomPayment(), paymentMapper.getCustomPaymnetRef(),
									user, terminal, currentCashDrawer, ticketTypeNo, isRefundPayment);
							transaction = RfPayTransactionDAO.getInstance().createRfPayTransaction(rfPaymentMapper, batchNo, session);
						}

						String transactionId = transaction == null ? "" : transaction.getId();
						LedgerEntryDAO.getInstance().saveRFPayLedgerEntry(ticketsMap, transactionId, session);
						tenderAmount -= rfTotalAmount;
						if (payOutAmount <= 0) {
							break;
						}
					}
				}
			}
			tx.commit();

			logJournal(transaction, isDoctorPayment, unpaidAmount, referrer);

			return totalAmount;

		} catch (Exception e1) {
			try {
				tx.rollback();
			} catch (Exception x) {
				PosLog.error(getClass(), x);
			}
			throw e1;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	private void logJournal(PosTransaction posTransaction, boolean isDoctorPayment, double unpaidAmount, Customer referrer) {
		if (posTransaction == null) {
			return;
		}

		try {
			String actionName = isDoctorPayment ? ActionHistory.LDF_PAY : ActionHistory.RF_PAY;

			StringBuilder builder = new StringBuilder();
			builder.append("Cash drawer id: " + posTransaction.getCashDrawerId()); //$NON-NLS-1$
			builder.append((isDoctorPayment ? ", LD id: " : ", Referrer id: ") + referrer.getId()); //$NON-NLS-1$ //$NON-NLS-2$

			double paidAmount = posTransaction.getAmount();

			builder.append(String.format(", Amount before: %s, Paid amount: %s, Amount after: %s, Payment type: %s", //$NON-NLS-1$
					NumberUtil.formatNumber(unpaidAmount), NumberUtil.formatNumber(paidAmount), NumberUtil.formatNumber(unpaidAmount - paidAmount),
					posTransaction.getPaymentType().getDisplayString()));

			if (posTransaction.getPaymentType().equals(PaymentType.CUSTOM_PAYMENT)) {
				builder.append(", Name: " + posTransaction.getCustomPaymentName()); //$NON-NLS-1$
				builder.append(", Ref: " + posTransaction.getCustomPaymentRef()); //$NON-NLS-1$
			}
			else if (posTransaction.getPaymentType().equals(PaymentType.BANK_ACCOUNT)) {
				builder.append(", Bank name: " + posTransaction.getBankAccountDisplay()); //$NON-NLS-1$
				builder.append(", Bank ID: " + posTransaction.getBankAccountId()); //$NON-NLS-1$
			}

			String batchNo = posTransaction.getBatchNo();
			if (StringUtils.isNotBlank(batchNo)) {
				builder.append(", BatchNo: " + batchNo);
			}

			String actionMessage = builder.toString();
			actionMessage += ", Transaction id: " + posTransaction.getId(); //$NON-NLS-1$
			ActionHistoryDAO.saveHistory(actionName, actionMessage);
		} catch (Exception e) {
			PosLog.error(getClass(), e);
		}
	}

	public List<PosTransaction> getTransactionsByEntityId(String id) {
		Pagination pagination = new Pagination(0, -1);
		loadTransactionsByEntityId(pagination, id);
		return pagination.getDataList();
	}

	public void loadTransactionsByEntityId(PaginationSupport model, String id) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_ENTITY_ID, id));
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));

			criteria.setProjection(Projections.rowCount());

			Number uniqueResult = (Number) criteria.uniqueResult();
			model.setNumRows(uniqueResult.intValue());
			criteria.setProjection(null);

			criteria.setFirstResult(model.getCurrentRowIndex());
			criteria.setMaxResults(model.getPageSize());

			criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));
			model.setRows(criteria.list());
		}
	}

	public List findPaidTransactionsByTicketId(String ticketId, Class clazz, Session session) {
		//try (Session session = createNewSession()) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.ilike(PosTransaction.PROP_TRANS_TICKET_IDS, ticketId, MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$

		criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));
		return criteria.list();
		//}
	}

	public boolean isPresent(String batchNo) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_BATCH_NO, batchNo));
			criteria.setMaxResults(1);
			List list = criteria.list();
			return list != null && !list.isEmpty();
		}
	}

	public static void createOthersCardSearchCriteria(Criteria criteria) {
		List<String> cardTypeNames = new ArrayList<>();
		cardTypeNames.add(CardTypeEnum.VISA.name());
		cardTypeNames.add(CardTypeEnum.MASTER_CARD.name());
		cardTypeNames.add(CardTypeEnum.MASTER_CARD.name().replaceAll("_", "")); //$NON-NLS-1$ //$NON-NLS-2$
		cardTypeNames.add(CardTypeEnum.MASTER_CARD.name().replaceAll("_", " ")); //$NON-NLS-1$ //$NON-NLS-2$
		cardTypeNames.add(CardTypeEnum.AMERICAN_EXPRESS.name());
		cardTypeNames.add(CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", ""));//$NON-NLS-1$ //$NON-NLS-2$
		cardTypeNames.add(CardTypeEnum.AMERICAN_EXPRESS.name().replaceAll("_", " ")); //$NON-NLS-1$ //$NON-NLS-2$
		cardTypeNames.add(CardTypeEnum.DISCOVER.name());
		for (String prop : cardTypeNames) {
			criteria.add(Restrictions.ne(PosTransaction.PROP_CARD_TYPE, prop).ignoreCase());
		}
	}

	public void doVoidTransaction(PosTransaction posTransaction, User user) {
		try (Session session = createNewSession()) {
			Transaction beginTransaction = session.beginTransaction();
			if (EntityType.match(posTransaction.getEntityType(), EntityType.PURCHASE)) {
				PurchaseOrder purchaseOrder = PurchaseOrderDAO.getInstance().get(posTransaction.getEntityId(), session);
				purchaseOrder.setPaidAmount(NumberUtil.round(purchaseOrder.getPaidAmount() - posTransaction.getAmount()));
				purchaseOrder.setDueAmount(NumberUtil.round(purchaseOrder.getDueAmount() + posTransaction.getAmount()));
				purchaseOrder.putFullPaid(Boolean.FALSE);
				if (purchaseOrder.getStatus() == PurchaseOrder.ORDER_CLOSED) {
					purchaseOrder.setStatus(PurchaseOrder.ORDER_FULLY_RECEIVED);
				}
				PurchaseOrderDAO.getInstance().update(purchaseOrder, session);

				posTransaction.setVoided(true);
				posTransaction.setVoidedByUser(user);
				posTransaction.setVoidDate(StoreDAO.getServerTimestamp());
				saveOrUpdate(posTransaction, session);
			}
			else {
				Ticket ticket = posTransaction.getTicket();
				ticket = TicketDAO.getInstance().loadFullTicket(ticket.getId(), ticket.getOutletId(), session);
				ticket.setPaidAmount(NumberUtil.round(ticket.getPaidAmount() - posTransaction.getAmount()));
				ticket.setDueAmount(NumberUtil.round(ticket.getDueAmount() + posTransaction.getAmount()));
				ticket.setClosed(false);
				ticket.setPaid(false);
				PosTransaction trans = ticket.getTransactions().stream().filter(t -> t.getId().equals(posTransaction.getId())).findFirst().get();
				trans.setVoided(true);
				posTransaction.setVoidedByUser(user);
				trans.setVoidDate(StoreDAO.getServerTimestamp());

				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}
			beginTransaction.commit();
		}
	}

	public List<PosTransaction> findPurchaseTransaction(Outlet outlet, Date from, Date to) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.in("class", PurchaseTransaction.class, PurchaseRefundTransaction.class));

			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			//criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));
			criteria.add(Restrictions.eqOrIsNull(PosTransaction.PROP_VOIDED, Boolean.FALSE));

			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));

			criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public List<PaymentReceivedReportData> findPayments(Date fromDate, Date toDate, User user, Outlet outlet, boolean isShowDueCollectionOnly,
			Customer referrer, AgentTypeEnum agentType) {
		Criteria criteria = null;

		try (Session session = createNewSession()) {

			criteria = session.createCriteria(PosTransaction.class).createAlias(PosTransaction.PROP_TICKET, "t");
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));
			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			criteria.add(Restrictions.isNotNull(PosTransaction.PROP_TICKET));

			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			if (referrer == null && agentType != null) {
				DetachedCriteria detachedCriteria = DetachedCriteria.forClass(Customer.class);

				if (agentType == AgentTypeEnum.INDOOR_DOCTOR) {
					detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, true));
					//addDetachedCriteriaPropertyFilter(detachedCriteria, Customer.PROP_PROPERTIES, "doctor.indor", true);
				}
				else if (agentType == AgentTypeEnum.OUTDOOR_DOCTOR) {
					detachedCriteria.add(Restrictions.eq(Customer.PROP_INDOR_DOCTOR, false));
					//addDetachedCriteriaPropertyFilter(detachedCriteria, Customer.PROP_PROPERTIES, "doctor.indor", false);
				}
				else {
					detachedCriteria.add(Restrictions.eq(Customer.PROP_AGENT_TYPE, agentType.name()));
					//addDetachedCriteriaPropertyFilter(detachedCriteria, Customer.PROP_PROPERTIES, "agent.type", agentType.name());
				}
				detachedCriteria.setProjection(Projections.property(Customer.PROP_ID));

				criteria.add(Property.forName("t." + Ticket.PROP_REFERRER_ID).in(detachedCriteria));
			}
			else if (referrer != null) {
				criteria.add(Restrictions.eq("t." + Ticket.PROP_REFERRER_ID, referrer.getId()));
			}

			criteria.add(Restrictions.eq(PosTransaction.PROP_TRANSACTION_TYPE, TransactionType.CREDIT.name()));

			addMultiUserFilter(user, criteria);

			ProjectionList pList = Projections.projectionList();
			pList.add(Projections.property(PosTransaction.PROP_PAYMENT_TYPE_STRING), PosTransaction.PROP_PAYMENT_TYPE_STRING);
			pList.add(Projections.property(PosTransaction.PROP_CUSTOM_PAYMENT_NAME), PosTransaction.PROP_CUSTOM_PAYMENT_NAME);
			pList.add(Projections.property(PosTransaction.PROP_OUTLET_ID), PosTransaction.PROP_OUTLET_ID);
			pList.add(Projections.property(PosTransaction.PROP_USER_ID), PosTransaction.PROP_USER_ID);
			pList.add(Projections.property(PosTransaction.PROP_ID), PosTransaction.PROP_ID);
			pList.add(Projections.property(PosTransaction.PROP_TICKET), PaymentReceivedReportData.PROP_TICKET);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TIME), PosTransaction.PROP_TRANSACTION_TIME);
			pList.add(Projections.property(PosTransaction.PROP_AMOUNT), PosTransaction.PROP_AMOUNT);
			pList.add(Projections.property(PosTransaction.PROP_TRANSACTION_TYPE), PosTransaction.PROP_TRANSACTION_TYPE);
			pList.add(Projections.property("t." + Ticket.PROP_REFERRER_ID), PaymentReceivedReportData.PROP_REFERRAL_ID);
			criteria.setProjection(pList);
			criteria.setResultTransformer(Transformers.aliasToBean(PaymentReceivedReportData.class));
			criteria.addOrder(Order.asc(PosTransaction.PROP_USER_ID));
			criteria.addOrder(Order.asc(PosTransaction.PROP_PAYMENT_TYPE_STRING));
			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			List<PaymentReceivedReportData> list = criteria.list();
			if (!isShowDueCollectionOnly) {
				return list;
			}
			return list.stream().filter(d -> (d.getTicket().getCreateDate()).before(DateUtil.startOfDay(d.getTransactionTime()))).collect(Collectors.toList());
		}
	}

	public List<PosTransaction> findProjectTransaction(Outlet outlet, Date from, Date to, Project project) {
		try (Session session = getSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			Disjunction disjunction = Restrictions.disjunction();
			disjunction.add(Property.forName("class").eq(CashTransaction.class));
			disjunction.add(Property.forName("class").eq(CreditCardTransaction.class));
			disjunction.add(Property.forName("class").eq(DebitCardTransaction.class));
			disjunction.add(Property.forName("class").eq(CustomPaymentTransaction.class));
			disjunction.add(Property.forName("class").eq(RefundTransaction.class));
			//disjunction.add(Property.forName("class").eq(SalesTransaction.class));
			disjunction.add(Property.forName("class").eq(PurchaseTransaction.class));
			disjunction.add(Property.forName("class").eq(ExpenseTransaction.class));
			criteria.add(disjunction);

			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			criteria.add(Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, from, to));

			if (project != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_PROJECT_ID, project.getId()));
			}

			criteria.addOrder(Order.asc(PosTransaction.PROP_TRANSACTION_TIME));

			return criteria.list();
		}
	}

	public void doConfirmTransaction(PosTransaction posTransaction) {
		try (Session session = createNewSession()) {
			Transaction tr = session.beginTransaction();

			String linkedBankAccountId = posTransaction.getLinkedBankAccountId();
			if (StringUtils.isNotBlank(linkedBankAccountId) && !posTransaction.isConfirmPayment()) {
				BankAccount bankAccount = BankAccountDAO.getInstance().get(posTransaction.getLinkedBankAccountId());
				if (bankAccount != null) {

					if (Arrays.asList(TransactionType.DEBIT.name(), TransactionType.OUT.name()).contains(posTransaction.getTransactionType())
							&& bankAccount.getBalance() < posTransaction.getAmount()) {
						throw new PosException("Insufficient balance");
					}

					List customPaymentRefJsonList = null;
					String note = "";
					if (posTransaction instanceof PurchaseTransaction) {
						note = "Paid to vendor";
					}
					else if (posTransaction instanceof PurchaseRefundTransaction) {
						note = "Refund from vendor";
					}
					else {
						customPaymentRefJsonList = posTransaction.getCustomPaymentRefJsonList();
					}

					BalanceUpdateTransactionDAO.getInstance().saveBankAccountTransaction(bankAccount, posTransaction.getPaymentType(), posTransaction,
							posTransaction.getAmount(), note, session,
							(customPaymentRefJsonList == null || customPaymentRefJsonList.isEmpty() ? null : customPaymentRefJsonList.toString()));
				}
			}

			posTransaction.setConfirmPayment(true);
			saveOrUpdate(posTransaction, session);

			tr.commit();
		}
	}

	public List<PosTransaction> loadAllExpenseAndPurchaseTransaction(Date fromDate, Date toDate, String recipientId, String reasonId, String projectId,
			String subCategoryId, Outlet selectedOutlet, String accountNo, String performByUserId) {
		return loadAllExpenseAndPurchaseTransaction(DateTypeOption.CREATE_DATE, fromDate, toDate, recipientId, reasonId, projectId, subCategoryId,
				selectedOutlet, accountNo, performByUserId, "");

	}

	public List<PosTransaction> loadAllExpenseAndPurchaseTransaction(Date fromDate, Date toDate, String recipientId, String reasonId, String projectId,
			String subCategoryId, Outlet selectedOutlet, String accountNo, String performByUserId, String coaCode, boolean isServiceOrder) {
		return loadAllExpenseAndPurchaseTransaction(DateTypeOption.CREATE_DATE, fromDate, toDate, recipientId, reasonId, projectId, subCategoryId,
				selectedOutlet, accountNo, performByUserId, coaCode, isServiceOrder);
	}

	public List<PosTransaction> loadAllExpenseAndPurchaseTransaction(DateTypeOption dateTypeOption, Date fromDate, Date toDate, String recipientId,
			String reasonId, String projectId, String subCategoryId, Outlet selectedOutlet, String accountNo, String performByUserId, String coaCode) {
		return loadAllExpenseAndPurchaseTransaction(dateTypeOption, fromDate, toDate, recipientId, reasonId, projectId, subCategoryId, selectedOutlet,
				accountNo, performByUserId, coaCode, false);
	}

	public List<PosTransaction> loadAllExpenseAndPurchaseTransaction(DateTypeOption dateTypeOption, Date fromDate, Date toDate, String recipientId,
			String reasonId, String projectId, String subCategoryId, Outlet selectedOutlet, String accountNo, String performByUserId, String coaCode,
			boolean isServiceOrder) {
		return loadAllExpenseAndPurchaseTransaction(dateTypeOption, fromDate, toDate, recipientId, reasonId, projectId, subCategoryId, selectedOutlet,
				accountNo, performByUserId, coaCode, isServiceOrder, false);
	}

	public List<PosTransaction> loadAllExpenseAndPurchaseTransaction(DateTypeOption dateTypeOption, Date fromDate, Date toDate, String recipientId,
			String reasonId, String projectId, String subCategoryId, Outlet selectedOutlet, String accountNo, String performByUserId, String coaCode,
			boolean isServiceOrder, boolean isShowExpenseReportView) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);

			criteria.add(Restrictions.or(Restrictions.eq("class", ExpenseTransaction.class), Restrictions.eq("class", PurchaseTransaction.class),
					Restrictions.eq("class", PurchaseRefundTransaction.class)));

			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));

			if (selectedOutlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, selectedOutlet.getId()));
			}

			if (StringUtils.isNotBlank(accountNo)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_ACCOUNT_MANAGER_ID, accountNo));
			}

			if (isShowExpenseReportView && fromDate != null && toDate != null) {
				Criterion eventTimeNotNull = Restrictions.isNotNull(PosTransaction.PROP_EVENT_TIME);
				Criterion eventTimeCriterion = Restrictions.between(PosTransaction.PROP_EVENT_TIME, fromDate, toDate);
				Criterion createDateCriterion = Restrictions.between(PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate);

				criteria.add(Restrictions.or(Restrictions.and(eventTimeNotNull, eventTimeCriterion),
						Restrictions.and(Restrictions.isNull(PosTransaction.PROP_EVENT_TIME), createDateCriterion)));
			}
			else if (fromDate != null && toDate != null) {
				criteria.add(
						Restrictions.between(dateTypeOption != null ? dateTypeOption.getPropString() : PosTransaction.PROP_TRANSACTION_TIME, fromDate, toDate));
			}

			if (StringUtils.isNotBlank(recipientId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_RECEPIENT_ID, recipientId));
			}

			if (StringUtils.isNotBlank(reasonId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_REASON_ID, reasonId));
			}

			if (StringUtils.isNotBlank(projectId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_PROJECT_ID, projectId));
			}

			if (StringUtils.isNotBlank(subCategoryId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_SUB_REASON_ID, subCategoryId));
			}

			if (StringUtils.isNotBlank(performByUserId)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_USER_ID, performByUserId));
			}
			if (isServiceOrder) {
				DetachedCriteria dCriteria = DetachedCriteria.forClass(PurchaseOrder.class);
				dCriteria.add(Restrictions.eq(PurchaseOrder.PROP_PURCHASE_ORDER_TYPE, PurchaseOrderType.SERVICE_ORDER.name()));
				dCriteria.setProjection(Projections.property(PurchaseOrder.PROP_ID));
				criteria.add(Subqueries.propertyIn(PosTransaction.PROP_ENTITY_ID, dCriteria));
			}

			if (StringUtils.isNotBlank(coaCode)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_COA_CODE, coaCode));
			}

			return criteria.list();

		}
	}

	public List<PosTransaction> getPosTransactionsByIds(List<String> ids) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			criteria.add(Restrictions.in(PosTransaction.PROP_ID, ids));
			return criteria.list();
		}
	}

	public List<PosTransaction> findAdvanceTransactions(String ticketId) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(AdvanceTransaction.class);
			criteria.add(Restrictions.ilike(AdvanceTransaction.PROP_EXTRA_PROPERTIES, "\"" + "ticketId" + "\":\"" + ticketId + "\"", MatchMode.ANYWHERE)); //$NON-NLS-1$ //$NON-NLS-2$
			criteria.add(Restrictions.isNull(AdvanceTransaction.PROP_TICKET));
			return criteria.list();
		}
	}

	public List<GrandTotalCollectionReportData> findUserWiseCollectionSalesTransactions(Outlet outlet, Date fromDate, Date toDate) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(PosTransaction.class);
			addDeletedFilter(criteria);

			criteria.add(Restrictions.ge(PosTransaction.PROP_TRANSACTION_TIME, fromDate));
			criteria.add(Restrictions.lt(PosTransaction.PROP_TRANSACTION_TIME, toDate));

			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, Boolean.FALSE));

			if (outlet != null) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_OUTLET_ID, outlet.getId()));
			}

			//@formatter:off
			Projection projection = Projections.sqlGroupProjection(
			    "user_id as userId, " + //$NON-NLS-1$
			    
			    "SUM(CASE WHEN transaction_reason in('SALES', 'SALES_ADVANCE') THEN amount ELSE 0 END) - " + //$NON-NLS-1$
			    "SUM(CASE WHEN transaction_reason in('SALES_REFUND', 'SALES_ADVANCE_REFUND') THEN amount ELSE 0 END) as netAmount, " + //$NON-NLS-1$
			    
			    "SUM(CASE WHEN transaction_reason in('SALES', 'SALES_ADVANCE') AND payment_type = 'CASH' THEN amount ELSE 0 END) - " + //$NON-NLS-1$
			    "SUM(CASE WHEN transaction_reason in('SALES_REFUND', 'SALES_ADVANCE_REFUND') AND payment_type = 'CASH' THEN amount ELSE 0 END) as cashAmount, " + //$NON-NLS-1$
			    
			    "SUM(CASE WHEN transaction_reason in('SALES', 'SALES_ADVANCE') AND payment_type != 'CASH' THEN amount ELSE 0 END) - " + //$NON-NLS-1$
			    "SUM(CASE WHEN transaction_reason in('SALES_REFUND', 'SALES_ADVANCE_REFUND') AND payment_type != 'CASH' THEN amount ELSE 0 END) as othersAmount", //$NON-NLS-1$
            
			    "user_id", //$NON-NLS-1$
			    new String[] { "userId", "netAmount", "cashAmount", "othersAmount" }, //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			    new Type[] { StringType.INSTANCE, DoubleType.INSTANCE, DoubleType.INSTANCE, DoubleType.INSTANCE }
			);
			//@formatter:on
			criteria.setProjection(projection);

			List<GrandTotalCollectionReportData> datas = new ArrayList<>();
			List<Object[]> results = criteria.list();

			for (Object[] row : results) {
				String userId = (String) row[0];
				Double netAmount = (Double) row[1];
				Double cashAmount = (Double) row[2];
				Double othersAmount = (Double) row[3];
				if (StringUtils.isBlank(userId)) {
					continue;
				}
				GrandTotalCollectionReportData data = new GrandTotalCollectionReportData();
				data.setParticularId(userId);
				data.setCashAmount(cashAmount);
				data.setOthersAmount(othersAmount);
				data.setTotalAmount(netAmount);
				datas.add(data);
			}

			return datas;
		}
	}


	public void findAdvanceTransactions(PaginationSupport dataModel, String customerID) {
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(AdvanceTransaction.class);
			criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
			if (StringUtils.isNotBlank(customerID)) {
				criteria.add(Restrictions.eq(PosTransaction.PROP_CUSTOMER_ID, customerID));
			}
			criteria.add(Restrictions.isNull(AdvanceTransaction.PROP_TICKET));

			dataModel.setNumRows(rowCount(criteria));

			criteria.setFirstResult(dataModel.getCurrentRowIndex());
			criteria.setMaxResults(dataModel.getPageSize());
			criteria.addOrder(Order.desc(PosTransaction.PROP_TRANSACTION_TIME));
			dataModel.setRows(criteria.list());
		}
	}

	public void voidAdvanceTransaction(PosTransaction advanceTransaction, User user) {
		voidAdvanceTransaction(advanceTransaction, null, user);
	}

	public void voidAdvanceTransaction(PosTransaction advanceTransaction, Ticket ticket, User user) {
		try (Session session = createNewSession()) {
			Transaction tx = session.beginTransaction();

			String customerId = advanceTransaction.getCustomerId();
			if (StringUtils.isBlank(customerId)) {
				throw new PosException("Customer is required.");
			}

			double advanceAmount = calculateTotalAmountForCustomer(AdvanceTransaction.class, customerId, session);
			double adjustmentAmount = calculateTotalAmountForCustomer(AdvanceAdjustmentTransaction.class, customerId, session);

			Double advanceTransactionAmount = advanceTransaction.getAmount();
			if ((advanceAmount - advanceTransactionAmount - (adjustmentAmount)) < 0) {
				throw new PosException("Cannot void this transaction. Adjustments would exceed remaining advances.");
			}

			// Update customer balance and log transaction
			Customer customer = CustomerDAO.getInstance().get(advanceTransaction.getCustomerId(), session);
			double previousBalance = customer.getBalance();
			double newBalance = NumberUtil.round(previousBalance - advanceTransactionAmount);
			if (newBalance < 0) {
				throw new PosException("Cannot void this transaction. Customer balance cannot be negative.");
			}

			//Mark Void main advance transaction
			advanceTransaction.setVoided(true);
			advanceTransaction.setVoidedByUser(user);
			advanceTransaction.setVoidDate(StoreDAO.getServerTimestamp());
			saveOrUpdate(advanceTransaction, session);

			//Adjust ticket advance amount
			if (ticket == null) {
				//String ticketId = advanceTransaction.getProperty("ticketId"); //$NON-NLS-1$
				String ticketId = advanceTransaction.getEntityId();
				if (StringUtils.isNotBlank(ticketId)) {
					ticket = TicketDAO.getInstance().get(ticketId, advanceTransaction.getOutletId(), session);
				}
			}

			if (ticket != null) {
				ticket.setAdvanceAmount(ticket.getAdvanceAmount() - advanceTransactionAmount);
				TicketDAO.getInstance().saveOrUpdate(ticket, session);
			}

			BalanceUpdateTransactionDAO.getInstance().saveBalanceUpdateTrans(BalanceType.CUSTOMER, ticket, advanceTransaction, TransactionType.DEBIT,
					advanceTransaction.getCustomerId(), advanceTransactionAmount, previousBalance, TransactionSubType.VOIDED_BALANCE, session);

			customer.setBalance(newBalance);
			CustomerDAO.getInstance().update(customer, session);

			//Adjust linked bank amount
			String linkedBankAccountId = advanceTransaction.getLinkedBankAccountId();
			if (StringUtils.isNotBlank(linkedBankAccountId) /*&& advanceTransaction.isConfirmPayment()*/) {
				BankAccount bankAccount = (BankAccount) DataProvider.get().getObjectOf(BankAccount.class, linkedBankAccountId);
				List customPaymentRefJsonList = advanceTransaction.getCustomPaymentRefJsonList();

				BalanceUpdateTransactionDAO.getInstance().saveBankAccountTransaction(bankAccount, advanceTransaction.getPaymentType(), advanceTransaction,
						advanceTransaction.getAmount(), "Adjustment for voided advance payment", session,
						customPaymentRefJsonList.isEmpty() ? null : customPaymentRefJsonList.toString());
			}

			LedgerEntryDAO.getInstance().handleCustomerDepositVoidedLedgerEntry(advanceTransaction, session);

			tx.commit();
		}
	}

	public double calculateTotalAmountForCustomer(Class clazz, String customerId) {
		try (Session session = createNewSession()) {
			return calculateTotalAmountForCustomer(clazz, customerId, session);
		}
	}

	public double calculateTotalAmountForCustomer(Class clazz, String customerId, Session session) {
		Criteria criteria = session.createCriteria(clazz);
		criteria.add(Restrictions.eq(PosTransaction.PROP_VOIDED, false));
		criteria.add(Restrictions.eq(PosTransaction.PROP_CUSTOMER_ID, customerId));
		criteria.setProjection(Projections.sum(PosTransaction.PROP_AMOUNT));
		return POSUtil.getDoubleAmount(criteria.uniqueResult());
	}

}