package com.floreantpos.model.dao;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.beanutils.PropertyUtils;
import org.hibernate.Criteria;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.criterion.ProjectionList;
import org.hibernate.criterion.Projections;
import org.hibernate.criterion.Restrictions;
import org.hibernate.transform.Transformers;
import org.hibernate.type.DoubleType;
import org.hibernate.type.Type;

import com.floreantpos.PosException;
import com.floreantpos.PosLog;
import com.floreantpos.model.InventoryClosingBalance;
import com.floreantpos.model.InventoryLocation;
import com.floreantpos.model.InventoryTransaction;
import com.floreantpos.model.InventoryTransactionType;
import com.floreantpos.model.MenuItem;
import com.floreantpos.model.util.DateUtil;
import com.floreantpos.util.NumberUtil;
import com.orocube.rest.service.server.BaseDataServiceDao;

public class InventoryClosingBalanceDAO extends BaseInventoryClosingBalanceDAO {

	/**
	 * Default constructor.  Can be used in place of getInstance()
	 */
	public InventoryClosingBalanceDAO() {
	}

	@Override
	protected Serializable save(Object obj, Session s) {
		updateTime(obj);
		return super.save(obj, s);
	}

	@Override
	protected void update(Object obj, Session s) {
		updateTime(obj);
		super.update(obj, s);
	}

	@Override
	protected void saveOrUpdate(Object obj, Session s) {
		updateTime(obj);
		super.saveOrUpdate(obj, s);
	}

	public void closeStock(Date closingMonth, Date closingDate) throws Exception {
		Session session = null;
		Transaction tx = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();
			List<InventoryClosingBalance> dataList = closeInventoryStock(session, DateUtil.endOfMonth(closingMonth), closingDate, true);
			if (dataList == null || dataList.isEmpty())
				throw new PosException("No data found");

			closingDate = DateUtil.startOfDay(closingDate);
			for (InventoryClosingBalance data : dataList) {
				data.setClosingDate(closingDate);
				session.saveOrUpdate(data);
			}
			tx.commit();
		} catch (Exception e) {
			if (tx != null)
				tx.rollback();
			throw e;
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public List<InventoryClosingBalance> closeInventoryStock(Session session, Date closingUpToDate, Date closingDate, boolean includeAllItems) {
		Criteria criteria = null;
		Map<String, InventoryClosingBalance> itemMap = getInventoryItems(session);
		Map<String, InventoryClosingBalance> dataMap = new HashMap<String, InventoryClosingBalance>();

		Date existingClosingDate = getExistingClosingDate(closingUpToDate, closingDate);
		if (existingClosingDate != null) {
			criteria = session.createCriteria(InventoryClosingBalance.class);
			if (existingClosingDate != null) {
				existingClosingDate = DateUtil.startOfDay(existingClosingDate);
				criteria.add(Restrictions.ge(InventoryClosingBalance.PROP_CLOSING_DATE, existingClosingDate));
				criteria.add(Restrictions.lt(InventoryClosingBalance.PROP_CLOSING_DATE, DateUtil.endOfDay(existingClosingDate)));
			}
			List dataList = criteria.list();
			if (dataList != null && dataList.size() > 0) {
				for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
					InventoryClosingBalance inventoryStockData = (InventoryClosingBalance) iterator.next();
					inventoryStockData.setBalance(0.0);
					inventoryStockData.setUnitCost(0.0);
					inventoryStockData.setTotal(0.0);
					dataMap.put(generateKey(inventoryStockData), inventoryStockData);
				}
			}
		}
		Date lastClosingDate = getLastClosingDate(closingUpToDate);
		if (lastClosingDate != null) {
			lastClosingDate = DateUtil.startOfDay(lastClosingDate);
			criteria = session.createCriteria(InventoryClosingBalance.class);
			if (lastClosingDate != null) {
				criteria.add(Restrictions.ge(InventoryClosingBalance.PROP_CLOSING_DATE, lastClosingDate));
				criteria.add(Restrictions.lt(InventoryClosingBalance.PROP_CLOSING_DATE, DateUtil.endOfDay(lastClosingDate)));
			}
			List dataList = criteria.list();
			if (dataList != null && dataList.size() > 0) {
				for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
					InventoryClosingBalance inventoryStockData = (InventoryClosingBalance) iterator.next();
					String key = generateKey(inventoryStockData);
					InventoryClosingBalance stockData = dataMap.get(key);
					if (stockData != null) {
						stockData.setBalance(stockData.getBalance() + inventoryStockData.getBalance());
					}
					else {
						InventoryClosingBalance nextClosingBalance = new InventoryClosingBalance();
						nextClosingBalance.setMenuItemId(inventoryStockData.getMenuItemId());
						nextClosingBalance.setLocationId(inventoryStockData.getLocationId());
						nextClosingBalance.setUnit(inventoryStockData.getUnit());
						nextClosingBalance.setBalance(inventoryStockData.getBalance());
						nextClosingBalance.setUnitCost(inventoryStockData.getUnitCost());
						dataMap.put(key, nextClosingBalance);
					}
				}
			}
		}
		populateInventoryTransactions(session, dataMap, closingDate, lastClosingDate);
		List stockDataList = new ArrayList(dataMap.values());

		for (Iterator iterator = stockDataList.iterator(); iterator.hasNext();) {
			InventoryClosingBalance data = (InventoryClosingBalance) iterator.next();
			if (data.getBalance() <= 0) {
				data.setUnitCost(0.0);
				data.setTotal(0.0);
			}
			else {
				data.setTotal(data.getUnitCost() * data.getBalance());
			}
			if (!includeAllItems && data.getBalance() == 0)
				iterator.remove();
			InventoryClosingBalance itemData = itemMap.get(data.getMenuItemId());
			if (itemData == null)
				continue;
			if (includeAllItems && itemData != null)
				itemMap.remove(data.getMenuItemId());
		}
		if (includeAllItems && itemMap.values().size() > 0)
			stockDataList.addAll(itemMap.values());
		return stockDataList;
	}

	private String generateKey(InventoryClosingBalance inventoryStockData) {
		return inventoryStockData.getMenuItemId() + inventoryStockData.getUnit() + inventoryStockData.getLocationId();
	}

	/**
	 * 
	 * @param session
	 * @param dataMap
	 * @param closingDate date for which closing balance will be generated.
	 * @param lastClosingDate last date for which closing balance was generated.
	 */
	private void populateInventoryTransactions(Session session, Map<String, InventoryClosingBalance> dataMap, Date closingDate, Date lastClosingDate) {
		Criteria criteria = session.createCriteria(InventoryLocation.class);
		criteria.setProjection(Projections.property(InventoryLocation.PROP_ID));
		List locationIds = criteria.list();
		if (locationIds != null) {
			for (Iterator locIterator = locationIds.iterator(); locIterator.hasNext();) {
				String locationId = (String) locIterator.next();

				criteria = session.createCriteria(InventoryTransaction.class);
				criteria.createAlias(InventoryTransaction.PROP_MENU_ITEM, "item");

				ProjectionList projections = Projections.projectionList();
				projections.add(Projections.property("item.id"), InventoryClosingBalance.PROP_MENU_ITEM_ID);
				projections.add(Projections.property(InventoryTransaction.PROP_UNIT), InventoryClosingBalance.PROP_UNIT);
				projections.add(Projections.sqlProjection("sum(quantity*tran_type) AS " + InventoryClosingBalance.PROP_BALANCE,
						new String[] { InventoryClosingBalance.PROP_BALANCE }, new Type[] { new DoubleType() }));
				projections.add(Projections.groupProperty(InventoryTransaction.PROP_UNIT));
				projections.add(Projections.groupProperty("item.id"));
				criteria.setProjection(projections);

				if (lastClosingDate != null)
					criteria.add(Restrictions.gt(InventoryTransaction.PROP_TRANSACTION_DATE, lastClosingDate));

				criteria.add(Restrictions.lt(InventoryTransaction.PROP_TRANSACTION_DATE, closingDate));
				if (locationId != null)
					criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_FROM_LOCATION_ID, locationId),
							Restrictions.eq(InventoryTransaction.PROP_TO_LOCATION_ID, locationId)));
				criteria.setResultTransformer(Transformers.aliasToBean(InventoryClosingBalance.class));
				List transactions = criteria.list();

				projections = Projections.projectionList();
				projections.add(Projections.property("item.id"), InventoryClosingBalance.PROP_MENU_ITEM_ID); //$NON-NLS-1$
				projections.add(Projections.property(InventoryTransaction.PROP_UNIT), InventoryClosingBalance.PROP_UNIT);
				projections.add(Projections.groupProperty(InventoryTransaction.PROP_UNIT));
				projections.add(Projections.groupProperty("item.id")); //$NON-NLS-1$
				projections.add(Projections.sqlProjection("sum(total)/sum(quantity) AS " + InventoryClosingBalance.PROP_UNIT_COST, //$NON-NLS-1$
						new String[] { InventoryClosingBalance.PROP_UNIT_COST }, new Type[] { new DoubleType() }));
				criteria.add(Restrictions.eq(InventoryTransaction.PROP_TYPE, InventoryTransactionType.IN.getType()));
				criteria.add(Restrictions.or(Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_NEW_STOCK),
						Restrictions.eq(InventoryTransaction.PROP_REASON, InventoryTransaction.REASON_PURCHASE)));
				criteria.setProjection(projections);
				criteria.setResultTransformer(Transformers.aliasToBean(InventoryClosingBalance.class));
				List avgCostTransactions = criteria.list();

				List<InventoryTransaction> dataList = new ArrayList<>();
				if (transactions != null && transactions.size() > 0) {
					dataList.addAll(transactions);
				}
				if (avgCostTransactions != null && avgCostTransactions.size() > 0) {
					dataList.addAll(avgCostTransactions);
				}
				if (dataList != null && dataList.size() > 0) {
					for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
						InventoryClosingBalance tData = (InventoryClosingBalance) iterator.next();
						tData.setLocationId(locationId);
						String key = generateKey(tData);
						InventoryClosingBalance stockData = dataMap.get(key);
						if (stockData != null) {
							stockData.setBalance(stockData.getBalance() + tData.getBalance());
							stockData.setUnitCost(NumberUtil.getCurrencyFormatWithoutSymbol(stockData.getUnitCost() + tData.getUnitCost()));
						}
						else {
							dataMap.put(key, tData);
						}
					}
				}
			}
		}
	}

	private Map<String, InventoryClosingBalance> getInventoryItems(Session session) {
		Criteria criteria = session.createCriteria(MenuItem.class);
		ProjectionList list = Projections.projectionList();
		list.add(Projections.property(MenuItem.PROP_ID), InventoryClosingBalance.PROP_MENU_ITEM_ID);
		list.add(Projections.property(MenuItem.PROP_UNIT_NAME), InventoryClosingBalance.PROP_UNIT);
		criteria.setProjection(list);
		criteria.add(Restrictions.eqOrIsNull(MenuItem.PROP_INVENTORY_ITEM, Boolean.TRUE));
		criteria.setResultTransformer(Transformers.aliasToBean(InventoryClosingBalance.class));
		List dataList = criteria.list();
		Map<String, InventoryClosingBalance> itemMap = new HashMap<String, InventoryClosingBalance>();
		if (dataList != null && dataList.size() > 0) {
			for (Iterator iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryClosingBalance inventoryStockData = (InventoryClosingBalance) iterator.next();
				itemMap.put(inventoryStockData.getMenuItemId(), inventoryStockData);
			}
		}
		return itemMap;
	}

	public Date getExistingClosingDate(Date previousMonthEndDate, Date closingDate) {
		Criteria criteria = getSession().createCriteria(getReferenceClass());
		Date from = DateUtil.startOfDay(previousMonthEndDate);
		Date to = DateUtil.endOfDay(closingDate);
		criteria.setProjection(Projections.property(InventoryClosingBalance.PROP_CLOSING_DATE));
		criteria.add(Restrictions.gt(InventoryClosingBalance.PROP_CLOSING_DATE, from));
		criteria.add(Restrictions.le(InventoryClosingBalance.PROP_CLOSING_DATE, to));
		criteria.setMaxResults(1);
		return (Date) criteria.uniqueResult();
	}

	public Date getLastClosingDate(Date closingDate) {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.setProjection(Projections.distinct(Projections.max(InventoryClosingBalance.PROP_CLOSING_DATE)));
			criteria.add(Restrictions.lt(InventoryClosingBalance.PROP_CLOSING_DATE, closingDate));
			return (Date) criteria.uniqueResult();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public Date getFirstInventoryTransactionDate() {
		Session session = null;
		try {
			session = createNewSession();
			Criteria criteria = session.createCriteria(InventoryTransaction.class);
			criteria.setProjection(Projections.distinct(Projections.min(InventoryTransaction.PROP_TRANSACTION_DATE)));
			return (Date) criteria.uniqueResult();
		} finally {
			if (session != null) {
				session.close();
			}
		}
	}

	public void saveOrUpdateInventoryClosingBalances(List<InventoryClosingBalance> dataList, boolean updateLastUpdateTime, boolean updateSyncTime)
			throws Exception {

		if (dataList == null)
			return;

		Transaction tx = null;
		Session session = null;
		try {
			session = createNewSession();
			tx = session.beginTransaction();

			for (Iterator<InventoryClosingBalance> iterator = dataList.iterator(); iterator.hasNext();) {
				InventoryClosingBalance item = (InventoryClosingBalance) iterator.next();
				InventoryClosingBalance existingItem = get(item.getId());
				if (existingItem != null) {
					if (!BaseDataServiceDao.get().shouldSave(item.getLastUpdateTime(), existingItem.getLastUpdateTime())) {
						PosLog.info(getClass(), item.getId() + " already updated"); //$NON-NLS-1$
						continue;
					}
					long version = existingItem.getVersion();
					PropertyUtils.copyProperties(existingItem, item);
					existingItem.setVersion(version);
					existingItem.setUpdateLastUpdateTime(updateLastUpdateTime);
					existingItem.setUpdateSyncTime(updateSyncTime);
					update(existingItem, session);
				}
				else {
					item.setUpdateLastUpdateTime(updateLastUpdateTime);
					item.setUpdateSyncTime(updateSyncTime);
					save(item, session);
				}
			}
			tx.commit();
		} catch (Exception e) {
			tx.rollback();
			throw e;
		} finally {
			closeSession(session);
		}
	}

	public List<InventoryClosingBalance> getClosingBalances(List<String> menuItemIds, Date lastClosingDate) {
		if (menuItemIds == null || menuItemIds.isEmpty()) {
			return null;
		}
		try (Session session = createNewSession()) {
			Criteria criteria = session.createCriteria(getReferenceClass());
			criteria.add(Restrictions.in(InventoryClosingBalance.PROP_MENU_ITEM_ID, menuItemIds));
			criteria.add(Restrictions.ge(InventoryClosingBalance.PROP_CLOSING_DATE, lastClosingDate));
			criteria.add(Restrictions.lt(InventoryClosingBalance.PROP_CLOSING_DATE, DateUtil.endOfDay(lastClosingDate)));
			return criteria.list();
		}
	}
}